/*____________________________________________________________________________
	Copyright (C) 1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	Note: In this file, the term "block" refers to an HFS or HFS+ allocation
	block. The term "sector" is used to refer to the 512 byte disk block
	construct used by Mac disk drivers. The exception is the DiskExtent data
	structure which uses the term "block" but really means sectors
	
	$Id: pgpMacVolumeWiping.c,v 1.13.12.1 1998/11/12 03:19:56 heller Exp $
____________________________________________________________________________*/

#include <string.h>

#include "HFSVolumes.h"

#include "MacBTree.h"
#include "MacDriverUtils.h"
#include "MacFiles.h"
#include "MacStrings.h"
#include "pgpBitUtils.h"
#include "pgpMacMemory.h"
#include "pgpMacVolumeWiping.h"
#include "pgpMem.h"

#define	kResourceForkType	0xFF
#define	kDataForkType		0
#define	kDiskSectorSize		512L

typedef PGPUInt8	ForkType;

#define kMemoryAllocationFlags	( kMacMemory_PreferTempMem |		\
									kMacMemory_UseApplicationHeap )
									
pgpFixBeforeShip( "Handle slow volumes" )
pgpFixBeforeShip( "Compute total blocks in extent during scan" )
pgpFixBeforeShip( "Sort problem fork info and implement binary search" )
pgpFixBeforeShip( "Handle Problem files" )
pgpFixBeforeShip( "Flush after each pass on mounted volumes" )

#define	kMaxIOBufferSize		(2L * 1024L * 1024L)
#define	kMinIOBufferSize		(64L * 1024L)

#define MacValidateParam( expr )	\
	if ( ! (expr ) )	\
	{\
		pgpAssert( expr );\
		return( paramErr );\
	}

#define MacValidatePtr( ptr )	\
			MacValidateParam( (ptr) != NULL )

#if PGP_DEBUG
	#define ASSERT(x)	AssertProc(x)

	static void
AssertProc(const char *msg)
{
	/*
	** A separate proc is used here because #defining ASSERT to be
	** pgpAssert would consume lots of stack space
	*/
	
	pgpDebugMsg( msg );
}
#else
	#define ASSERT(x)	{}
#endif

#define	Require(condition)											\
																	\
	if( ! (condition) ) {											\
		ASSERT( "Catalog validation failure: " #condition );		\
		return( kWipingError_InvalidVolumeDataStructure );			\
	}	

#pragma options align=mac68k

typedef struct ExtentRecord
{
	union
	{
		HFSExtentRecord		hfs;
		HFSPlusExtentRecord	hfsPlus;
	};

} ExtentRecord;

typedef union HFSCatalogRecord
{
	SInt16				type;
	
	HFSCatalogFile		file;
	HFSCatalogFolder	folder;
	HFSCatalogThread	thread;

} HFSCatalogRecord;

typedef union HFSPlusCatalogRecord
{
	SInt16					type;
	
	HFSPlusCatalogFile		file;
	HFSPlusCatalogFolder	folder;
	HFSPlusCatalogThread	thread;

} HFSPlusCatalogRecord;

typedef struct HFSCatalogFileIndexRecord
{
	HFSCatalogKey	key;
	UInt32			nodeIndex;
	
} HFSCatalogFileIndexRecord;

typedef struct HFSPlusCatalogFileIndexRecord
{
	HFSPlusCatalogKey	key;
	UInt32				nodeIndex;
	
} HFSPlusCatalogFileIndexRecord;

typedef struct HFSExtentsFileIndexRecord
{
	HFSExtentKey	key;
	UInt32			nodeIndex;

} HFSExtentsFileIndexRecord;

typedef struct BTreeFileInfo
{
	short		fileRefNum;				/* 0 = unmounted */
	PGPUInt32	fileSize;				/* in bytes */
	PGPUInt32	numDiskExtents;
	DiskExtent	*diskExtents;
	PGPUInt32	nodeSize;
	PGPUInt32	numNodes;
	PGPUInt32	numFreeNodes;
	PGPUInt32	treeDepth;
	PGPByte		*nodeBitmap;
	PGPByte		*nodeBuffer;			/* nodeSize bytes */
	
} BTreeFileInfo;

typedef struct ForkExtent
{
	struct ForkExtent	*next;
	
	PGPUInt32		firstExtentBlock;
	ExtentRecord	extents;

} ForkExtent;

typedef struct ProblemForkInfo
{
	PGPUInt32	fileID;
	ForkType	forkType;
	PGPUInt32	forkLogicalLength;
	PGPUInt32	forkPhysicalLength;
	ForkExtent	*extentList;
	
} ProblemForkInfo;

typedef struct WipeRange
{
	PGPUInt32	startSector;
	PGPUInt16	firstSectorOffset;
	PGPUInt16	numSectors;
	
} WipeRange;

typedef struct WipeControlBlock
{
	PGPInt16			vRefNum;				/* 0 = unmounted */
	PGPInt16			driveNumber;
	PGPInt16			driverRefNum;
	PGPUInt16			signature;
	
	PGPUInt32			numBlocks;
	PGPUInt32			numFreeBlocks;
	PGPUInt32			blockSize;				/* Allocation block size */
	PGPUInt32			numSectorsPerBlock;
	PGPUInt32			numVolumeSectors;
	PGPUInt32			firstSectorBias;
	
	PGPUInt32			numSectorsToWipe;
	PGPUInt32			numSectorsWiped;
	PGPUInt32			passIndex;
	PGPUInt32			totalPasses;
	
	PGPByte				*volumeBitmap;
	
	PGPByte				*sectorBuffer;
	PGPWipingPattern	*patternBuffer;		/* Holds current wipe pattern */
	PGPByte				*ioBuffer;
	PGPUInt32			ioBufferSize;
	
	BTreeFileInfo		catalogFileInfo;
	BTreeFileInfo		extentsFileInfo;
	
	PGPUInt32			numProblemForks;
	ProblemForkInfo		*problemForkList;
	
	PGPUInt32			numWipeRanges;
	PGPUInt32			numAllocatedWipeRanges;
	WipeRange			*wipeRangeList;
	
	PGPVolumeWipingEventHandler	eventHandler;
	PGPUserValue				eventUserValue;
	
	union
	{
		struct
		{
			HFSMasterDirectoryBlock	mdb;
		
		} hfs;
		
		struct
		{
			HFSPlusVolumeHeader 	vh;
		
		} hfsPlus;
	};

#if PGP_DEBUG
	PGPUInt32			readCount;
	PGPUInt32			readSectorCount;
	PGPUInt32			writeCount;
	PGPUInt32			writeSectorCount;
#endif
		
} WipeControlBlock;

typedef union
{
	ParamBlockRec	pb;
	XIOParam		xpb;
	
} IOPB;

typedef void	(*WipeBTreeNodeProcPtr)(WipeControlBlock *wcb,
					NodeDescriptor *nodeDesc, PGPUInt32 nodeSize );

#pragma options align=reset


	OSStatus
pgpCanWipeVolumeFreeSpace(short volRefNum)
{
	OSStatus		err = noErr;
	XVolumeParam	xpb;
	Str32			volumeName;
	
	if( volRefNum >= 0)
	{
		pgpDebugMsg( "pgpWipeVolumeFreeSpace(): Bad vRefNum" );
		return( paramErr );
	}

	if( FileSharingIsActive() )
	{
		pgpDebugMsg( "pgpWipeVolumeFreeSpace(): Filesharing active" );
		return( vLckdErr );
	}
	
	pgpClearMemory( &xpb, sizeof( xpb ) );
	
	xpb.ioVRefNum = volRefNum;
	xpb.ioNamePtr = volumeName;
	
	err = pgpPBXGetVolInfo( &xpb );
	if( IsntErr( err ) )
	{
		if( ( xpb.ioVAtrb & kVolumeAttributeLockMask ) != 0 )
		{
			/* Volume is locked */
			err = vLckdErr;
		}
		else if( xpb.ioVDrvInfo <= 0 || xpb.ioVDRefNum >= 0 )
		{
			/* Volume is offline */
			
			err = volOffLinErr;
		}
		else if( xpb.ioVFSID != 0 )
		{
			/* Volume is not HFS or HFS Plus or is remote */
			err = extFSErr;
		}
		else
		{
			const VCB *vcb;
			
			vcb = GetVCBForVolume( volRefNum );
			if( IsntNull( vcb ) )
			{
				if( vcb->vcbSigWord != kHFSSigWord &&
					vcb->vcbSigWord != kHFSPlusSigWord )
				{
					err = wrgVolTypErr;
				}
				else if( vcb->vcbXTRef <= 0 || vcb->vcbCTRef <= 0 )
				{
					/* No catalog or BTree files. */
					err = wrgVolTypErr;
				}
			}
			else
			{
				err = wrgVolTypErr;
			}
		}
	}
	
	return( err );
}

	static OSStatus
SendEvent(
	const WipeControlBlock 	*wcb,
	const VolumeWipeEvent	*event)
{
	OSStatus	err = noErr;
	
	if( IsntNull( wcb->eventHandler ) )
		err = (*wcb->eventHandler)( event, wcb->eventUserValue );
	
	return( err );
}

	static OSStatus
DoProgressEvent(const WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	if( IsntNull( wcb->eventHandler ) )
	{
		VolumeWipeEvent	event;
		
		event.type = kVolumeWipeEvent_WipeProgress;

		event.progress.passIndex 			= wcb->passIndex;
		event.progress.numSectorsWiped 		= wcb->numSectorsWiped;
		event.progress.totalSectorsToWipe 	= wcb->numSectorsToWipe;

		if( event.progress.numSectorsWiped >
				event.progress.totalSectorsToWipe )
		{
			event.progress.numSectorsWiped =
				event.progress.totalSectorsToWipe;
		}
		
		err = SendEvent( wcb, &event );
	}
	
	return( err );
}

	static OSStatus
WCBFillIOPB(
	const WipeControlBlock 	*wcb,
	PGPUInt32				startingSector,
	PGPUInt32				numSectors,
	PGPByte					*buffer,
	IOPB					*iopb)
{
	OSStatus	err = noErr;
	
	if( wcb->numVolumeSectors != 0 &&
		startingSector + numSectors > wcb->numVolumeSectors )
	{
		return( fileBoundsErr );
	}
	
	pgpClearMemory( iopb, sizeof( *iopb ) );
	
	iopb->pb.ioParam.ioCompletion	= NULL;
	iopb->pb.ioParam.ioVRefNum		= wcb->driveNumber;
	iopb->pb.ioParam.ioRefNum		= wcb->driverRefNum;
	iopb->pb.ioParam.ioBuffer		= (Ptr) buffer;
	iopb->pb.ioParam.ioReqCount		= numSectors * kDiskSectorSize;
	iopb->pb.ioParam.ioPosMode		= fsFromStart;
	
	if( NeedWidePositioning( startingSector, numSectors ) )
	{
		UInt64	startingOffset;
		
		iopb->xpb.ioPosMode 	|= kWidePosOffsetMask;
		startingOffset			= ( wcb->firstSectorBias +
									(UInt64) startingSector ) *
										(UInt64) kDiskSectorSize;
		iopb->xpb.ioWPosOffset	= *((wide *) &startingOffset);
	}
	else
	{
		iopb->pb.ioParam.ioPosOffset = (wcb->firstSectorBias + startingSector) *
												kDiskSectorSize;
	}
	
	return( err );
}

	static OSStatus
WCBReadSectors(
	WipeControlBlock 	*wcb,
	PGPUInt32			startingSector,
	PGPUInt32			numSectors,
	PGPByte				*buffer)
{
	IOPB		iopb;
	OSStatus	err;
	
	err = WCBFillIOPB( wcb, startingSector, numSectors, buffer, &iopb );
	if( IsntErr( err ) )
	{
#if PGP_DEBUG
		wcb->readCount			+= 1;
		wcb->readSectorCount	+= numSectors;
#endif

		err = PBReadSync( &iopb.pb );
	}
	
	pgpAssert( IsntErr( err ) );

	return( err );
}

	static OSStatus
WCBWriteSectors(
	WipeControlBlock 	*wcb,
	PGPUInt32			startingSector,
	PGPUInt32			numSectors,
	PGPByte				*buffer)
{
	IOPB		iopb;
	OSStatus	err;
	
	err = WCBFillIOPB( wcb, startingSector, numSectors, buffer, &iopb );
	if( IsntErr( err ) )
	{
#if PGP_DEBUG
		wcb->writeCount			+= 1;
		wcb->writeSectorCount	+= numSectors;
#endif

		err = PBWriteSync( &iopb.pb );
	}
	
	pgpAssert( IsntErr( err ) );

	return( err );
}

	static OSStatus
ValidateHFSMasterDirectoryBlock(const HFSMasterDirectoryBlock *mdb)
{
	Require( mdb->drSigWord == kHFSSigWord );
	Require( mdb->drCrDate != 0 );
	Require( mdb->drLsMod != 0 );
	Require( mdb->drLsMod >= mdb->drCrDate );
	Require( mdb->drVBMSt >= 3 );
	Require( mdb->drAllocPtr <= mdb->drNmAlBlks );
	Require( mdb->drNmAlBlks != 0 );
	Require( ( mdb->drAlBlkSiz % 512L ) == 0 );
	Require( ( mdb->drClpSiz % mdb->drAlBlkSiz ) == 0 );
	Require( mdb->drAlBlSt > mdb->drVBMSt );
	Require( mdb->drNxtCNID >= kHFSFirstUserCatalogNodeID );
	Require( mdb->drFreeBks < mdb->drNmAlBlks );
	Require( mdb->drVN[0] != 0 );
	Require( mdb->drVN[0] < sizeof( mdb->drVN ) );
	Require(( mdb->drXTFlSize % mdb->drAlBlkSiz ) == 0 );
	Require(( mdb->drCTFlSize % mdb->drAlBlkSiz ) == 0 );
	
	return( noErr );
}

	static OSStatus
ValidateHFSPlusVolumeHeader(const HFSPlusVolumeHeader *vh)
{
	Require( vh->signature == kHFSPlusSigWord );
	Require( vh->createDate != 0 );
	Require( vh->modifyDate != 0 );
	Require( vh->modifyDate >= vh->createDate );
	Require( vh->nextAllocation <= vh->totalBlocks );
	Require( vh->totalBlocks != 0 );
	Require( vh->freeBlocks < vh->totalBlocks );
	Require( ( vh->blockSize % 512L ) == 0 );
	Require( vh->nextCatalogID >= kHFSFirstUserCatalogNodeID );

	pgpAssertMsg( kHFSPlusVersion == 4,
				"HFSPlus volume format changed. Check all code" );
	
	if( vh->version != kHFSPlusVersion ||
		vh->attributesFile.logicalSize != 0 )
	{
		return( kWipingError_UnsupportedVolumeFormat );
	}
	
	return( noErr );
}

	static OSStatus
WCBReadWriteFileSectors(
	WipeControlBlock 	*wcb,
	PGPBoolean			doRead,
	PGPUInt32			numExtents,
	const DiskExtent	*extentList,
	PGPUInt32			startingSector,
	PGPSize				numSectorsToReadWrite,
	PGPByte				*buffer)
{
	OSStatus	err = noErr;
	
	pgpAssertAddrValid( extentList, DiskExtent );
	pgpAssertAddrValid( buffer, VoidAlign );
	
	if( numSectorsToReadWrite != 0 )
	{
		PGPUInt32		extentIndex;
		uchar			*curBuffer;
		PGPUInt32		curFileSector;
		PGPUInt32		sectorsRemaining;
		
		curFileSector		= 0;
		curBuffer			= buffer;
		sectorsRemaining	= numSectorsToReadWrite;
		
		for( extentIndex = 0; extentIndex < numExtents; extentIndex++ )
		{
			const DiskExtent	*curExtent;
			
			curExtent = &extentList[extentIndex];
			
			if( curFileSector + curExtent->numBlocks > startingSector )
			{
				PGPUInt32	readWriteStartSector;
				PGPUInt32	numReadWrittenSectors;
				
				readWriteStartSector 	= curExtent->diskBlockIndex;
				numReadWrittenSectors	= curExtent->numBlocks;

				if( curFileSector < startingSector )
				{
					// File offset starts in this extent.
					readWriteStartSector  += ( startingSector - curFileSector );
					numReadWrittenSectors -= ( startingSector - curFileSector );
				}
				
				if( numReadWrittenSectors > sectorsRemaining )
					numReadWrittenSectors = sectorsRemaining;
				
				if( doRead )
				{
					err = WCBReadSectors( wcb, readWriteStartSector, 
								numReadWrittenSectors, curBuffer );
				}
				else
				{
					err = WCBWriteSectors( wcb, readWriteStartSector, 
								numReadWrittenSectors, curBuffer );
				}
				
				sectorsRemaining	-= numReadWrittenSectors;
				curBuffer 			+= ( numReadWrittenSectors *
												kDiskSectorSize );
				
				if( IsErr( err ) || sectorsRemaining == 0 )
					break;
			}
			
			curFileSector += curExtent->numBlocks;
		}
	}
	
	return( err );
}

	
	static OSStatus
WCBReadBTreeFile(
	WipeControlBlock 	*wcb,
	const BTreeFileInfo	*fileInfo,
	PGPUInt32			startingSector,
	PGPUInt32			numSectorsToRead,
	PGPByte				*buffer)
{
	OSStatus	err;
	
	if( fileInfo->fileRefNum != 0 )
	{
		UInt32	bytesRead;
		
		/* Go through the File Manager because of caching issues */

		err = FSReadAtOffset( fileInfo->fileRefNum, startingSector *
						kDiskSectorSize, numSectorsToRead * kDiskSectorSize,
						buffer, &bytesRead, TRUE );
	}
	else
	{
		err = WCBReadWriteFileSectors( wcb, TRUE, fileInfo->numDiskExtents,
						fileInfo->diskExtents, startingSector,
						numSectorsToRead, buffer );
	}

	return( err );
}

	static OSStatus
WCBWriteBTreeFile(
	WipeControlBlock 	*wcb,
	const BTreeFileInfo	*fileInfo,
	PGPUInt32			startingSector,
	PGPUInt32			numSectorsToRead,
	PGPByte				*buffer)
{
	OSStatus	err;
	
	err = WCBReadWriteFileSectors( wcb, FALSE, fileInfo->numDiskExtents,
						fileInfo->diskExtents, startingSector,
						numSectorsToRead, buffer );

	return( err );
}

	static OSStatus
WCBReadBTreeNodes(
	WipeControlBlock 	*wcb,
	const BTreeFileInfo	*fileInfo,
	PGPUInt32			startingNode,
	PGPUInt32			numNodesToRead,
	PGPByte				*buffer)
{
	PGPUInt32	sectorsPerNode;
	
	sectorsPerNode = fileInfo->nodeSize / kDiskSectorSize;
	
	return( WCBReadBTreeFile( wcb, fileInfo, startingNode * sectorsPerNode,
					numNodesToRead * sectorsPerNode, buffer ) );
}

	static OSStatus
WCBWriteBTreeNodes(
	WipeControlBlock 	*wcb,
	const BTreeFileInfo	*fileInfo,
	PGPUInt32			startingNode,
	PGPUInt32			numNodesToWrute,
	PGPByte				*buffer)
{
	PGPUInt32	sectorsPerNode;
	
	sectorsPerNode = fileInfo->nodeSize / kDiskSectorSize;
	
	return( WCBWriteBTreeFile( wcb, fileInfo, startingNode * sectorsPerNode,
					numNodesToWrute * sectorsPerNode, buffer ) );
}

	static OSStatus
ValidateBTreeHeaderNode(
	const BTreeHeaderNode	*header,
	PGPSize					bTreeFileSize)
{
	Require( header->descriptor.ndType == kBTreeHeaderNodeType )
	Require( header->record.bthFree < header->record.bthNNodes );
	Require( ( header->record.bthNodeSize % 512L ) == 0 );
	Require( header->record.bthNNodes * header->record.bthNodeSize ==
					bTreeFileSize );

	if( header->record.bthNRecs == 0 )
	{
		/* Verify as the header of an empty tree */

		Require( header->record.bthDepth == 0 );
		Require( header->record.bthRoot == 0 );
		Require( header->record.bthFNode == 0 );
		Require( header->record.bthLNode == 0 );
		Require( header->record.bthFree < header->record.bthNNodes );
		
	}
	else
	{
		Require( header->record.bthDepth != 0 );
		Require( header->record.bthRoot > 0 &&
						header->record.bthRoot < header->record.bthNNodes );
		Require( header->record.bthFNode > 0 &&
						header->record.bthFNode < header->record.bthNNodes );
		Require( header->record.bthLNode > 0 &&
						header->record.bthLNode < header->record.bthNNodes );
	}
	
	return( noErr );
}

	static OSStatus
WCBOpenBTreeFile(
	WipeControlBlock 	*wcb,
	short				fileRefNum,
	PGPUInt32			fileSize,
	BTreeFileInfo		*fileInfo)
{
	OSStatus	err = noErr;
	
	MacValidateParam( fileRefNum > 0 );
	
	pgpClearMemory( fileInfo, sizeof( *fileInfo ) );
	
	fileInfo->fileRefNum 	= fileRefNum;
	fileInfo->fileSize		= fileSize;
	
	err = GetDiskExtentsForFork( fileRefNum, &fileInfo->diskExtents,
					&fileInfo->numDiskExtents );
	if( IsntErr( err ) )
	{
		PGPUInt32	calcFileSize = 0;
		PGPUInt32	extentIndex;
		
		for( extentIndex = 0; extentIndex < fileInfo->numDiskExtents;
					++extentIndex)
		{
			calcFileSize += fileInfo->diskExtents[extentIndex].numBlocks *
								kDiskSectorSize;
								
			/*
			** The extents already have the first sector bias added because
			** they were obtained using Lg2Phys. Compensate.
			*/
			
			Require( fileInfo->diskExtents[extentIndex].diskBlockIndex >=
						wcb->firstSectorBias );
			
			fileInfo->diskExtents[extentIndex].diskBlockIndex -=
							wcb->firstSectorBias;	
		}
		
		if( calcFileSize != fileSize )
		{
			pgpDebugMsg(
				"WCBOpenBTreeFile(): Missing or extra extents" );
			err = eofErr;
		}
		
		if( IsntErr( err ) )
		{
			err = WCBReadBTreeFile( wcb, fileInfo, 0, 1, wcb->ioBuffer );
			if( IsntErr( err ) )
			{
				BTreeHeaderNode		header;
				
				header = *((BTreeHeaderNode *) wcb->ioBuffer);

				fileInfo->nodeSize 	= (PGPUInt16) header.record.bthNodeSize;
				fileInfo->numNodes 		= (PGPUInt16) header.record.bthNNodes;
				fileInfo->numFreeNodes	= (PGPUInt16) header.record.bthFree;
				fileInfo->treeDepth		= header.record.bthDepth;
				
				if( fileInfo->nodeSize < wcb->ioBufferSize )
				{
					err = ValidateBTreeHeaderNode( &header, fileSize );
				}
				else
				{
					/* We need to have a buffer which holds one complete node */
					err = memFullErr;
				}
				
				if( IsntErr( err ) )
				{
					PGPSize	allocationSize;
					
					/* Allocate BTree bitmap and node buffers */
					
					allocationSize = ( fileInfo->numNodes + 7 ) / 8;
					
					fileInfo->nodeBitmap = (PGPByte *) pgpAllocMac(
												allocationSize,
												kMemoryAllocationFlags |
												kMacMemory_ClearBytes);
					
					fileInfo->nodeBuffer = (PGPByte *) pgpAllocMac(
												fileInfo->nodeSize,
												kMemoryAllocationFlags );
					
					if( IsNull( fileInfo->nodeBitmap ) ||
						IsNull( fileInfo->nodeBuffer ) )
					{
						err = memFullErr;
					}
				}
				
				if( IsntErr( err ) )
				{
					PGPUInt32	remainingBytes;
					PGPUInt32	nodeIndex;
					PGPByte		*curBitmapByte;
					
					remainingBytes 	= ( fileInfo->numNodes + 7 ) / 8;
					curBitmapByte 	= fileInfo->nodeBitmap;
					nodeIndex		= 0;
					
					while( remainingBytes != 0 && IsntErr( err ) )
					{
						err = WCBReadBTreeNodes( wcb, fileInfo, nodeIndex,
									1, fileInfo->nodeBuffer );
						if( IsntErr(  err ) )
						{
							PGPUInt32		bytesToCopy;
							PGPUInt32		copyOffset;
							NodeDescriptor	*nodeDesc;
							
							/*
							** The B-Tree header node contains first map bits
							** at offset 0xF8. Additional map nodes contain
							** map bits at offset 0x0e. See IM Files for
							** details. Note that IM Files claims there are 494
							** bytes in a map node. This is incorrect.
							*/
							
							if( nodeIndex == 0 )
							{
								bytesToCopy = fileInfo->nodeSize - 256;
								copyOffset	= 0xF8;
							}
							else
							{
								bytesToCopy = fileInfo->nodeSize - 20;
								copyOffset	= 0x0E;
							}
							
							if( bytesToCopy > remainingBytes )
								bytesToCopy = remainingBytes;
								
							pgpCopyMemory( &fileInfo->nodeBuffer[copyOffset],
									curBitmapByte, bytesToCopy );
									
							remainingBytes 	-= bytesToCopy;
							curBitmapByte	+= bytesToCopy;
							
							nodeDesc = (NodeDescriptor *) fileInfo->nodeBuffer;
							nodeIndex = nodeDesc->ndFLink;
						}
					}
					
					if( IsntErr( err ) )
					{
						PGPBitCount	numFreeNodes;
						
						/* Validate the bitmap count */
						numFreeNodes = pgpCountClearBitsInRange(
											fileInfo->nodeBitmap, 0,
											fileInfo->numNodes );
						if( numFreeNodes != fileInfo->numFreeNodes )
						{
							pgpDebugMsg( "BTree node map inconsistent" );
							err = kWipingError_InvalidVolumeDataStructure;
						}
					}
				}
			}
		}
	}

	return( err );
}

	static void
WCBCloseBTreeFile(
	const WipeControlBlock 	*wcb,
	BTreeFileInfo			*fileInfo)
{
	(void) wcb;
	
	if( IsntNull( fileInfo->diskExtents) )
		FreeDiskExtentList( fileInfo->diskExtents );
			
	if( IsntNull( fileInfo->nodeBitmap ) )
		pgpFreeMac( fileInfo->nodeBitmap );

	if( IsntNull( fileInfo->nodeBuffer) )
		pgpFreeMac( fileInfo->nodeBuffer );
		
	pgpClearMemory( fileInfo, sizeof( *fileInfo ) );
}

	static PGPUInt32
WCBBlockToSector(
	const WipeControlBlock 	*wcb,
	PGPUInt32				block)
{
	if( wcb->signature == kHFSSigWord )
	{
		pgpAssert( block < wcb->numBlocks );
		
		return( ( block * wcb->numSectorsPerBlock ) + wcb->hfs.mdb.drAlBlSt );
	}
	else if( wcb->signature == kHFSPlusSigWord )
	{
		pgpAssert( block < wcb->numBlocks );
		
		return( block * wcb->numSectorsPerBlock );
	}
	else
	{
		pgpDebugMsg( "\pWCBBlockToSector(): Unknown volume format" );
		return( 0xFFFFFFFF );
	}
}

#pragma mark [ --- Wipe Ranges --- ]

#define	kWipeRangeGrowQuantum	2000

	static OSStatus
RememberWipeRange(
	WipeControlBlock	*wcb,
	PGPUInt32			startSector,
	PGPUInt32			firstSectorOffset,
	PGPUInt32			numSectors)
{
	OSStatus	err = noErr;
	
	pgpAssert( firstSectorOffset < kDiskSectorSize );
	pgpAssert( numSectors < (1L << 16) );
	
	if( wcb->numWipeRanges == wcb->numAllocatedWipeRanges )
	{
		/* Grow/allocate the list */
	
		wcb->numAllocatedWipeRanges += kWipeRangeGrowQuantum;
		
		err = pgpReallocMac( &wcb->wipeRangeList, wcb->numAllocatedWipeRanges *
					sizeof( WipeRange ), kMemoryAllocationFlags );
	}

	if( IsntErr( err ) )
	{
		WipeRange	*range;
			
		range = &wcb->wipeRangeList[wcb->numWipeRanges];
		
		range->startSector 			= startSector;
		range->firstSectorOffset 	= firstSectorOffset;
		range->numSectors 			= numSectors;
		
		++wcb->numWipeRanges;
	}
	
	return( err );
}

#define HeapSortName	SortWipeRanges
#define HeapSortItem	WipeRange

#include "pgpHeapSort.h"

#undef HeapSortName
#undef HeapSortItem

	static int
CompareWipeRanges(
	WipeRange	*range1,
	WipeRange	*range2,
	void 		*userValue)
{
	(void) userValue;
	
	if( range1->startSector < range2->startSector )
	{
		return( -1 );
	}
	else if( range1->startSector > range2->startSector )
	{
		return( 1 );
	}
	else
	{
		pgpDebugMsg( "CompareWipeRanges(): Overlaping wipe ranges" );
	}
	
	return( 0 );
}
	
#pragma mark [ --- Problem Forks --- ]

	static ProblemForkInfo *
FindProblemForkInfo(
	const WipeControlBlock	*wcb,
	PGPUInt32				fileID,
	ForkType				forkType)
{
	ProblemForkInfo	*forkInfo = NULL;
	
	if( IsntNull( wcb->problemForkList ) )
	{
		PGPUInt32	forkIndex;
		
		for( forkIndex = 0; forkIndex < wcb->numProblemForks; forkIndex++ )
		{
			if( wcb->problemForkList[forkIndex].fileID == fileID &&
				wcb->problemForkList[forkIndex].forkType == forkType )
			{
				forkInfo = &wcb->problemForkList[forkIndex];
				break;
			}
		}
	}
	
	pgpAssert( IsntNull( forkInfo ) );
	
	return( forkInfo );
}

	static OSStatus
RememberProblemForkExtents(
	ProblemForkInfo		*forkInfo,
	PGPUInt32			firstExtentBlock,
	const ExtentRecord	*extents)
{
	ForkExtent	*extentInfo;
	OSStatus	err = noErr;
	
	extentInfo = (ForkExtent *) pgpAllocMac( sizeof( *extentInfo ),
												kMemoryAllocationFlags );
	if( IsntNull( extentInfo ) )
	{
		ForkExtent	*cur;
		ForkExtent	*prev;
			
		extentInfo->firstExtentBlock 	= firstExtentBlock;
		extentInfo->extents				= *extents;
		extentInfo->next				= NULL;
		
		/*
		** Add the extent list record to the forkInfo in sorted
		** firstExtenBlock order.
		*/
		
		cur 	= forkInfo->extentList;
		prev	= NULL;
		
		while( IsntNull( cur ) )
		{
			if( cur->firstExtentBlock > firstExtentBlock )
			{
				/* Insert at prev */
				break;
			}
			
			prev = cur;
			cur  = cur->next;
		}
		
		if( IsntNull( prev ) )
		{
			prev->next = extentInfo;
		}
		else
		{
			forkInfo->extentList = extentInfo;
		}
		
		extentInfo->next = cur;
	}
	else
	{
		err = memFullErr;
	}

	return( err );
}

	static OSStatus
RememberProblemFork(
	WipeControlBlock 	*wcb,
	PGPUInt32			fileID,
	ForkType			forkType,
	PGPUInt32			forkLogicalLength,
	PGPUInt32			forkPhysicalLength,
	const ExtentRecord	*firstExtentRecord)
{
	OSStatus	err = noErr;
	PGPUInt32	newDataSize;
	
	newDataSize	= (wcb->numProblemForks + 1) * sizeof( ProblemForkInfo );
	
	err = pgpReallocMac( &wcb->problemForkList, newDataSize,
				kMemoryAllocationFlags );
	if( IsntErr( err ) )
	{
		ProblemForkInfo	*info;

		info = &wcb->problemForkList[wcb->numProblemForks];
		
		info->fileID				= fileID;
		info->forkType				= forkType;
		info->forkLogicalLength		= forkLogicalLength;
		info->forkPhysicalLength	= forkPhysicalLength;
		info->extentList			= NULL;

		err = RememberProblemForkExtents( info, 0, firstExtentRecord );
		
		++wcb->numProblemForks;
	}
	
	return( err );
}

	static OSStatus
ValidateHFSProblemForkList(WipeControlBlock *wcb)
{
	PGPUInt32	forkIndex;
	OSStatus	err = noErr;
	
	/*
	** Validate the integrity of the problem forks. We check to make sure
	** that all allocation blocks are accounted for and none overlap.
	*/
	
	for( forkIndex = 0; forkIndex < wcb->numProblemForks; forkIndex++ )
	{
		ProblemForkInfo	*forkInfo;
		PGPUInt32		foundBlocks;
		ForkExtent		*curExtent;
		
		forkInfo 	= &wcb->problemForkList[forkIndex];
		foundBlocks	= 0;
		
		curExtent = forkInfo->extentList;
		while( IsntNull( curExtent ) )
		{
			PGPUInt32	extentIndex;
			
			Require( curExtent->firstExtentBlock == foundBlocks );
			
			for( extentIndex = 0; extentIndex < kHFSExtentDensity;
						extentIndex++ )
			{
				foundBlocks += curExtent->extents.hfs[extentIndex].blockCount;
			}
			
			curExtent = curExtent->next;
		}
		
		Require( forkInfo->forkPhysicalLength == foundBlocks *
					wcb->blockSize );
	}

	/* Compute the wiping range(s) for each of the forks in the list */

	for( forkIndex = 0; forkIndex < wcb->numProblemForks; forkIndex++ )
	{
		ProblemForkInfo	*forkInfo;
		PGPUInt32		remainingBytes;
		ForkExtent		*curExtent;
		
		forkInfo		= &wcb->problemForkList[forkIndex];
		remainingBytes	= forkInfo->forkLogicalLength;
		
		curExtent = forkInfo->extentList;
		while( IsntNull( curExtent ) &&
				IsntErr( err ) )
		{
			PGPUInt32	extentIndex;
			
			for( extentIndex = 0; extentIndex < kHFSExtentDensity;
						extentIndex++ )
			{
				HFSExtentDescriptor	*desc;
				PGPUInt32			startSector			= 0;
				PGPUInt32			firstSectorOffset 	= 0;
				PGPUInt32			numSectors 			= 0;
				
				desc = &curExtent->extents.hfs[extentIndex];
				
				if( remainingBytes != 0 )
				{
					PGPUInt32	descBytes;
					
					Require( desc->blockCount != 0 );
					
					descBytes = (PGPUInt32) desc->blockCount *
									wcb->blockSize;
					if( descBytes <= remainingBytes )
					{
						/* All bytes in this range are in use */
						
						remainingBytes -= descBytes;
					}
					else
					{
						/* The logical eof is in this extent descriptor */
				
						PGPUInt32	usedSectors;
						
						startSector	= WCBBlockToSector( wcb, desc->startBlock );
						numSectors	= (PGPUInt32) desc->blockCount *
											wcb->numSectorsPerBlock;
						usedSectors	= remainingBytes / kDiskSectorSize;
						
						startSector 		+= usedSectors;
						numSectors			-= usedSectors;
						firstSectorOffset	= remainingBytes % kDiskSectorSize;
					
						remainingBytes = 0;
					}
				}
				else if( desc->blockCount != 0 )
				{
					/*
					** Entire extent descriptor is in free space. Wipe
					** the whole thing
					*/
					
					startSector 		= WCBBlockToSector( wcb,
												desc->startBlock );
					firstSectorOffset 	= 0;
					numSectors			= (PGPUInt32) desc->blockCount * 
												wcb->numSectorsPerBlock;
				}
				else
				{
					/* All remaining descriptors should have no counts */
					
					break;
				}
				
				if( numSectors != 0 )
				{
					err = RememberWipeRange( wcb, startSector,
								firstSectorOffset, numSectors );
					
					wcb->numSectorsToWipe += numSectors;
					
					if( IsErr( err ) )
					{
						break;
					}
				}
			}
			
			curExtent = curExtent->next;
		}
	}
		
	return( err );
}

	static OSStatus
ValidateHFSPlusProblemForkList(WipeControlBlock *wcb)
{
	PGPUInt32	forkIndex;
	OSStatus	err = noErr;
	
	/*
	** Validate the integrity of the problem forks. We check to make sure
	** that all allocation blocks are accounted for and none overlap.
	*/
	
	for( forkIndex = 0; forkIndex < wcb->numProblemForks; forkIndex++ )
	{
		ProblemForkInfo	*forkInfo;
		PGPUInt32		foundBlocks;
		ForkExtent		*curExtent;
		
		forkInfo 	= &wcb->problemForkList[forkIndex];
		foundBlocks	= 0;
		
		curExtent = forkInfo->extentList;
		while( IsntNull( curExtent ) )
		{
			PGPUInt32	extentIndex;
			
			Require( curExtent->firstExtentBlock == foundBlocks );
			
			for( extentIndex = 0; extentIndex < kHFSPlusExtentDensity;
						extentIndex++ )
			{
				foundBlocks += 
					curExtent->extents.hfsPlus[extentIndex].blockCount;
			}
			
			curExtent = curExtent->next;
		}
		
		Require( forkInfo->forkPhysicalLength == foundBlocks *
					wcb->blockSize );
	}

	/* Compute the wiping range(s) for each of the forks in the list */

	for( forkIndex = 0; forkIndex < wcb->numProblemForks; forkIndex++ )
	{
		ProblemForkInfo	*forkInfo;
		PGPUInt32		remainingBytes;
		ForkExtent		*curExtent;
		
		forkInfo		= &wcb->problemForkList[forkIndex];
		remainingBytes	= forkInfo->forkLogicalLength;
		
		curExtent = forkInfo->extentList;
		while( IsntNull( curExtent ) &&
				IsntErr( err ) )
		{
			PGPUInt32	extentIndex;
			
			for( extentIndex = 0; extentIndex < kHFSPlusExtentDensity;
						extentIndex++ )
			{
				HFSPlusExtentDescriptor	*desc;
				PGPUInt32				startSector			= 0;
				PGPUInt32				firstSectorOffset 	= 0;
				PGPUInt32				numSectors 			= 0;
				
				desc = &curExtent->extents.hfsPlus[extentIndex];
				
				if( remainingBytes != 0 )
				{
					PGPUInt32	descBytes;
					
					Require( desc->blockCount != 0 );
					
					descBytes = desc->blockCount * wcb->blockSize;
					if( descBytes <= remainingBytes )
					{
						/* All bytes in this range are in use */
						
						remainingBytes -= descBytes;
					}
					else
					{
						/* The logical eof is in this extent descriptor */
				
						PGPUInt32	usedSectors;
						
						startSector	= WCBBlockToSector( wcb, desc->startBlock );
						numSectors	= desc->blockCount *
											wcb->numSectorsPerBlock;
						usedSectors	= remainingBytes / kDiskSectorSize;
						
						startSector 		+= usedSectors;
						numSectors			-= usedSectors;
						firstSectorOffset	= remainingBytes % kDiskSectorSize;
					
						remainingBytes = 0;
					}
				}
				else if( desc->blockCount != 0 )
				{
					/*
					** Entire extent descriptor is in free space. Wipe
					** the whole thing
					*/
					
					startSector 		= WCBBlockToSector( wcb,
												desc->startBlock );
					firstSectorOffset 	= 0;
					numSectors			= desc->blockCount * 
												wcb->numSectorsPerBlock;
				}
				else
				{
					/* All remaining descriptors should have no counts */
					
					break;
				}
				
				if( numSectors != 0 )
				{
					err = RememberWipeRange( wcb, startSector,
								firstSectorOffset, numSectors );
					
					wcb->numSectorsToWipe += numSectors;
					
					if( IsErr( err ) )
					{
						break;
					}
				}
			}
			
			curExtent = curExtent->next;
		}
	}
		
	return( err );
}

	static OSStatus
ValidateProblemForkList(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	if( wcb->signature == kHFSSigWord )
	{
		err = ValidateHFSProblemForkList( wcb );
	}
	else if( wcb->signature == kHFSPlusSigWord )
	{
		err = ValidateHFSPlusProblemForkList( wcb );
	}
	else
	{
		pgpDebugMsg( "\pValidateProblemForkList(): Unknown volume format" );
		err = kWipingError_UnsupportedVolumeFormat;
	}
	
	return( err );
}

#pragma mark [ --- HFS Scanning --- ]

	static OSStatus
ScanHFSExtentRecord(
	const WipeControlBlock 	*wcb,
	const HFSExtentRecord	record,
	PGPUInt32				*foundForkBlocks)
{
	PGPUInt32	extentIndex;
	
	*foundForkBlocks = 0;
	
	for( extentIndex = 0; extentIndex < kHFSExtentDensity; extentIndex++)
	{
		const HFSExtentDescriptor	*extent = &record[extentIndex];
		PGPUInt32					overlappingBlockCount;
		
		if( extent->blockCount != 0 )
		{
			Require( (PGPUInt32) extent->startBlock +
						(PGPUInt32) extent->blockCount <=
							wcb->numBlocks );
			
			*foundForkBlocks += extent->blockCount;
			
			overlappingBlockCount = pgpCountAndSetBitRange(
											wcb->volumeBitmap,
											extent->startBlock,
											extent->blockCount );
			Require( overlappingBlockCount == 0 );
		}
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSFileFork(
	WipeControlBlock 		*wcb,
	PGPUInt32				fileID,
	ForkType				forkType,
	const HFSExtentRecord	extentRecord,
	PGPUInt32				forkLogicalLength,
	PGPUInt32				forkPhysicalLength)
{
	OSStatus	err = noErr;
	PGPInt32	extentIndex;
	
	Require( forkLogicalLength <= forkPhysicalLength );
	Require( ( forkPhysicalLength % wcb->blockSize ) == 0 );

	if( forkPhysicalLength != 0 )
	{
		PGPUInt32	foundForkBlocks;

		err = ScanHFSExtentRecord( wcb, extentRecord, &foundForkBlocks );
		if( IsntErr( err ) )
		{
			PGPUInt32	totalForkBlocks;

			totalForkBlocks = forkPhysicalLength / wcb->blockSize;

			if( foundForkBlocks == totalForkBlocks )
			{
				/*
				** Simplest case. All of the files allocation blocks are
				** accounted for in the catalog. We now check to see if
				** the file's logical EOF is in the last allocation block of
				** the file. If so, we remember the remainder of the last
				** allocation block for wiping later
				*/
				
				if( forkPhysicalLength - forkLogicalLength <
							wcb->blockSize )
				{
					if( forkPhysicalLength != forkLogicalLength )
					{
						/*
						** Get the last allocation block and remember
						** the range.
						*/
					
						for( extentIndex = kHFSExtentDensity - 1;
								extentIndex >= 0; --extentIndex )
						{
							HFSExtentDescriptor	desc;
							
							desc = extentRecord[extentIndex];
							
							if( desc.blockCount != 0 )
							{
								PGPUInt32	lastBlock;
								PGPUInt32	usedBytes;
								PGPUInt32	startSector;
								PGPUInt32	skipSectors;
								PGPUInt32	byteOffset;
								PGPUInt32	numSectors;
								
								lastBlock 	= desc.startBlock +
													desc.blockCount - 1;
								usedBytes	= wcb->blockSize - (
												forkPhysicalLength -
													forkLogicalLength );
								skipSectors	= usedBytes / kDiskSectorSize;
								byteOffset	= usedBytes % kDiskSectorSize;
								numSectors	= wcb->numSectorsPerBlock -
												skipSectors;
								
								startSector	= WCBBlockToSector( wcb,
													lastBlock );
								startSector	+= skipSectors;
								
								err = RememberWipeRange( wcb, startSector, 
										byteOffset, numSectors );
								
								wcb->numSectorsToWipe += numSectors;
									
								break;
							}
						}
					}
					else
					{
						/*
						** Fork uses all of the allocation block.
						** No wiping needed
						*/
					}
				}
				else
				{
					/*
					** This is a problem file. The logical EOF is more than one
					** allocation block from the physical EOF. Add it to our
					** problem file list
					*/
					
					err = RememberProblemFork( wcb, fileID, forkType,
							forkLogicalLength, forkPhysicalLength,
							(ExtentRecord *) extentRecord );
				}
			}
			else if( foundForkBlocks < totalForkBlocks )
			{
				/*
				** Not all of the allocation blocks are in the catalog.
				** Remember this file for later analysis when we scan
				** the extents file.
				*/
				
				err = RememberProblemFork( wcb, fileID, forkType,
							forkLogicalLength, forkPhysicalLength,
							(ExtentRecord *) extentRecord );
			}
			else
			{
				/* Too many blocks. Damaged. */
				err = kWipingError_InvalidVolumeDataStructure;
			}
		}
	}
	else
	{
		for( extentIndex = 0; extentIndex < kHFSExtentDensity; extentIndex++ )
		{
			const HFSExtentDescriptor	*extent = &extentRecord[extentIndex];

			Require( extent->startBlock == 0 );
			Require( extent->blockCount == 0 );
		}
	}
	
	return( err );
}

	static OSStatus
ScanHFSCatalogFileRecord(
	WipeControlBlock 		*wcb,
	const HFSCatalogFile	*file)
{	
	OSStatus	err = noErr;
	
	Require( file->recordType == kHFSFileRecord );
	Require( file->fileID < wcb->hfs.mdb.drNxtCNID );

	/*
	** Validate that we do not have too many extents and mark the used
	** extents in our volume bitmap.
	*/
	
	err = ScanHFSFileFork( wcb, file->fileID, kDataForkType, file->dataExtents,
				file->dataLogicalSize, file->dataPhysicalSize );
	if( IsntErr( err ) )
	{
		err = ScanHFSFileFork( wcb, file->fileID, kResourceForkType,
					file->rsrcExtents, file->rsrcLogicalSize,
					file->rsrcPhysicalSize );
	}
		
	return( err );
}

	static OSStatus
ScanHFSCatalogFileLeafNode(
	WipeControlBlock 		*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	Require( nodeDesc->ndType == kBTreeLeafNodeType );
	Require( nodeDesc->ndNHeight == 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->catalogFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSCatalogKey		*key;
		PGPUInt32			dataOffset;
		HFSCatalogRecord	*data;
		
		key = (HFSCatalogKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		Require( key->parentID < wcb->hfs.mdb.drNxtCNID );
		Require( key->nodeName[0] <= kHFSMaxFileNameChars );
		Require( key->keyLength >= kHFSCatalogKeyMinimumLength );
		Require( key->keyLength <= kHFSCatalogKeyMaximumLength );
	
		/* Data is padded to an even boundary */
		dataOffset = key->keyLength + 1;
		if( ( dataOffset & 1 ) != 0 )
			++dataOffset;
			
		data = (HFSCatalogRecord *) ( (PGPByte *) key + dataOffset );
		switch( data->type )
		{
			OSStatus	err;
			
			case kHFSFileRecord:
				err = ScanHFSCatalogFileRecord( wcb, &data->file );
				if( IsErr( err ) )
					return( err );
				break;
				
			case kHFSFolderRecord:
				Require( data->folder.folderID < wcb->hfs.mdb.drNxtCNID );
				break;
				
			case kHFSFolderThreadRecord:
			case kHFSFileThreadRecord:
				Require( data->thread.parentID < wcb->hfs.mdb.drNxtCNID );
				Require( data->thread.nodeName[0] <= kHFSMaxFileNameChars );
				break;
				
			default:
				pgpDebugMsg( "Invalid leaf record type" );
				return( kWipingError_InvalidVolumeDataStructure );
				break;
		}
			
		--recordOffsetPtr;
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSCatalogFileIndexNode(
	const WipeControlBlock 	*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	Require( nodeDesc->ndType == kBTreeIndexNodeType );
	Require( nodeDesc->ndNHeight != 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->catalogFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSCatalogFileIndexRecord		*record;
		
		record = (HFSCatalogFileIndexRecord *) ((PGPByte *) nodeDesc +
						*recordOffsetPtr);
		
		Require( record->key.parentID < wcb->hfs.mdb.drNxtCNID );
		Require( record->key.nodeName[0] <= kHFSMaxFileNameChars );
		Require( record->key.keyLength == kHFSCatalogKeyMaximumLength );
		Require( record->nodeIndex > 0 );
		Require( record->nodeIndex < wcb->catalogFileInfo.numNodes );
		Require( pgpTestBit( wcb->catalogFileInfo.nodeBitmap,
					record->nodeIndex ) );
				
		--recordOffsetPtr;
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSCatalogFile(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	/* Account for catalog allocation blocks. */
	
	err = ScanHFSFileFork( wcb, kHFSCatalogFileID, kDataForkType,
				wcb->hfs.mdb.drCTExtRec, wcb->hfs.mdb.drCTFlSize,
				wcb->hfs.mdb.drCTFlSize );
	if( IsntErr( err ) )
	{
		PGPUInt32	remainingNodes;
		PGPUInt32	btNodeIndex;
		PGPUInt32	readStartNode;
		PGPUInt32	numHeaderNodes;
		
		/*
		** Loop over all records in the catalog. For each record, we validate
		** the format of each record and set bits in the volume bit map. Error
		** if there are already any bits set in the volume bit map.
		*/
		
		btNodeIndex		= 0;
		readStartNode	= 0;
		numHeaderNodes	= 0;
		
		remainingNodes = wcb->catalogFileInfo.numNodes;
		while( remainingNodes != 0 && IsntErr( err ) )
		{
			PGPUInt32	nodesToRead;
			
			nodesToRead = wcb->ioBufferSize / wcb->catalogFileInfo.nodeSize;
			if( nodesToRead > remainingNodes )
				nodesToRead = remainingNodes;
			
			err = WCBReadBTreeNodes( wcb, &wcb->catalogFileInfo, 
						readStartNode, nodesToRead, wcb->ioBuffer );
			if( IsntErr( err ) )
			{
				PGPUInt32		nodeIndex;
				NodeDescriptor	*nodeDesc = (NodeDescriptor *) wcb->ioBuffer;
				
				for( nodeIndex = 0; nodeIndex < nodesToRead; nodeIndex++ )
				{
					if( pgpTestBit( wcb->catalogFileInfo.nodeBitmap,
								btNodeIndex ) )
					{
						switch( nodeDesc->ndType )
						{
							case kBTreeLeafNodeType:
								err = ScanHFSCatalogFileLeafNode( wcb,
											nodeDesc );
								++wcb->numSectorsToWipe;
								break;
								
							case kBTreeIndexNodeType:
								err = ScanHFSCatalogFileIndexNode( wcb,
											nodeDesc );
								++wcb->numSectorsToWipe;
								break;
							
							case kBTreeMapNodeType:
								/* Don't wipe or validate map nodes */
								break;
								
							case kBTreeHeaderNodeType:
								/* Already validated header node */
								++numHeaderNodes;
								if( numHeaderNodes > 1 )
								{
								pgpDebugMsg(
								"Found multiple catalogB-Tree header nodes" );
								err = kWipingError_InvalidVolumeDataStructure;
								}
								break;
						
							default:
								pgpDebugMsg( "Unknown catalog node type" );
								err = kWipingError_InvalidVolumeDataStructure;
								break;
						}
					}
					else
					{
						/* Free B-Tree node */
						++wcb->numSectorsToWipe;
					}
					
					if( IsntErr( err ) )
					{
						VolumeWipeEvent	event;
		
						event.type = kVolumeWipeEvent_GatheringInfo;
	
						err = SendEvent( wcb, &event );
					}
					
					if( IsErr( err ) )
						break;
				
					++btNodeIndex;
					
					nodeDesc = (NodeDescriptor *) ((PGPByte *) nodeDesc +
										wcb->catalogFileInfo.nodeSize);
				}
			}
			
			remainingNodes 	-= nodesToRead;
			readStartNode	+= nodesToRead;
		}
	}
	
	return( err );
}

	static OSStatus
ScanHFSExtentsFileLeafNode(
	const WipeControlBlock 	*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	OSStatus	err = noErr;
	
	Require( nodeDesc->ndType == kBTreeLeafNodeType );
	Require( nodeDesc->ndNHeight == 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->extentsFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSExtentKey		*key;
		HFSExtentDescriptor	*data;
		PGPUInt32			foundForkBlocks;
		ProblemForkInfo		*forkInfo;
		
		key = (HFSExtentKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		Require( key->keyLength == kHFSExtentKeyMaximumLength );
		Require( key->forkType == kDataForkType ||
					key->forkType == kResourceForkType );
		Require( key->fileID < wcb->hfs.mdb.drNxtCNID );
	
		data = (HFSExtentDescriptor *) ( (PGPByte *) key + key->keyLength + 1 );
		
		/* Find the fork in the problem file list. */
		forkInfo = FindProblemForkInfo( wcb, key->fileID, key->forkType );
		if( IsntNull( forkInfo ) )
		{
			err = ScanHFSExtentRecord( wcb, data, &foundForkBlocks );
			if( IsntErr( err ) )
			{
				/* Accumulate these extents into the problem fork list */
				err = RememberProblemForkExtents( forkInfo, key->startBlock,
							(ExtentRecord *) data );
			}
		}
		else
		{
			pgpDebugMsg("\pFound extent record with no matching catalog entry");
			err = kWipingError_InvalidVolumeDataStructure;
		}
		
		if( IsErr( err ) )
			break;
			
		--recordOffsetPtr;
	}
	
	return( err );
}

	static OSStatus
ScanHFSExtentsFileIndexNode(
	const WipeControlBlock 	*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	Require( nodeDesc->ndType == kBTreeIndexNodeType );
	Require( nodeDesc->ndNHeight != 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->extentsFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSExtentsFileIndexRecord		*record;
		
		record = (HFSExtentsFileIndexRecord *) ((PGPByte *) nodeDesc +
						*recordOffsetPtr);
		
		Require( record->key.keyLength == kHFSExtentKeyMaximumLength );
		Require( record->key.forkType == kDataForkType ||
					record->key.forkType == kResourceForkType );
		Require( record->key.fileID < wcb->hfs.mdb.drNxtCNID );
		Require( record->nodeIndex > 0 );
		Require( record->nodeIndex < wcb->extentsFileInfo.numNodes );
		Require( pgpTestBit( wcb->extentsFileInfo.nodeBitmap,
					record->nodeIndex ) );
					
		--recordOffsetPtr;
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSExtentsFile(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	/* Account for extents file allocation blocks. */
	
	err = ScanHFSFileFork( wcb, kHFSExtentsFileID, kDataForkType,
				wcb->hfs.mdb.drXTExtRec, wcb->hfs.mdb.drXTFlSize,
				wcb->hfs.mdb.drXTFlSize );
	
	if( IsntErr( err ) )
	{
		PGPUInt32	remainingNodes;
		PGPUInt32	btNodeIndex;
		PGPUInt32	readStartNode;
		PGPUInt32	numHeaderNodes;
		
		/*
		** Loop over all records in the catalog. For each record, we validate
		** the format of each record and set bits in the volume bit map. Error
		** if there are already any bits set in the volume bit map.
		*/
		
		btNodeIndex		= 0;
		readStartNode	= 0;
		numHeaderNodes	= 0;
		
		remainingNodes = wcb->extentsFileInfo.numNodes;
		while( remainingNodes != 0 && IsntErr( err ) )
		{
			PGPUInt32	nodesToRead;
			
			nodesToRead = wcb->ioBufferSize / wcb->extentsFileInfo.nodeSize;
			if( nodesToRead > remainingNodes )
				nodesToRead = remainingNodes;
				
			err = WCBReadBTreeNodes( wcb, &wcb->extentsFileInfo, 
						readStartNode, nodesToRead, wcb->ioBuffer );
			if( IsntErr( err ) )
			{
				PGPUInt32		nodeIndex;
				NodeDescriptor	*nodeDesc = (NodeDescriptor *) wcb->ioBuffer;
				
				for( nodeIndex = 0; nodeIndex < nodesToRead; nodeIndex++ )
				{
					if( pgpTestBit( wcb->extentsFileInfo.nodeBitmap,
							btNodeIndex ) )
					{
						switch( nodeDesc->ndType )
						{
							case kBTreeLeafNodeType:
								err = ScanHFSExtentsFileLeafNode( wcb,
											nodeDesc );
								++wcb->numSectorsToWipe;
								break;
								
							case kBTreeIndexNodeType:
								err = ScanHFSExtentsFileIndexNode( wcb,
											nodeDesc );
								++wcb->numSectorsToWipe;
								break;
							
							case kBTreeMapNodeType:
								/* Don't wipe or validate map nodes */
								break;
								
							case kBTreeHeaderNodeType:
								/* Already validated header node */
								++numHeaderNodes;
								if( numHeaderNodes > 1 )
								{
								pgpDebugMsg(
								"Found multiple extents-Tree header nodes" );
								err = kWipingError_InvalidVolumeDataStructure;
								}
								break;
						
							default:
								pgpDebugMsg( "Unknown catalog node type" );
								err = kWipingError_InvalidVolumeDataStructure;
								break;
						}
					}
					else
					{
						/* Free B-Tree node */
						++wcb->numSectorsToWipe;
					}
					
					if( IsntErr( err ) )
					{
						VolumeWipeEvent	event;
		
						event.type = kVolumeWipeEvent_GatheringInfo;
	
						err = SendEvent( wcb, &event );
					}
					
					if( IsErr( err ) )
						break;
				
					++btNodeIndex;
					
					nodeDesc = (NodeDescriptor *) ((PGPByte *) nodeDesc +
										wcb->extentsFileInfo.nodeSize);
				}
			}
			
			remainingNodes 	-= nodesToRead;
			readStartNode	+= nodesToRead;
		}
	}
	
	return( err );
}

	static OSStatus
GatherHFSVolumeInfo(WipeControlBlock *wcb)
{
	OSStatus	err;
	
	/* Read the MDB (Sector 2) */
	err = WCBReadSectors( wcb, 2, 1, wcb->ioBuffer );
	if( IsntErr( err ) )
	{
		wcb->hfs.mdb = *((HFSMasterDirectoryBlock *) wcb->ioBuffer);
		
		err = ValidateHFSMasterDirectoryBlock( &wcb->hfs.mdb );
		if( IsntErr( err ) )
		{
			/* Compute some common values based on MDB */
			
			wcb->numBlocks			= wcb->hfs.mdb.drNmAlBlks;
			wcb->numFreeBlocks		= (UInt16) wcb->hfs.mdb.drFreeBks;
			wcb->blockSize 			= wcb->hfs.mdb.drAlBlkSiz;
			wcb->numSectorsPerBlock	= wcb->blockSize / kDiskSectorSize;
			wcb->numVolumeSectors 	= ( wcb->numBlocks *
											wcb->numSectorsPerBlock ) +
												wcb->hfs.mdb.drAlBlSt;
		}
	}
	
	if( IsntErr( err ) )
	{
		/* "Open" BTree files and get extent info for each */
		err = WCBOpenBTreeFile( wcb, wcb->catalogFileInfo.fileRefNum,
						wcb->hfs.mdb.drCTFlSize,
						&wcb->catalogFileInfo );
		if( IsntErr( err ) )
		{
			err = WCBOpenBTreeFile( wcb, wcb->extentsFileInfo.fileRefNum,
						wcb->hfs.mdb.drXTFlSize, &wcb->extentsFileInfo );
		}
	}
	
	return( err );
}

#pragma mark [ --- HFS+ Scanning --- ]

	static OSStatus
ScanHFSPlusExtentRecord(
	const WipeControlBlock 		*wcb,
	const HFSPlusExtentRecord	record,
	PGPUInt32					*foundForkBlocks)
{
	PGPUInt32	extentIndex;
	
	*foundForkBlocks = 0;
	
	for( extentIndex = 0; extentIndex < kHFSPlusExtentDensity; extentIndex++)
	{
		const HFSPlusExtentDescriptor	*extent = &record[extentIndex];
		PGPUInt32						overlappingBlockCount;
		
		if( extent->blockCount != 0 )
		{
			Require( extent->startBlock + extent->blockCount <=
						wcb->numBlocks );
			
			*foundForkBlocks += extent->blockCount;
			
			overlappingBlockCount = pgpCountAndSetBitRange(
											wcb->volumeBitmap,
											extent->startBlock,
											extent->blockCount );
											
			Require( overlappingBlockCount == 0 );
		}
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSPlusFileFork(
	WipeControlBlock 		*wcb,
	PGPUInt32				fileID,
	ForkType				forkType,
	const HFSPlusForkData	*forkData)
{
	OSStatus	err = noErr;
	PGPInt32	extentIndex;
	PGPUInt64	forkPhysicalLength;
	
	forkPhysicalLength = (PGPUInt64) forkData->totalBlocks * wcb->blockSize;
	
	Require( forkData->logicalSize <= forkPhysicalLength );

	if( forkData->totalBlocks != 0 )
	{
		PGPUInt32	foundForkBlocks;

		err = ScanHFSPlusExtentRecord( wcb, forkData->extents,
					&foundForkBlocks );
		if( IsntErr( err ) )
		{
			if( foundForkBlocks == forkData->totalBlocks )
			{
				/*
				** Simplest case. All of the files allocation blocks are
				** accounted for in the catalog. We now check to see if
				** the file's logical EOF is in the last allocation block of
				** the file. If so, we remember the remainder of the last
				** allocation block for wiping later
				*/
				
				if( forkPhysicalLength - forkData->logicalSize <
							wcb->blockSize )
				{
					if( forkPhysicalLength != forkData->logicalSize )
					{
						/*
						** Get the last allocation block and remember the
						** range.
						*/
					
						for( extentIndex = kHFSPlusExtentDensity - 1;
								extentIndex >= 0; --extentIndex )
						{
							HFSPlusExtentDescriptor	desc;
							
							desc = forkData->extents[extentIndex];
							
							if( desc.blockCount != 0 )
							{
								PGPUInt32	lastBlock;
								PGPUInt32	usedBytes;
								PGPUInt32	startSector;
								PGPUInt32	skipSectors;
								PGPUInt32	byteOffset;
								PGPUInt32	numSectors;
								
								lastBlock 	= desc.startBlock +
													desc.blockCount - 1;
								usedBytes	= wcb->blockSize - (
												forkPhysicalLength -
													forkData->logicalSize );
								skipSectors	= usedBytes / kDiskSectorSize;
								byteOffset	= usedBytes % kDiskSectorSize;
								numSectors	= wcb->numSectorsPerBlock -
												skipSectors;
								
								startSector	= WCBBlockToSector( wcb,
													lastBlock );
								startSector	+= skipSectors;
								
								err = RememberWipeRange( wcb, startSector, 
										byteOffset, numSectors );
								
								wcb->numSectorsToWipe += numSectors;
									
								break;
							}
						}
					}
					else
					{
						/*
						** Fork uses all of the allocation block.
						** No wiping needed
						*/
					}
				}
				else
				{
					/*
					** This is a problem file. The logical EOF is more than one
					** allocation block from the physical EOF. Add it to our
					** problem file list
					*/
					
					err = RememberProblemFork( wcb, fileID, forkType,
							forkData->logicalSize, forkPhysicalLength,
							(ExtentRecord *) forkData->extents );
				}
			}
			else if( foundForkBlocks < forkData->totalBlocks )
			{
				/*
				** Not all of the allocation blocks are in the catalog.
				** Remember this file for later analysis when we scan
				** the extents file.
				*/
				
				err = RememberProblemFork( wcb, fileID, forkType,
							forkData->logicalSize, forkPhysicalLength,
							(ExtentRecord *) forkData->extents );
			}
			else
			{
				/* Too many blocks. Damaged. */
				err = kWipingError_InvalidVolumeDataStructure;
			}
		}
	}
	else
	{
		for( extentIndex = 0; extentIndex < kHFSPlusExtentDensity;
				extentIndex++ )
		{
			const HFSPlusExtentDescriptor	*extent;

			extent = &forkData->extents[extentIndex];
			
			Require( extent->startBlock == 0 );
			Require( extent->blockCount == 0 );
		}
	}
	
	return( err );
}

	static OSStatus
ScanHFSPlusCatalogFileRecord(
	WipeControlBlock 			*wcb,
	const HFSPlusCatalogFile	*file)
{	
	OSStatus	err = noErr;
	
	Require( file->recordType == kHFSPlusFileRecord );
	Require( file->fileID < wcb->hfsPlus.vh.nextCatalogID );

	/*
	** Validate that we do not have too many extents and mark the used
	** extents in our volume bitmap.
	*/
	
	err = ScanHFSPlusFileFork( wcb, file->fileID, kDataForkType,
				&file->dataFork );
	if( IsntErr( err ) )
	{
		err = ScanHFSPlusFileFork( wcb, file->fileID, kResourceForkType,
					&file->resourceFork );
	}
		
	return( err );
}

	static OSStatus
ValidateHFSPlusCatalogKey(
	const WipeControlBlock 	*wcb,
	const HFSPlusCatalogKey	*key)
{
	Require( key->parentID < wcb->hfsPlus.vh.nextCatalogID );
	Require( key->nodeName.length <= kHFSPlusMaxFileNameChars );
	Require( key->keyLength >= kHFSPlusCatalogKeyMinimumLength );
	Require( key->keyLength <= kHFSPlusCatalogKeyMaximumLength );

	return( noErr );
}

	static OSStatus
ScanHFSPlusCatalogFileLeafNode(
	WipeControlBlock 		*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	Require( nodeDesc->ndType == kBTreeLeafNodeType );
	Require( nodeDesc->ndNHeight == 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->catalogFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSPlusCatalogKey		*key;
		PGPUInt32				dataOffset;
		HFSPlusCatalogRecord	*data;
		OSStatus				err;
		
		key = (HFSPlusCatalogKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		err = ValidateHFSPlusCatalogKey( wcb, key );
		if( IsErr( err ) )
			return( err );
			
		/* Data is padded to an even boundary */
		dataOffset = key->keyLength + 1;
		if( ( dataOffset & 1 ) != 0 )
			++dataOffset;
			
		data = (HFSPlusCatalogRecord *) ( (PGPByte *) key + dataOffset );
		switch( data->type )
		{
			OSStatus	err;
			
			case kHFSPlusFileRecord:
				err = ScanHFSPlusCatalogFileRecord( wcb, &data->file );
				if( IsErr( err ) )
					return( err );
				break;
				
			case kHFSPlusFolderRecord:
				Require( data->folder.folderID <
					wcb->hfsPlus.vh.nextCatalogID );
				break;
				
			case kHFSPlusFolderThreadRecord:
			case kHFSPlusFileThreadRecord:
				Require( data->thread.parentID <
					wcb->hfsPlus.vh.nextCatalogID );
				Require( data->thread.nodeName.length <=
					kHFSPlusMaxFileNameChars );
				break;
				
			default:
				pgpDebugMsg( "Invalid leaf record type" );
				return( kWipingError_InvalidVolumeDataStructure );
				break;
		}
			
		--recordOffsetPtr;
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSPlusCatalogFileIndexNode(
	const WipeControlBlock 	*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	Require( nodeDesc->ndType == kBTreeIndexNodeType );
	Require( nodeDesc->ndNHeight != 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->catalogFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSPlusCatalogKey	*key;
		OSStatus			err;
		PGPUInt32			nodeIndexOffset;
		UInt32				*nodeIndexPtr;
		
		key = (HFSPlusCatalogKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		err = ValidateHFSPlusCatalogKey( wcb, key );
		if( IsErr( err ) )
			return( err );
		
		/* Data is padded to an even boundary */
		nodeIndexOffset = key->keyLength + 1;
		if( ( nodeIndexOffset & 1 ) != 0 )
			++nodeIndexOffset;
			
		nodeIndexPtr = (UInt32 *) ( (PGPByte *) key + nodeIndexOffset );
		
		Require( *nodeIndexPtr > 0 );
		Require( *nodeIndexPtr < wcb->catalogFileInfo.numNodes );
		Require( pgpTestBit( wcb->catalogFileInfo.nodeBitmap,
					*nodeIndexPtr ) );
				
		--recordOffsetPtr;
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSPlusCatalogFile(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	/* Account for catalog allocation blocks. */
	
	err = ScanHFSPlusFileFork( wcb, kHFSCatalogFileID, kDataForkType,
				&wcb->hfsPlus.vh.catalogFile );
	if( IsntErr( err ) )
	{
		PGPUInt32	remainingNodes;
		PGPUInt32	btNodeIndex;
		PGPUInt32	readStartNode;
		PGPUInt32	numHeaderNodes;
		PGPUInt32	sectorsPerNode;
		
		/*
		** Loop over all records in the catalog. For each record, we validate
		** the format of each record and set bits in the volume bit map. Error
		** if there are already any bits set in the volume bit map.
		*/
		
		btNodeIndex		= 0;
		readStartNode	= 0;
		numHeaderNodes	= 0;
		sectorsPerNode	= wcb->catalogFileInfo.nodeSize / kDiskSectorSize;
		
		remainingNodes = wcb->catalogFileInfo.numNodes;
		while( remainingNodes != 0 && IsntErr( err ) )
		{
			PGPUInt32	nodesToRead;
			
			nodesToRead = wcb->ioBufferSize / wcb->catalogFileInfo.nodeSize;
			if( nodesToRead > remainingNodes )
				nodesToRead = remainingNodes;
				
			err = WCBReadBTreeNodes( wcb, &wcb->catalogFileInfo, 
						readStartNode, nodesToRead, wcb->ioBuffer );
			if( IsntErr( err ) )
			{
				PGPUInt32		nodeIndex;
				NodeDescriptor	*nodeDesc = (NodeDescriptor *) wcb->ioBuffer;
				
				for( nodeIndex = 0; nodeIndex < nodesToRead; nodeIndex++ )
				{
					if( pgpTestBit( wcb->catalogFileInfo.nodeBitmap,
								btNodeIndex ) )
					{
						switch( nodeDesc->ndType )
						{
							case kBTreeLeafNodeType:
								err = ScanHFSPlusCatalogFileLeafNode( wcb,
											nodeDesc );
								wcb->numSectorsToWipe += sectorsPerNode;
								break;
								
							case kBTreeIndexNodeType:
								err = ScanHFSPlusCatalogFileIndexNode( wcb,
											nodeDesc );
								wcb->numSectorsToWipe += sectorsPerNode;
								break;
							
							case kBTreeMapNodeType:
								/* Don't wipe or validate map nodes */
								break;
								
							case kBTreeHeaderNodeType:
								/* Already validated header node */
								++numHeaderNodes;
								if( numHeaderNodes > 1 )
								{
								pgpDebugMsg(
								"Found multiple catalogB-Tree header nodes" );
								err = kWipingError_InvalidVolumeDataStructure;
								}
								break;
						
							default:
								pgpDebugMsg( "Unknown catalog node type" );
								err = kWipingError_InvalidVolumeDataStructure;
								break;
						}
					}
					else
					{
						/* Free B-Tree node */
						wcb->numSectorsToWipe += sectorsPerNode;
					}
					
					if( IsntErr( err ) )
					{
						VolumeWipeEvent	event;
		
						event.type = kVolumeWipeEvent_GatheringInfo;
	
						err = SendEvent( wcb, &event );
					}
					
					if( IsErr( err ) )
						break;
				
					++btNodeIndex;
					
					nodeDesc = (NodeDescriptor *) ((PGPByte *) nodeDesc +
										wcb->catalogFileInfo.nodeSize);
				}
			}
			
			remainingNodes 	-= nodesToRead;
			readStartNode	+= nodesToRead;
		}
	}
	
	return( err );
}

	static OSStatus
ScanHFSPlusExtentsFileLeafNode(
	const WipeControlBlock 	*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	OSStatus	err = noErr;
	
	Require( nodeDesc->ndType == kBTreeLeafNodeType );
	Require( nodeDesc->ndNHeight == 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->extentsFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSPlusExtentKey		*key;
		HFSPlusExtentDescriptor	*data;
		PGPUInt32				foundForkBlocks;
		ProblemForkInfo			*forkInfo;
		
		key = (HFSPlusExtentKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		Require( key->keyLength == kHFSPlusExtentKeyMaximumLength );
		Require( key->forkType == kDataForkType ||
					key->forkType == kResourceForkType );
		Require( key->fileID < wcb->hfsPlus.vh.nextCatalogID );
	
		data = (HFSPlusExtentDescriptor *) ( (PGPByte *) key +
					key->keyLength + 2 );
		
		/* Find the fork in the problem file list. */
		forkInfo = FindProblemForkInfo( wcb, key->fileID, key->forkType );
		if( IsntNull( forkInfo ) )
		{
			err = ScanHFSPlusExtentRecord( wcb, data, &foundForkBlocks );
			if( IsntErr( err ) )
			{
				/* Accumulate these extents into the problem fork list */
				err = RememberProblemForkExtents( forkInfo, key->startBlock,
							(ExtentRecord *) data );
			}
		}
		else
		{
			pgpDebugMsg("\pFound extent record with no matching catalog entry");
			err = kWipingError_InvalidVolumeDataStructure;
		}
		
		if( IsErr( err ) )
			break;
			
		--recordOffsetPtr;
	}
	
	return( err );
}

	static OSStatus
ScanHFSPlusExtentsFileIndexNode(
	const WipeControlBlock 	*wcb,
	const NodeDescriptor 	*nodeDesc)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	Require( nodeDesc->ndType == kBTreeIndexNodeType );
	Require( nodeDesc->ndNHeight != 1 );
	Require( nodeDesc->ndNRecs != 0 );

	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc +
								wcb->extentsFileInfo.nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSPlusExtentKey	*key;
		PGPUInt32			nodeIndexOffset;
		UInt32				*nodeIndexPtr;
		
		key = (HFSPlusExtentKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		Require( key->keyLength == kHFSExtentKeyMaximumLength );
		Require( key->forkType == kDataForkType ||
					key->forkType == kResourceForkType );
		Require( key->fileID < wcb->hfsPlus.vh.nextCatalogID );
					
		/* Data is padded to an even boundary */
		nodeIndexOffset = key->keyLength + 1;
		if( ( nodeIndexOffset & 1 ) != 0 )
			++nodeIndexOffset;
			
		nodeIndexPtr = (UInt32 *) ( (PGPByte *) key + nodeIndexOffset );
		
		Require( *nodeIndexPtr > 0 );
		Require( *nodeIndexPtr < wcb->extentsFileInfo.numNodes );
		Require( pgpTestBit( wcb->extentsFileInfo.nodeBitmap,
					*nodeIndexPtr ) );
					
		--recordOffsetPtr;
	}
	
	return( noErr );
}

	static OSStatus
ScanHFSPlusExtentsFile(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	/* Account for extents file allocation blocks. */
	
	err = ScanHFSPlusFileFork( wcb, kHFSExtentsFileID, kDataForkType,
				&wcb->hfsPlus.vh.extentsFile );
	
	if( IsntErr( err ) )
	{
		PGPUInt32	remainingNodes;
		PGPUInt32	btNodeIndex;
		PGPUInt32	readStartNode;
		PGPUInt32	numHeaderNodes;
		PGPUInt32	sectorsPerNode;
		
		/*
		** Loop over all records in the catalog. For each record, we validate
		** the format of each record and set bits in the volume bit map. Error
		** if there are already any bits set in the volume bit map.
		*/
		
		btNodeIndex		= 0;
		readStartNode	= 0;
		numHeaderNodes	= 0;
		sectorsPerNode	= wcb->extentsFileInfo.nodeSize / kDiskSectorSize;
		
		remainingNodes = wcb->extentsFileInfo.numNodes;
		while( remainingNodes != 0 && IsntErr( err ) )
		{
			PGPUInt32	nodesToRead;
			
			nodesToRead = wcb->ioBufferSize / wcb->extentsFileInfo.nodeSize;
			if( nodesToRead > remainingNodes )
				nodesToRead = remainingNodes;
				
			err = WCBReadBTreeNodes( wcb, &wcb->extentsFileInfo, 
						readStartNode, nodesToRead, wcb->ioBuffer );
			if( IsntErr( err ) )
			{
				PGPUInt32		nodeIndex;
				NodeDescriptor	*nodeDesc = (NodeDescriptor *) wcb->ioBuffer;
				
				for( nodeIndex = 0; nodeIndex < nodesToRead; nodeIndex++ )
				{
					if( pgpTestBit( wcb->extentsFileInfo.nodeBitmap,
							btNodeIndex ) )
					{
						switch( nodeDesc->ndType )
						{
							case kBTreeLeafNodeType:
								err = ScanHFSPlusExtentsFileLeafNode( wcb,
											nodeDesc );
								wcb->numSectorsToWipe += sectorsPerNode;
								break;
								
							case kBTreeIndexNodeType:
								err = ScanHFSPlusExtentsFileIndexNode( wcb,
											nodeDesc );
								wcb->numSectorsToWipe += sectorsPerNode;
								break;
							
							case kBTreeMapNodeType:
								/* Don't wipe or validate map nodes */
								break;
								
							case kBTreeHeaderNodeType:
								/* Already validated header node */
								++numHeaderNodes;
								if( numHeaderNodes > 1 )
								{
								pgpDebugMsg(
								"Found multiple extents-Tree header nodes" );
								err = kWipingError_InvalidVolumeDataStructure;
								}
								break;
						
							default:
								pgpDebugMsg( "Unknown catalog node type" );
								err = kWipingError_InvalidVolumeDataStructure;
								break;
						}
					}
					else
					{
						/* Free B-Tree node */
						wcb->numSectorsToWipe += sectorsPerNode;
					}
					
					if( IsntErr( err ) )
					{
						VolumeWipeEvent	event;
		
						event.type = kVolumeWipeEvent_GatheringInfo;
	
						err = SendEvent( wcb, &event );
					}
					
					if( IsErr( err ) )
						break;
				
					++btNodeIndex;
					
					nodeDesc = (NodeDescriptor *) ((PGPByte *) nodeDesc +
										wcb->extentsFileInfo.nodeSize);
				}
			}
			
			remainingNodes 	-= nodesToRead;
			readStartNode	+= nodesToRead;
		}
	}
	
	return( err );
}

	static OSStatus
GatherHFSPlusVolumeInfo(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	/*
	** The volume may be an HFS+ volume in an HFS wrapper or an entire
	** HFS+ volume. Read the MDB (Sector 2)
	*/
	
	err = WCBReadSectors( wcb, 2, 1, wcb->ioBuffer );
	if( IsntErr( err ) )
	{
		HFSMasterDirectoryBlock	*mdb;
		
		mdb = (HFSMasterDirectoryBlock *) wcb->ioBuffer;
		
		if( mdb->drSigWord == kHFSSigWord )
		{
			err = ValidateHFSMasterDirectoryBlock( mdb );
			if( IsntErr( err ) )
			{
				PGPUInt32	sectorsPerBlock;
				
				/*
				** The MDB on an HFS+ wrapped volume will tell us where
				** the HFS+ volume starts
				*/
				
				Require( mdb->drEmbedSigWord == kHFSPlusSigWord );
				
				sectorsPerBlock = mdb->drAlBlkSiz / kDiskSectorSize;
				
				wcb->firstSectorBias = ( mdb->drEmbedExtent.startBlock *
											sectorsPerBlock ) + mdb->drAlBlSt;
			}
			
			if( IsntErr( err ) )
			{
				/*
				** Now that wcb->firstSectorBias is set up correctly,
				** read sector 2 again to get the HFSPlusVolumeHeader
				*/
				
				err = WCBReadSectors( wcb, 2, 1, wcb->ioBuffer );
			}
		}
		
		if( IsntErr( err ) )
		{
			wcb->hfsPlus.vh = *((HFSPlusVolumeHeader *) wcb->ioBuffer);
			
			err = ValidateHFSPlusVolumeHeader( &wcb->hfsPlus.vh );
			if( IsntErr( err ) )
			{
				/* Compute some common values based on MDB */
				
				wcb->numBlocks			= wcb->hfsPlus.vh.totalBlocks;
				wcb->numFreeBlocks		= wcb->hfsPlus.vh.freeBlocks;
				wcb->blockSize 			= wcb->hfsPlus.vh.blockSize;
				wcb->numSectorsPerBlock	= wcb->blockSize / kDiskSectorSize;
				wcb->numVolumeSectors 	= wcb->numSectorsPerBlock *
											wcb->numBlocks;
			}
		}
	}

	if( IsntErr( err ) )
	{
		/* "Open" BTree files and get extent info for each */
		err = WCBOpenBTreeFile( wcb, wcb->catalogFileInfo.fileRefNum,
						wcb->hfsPlus.vh.catalogFile.logicalSize,
						&wcb->catalogFileInfo );
		if( IsntErr( err ) )
		{
			err = WCBOpenBTreeFile( wcb, wcb->extentsFileInfo.fileRefNum,
						wcb->hfsPlus.vh.extentsFile.logicalSize,
						&wcb->extentsFileInfo );
		}
	}

	return( err );
}

#pragma mark [ --- Common Wiping --- ]

	static void
WipeMemory(
	WipeControlBlock	*wcb,
	PGPUInt32			wipeCount,
	PGPByte				*memory,
	PGPUInt32			patternOffset)
{
	PGPUInt32	bytesRemaining;
	PGPByte		*cur;
	PGPByte		*curPattern;
	PGPUInt32	bytesToWipe;
	
	pgpAssertAddrValid( wcb, WipeControlBlock );
	pgpAssertAddrValid( memory, PGPByte );
	pgpAssert( kNumPGPWipingPatternBytes == sizeof( PGPWipingPattern ) );
	
	if( wipeCount == 0 )
		return;
	
	if( patternOffset != 0 )
		patternOffset = patternOffset % kNumPGPWipingPatternBytes;
		
	cur				= memory;
	curPattern		= &wcb->patternBuffer->bytes[patternOffset];
	bytesToWipe		= kNumPGPWipingPatternBytes - patternOffset;
	bytesRemaining	= wipeCount;
	
	while( bytesRemaining != 0 )
	{
		if( bytesToWipe > bytesRemaining )
			bytesToWipe = bytesRemaining;
			
		pgpCopyMemory( curPattern, cur, bytesToWipe );
	
		cur				+= bytesToWipe;
		bytesRemaining	-= bytesToWipe;
		curPattern		= &wcb->patternBuffer->bytes[0];
		bytesToWipe 	= kNumPGPWipingPatternBytes;
	}
}

/* This procedure assumes that wcb->ioBuffer contains the wiping pattern */

	static OSStatus
WipeSectors(
	WipeControlBlock	*wcb,
	PGPUInt32			startingSector,
	PGPUInt32			numSectors,
	PGPUInt32			firstSectorOffset)
{
	OSStatus	err = noErr;
	PGPUInt32	sectorsPerBuffer;
	PGPUInt32	curSector;
	
	if( numSectors == 0 )
		return( noErr );
	
	curSector 			= startingSector;
	sectorsPerBuffer 	= wcb->ioBufferSize / kDiskSectorSize;
	
	if( firstSectorOffset != 0 )
	{
		pgpAssert( firstSectorOffset < kDiskSectorSize );
		
		err = WCBReadSectors( wcb, curSector, 1, wcb->sectorBuffer );
		if( IsntErr( err ) )
		{
			WipeMemory( wcb, kDiskSectorSize - firstSectorOffset,
							wcb->sectorBuffer + firstSectorOffset,
							firstSectorOffset );
			
			err = WCBWriteSectors( wcb, startingSector, 1, wcb->sectorBuffer );
			if( IsntErr( err ) )
			{
				++wcb->numSectorsWiped;
				--numSectors;
				++curSector;
			
				err = DoProgressEvent( wcb );
			}
		}
	}
	
	if( IsntErr( err ) )
	{
		while( numSectors != 0 )
		{
			PGPUInt32	sectorsToWipe;
			
			sectorsToWipe = numSectors;
			if( sectorsToWipe > sectorsPerBuffer )
				sectorsToWipe = sectorsPerBuffer;
				
			err = WCBWriteSectors( wcb, curSector, sectorsToWipe,
							wcb->ioBuffer );
			if( IsntErr( err ) )
			{
				wcb->numSectorsWiped += sectorsToWipe;
				
				err = DoProgressEvent( wcb );
			}
			
			if( IsErr( err ) )
				break;
			
			numSectors 	-= sectorsToWipe;
			curSector	+= sectorsToWipe;
		}
	}
	
	return( err );
}

	static OSStatus
WipeFreeBlocks(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	PGPUInt32	blockIndex;
	
	/* Fill the ioBuffer with the wiping pattern */
	pgpCopyPattern( &wcb->patternBuffer->bytes[0],
				kNumPGPWipingPatternBytes, wcb->ioBuffer,
				wcb->ioBufferSize );
	
	for( blockIndex = 0; blockIndex < wcb->numBlocks; ++blockIndex )
	{
		if( ! pgpTestBit( wcb->volumeBitmap, blockIndex ) )
		{
			PGPUInt32	usedBlockIndex;
			PGPUInt32	startingSector;
			PGPUInt32	numSectors;
			
			/* Find next used block */	
	
			usedBlockIndex = blockIndex + 1;
			while( usedBlockIndex < wcb->numBlocks )
			{
				if( pgpTestBit( wcb->volumeBitmap, usedBlockIndex ) )
				{
					break;
				}
				else
				{
					++usedBlockIndex;
				}
			}
			
			/* Convert alloc blocks to disk blocks */
			
			startingSector 	= WCBBlockToSector( wcb, blockIndex );
			numSectors		= ( usedBlockIndex - blockIndex ) *
									wcb->numSectorsPerBlock;
			
			err = WipeSectors( wcb, startingSector, numSectors, 0 );
			if( IsErr( err ) )
				break;
			
			/* for() loop will increment blockIndex */
			blockIndex = usedBlockIndex; 
		}
	}
	
	return( err );
}

	static OSStatus
WipeRanges(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	PGPUInt32	rangeIndex;
	
	/* Fill the ioBuffer with the wiping pattern */
	pgpCopyPattern( &wcb->patternBuffer->bytes[0],
				kNumPGPWipingPatternBytes, wcb->ioBuffer,
				wcb->ioBufferSize );
	
	/* Sort the range list for better seeking performance */
	if( IsntNull( wcb->wipeRangeList ) )
	{
		SortWipeRanges( wcb->wipeRangeList, wcb->numWipeRanges,
					CompareWipeRanges, NULL );
	}
	
	for( rangeIndex = 0; rangeIndex < wcb->numWipeRanges; ++rangeIndex )
	{
		WipeRange	*range = &wcb->wipeRangeList[rangeIndex];
		
		err = WipeSectors( wcb, range->startSector, range->numSectors,
					range->firstSectorOffset );
		if( IsErr( err ) )
			break;
	}

	return( err );
}

	static void
WipeBTreeNodeFreeSpace(
	WipeControlBlock 	*wcb,
	NodeDescriptor 		*nodeDesc,
	PGPUInt32			nodeSize)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	PGPByte		*wipeStart;
	PGPByte		*wipeEndPlusOne;
	PGPUInt32	wipeCount;
	PGPUInt32	patternOffset;
	
	/* Index past the last record so we can wipe the remaining free space */
	
	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc + nodeSize - 2);
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		--recordOffsetPtr;
	}
	
	/*
	** Wipe remaining free bytes in node, if any. *recordOffsetPtr points to
	** start of free space in node or itself if there is no free space
	*/
	
	wipeStart 		= (PGPByte *) nodeDesc + *recordOffsetPtr;
	wipeEndPlusOne	= (PGPByte *) recordOffsetPtr;
	
	if( wipeStart < wipeEndPlusOne )
	{
		wipeCount 		= wipeEndPlusOne - wipeStart;
		patternOffset	= wipeStart - (PGPByte *) nodeDesc;
		
		WipeMemory( wcb, wipeCount, wipeStart, patternOffset );
	}
}

	static OSStatus
WipeBTreeFile(
	WipeControlBlock 		*wcb,
	const BTreeFileInfo		*bTreeInfo,
	WipeBTreeNodeProcPtr	wipeNodeProc)
{
	OSStatus	err = noErr;
	PGPUInt32	remainingNodes;
	PGPUInt32	btNodeIndex;
	PGPUInt32	readStartNode;
	PGPUInt32	sectorsPerNode;
	
	/*
	** Loop over all records in the catalog. For each record, we call the
	** node wiping proc for leaf and index nodes.
	*/
	
	btNodeIndex		= 0;
	readStartNode	= 0;
	sectorsPerNode	= bTreeInfo->nodeSize / kDiskSectorSize;
	
	remainingNodes = bTreeInfo->numNodes;
	while( remainingNodes != 0 && IsntErr( err ) )
	{
		PGPUInt32	nodesToRead;
		
		nodesToRead = wcb->ioBufferSize / bTreeInfo->nodeSize;
		if( nodesToRead > remainingNodes )
			nodesToRead = remainingNodes;
			
		err = WCBReadBTreeNodes( wcb, bTreeInfo, readStartNode, nodesToRead,
						wcb->ioBuffer );
		if( IsntErr( err ) )
		{
			PGPUInt32		nodeIndex;
			NodeDescriptor	*nodeDesc = (NodeDescriptor *) wcb->ioBuffer;
			
			for( nodeIndex = 0; nodeIndex < nodesToRead; nodeIndex++ )
			{
				if( pgpTestBit( bTreeInfo->nodeBitmap, btNodeIndex ) )
				{
					switch( nodeDesc->ndType )
					{
						case kBTreeLeafNodeType:
						case kBTreeIndexNodeType:
							(*wipeNodeProc)( wcb, nodeDesc,
												bTreeInfo->nodeSize );
							wcb->numSectorsWiped += sectorsPerNode;
							break;
						
						case kBTreeMapNodeType:
						case kBTreeHeaderNodeType:
							/* Don't wipe header or map nodes */
							break;
					
						default:
							pgpDebugMsg( "Unknown wipe node type" );
							break;
					}
				}
				else
				{
					WipeMemory( wcb, bTreeInfo->nodeSize,
										(PGPByte *) nodeDesc, 0 );
					wcb->numSectorsWiped += sectorsPerNode;
				}
				
				err = DoProgressEvent( wcb );
				if( IsErr( err ) )
					break;
			
				++btNodeIndex;
				
				nodeDesc = (NodeDescriptor *) ((PGPByte *) nodeDesc +
									bTreeInfo->nodeSize);
			}
			
			if( IsntErr( err ) )
			{
				err = WCBWriteBTreeNodes( wcb, bTreeInfo, readStartNode,
								nodesToRead, wcb->ioBuffer );
			}
		}
		
		remainingNodes 	-= nodesToRead;
		readStartNode	+= nodesToRead;
	}
	
	return( err );
}


#pragma mark [ --- HFS Wiping --- ]

	static void
WipeHFSCatalogFileLeafNode(
	WipeControlBlock	*wcb,
	NodeDescriptor 		*nodeDesc,
	PGPUInt32			nodeSize)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	PGPByte		*wipeStart;
	PGPUInt32	wipeCount;
	PGPUInt32	patternOffset;
	
	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc + nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSCatalogKey		*key;
		PGPUInt32			dataOffset;
		HFSCatalogRecord	*data;
		
		key = (HFSCatalogKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		/* Data is padded to an even boundary */
		dataOffset = key->keyLength + 1;
		if( ( dataOffset & 1 ) != 0 )
			++dataOffset;
			
		data = (HFSCatalogRecord *) ( (PGPByte *) key + dataOffset );
		switch( data->type )
		{
			case kHFSFileRecord:
			case kHFSFolderRecord:
				/* No free space to wipe in these records */
				break;
				
			case kHFSFolderThreadRecord:
			case kHFSFileThreadRecord:
			{
				PGPUInt32	nameLength;
				
				nameLength	= data->thread.nodeName[0];
				wipeCount 	= kHFSMaxFileNameChars - nameLength;
				
				if( wipeCount > 0 )
				{
					wipeStart 		= &data->thread.nodeName[nameLength] + 1;
					patternOffset	= wipeStart - (PGPByte *) nodeDesc;
					
					WipeMemory( wcb, wipeCount, wipeStart, patternOffset );
				}
			
				break;
			}
				
			default:
				pgpDebugMsg( "Invalid leaf record type" );
				break;
		}
			
		--recordOffsetPtr;
	}
	
	WipeBTreeNodeFreeSpace( wcb, nodeDesc, nodeSize );
}

	static void
WipeHFSCatalogFileIndexNode(
	WipeControlBlock 	*wcb,
	NodeDescriptor 		*nodeDesc,
	PGPUInt32			nodeSize)
{
	PGPUInt16	*recordOffsetPtr;
	PGPUInt32	recordIndex;
	
	recordOffsetPtr = (PGPUInt16 *) ((PGPByte *) nodeDesc + nodeSize - 2 );
	
	for( recordIndex = 0; recordIndex < nodeDesc->ndNRecs; recordIndex++ )
	{
		HFSCatalogKey	*key;
		PGPUInt32		nameLength;
		PGPUInt32		wipeCount;
		
		key = (HFSCatalogKey *) ((PGPByte *) nodeDesc + *recordOffsetPtr);
		
		nameLength	= key->nodeName[0];
		wipeCount 	= kHFSMaxFileNameChars - nameLength;
		
		if( wipeCount > 0 )
		{
			PGPByte		*wipeStart;
			PGPUInt32	patternOffset;

			wipeStart 		= &key->nodeName[nameLength] + 1;
			patternOffset	= wipeStart - (PGPByte *) nodeDesc;
			
			WipeMemory( wcb, wipeCount, wipeStart, patternOffset );
		}
			
		--recordOffsetPtr;
	}
	
	WipeBTreeNodeFreeSpace( wcb, nodeDesc, nodeSize );
}

	static void
WipeHFSCatalogFileNode(
	WipeControlBlock 	*wcb,
	NodeDescriptor		*nodeDesc,
	PGPUInt32			nodeSize)
{
	if( nodeDesc->ndType == kBTreeLeafNodeType )
	{
		WipeHFSCatalogFileLeafNode( wcb, nodeDesc, nodeSize );
	}
	else if( nodeDesc->ndType == kBTreeIndexNodeType )
	{
		WipeHFSCatalogFileIndexNode( wcb, nodeDesc, nodeSize );
	}
	else
	{
		pgpDebugMsg( "WipeHFSCatalogNode(): Invalid node type" );
	}
}

#pragma mark [ --- Processing --- ]

	static OSStatus
BeginWipe(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	UInt32		bufferSize;
		
	/* Allocate our working buffer */
	wcb->ioBuffer = (PGPByte *) pgpNewPtrMost( kMaxIOBufferSize,
										kMinIOBufferSize,
										kMemoryAllocationFlags,
										&bufferSize );
	if( IsntNull( wcb->ioBuffer ) )
	{
		wcb->ioBufferSize = bufferSize;
	}
	else
	{
		err = memFullErr;
	}
	
	if( IsntErr( err ) )
	{
		if( wcb->signature == kHFSSigWord )
		{
			err = GatherHFSVolumeInfo( wcb );
		}
		else if( wcb->signature == kHFSPlusSigWord )
		{
			err = GatherHFSPlusVolumeInfo( wcb );
		}
		else
		{
			pgpDebugMsg( "Unknown volume signature" );
			err = kWipingError_UnsupportedVolumeFormat;
		}
	}
	
	if( IsntErr( err ) )
	{
		PGPSize	allocationSize;
		
		/* Allocate volume bit map */
		
		allocationSize = ( wcb->numBlocks + 7 ) / 8;
		
		wcb->volumeBitmap = (PGPByte *) pgpAllocMac( allocationSize,
												kMemoryAllocationFlags |
												kMacMemory_ClearBytes);
		if( IsNull( wcb->volumeBitmap ) )
			err = memFullErr;
	}
	
	if( IsntErr( err ) )
	{
		/* Try to unmount volume. An error is fatal if the volume is HFS+ */
		
		err = UnmountVol( NULL, wcb->vRefNum );
		if( IsntErr( err ) )
		{
			wcb->vRefNum = 0;
			
			wcb->catalogFileInfo.fileRefNum	= 0;
			wcb->extentsFileInfo.fileRefNum	= 0;
		}
		else
		{
			VolumeWipeEvent	event;
			OSStatus		tempErr;
			
			event.type = kVolumeWipeEvent_VolumeUnmountFailure;
			event.unmountFailure.vRefNum		= wcb->vRefNum;
			event.unmountFailure.unmountError	= err;
			
			tempErr = SendEvent( wcb, &event );
			
			if( wcb->signature == kHFSSigWord )
			{
				/* Not fatal if we cannot unmount HFS volume*/
				err = tempErr;
			}
			else
			{
				/* Propogate error */
			}
		}
	}
	
	if( IsntErr( err ) )
	{
		if( wcb->signature == kHFSSigWord )
		{
			err = ScanHFSCatalogFile( wcb );
			if( IsntErr( err ) )
			{
				err = ScanHFSExtentsFile( wcb );
			}
		}
		else if( wcb->signature == kHFSPlusSigWord )
		{
			err = ScanHFSPlusCatalogFile( wcb );
			if( IsntErr( err ) )
			{
				err = ScanHFSPlusExtentsFile( wcb );
			}
			
			/*
			** Account for extra HFS+ volume format files. Note that the
			** attributes file is not accounted for because
			** GatherHFSPlusVolumeInfo*() should have already rejected a 
			** non-empty attributes file
			*/
			
			if( IsntErr( err ) )
			{
				err = ScanHFSPlusFileFork( wcb, kHFSAllocationFileID,
						kDataForkType, &wcb->hfsPlus.vh.allocationFile );
				if( IsntErr( err ) )
				{
					err = ScanHFSPlusFileFork( wcb, kHFSStartupFileID,
						kDataForkType, &wcb->hfsPlus.vh.startupFile );
				}
			}
			
			/*
			** Finally, account for volume header and alternate header.
			** These items are in the allocation block space but are
			** not actually noted in the volume header
			*/
			
			if( IsntErr( err ) )
			{
				PGPUInt32	numHeaderBlocks;
				PGPUInt32	numAltHeaderBlocks;
				
				numHeaderBlocks = (2 + wcb->numSectorsPerBlock ) / 
										wcb->numSectorsPerBlock;
				pgpSetBitRange( wcb->volumeBitmap, 0, 
						numHeaderBlocks );
						
				numAltHeaderBlocks = (wcb->numSectorsPerBlock + 1) / 
										wcb->numSectorsPerBlock;
				pgpSetBitRange( wcb->volumeBitmap,
						wcb->numBlocks - numAltHeaderBlocks,
						numAltHeaderBlocks );
			}
		}
	}

	if( IsntErr( err ) && wcb->numProblemForks != 0 )
	{
		err = ValidateProblemForkList( wcb );
	}

	if( IsntErr( err ) )
	{
		PGPBitCount	numFreeBlocks;
		
		/*
		** Verify that the number of free blocks matches the count from our
		** volume bitmap
		*/
		
		numFreeBlocks = pgpCountClearBitsInRange( wcb->volumeBitmap,
										0, wcb->numBlocks );
		if( numFreeBlocks == wcb->numFreeBlocks )
		{
			/* Account for the free allocation blocks in our wipe count */
			wcb->numSectorsToWipe += numFreeBlocks * 
										wcb->numSectorsPerBlock;
		}
		else
		{
			pgpDebugMsg( "Volume bitmap mismatch" );
			err = kWipingError_InvalidVolumeDataStructure;
		}
	}
	
	return( err );
}

	static void
EndWipe(WipeControlBlock *wcb)
{
	OSStatus		err;

	/* Try to mount volume, even if there was an error */
	if( wcb->vRefNum == 0 )
	{
		ParamBlockRec	pb;
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		pb.volumeParam.ioVRefNum = wcb->driveNumber;
		
		err = PBMountVol( &pb );
		if( IsErr( err ) )
		{
			VolumeWipeEvent	event;
			
			event.type = kVolumeWipeEvent_VolumeMountFailure;
			
			event.mountFailure.driveNumber	= wcb->driveNumber;
			event.mountFailure.driverRefNum	= wcb->driverRefNum;
			event.mountFailure.mountError	= err;
			
			err = SendEvent( wcb, &event );
		}
	}
}

	static OSStatus
InitWipeControlBlock(WipeControlBlock *wcb)
{
	OSStatus	err = noErr;
	
	if( IsNull( wcb ) )
		return( paramErr );
		
	pgpClearMemory( wcb, sizeof( *wcb ) );
	
	wcb->sectorBuffer = (PGPByte *) pgpAllocMac( kDiskSectorSize,
								kMemoryAllocationFlags);
	if( IsntNull( wcb->sectorBuffer ) )
	{
		wcb->patternBuffer = (PGPWipingPattern *) pgpAllocMac(
								kNumPGPWipingPatternBytes,
								kMemoryAllocationFlags);
	}
	else
	{
		err = memFullErr;
	}
	
	return( err );
}

	static void
CleanupWipeControlBlock(WipeControlBlock *wcb)
{
	if( IsntNull( wcb ) )
	{
		if( IsntNull( wcb->sectorBuffer ) )
			pgpFreeMac( wcb->sectorBuffer );
		
		if( IsntNull( wcb->patternBuffer ) )
			pgpFreeMac( wcb->patternBuffer );
		
		if( IsntNull( wcb->volumeBitmap ) )
			pgpFreeMac( wcb->volumeBitmap );
		
		if( IsntNull( wcb->ioBuffer ) )
			pgpFreeMac( wcb->ioBuffer );
		
		if( IsntNull( wcb->wipeRangeList ) )
			pgpFreeMac( wcb->wipeRangeList );
			
		if( IsntNull( wcb->problemForkList ) )
		{
			PGPUInt32	index;
			
			for( index = 0; index < wcb->numProblemForks; index++ )
			{
				ForkExtent	*cur;
				
				cur = wcb->problemForkList[index].extentList;
				while( IsntNull( cur ) )
				{
					ForkExtent	*next;

					next = cur->next;
					pgpFreeMac( cur );
					cur = next;
				}
			}
			
			pgpFreeMac( wcb->problemForkList );
		}
			
		WCBCloseBTreeFile( wcb, &wcb->catalogFileInfo );
		WCBCloseBTreeFile( wcb, &wcb->extentsFileInfo );
			
		pgpClearMemory( wcb, sizeof( *wcb ) );
	}
}

	static OSStatus
DoWipePass(
	WipeControlBlock 	*wcb,
	PGPUInt32			passIndex)
{
	OSStatus		err = noErr;
	OSStatus		tempErr;
	VolumeWipeEvent	event;
	
	wcb->passIndex = passIndex;
	
	event.type 							= kVolumeWipeEvent_BeginPass;
	event.beginPass.passIndex			= passIndex;
	event.beginPass.totalPasses			= wcb->totalPasses;
	event.beginPass.totalSectorsToWipe	= wcb->numSectorsToWipe;
	
	err = SendEvent( wcb, &event );
	if( IsntErr( err ) )
	{
		wcb->numSectorsWiped = 0;	/* Reset our count */
	
		if( wcb->signature == kHFSSigWord )
		{
			err = WipeBTreeFile( wcb, &wcb->catalogFileInfo,
							WipeHFSCatalogFileNode );
			if( IsntErr( err ) )
			{
				err = WipeBTreeFile( wcb, &wcb->extentsFileInfo,
								WipeBTreeNodeFreeSpace );
			}
		}
		else if( wcb->signature == kHFSPlusSigWord )
		{
			err = WipeBTreeFile( wcb, &wcb->catalogFileInfo,
							WipeBTreeNodeFreeSpace );
			if( IsntErr( err ) )
			{
				err = WipeBTreeFile( wcb, &wcb->extentsFileInfo,
								WipeBTreeNodeFreeSpace );
			}
		}
		else
		{
			pgpDebugMsg( "DoWipePass(): Unknown volume type" );
			err = kWipingError_UnsupportedVolumeFormat;
		}
		
		if( IsntErr( err ) )
		{
			err = WipeRanges( wcb );
			if( IsntErr( err ) )
			{
				err = WipeFreeBlocks( wcb );
			}
		}
	}

	if( wcb->vRefNum != 0 )
		(void) FlushVol( NULL, wcb->vRefNum );
	
	event.type 					= kVolumeWipeEvent_EndPass;
	event.endPass.passIndex		= passIndex;
	event.endPass.totalPasses	= wcb->totalPasses;
	event.endPass.passError		= err;
	
	tempErr = SendEvent( wcb, &event );
	if( IsntErr( err ) )
		err = tempErr;
	
	return( err );

}

	OSStatus
pgpWipeVolumeFreeSpace(
	short 						volRefNum,
	ConstStr255Param			volName,	/* Double check (optional)*/
	PGPUInt32 					numPasses,
	PGPUInt32					numPatterns,
	const PGPWipingPattern 		patternList[],
	PGPVolumeWipingEventHandler	eventHandler,
	PGPUserValue				eventUserValue)
{
	OSStatus		err = noErr;
	
	if( volRefNum >= 0 ||
		numPasses == 0 || 
		numPatterns == 0 ||
		IsNull( patternList ) )
	{
		pgpDebugMsg( "pgpWipeVolumeFreeSpace(): Parameter error" );
		return( paramErr );
	}

	if( IsntNull( eventHandler ) )
	{
		VolumeWipeEvent	event;
		
		event.type = kVolumeWipeEvent_Begin;
		
		err = (*eventHandler)( &event, eventUserValue );
	}
	
	if( IsntErr( err ) )
	{	
		err = pgpCanWipeVolumeFreeSpace( volRefNum );
		if( IsntErr( err ) )
		{
			XVolumeParam	xpb;
			Str32			volumeName;

			pgpClearMemory( &xpb, sizeof( xpb ) );
			
			xpb.ioVRefNum = volRefNum;
			xpb.ioNamePtr = volumeName;
			
			err = pgpPBXGetVolInfo( &xpb );
			if( IsntErr( err ) && IsntNull( volName ) )
			{
				if( ! PStringsAreEqual( volName, volumeName ) )
				{
					err = nsvErr;
				}
			}
				
			if( IsntErr( err ) )
			{
				WipeControlBlock	wcb;

				err = InitWipeControlBlock( &wcb );
				if( IsntErr( err ) )
				{
					const VCB	*vcb;
					
					wcb.vRefNum			= volRefNum;
					wcb.driveNumber		= xpb.ioVDrvInfo;
					wcb.driverRefNum	= xpb.ioVDRefNum;
					wcb.eventHandler	= eventHandler;
					wcb.eventUserValue	= eventUserValue;
					wcb.totalPasses		= numPasses;
					
					vcb = GetVCBForVolume( volRefNum );
					pgpAssert( IsntNull( vcb ) );
					
					wcb.catalogFileInfo.fileRefNum	= vcb->vcbCTRef;
					wcb.extentsFileInfo.fileRefNum	= vcb->vcbXTRef;
					wcb.signature					= vcb->vcbSigWord;
					
					(void) FlushVol( NULL, volRefNum );
					
					err = BeginWipe( &wcb );
					if( IsntErr( err ) )
					{
						PGPUInt32	wipeIndex;
						
						for(wipeIndex = 1; wipeIndex <= numPasses; wipeIndex++)
						{
							*wcb.patternBuffer =
									patternList[(wipeIndex - 1) % numPatterns];
							
							err = DoWipePass( &wcb, wipeIndex );
							if( IsErr( err ) )
								break;
						}
					}
					
					EndWipe( &wcb );
				}
				
				CleanupWipeControlBlock( &wcb );
			}
		}
	}
	
	if( IsntNull( eventHandler ) )
	{
		VolumeWipeEvent	event;
		OSStatus		tempErr;
		
		event.type 			= kVolumeWipeEvent_End;
		event.end.wipeError	= err;
		
		tempErr = (*eventHandler)( &event, eventUserValue );
		if( IsntErr( err ) )
			err = tempErr;
	}
	
	pgpAssert( IsntErr( err ) );
		
	return( err );
}