/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	
	
	$Id: PassphraseCache.cp,v 1.15.8.1 1998/11/12 03:09:52 heller Exp $
____________________________________________________________________________*/

#include <Timer.h>
#include <string.h>
 
#include "pgpKeys.h"
#include "pgpUtilities.h"
 
#include "PassphraseCache.h"
#include "pgpMacMemory.h"
#include "MacSecureMemory.h"
#include "MacStrings.h"
#include "MacEnvirons.h"

#include "SetupA4.h"
#include "A4Stuff.h"

const SInt32	kPassbufferSize = 256;

static char			sScrambler[256];
static PGPBoolean	sScramblerInitialized = false;
 
CPassphraseCache::CPassphraseCache( PGPContextRef context )
{
	RememberA4();
	
	const ulong	kDefaultEraseDelayMilliseconds	=
				kDefaultEraseDelaysSeconds * 1000;
	
	mPassbuffer				= nil;
	mPassbufferValid		= false;
	mEraseTaskIsInstalled	= false;
	mEraseDelayMilliseconds	= kDefaultEraseDelayMilliseconds;
	
	mEraseTask.upp	= NewTimerProc( sEraseTask );
	pgpAssert( IsntNull( mEraseTask.upp ) );
	
	mPassbuffer	= NewPtrSys( kPassbufferSize );
	pgpAssert( IsntNull( mPassbuffer ) );
	if ( VirtualMemoryIsOn() && IsntNull( mPassbuffer ) )
	{
		HoldMemory( mPassbuffer, kPassbufferSize );
	}
	
	if (! sScramblerInitialized) {
		PGPContextGetRandomBytes( context, sScrambler, sizeof( char[256] ) );
		sScramblerInitialized = true;
	}
}


CPassphraseCache::~CPassphraseCache( )
{
	RemoveEraseTask();
	
	if ( IsntNull( mPassbuffer ) )
	{
		pgpClearMemory( mPassbuffer, kPassbufferSize );
	}
	
	if ( IsntNull( mEraseTask.upp ) )
	{
		DisposeRoutineDescriptor( mEraseTask.upp );
	}
	
	if ( VirtualMemoryIsOn() && IsntNull( mPassbuffer ) )
	{
		UnholdMemory( mPassbuffer, kPassbufferSize );
		DisposePtr( mPassbuffer );
	}
}


	void
CPassphraseCache::EraseTask( void )
{
	Forget();
}


#if TARGET_RT_MAC_CFM	// [

	pascal void
CPassphraseCache::sEraseTask( MyTMTask *myTask )
{
	pgpAssert( myTask->magic == MyTMTask::kMagic );
	
	myTask->thisObject->EraseTask();
}

#else		// ][

	pascal void
CPassphraseCache::sEraseTask( void )
{
	MyTMTask *	myTask	= (MyTMTask *)GetA1();
	EnterCodeResource();

	pgpAssert( myTask->magic == MyTMTask::kMagic );
	myTask->thisObject->EraseTask();
	
	ExitCodeResource();
}


#endif		// ]


	void
CPassphraseCache::RemoveEraseTask(  )
{
	if ( mEraseTaskIsInstalled )
	{
		RmvTime( (QElemPtr) &mEraseTask.tmTask );
		mEraseTaskIsInstalled	= false;
	}
}


	void
CPassphraseCache::InstallEraseTask(  )
{
	pgpAssert( ! mEraseTaskIsInstalled );
	
	if ( ! mEraseTaskIsInstalled )
	{
		TMTask *		task	= &mEraseTask.tmTask;
		
		pgpClearMemory( task, sizeof( *task ) );
		
		mEraseTask.thisObject	= this;
	#if PGP_DEBUG
		mEraseTask.magic		= MyTMTask::kMagic;
	#endif
		
		task->tmAddr	= mEraseTask.upp;
		InsTime( (QElemPtr) task );
		PrimeTime( (QElemPtr) task, mEraseDelayMilliseconds );
		
		mEraseTaskIsInstalled	= true;
	}
}

	void
CPassphraseCache::RememberData(
	PGPBoolean		isPassphrase,
	const PGPByte 	*data,
	PGPSize			dataLength)
{
	PGPUInt32	i;
	
	mPassbufferValid	= TRUE;
	mIsPassphrase		= isPassphrase;
	mDataLength 		= (dataLength < kPassbufferSize) ? dataLength : kPassbufferSize;
	pgpCopyMemory( data, mPassbuffer, dataLength );
	
	for( i = 0; i < dataLength; i++)
		mPassbuffer[i] ^= sScrambler[i];
}

	
	void
CPassphraseCache::RememberPassphrase( const char *	passphrase )
{
	if ( IsNull( mPassbuffer ) )
		return;
		
	RemoveEraseTask();
	
	if ( mEraseDelayMilliseconds != 0 )
	{
		RememberData( TRUE, (PGPByte *) passphrase, strlen( passphrase ) );
		InstallEraseTask();
	}
	else
	{
		mPassbufferValid	= false;
		pgpClearMemory( mPassbuffer, kPassbufferSize );
	}
}

	
	void
CPassphraseCache::RememberPasskey(
	const void *	passkey,
	PGPSize			passkeyLength )
{
	if ( IsNull( mPassbuffer ) )
		return;
		
	RemoveEraseTask();
	
	if ( mEraseDelayMilliseconds != 0 )
	{
		RememberData( FALSE, (PGPByte *) passkey, passkeyLength );
		InstallEraseTask();
	}
	else
	{
		mPassbufferValid	= false;
		pgpClearMemory( mPassbuffer, kPassbufferSize );
	}
}


	void
CPassphraseCache::Forget( )
{
	if ( IsNull( mPassbuffer ) )
		return;
		
	RemoveEraseTask();
	
	pgpClearMemory( mPassbuffer, kPassbufferSize );
	mPassbufferValid	= false;
}



	Boolean
CPassphraseCache::GetPassphraseOrPasskey(
	void *			buffer,
	PGPBoolean *	isPassphrase,
	PGPSize *		passkeyLength)
{
	*passkeyLength = 0;
	
	if ( mPassbufferValid )
	{
		*isPassphrase = mIsPassphrase;
		
		pgpCopyMemory( mPassbuffer, buffer, mDataLength );
		for (UInt16 i = 0; i < mDataLength; i++)
			((char *) buffer)[i] ^= sScrambler[i];
		
		if ( mIsPassphrase )
		{
			((char *) buffer)[mDataLength] = 0;
		}
		else
		{
			*passkeyLength = mDataLength;
		}
	}
	
	return( mPassbufferValid );
}

	PGPBoolean
CPassphraseCache::GetPGPOptionList(
	PGPContextRef		context,
	PGPOptionListRef 	*optionList)
{
	PGPBoolean			haveData;
	CSecureCString256	buffer;
	PGPBoolean			isPassPhrase;
	PGPSize				passKeyLength;
	
	*optionList = kInvalidPGPOptionListRef;
	
	haveData = GetPassphraseOrPasskey( buffer, &isPassPhrase, &passKeyLength );
	if( haveData )
	{
		if( isPassPhrase )
		{
			*optionList = PGPOPassphrase( context, buffer );
		}
		else
		{
			*optionList = PGPOPasskeyBuffer( context, buffer, passKeyLength );
		}
	}
	
	return( haveData );
}

#pragma mark -


CSignPassphraseCache::CSignPassphraseCache( PGPContextRef context )
	: CPassphraseCache( context )
{
	MacDebug_FillWithGarbage( &mKeyID, sizeof( mKeyID ) );
}


/*____________________________________________________________________________
	Retrieve a cached passphrase, together with its corresponding key.
	Fetch the key using its key ID and its PGPPublicKeyAlgorithm
	
	Caller must call PGPFreeKey() on *signingKeyOut
____________________________________________________________________________*/
	Boolean
CSignPassphraseCache::GetPassphraseOrPasskey(
	PGPKeySetRef	allKeys,
	void *			buffer,
	PGPBoolean *	isPassphrase,
	PGPSize *		passkeyLength,
	PGPKeyRef *		keyRefOut )
{
	Boolean		haveCache	= FALSE;
	PGPError	err	= kPGPError_NoErr;
	
	*keyRefOut	= kInvalidPGPKeyRef;
	((char *) buffer)[ 0 ]	= '\0';
	
	haveCache	= inherited::GetPassphraseOrPasskey( buffer, isPassphrase,
							passkeyLength );
	if ( haveCache )
	{
		err	= PGPGetKeyByKeyID( allKeys,
					&mKeyID, mPublicKeyAlg, keyRefOut );
		
		haveCache	= IsntPGPError( err );
	}
	
	return( haveCache );
}

/*____________________________________________________________________________
	Remember a passphrase, together with its corresponding key.
	Remember the key by turning its key ID into a string and also remembering
	its PGPPublicKeyAlgorithm
____________________________________________________________________________*/
	void
CSignPassphraseCache::RememberPassphrase(
	const char *		passphrase,
	PGPKeyRef			key)
{
	PGPError	err	= kPGPError_NoErr;
	
	pgpAssertAddrValid( passphrase, char );
	
	err	= PGPGetKeyIDFromKey( key, &mKeyID );
	if ( IsntPGPError( err ) )
	{
		PGPInt32	number;
		
		err	= PGPGetKeyNumber( key, kPGPKeyPropAlgID, &number );
		mPublicKeyAlg	= (PGPPublicKeyAlgorithm)number;
	}
	
	pgpAssertNoErr( err );
	
	inherited::RememberPassphrase( passphrase );
}


	void
CSignPassphraseCache::RememberPasskey(
	const void *		passkey,
	PGPSize				passkeyLength,
	PGPKeyRef			key)
{
	PGPError	err	= kPGPError_NoErr;
	
	err	= PGPGetKeyIDFromKey( key, &mKeyID );
	if ( IsntPGPError( err ) )
	{
		PGPInt32	number;
		
		err	= PGPGetKeyNumber( key, kPGPKeyPropAlgID, &number );
		mPublicKeyAlg	= (PGPPublicKeyAlgorithm)number;
	}
	
	pgpAssertNoErr( err );
	
	inherited::RememberPasskey( passkey, passkeyLength );
}

#pragma mark -




CSignPassphraseCache *	gSigningPassphraseCache		= nil;
CSignPassphraseCache *	gDecryptionPassphraseCache	= nil;
	
	void
InitPassphraseCaches( PGPContextRef context )
{
	if ( IsNull( gSigningPassphraseCache ) )
	{
		gSigningPassphraseCache	= new CSignPassphraseCache( context );
	}
	
	if ( IsNull( gDecryptionPassphraseCache ) )
	{
		gDecryptionPassphraseCache	= new CSignPassphraseCache( context );
	}
}

	void
DisposePassphraseCaches( void )
{
	delete gSigningPassphraseCache;
	gSigningPassphraseCache	= nil;
	
	delete gDecryptionPassphraseCache;
	gDecryptionPassphraseCache	= nil;
}
	












