/*____________________________________________________________________________
	Copyright (C) 1996-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: CTuner.cp,v 1.4.8.2 1998/11/13 02:49:39 heller Exp $
____________________________________________________________________________*/

#include <Timer.h>
#include "pgpMacMemory.h"
#include "MacStrings.h"
#include "MacFiles.h"
#include "UInt64.h"
#include <FSM.h>

#include "CTuner.h"
#include "DriverAPI.h"
#include "PGPDisk.h"



static const ushort	kNumIOsForTest	= 10;


#define MIN(a, b)	( (a) <= (b) ? (a) : (b) )
	inline long
ABS( long a )
	{
	return( ( a >= 0 ) ? a : -a );
	}


	
	static void
InitTuner(
	TunerStruct *	tuner,
	ulong			maxBlocks)
	{
	pgpAssert( maxBlocks != 0 );
	
	for( ushort index = 0; index < TunerStruct::kNumTune; ++index )
		{
		tuner->maxBlocks[ index ]	= maxBlocks;
		}
	}
	
	
	static OSErr
ChangeDriverMaxReadBlocks(
	short	vRefNum,
	short	maxBlocks )
	{
	TunerStruct	tuner;
	OSErr		err	= noErr;
	
	InitTuner( &tuner, maxBlocks );
	err	= ChangeDriveTuner( vRefNum, &tuner, IOOptionsStruct::kReadTuner );
	return( err );
	}
	
		
	static OSErr
ChangeDriverMaxWriteBlocks(
	short	vRefNum,
	short	maxBlocks )
	{
	TunerStruct	tuner;
	OSErr		err	= noErr;
	
	InitTuner( &tuner, maxBlocks );
	err	= ChangeDriveTuner( vRefNum, &tuner, IOOptionsStruct::kWriteTuner );
	return( err );
	}
	
		
CTuner::CTuner(void)
	{
	mVRefNum		= 0;
	mDirection		= kReadDirection;
	mBuffer			= nil;
	mFileRefNum		= 0;
	mTestFileSize	= 0;
	mCallback		= nil;
	mNumTests		= 0;
	mCurTestIndex	= 0;
	}
	
	
CTuner::~CTuner( )
	{
	}
	

	static OSErr
AllocateSpaceToFile(
	short	fileRefNum,
	ulong	fileSize)
	{
	OSErr	err	= noErr;
	
	long	count	= fileSize;
	err	= AllocContig( fileRefNum, &count);
	if ( IsntErr( err ) )
		{
		err	= SetEOF( fileRefNum, fileSize );
		}
		
	return( err );
	}
	
	
/*___________________________________________________________________________
	Determine the kilobytes per second that can be transferred using the given
	ioSize and using a file of the specified size.
___________________________________________________________________________*/
	OSStatus
CTuner::TestKPerSecond(
	const short			fileRefNum,
	void *				buffer,
	const ulong			ioSize,
	const ulong			testFileSize,
	ulong *				kPerSec)
	{
	OSStatus	status	= noErr;
	UnsignedWide	startMS;
	UnsignedWide	stopMS;
	
	AssertFileRefNumIsValid( fileRefNum, "TestKPerSecond" );
	pgpAssertAddrValid( buffer, VoidAlign );
	pgpAssert( testFileSize != 0 );
	pgpAssert( ioSize != 0 );
	pgpAssert( ( ioSize % kDiskBlockSize ) == 0 );
	
	(void)SetEOF( fileRefNum, 0UL );
	(void)SetFPos( fileRefNum, fsFromStart, 0 );
	status	= AllocateSpaceToFile( fileRefNum, testFileSize );
	if ( IsntErr( status ) )
		{
		AssertNoErr( status, "CTuner::TestKPerSecond" );
		
		ulong	bytesDone	= 0;
		ulong	bytesToDo	= ( testFileSize / ioSize ) * ioSize;
		pgpAssert( bytesToDo >= ioSize );
		
		Microseconds( &startMS );
		while ( bytesDone < bytesToDo )
			{
			ParamBlockRec	pb;
			
			pgpAssert( ( bytesDone % kDiskBlockSize ) == 0 );
			
			pb.ioParam.ioVRefNum	= 0;
			pb.ioParam.ioRefNum		= fileRefNum;
			pb.ioParam.ioBuffer		= (Ptr)buffer;
			pb.ioParam.ioReqCount	= ioSize;
			pb.ioParam.ioPosMode	= fsFromStart | (1UL << noCacheBit);
			pb.ioParam.ioPosOffset	= bytesDone;
			
			if ( mDirection == kReadDirection )
				{
				status	= PBReadSync( &pb );
				}
			else
				{
				status	= PBWriteSync( &pb );
				}
			
			if ( IsErr( status ) )
				break;
			
			pgpAssert( pb.ioParam.ioActCount == ioSize );
			bytesDone	+= ioSize;
			}
		
		Microseconds( &stopMS );
		
		UInt64	elapsedMS			= (UInt64)stopMS - startMS;
		pgpAssert( elapsedMS.hi == 0 );
		float	elapsedMilliSecs	= elapsedMS.lo / (float)1000;
		float	elapsedSecs			= elapsedMilliSecs / (float)1000;
		
		*kPerSec	= (float)( bytesDone / 1024UL ) / elapsedSecs;
		}
		
	AssertNoErr( status, "CTuner::TestKPerSecond" );
	return( status );
	}




	OSStatus
CTuner::CallCallback( )
	{
	OSStatus	err	= noErr;
	
	if ( IsntNull( mCallback ) )
		{
		ushort	percentDone	= ((ulong)mCurTestIndex * 100 ) / mNumTests;
		
		err	= (*mCallback)( mCallbackData, percentDone );
		}
			
	return( err );
	}
	
	
/*___________________________________________________________________________
	Determine the optimal number of chunks to use for a I/O request of size
	'ioSize' by steadily inceasing the block limit for the driver.
	
	Don't change this routine casually; other algorithms were tried and
	may not work well.
___________________________________________________________________________*/
	OSStatus
CTuner::FindOptimalBlocks(
	const ulong			ioSize,
	const ulong			bestBlocksIn,
	const ulong			bestKPerSecIn,
	ulong *				optimalBlocks)
	{
	OSStatus	err	= noErr;
	ulong		bestBlocks		= bestBlocksIn;
	ulong		bestKPerSecond	= bestKPerSecIn;
	
	pgpAssert( bestBlocks != 0 );
	
	*optimalBlocks	= bestBlocks;
	
	ulong	testBlocks	= bestBlocks;
	
	// loop until an error or until our block limit reaches our ioSize
	while ( IsntErr( err ) )
		{
		ulong	testKPerSecond;
		
		err	= CallCallback();
		if ( IsErr( err ) )
			break;

		if ( mDirection == kReadDirection )
			err	= ChangeDriverMaxReadBlocks( mVRefNum, testBlocks );
		else
			err	= ChangeDriverMaxWriteBlocks( mVRefNum, testBlocks );
		if ( IsErr( err ) )
			break;
		
		err	= TestKPerSecond( mFileRefNum, mBuffer, ioSize, mTestFileSize,
					&testKPerSecond );
		if ( IsErr( err ) )
			break;
			
		if ( testKPerSecond >= bestKPerSecond )
			{
			bestBlocks		= testBlocks;
			bestKPerSecond	= testKPerSecond;
			}
		else
			{
			break;
			
			// if the numbers are with 3% of each other, keep going because
			// it could just be timing variation.
			ulong	difference	= ABS( testKPerSecond - bestKPerSecond );
			
			if ( ( ( difference * 100 ) / bestKPerSecond ) >= 3 )
				{
				break;	// it got significantly slower
				}
			}
		
		// no point in testing a limit with more blocks than our ioSize
		pgpAssert( testBlocks <= ioSize / kDiskBlockSize );
		if ( testBlocks == ioSize / kDiskBlockSize )
			break;
			
		// now increase the number of test blocks 
		if ( testBlocks >= 16 )
			{
			// keep size leaps smaller for more accurate results
			testBlocks	+= testBlocks / 4;
			}
		else
			{
			// IMPORTANT: timing variations with smaller sizes make it
			// important to scale by 2 or algorithm won't work well
			testBlocks	*= 2;
			}
		
		// limit the limit to the size of the I/O
		if ( testBlocks > ioSize / kDiskBlockSize )
			testBlocks	= ioSize / kDiskBlockSize;
		}
	
	*optimalBlocks	= bestBlocks;
					
	return( err );
	}
	
	
	
	OSStatus
CTuner::OpenTestFile(
	FSSpec *	testSpec,
	short *		fileRefNum )
	{
	OSStatus	status;
	
	testSpec->vRefNum	= mVRefNum;
	testSpec->parID		= fsRtDirID;
	CopyPString( "\pPerformanceTestFile\r", testSpec->name );
	
	(void)FSpDelete(testSpec );
	(void)FSpCreate( testSpec, 'TeSt', 'xxxX', smSystemScript );
	status	= FSpOpenDF( testSpec, fsRdWrPerm, fileRefNum );
	
	return( status );
	}
	
	

/*___________________________________________________________________________
	Determine a test file that satisfies the following:
		- fits in the available disk space
___________________________________________________________________________*/
	OSErr
CTuner::DetermineTestFileSize(
	ulong	ioSize,
	ulong *	testFileSize)
	{
	OSErr	err	= noErr;
	
	const ulong	kMegabyte	= 1024UL * 1024UL;
	*testFileSize	= 50 * kMegabyte;
	
#if 0
	if ( *testFileSize > 10 * ioSize )
		*testFileSize	= 10 * ioSize;
#endif
		
	if ( *testFileSize < ioSize )
		*testFileSize	= ioSize;
	
	ulong	blocksOnDisk= GetVolumeFreeAllocBlocks( mVRefNum ) *
								(GetVolumeAllocBlockSize( mVRefNum ) / 512);
	if ( blocksOnDisk > kMaximumBlocksIn4GB )
		blocksOnDisk	= kMaximumBlocksIn4GB;
		
	if ( *testFileSize > blocksOnDisk * kDiskBlockSize )
		{
		*testFileSize	= blocksOnDisk * kDiskBlockSize;
		}
		
	return( err );
	}




/*___________________________________________________________________________
	Determine the optimal number of blocks to use when making a request to
	our driver.
	
	testing starts assuming 'minBlocks' is the minimum number of blocks to
	use.
___________________________________________________________________________*/
	OSStatus
CTuner::TuneSize(
	const ulong		ioBlocks,
	const ulong		minBlocks,
	ulong *			optimalBlocks )
	{
	OSStatus	err;
	FSSpec		testSpec;
	
	*optimalBlocks	= minBlocks;
				
	err	= OpenTestFile( &testSpec, &mFileRefNum );
	if ( IsntErr( err ) )
		{
		ulong	ioSize	= ioBlocks * kDiskBlockSize;
		void *	buffer	= nil;
		
		buffer	= pgpAllocMac( ioSize, kMacMemory_PreferTempMem |
						kMacMemory_UseApplicationHeap);
		if ( IsntNull( buffer ) )
			{
			mBuffer	= buffer;
			
			err	= DetermineTestFileSize( ioSize, &mTestFileSize );
		
			if ( IsntErr( err ) && mTestFileSize >= ioSize )
				{
				err	= FindOptimalBlocks( ioSize, minBlocks, 0UL,
							optimalBlocks);
				}
	
			pgpFreeMac( buffer );
			}
		else
			{
			err	= memFullErr;
			}
		
		FSClose( mFileRefNum );
		mFileRefNum	= 0;
		(void)FSpDelete( &testSpec );
		}
	
	AssertNoErr( err, "CTuner::TuneSize" );
	return( err );
	}
	


/*___________________________________________________________________________
	Find optimal tuning for all I/O sizes in our tuning table
___________________________________________________________________________*/
	OSStatus
CTuner::Tune(
	short			vRefNum,
	IODirection		direction,
	TunerStruct *	optimalTune,
	TunerCallback	callback)
	{
	pgpAssert( vRefNum < 0 );
	OSErr			err	= noErr;
	const ushort	kSkipTune	= 7;
	
	mVRefNum	= vRefNum;
	mDirection	= direction;
	mCallback	= callback;
	
	// in case our tests can't be performed, we want some reasonable tuning
	if ( direction == kReadDirection )
		*optimalTune	= kDefaultReadTunerStruct;
	else
		*optimalTune	= kDefaultWriteTunerStruct128;
	
	// start at index 2 ( 8 blocks) because 1, 2 blocks rarely improve
	for( ushort tuneIndex = kSkipTune; tuneIndex < TunerStruct::kNumTune;
				++tuneIndex )
		{
		ushort	startBlocks;
		ulong	numBlocks;
		ulong	optimalBlocks;
		
		++mCurTestIndex;
		
		// assume that larger sizes need at least as many chunks as the next
		// smaller one does
		pgpAssert( tuneIndex >= 1 );
		startBlocks	= optimalTune->maxBlocks[ tuneIndex - 1];
		numBlocks	= 1UL << tuneIndex;
		
		pgpAssert( startBlocks <= numBlocks );
		err	= TuneSize( numBlocks, startBlocks, &optimalBlocks );
		if ( IsErr( err ) )
			{
			// propogate last optimal size upward
			while ( tuneIndex < TunerStruct::kNumTune )
				{
				ulong	lastOptimal;
				
				lastOptimal	= optimalTune->maxBlocks[ tuneIndex - 1 ];
				
				if ( optimalTune->maxBlocks[ tuneIndex ] >= lastOptimal )
					{
					// presumably larger items have larger maxes already
					break;
					}
					
				optimalTune->maxBlocks[ tuneIndex ] 	= lastOptimal;
				}
			break;
			}
		else
			{
			optimalTune->maxBlocks[ tuneIndex ]	= optimalBlocks;
			}
		}

	AssertNoErr( err, "CTuner::Tune" );
	return( noErr );
	}



	OSStatus
CTuner::Tune(
	short			vRefNum,
	TunerCallback	callback,
	long			callbackData)
	{
	TunerStruct		optimalTune;
	
	mNumTests		= TunerStruct::kNumTune;
	mCurTestIndex	= 0;
	mCallbackData	= callbackData;

#if 0
	(void)Tune( vRefNum, kReadDirection, &optimalTune, callback );
	(void)ChangeDriveTuner( vRefNum, &optimalTune,
			IOOptionsStruct::kReadTuner );
	SetA0( &optimalTune );
	pgpDebugPStr( "\p optimal read tune;dm A0 #32;g" );
#endif

	(void)Tune( vRefNum, kWriteDirection, &optimalTune, callback );
	(void)ChangeDriveTuner( vRefNum, &optimalTune,
			IOOptionsStruct::kWriteTuner );
	SetA0( &optimalTune );
	pgpDebugPStr( "\p optimal write tune;dm A0 #32;g" );


	// make sure callback sees that we're done
	mCurTestIndex	= mNumTests;
	(void)CallCallback( );
	
	return( noErr );
	}














