/*____________________________________________________________________________
	Copyright (C) 1994-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: randpool.c,v 1.4.2.1.2.1 1998/11/12 03:05:50 heller Exp $
____________________________________________________________________________*/

#include <stdlib.h>
#include <string.h>
#include <Timer.h>

#include "randpool.h"
#include "SHA.h"


#define RANDKEYWORDS 16	/* This is a parameter of the the SHA1 algorithm */

/* The pool must be a multiple of the 16-byte (128-bit) SHA1 block size */
#define RANDPOOLWORDS ((RANDPOOLBITS+127 & ~127) >> 5)

#if RANDPOOLWORDS <= RANDKEYWORDS
#error Random pool too small - please increase RANDPOOLBITS in randpool.h
#endif

/* Must be word-aligned, so make it words.  Cast to bytes as needed. */
static word32 randPool[RANDPOOLWORDS];	/* Random pool */
static word32 randKey[RANDKEYWORDS];	/* Next stirring key */
static unsigned randPoolGetPos = sizeof(randPool); /* Position to get from */
static unsigned randKeyAddPos = 0;	/* Position to add to */

//	Adds random stuff to the pool. Data from the mouse position and the
//	system clock is used.

Boolean AddRandomEntropy()
{
	static Point 	oldMousePos = {0,0};
	Point 			mousePos;
	UnsignedWide 	curms;
	Boolean			added = FALSE;
	
	//	Add any new mouse position to the random pool.
	GetMouse(&mousePos);
	LocalToGlobal(&mousePos);
	if(!EqualPt(mousePos, oldMousePos))
	{
		oldMousePos = mousePos;
		randPoolAddBytes((uchar *)&mousePos, sizeof(Point));
		
		added = TRUE;
	}
	
	//	Add current Microseconds since startup time
	//	This is a 64 bit value
	Microseconds(&curms);
	randPoolAddBytes((uchar *)&curms, 8);
	
	return( added );
}


/*
 * "Stir in" any random seed material before removing any random bytes.
 * This also ensures that no sensitive data remains in memory that can
 * be recovered later.
 *
 * The transformation is carried out by "encrypting" the data in CFB
 * mode with MD5 as the block cipher.  Then, to make certain the stirring
 * operation is strictly one-way, we destroy the key, reinitializing it
 * with 64 bytes from the beginning of the pool.  These bytes are not
 * returned by randPoolGetBytes().
 *
 * To make this useful for pseudo-random (that is, repeatable) operations,
 * the MD5 transformation is always done with a consistent byte order.
 * MD5Transform itself works with 32-bit words, not bytes, so the pool,
 * usually an array of bytes, is transformed into an array of 32-bit words,
 * taking each group of 4 bytes in little-endian order.  At the end of the
 * stirring, the transformation is reversed.
 */
void
randPoolStir(void)
{
	int i;
	word32 iv[4];

	/* Start IV from last block of randPool */
	BlockMoveData(randPool+RANDPOOLWORDS-4, iv, sizeof(iv));

	/* CFB pass */
	for (i = 0; i < RANDPOOLWORDS; i += 4)
	{
		SHA			sha;
		SHA::Digest	digest;
		
		sha.Update( (const uchar *) iv, sizeof( iv ) );
		sha.Update( (const uchar *) &randKey[ 0 ], sizeof( randKey ) );
		sha.Final( &digest );
		BlockMoveData( &digest, iv, sizeof( iv ) );

		iv[0] = randPool[i  ] ^= iv[0];
		iv[1] = randPool[i+1] ^= iv[1];
		iv[2] = randPool[i+2] ^= iv[2];
		iv[3] = randPool[i+3] ^= iv[3];
	}

	/* Wipe iv from memory */
	for(i=0;i<4;i++)
		iv[i]=0;

	/* Get new key */
	BlockMoveData(randPool, randKey, sizeof(randKey));

	/* Set up pointers for future addition or removal of random bytes */
	randKeyAddPos = 0;
	randPoolGetPos = sizeof(randKey);
}

/*
 * Make a deposit of information (entropy) into the pool.  This is done
 * by XORing the data into the key which used to encrypt the pool.
 * Before any bytes are retrieved from the pool, the altered key will be
 * used to encrypt the whole pool, causing all bits in the pool to
 * depend on the new information.
 *
 * The bits deposited need not have any particular distribution;
 * the stirring operation transforms them into uniformly-distributed bits.
 */
void
randPoolAddBytes(byte const *buf, unsigned len)
{
	byte *p = (byte *)randKey+randKeyAddPos;
	unsigned t = sizeof(randKey) - randKeyAddPos;

	while (len > t) {
		len -= t;
		while (t--)
			*p++ ^= *buf++;
		randPoolStir();		/* sets randKeyAddPos to 0 */
		p = (byte *)randKey;
		t = sizeof(randKey);
	}

	if (len) {
		randKeyAddPos += len;
		do
			*p++ ^= *buf++;
		while (--len);
		randPoolGetPos = sizeof(randPool); /* Force stir on get */
	}
}

/*
 * Withdraw some bits from the pool.  Regardless of the distribution of the
 * input bits, the bits returned are uniformly distributed, although they
 * cannot, of course, contain more Shannon entropy than the input bits.
 */
void
randPoolGetBytes(byte *buf, unsigned len)
{
	unsigned t;

	while (len > (t = sizeof(randPool) - randPoolGetPos)) {
		BlockMoveData((byte const *)randPool+randPoolGetPos, buf, t);
		buf += t;
		len -= t;
		randPoolStir();
	}

	BlockMoveData((byte const *)randPool+randPoolGetPos, buf, len);
	randPoolGetPos += len;
}

/* Get a single byte */
byte
randPoolGetByte(void)
{
	if (randPoolGetPos == sizeof(randPool))
		randPoolStir();

	return ((byte const *)randPool)[randPoolGetPos++];
}
