/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 *
 */

package cryptix.pgp;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.Date;

abstract class Certificate extends Packet
{
	private static final int VERSION = 2;
	private static final int LENGTH = 7;
	
	protected Date timeStamp;
	protected int daysValid;

	Certificate()
	{
		this( new Date(), 0 ); // today and forever.
	}

	Certificate( Date time0, int valid0 )
	{
		timeStamp = time0;
		daysValid = valid0;
	}
	
	Certificate( DataInput in, int length )
		throws IOException
	{
		super( in, length );
	}

	public void
	read( DataInput in, int length )
		throws IOException
	{
		if ( length < LENGTH )
			throw new FormatException( "Length too short." );
		in.readByte(); // version byte..
		timeStamp = new Date( ( (long)in.readInt() & 0xFFFFFFFFL ) * 1000L );
		daysValid = in.readShort();
	}

	public int
	write( DataOutput out )
		throws IOException
	{
		out.write( VERSION );
		out.writeInt( (int)( timeStamp.getTime() / 1000L ) );
		out.writeShort( daysValid );
		return LENGTH;
	}
	
	public String
	toString()
	{
		return "version: " + VERSION + " created:" + timeStamp + 
		" valid for" + ( daysValid == 0 ? "ever" : ( " " + daysValid + " days" ) );
	}
}
