// haval.cpp - modified by Wei Dai from Yuliang Zheng's HAVAL.c
// distributed with Yuliang Zheng's permission

/*
 *  Reference:
 *       Y. Zheng, J. Pieprzyk and J. Seberry:
 *       ``HAVAL --- a one-way hashing algorithm with variable
 *       length of output'', Advances in Cryptology --- AUSCRYPT'92,
 *       Lecture Notes in Computer Science, Springer-Verlag, 1993.
 *
 *  Author:     Yuliang Zheng
 *              Department of Computer Science
 *              University of Wollongong
 *              Wollongong, NSW 2522, Australia
 *              Email: yuliang@cs.uow.edu.au
 *              Voice: +61 42 21 4331 (office)
 */

#include "haval.h"
#include "misc.h"

static const int HAVAL_LAST_BLOCK = 118;

HAVAL::HAVAL(unsigned int digestSize, unsigned int pass)
    : digestSize(digestSize), pass(pass),
	  buffer(DIGESTSIZE/4 + DATASIZE/4),
      digest(buffer), data(digest+DIGESTSIZE/4)
{
	assert(digestSize >= 16 && digestSize <= 32 && digestSize%4==0);
	assert(pass >= 3 && pass <= 5);

    Init();
}

/* initialization */
void HAVAL::Init()
{
    countLo = countHi = 0;					/* clear count */
    digest[0] = 0x243F6A88;            /* initial digest */
    digest[1] = 0x85A308D3;
    digest[2] = 0x13198A2E;
    digest[3] = 0x03707344;
    digest[4] = 0xA4093822;
    digest[5] = 0x299F31D0;
    digest[6] = 0x082EFA98;
    digest[7] = 0xEC4E6C89;
}

inline void HAVAL::vTransform(word32 *buf, const word32 *in)
{
	if (pass==3)
		HAVAL3::Transform(buf, in);
	else if (pass==4)
		HAVAL4::Transform(buf, in);
	else
		HAVAL5::Transform(buf, in);
}

inline void HAVAL::do_block (const word32 *input)
{
#ifdef LITTLE_ENDIAN
    vTransform(digest, input);
#else
    byteReverse(digest, input, DATASIZE);
    vTransform(digest, data);
#endif
}

void HAVAL::Update (const byte *input, unsigned int len)
{
    word32 tmp = countLo;
    if ((countLo = tmp + ((word32)len << 3)) < tmp)
        countHi++;             /* Carry from low to high */
    countHi += len >> 29;

    unsigned int num = (unsigned int)(tmp >> 3) % DATASIZE;

    if (num)
    {
        if ((num+len) >= DATASIZE)
        {
            memcpy((byte *)data+num, input, DATASIZE-num);
            do_block(data);
            input+=(DATASIZE-num);
            len-=(DATASIZE-num);
            // drop through and do the rest
        }
        else
        {
            memcpy((byte *)data+num, input, len);
            return;
        }
    }

    // we now can process the input data in blocks of DATASIZE
    // chars and save the leftovers to this->data.
    if (len >= DATASIZE)
    {
        if ((int)input&3)   // test for 32-bit word alignment
            do
            {   // copy input first if it's not aligned correctly
                memcpy(data, input, DATASIZE);
                do_block(data);
                input+=DATASIZE;
                len-=DATASIZE;
            } while (len >= DATASIZE);
        else
            do
            {
                do_block((word32 *)input);
                input+=DATASIZE;
                len-=DATASIZE;
            } while (len >= DATASIZE);
    }

    memcpy(data, input, len);
}

/* tailor the last output */
void HAVAL::Tailor(unsigned int FPTLEN)
{
	word32 temp;

	switch (FPTLEN)
	{
	case 128:
		temp = (digest[7] & 0x000000FF) | 
			   (digest[6] & 0xFF000000) | 
			   (digest[5] & 0x00FF0000) | 
			   (digest[4] & 0x0000FF00);
		digest[0] += rotr(temp,  8U);

		temp = (digest[7] & 0x0000FF00) | 
			   (digest[6] & 0x000000FF) | 
			   (digest[5] & 0xFF000000) | 
			   (digest[4] & 0x00FF0000);
		digest[1] += rotr(temp, 16U);

		temp  = (digest[7] & 0x00FF0000) | 
			    (digest[6] & 0x0000FF00) | 
			    (digest[5] & 0x000000FF) | 
			    (digest[4] & 0xFF000000);
		digest[2] += rotr(temp, 24U);

		temp = (digest[7] & 0xFF000000) | 
			   (digest[6] & 0x00FF0000) | 
			   (digest[5] & 0x0000FF00) | 
			   (digest[4] & 0x000000FF);
		digest[3] += temp;
		break;

	case 160:
		temp = (digest[7] &  (word32)0x3F) | 
			   (digest[6] & ((word32)0x7F << 25)) |  
			   (digest[5] & ((word32)0x3F << 19));
		digest[0] += rotr(temp, 19U);

		temp = (digest[7] & ((word32)0x3F <<  6)) | 
			   (digest[6] &  (word32)0x3F) |  
			   (digest[5] & ((word32)0x7F << 25));
		digest[1] += rotr(temp, 25U);

		temp = (digest[7] & ((word32)0x7F << 12)) | 
			   (digest[6] & ((word32)0x3F <<  6)) |  
			   (digest[5] &  (word32)0x3F);
		digest[2] += temp;

		temp = (digest[7] & ((word32)0x3F << 19)) | 
			   (digest[6] & ((word32)0x7F << 12)) |  
			   (digest[5] & ((word32)0x3F <<  6));
		digest[3] += temp >> 6; 

		temp = (digest[7] & ((word32)0x7F << 25)) | 
			   (digest[6] & ((word32)0x3F << 19)) |  
			   (digest[5] & ((word32)0x7F << 12));
		digest[4] += temp >> 12;
		break;

	case 192:
		temp = (digest[7] &  (word32)0x1F) | 
			   (digest[6] & ((word32)0x3F << 26));
		digest[0] += rotr(temp, 26U);

		temp = (digest[7] & ((word32)0x1F <<  5)) | 
			   (digest[6] &  (word32)0x1F);
		digest[1] += temp;

		temp = (digest[7] & ((word32)0x3F << 10)) | 
			   (digest[6] & ((word32)0x1F <<  5));
		digest[2] += temp >> 5;

		temp = (digest[7] & ((word32)0x1F << 16)) | 
			   (digest[6] & ((word32)0x3F << 10));
		digest[3] += temp >> 10;

		temp = (digest[7] & ((word32)0x1F << 21)) | 
			   (digest[6] & ((word32)0x1F << 16));
		digest[4] += temp >> 16;

		temp = (digest[7] & ((word32)0x3F << 26)) | 
			   (digest[6] & ((word32)0x1F << 21));
		digest[5] += temp >> 21;
		break;

	case 224:
		digest[0] += (digest[7] >> 27) & 0x1F;
		digest[1] += (digest[7] >> 22) & 0x1F;
		digest[2] += (digest[7] >> 18) & 0x0F;
		digest[3] += (digest[7] >> 13) & 0x1F;
		digest[4] += (digest[7] >>  9) & 0x0F;
		digest[5] += (digest[7] >>  4) & 0x1F;
		digest[6] +=  digest[7]        & 0x0F;
		break;

	case 256:
		break;

	default:
		assert(FALSE);
	}
}

void HAVAL::Final (byte *hash)
{
    unsigned int num = (unsigned int)(countLo >> 3) % DATASIZE;
    // this->data should definitly have room for at least one more byte.
    ((byte *)data)[num++]=0x80;
    if (num > HAVAL_LAST_BLOCK)
    {
        memset((byte *)data+num, 0, DATASIZE-num);
        do_block(data);
        memset(data, 0, HAVAL_LAST_BLOCK);
    }
    else
    {
        memset((byte *)data+num, 0, HAVAL_LAST_BLOCK-num);
        CorrectEndianess(data, data, HAVAL_LAST_BLOCK+2);
    }

	data[29] &= 0xffff;
	data[29] |= (digestSize<<25) | (pass<<19) | (VERSION<<16);
    data[30] = countLo;
    data[31] = countHi;

    vTransform(digest, data);
	Tailor(digestSize*8);
    CorrectEndianess(digest, digest, digestSize);
    memcpy(hash, digest, digestSize);

    // reinit for next use
    Init();
}

/*
#define f_1(x6, x5, x4, x3, x2, x1, x0)          \
           ((x1) & ((x0) ^ (x4)) ^ (x2) & (x5) ^ \
            (x3) & (x6) ^ (x0))
*/

#define f_1(x6, x5, x4, x3, x2, x1, x0)          \
    ((x1&(x0^x4)) ^ (x2&x5) ^ (x3&x6) ^ x0)

/*
#define f_2(x6, x5, x4, x3, x2, x1, x0)                         \
           ((x2) & ((x1) & ~(x3) ^ (x4) & (x5) ^ (x6) ^ (x0)) ^ \
            (x4) & ((x1) ^ (x5)) ^ (x3) & (x5) ^ (x0))
*/

#define f_2(x6, x5, x4, x3, x2, x1, x0)                         \
	(((x4&x5)|x2) ^ (x0|x2) ^ (x2&((x1&~x3)^x6)) ^ (x3&x5) ^ (x1&x4))

/*
#define f_3(x6, x5, x4, x3, x2, x1, x0)          \
           ((x3) & ((x1) & (x2) ^ (x6) ^ (x0)) ^ \
            (x1) & (x4) ^ (x2) & (x5) ^ (x0))
*/

#define f_3(x6, x5, x4, x3, x2, x1, x0)          \
    ((x3 & ((x1&x2) ^ x6 ^ x0)) ^ (x1&x4) ^ (x2&x5) ^ x0)

/*
#define f_4(x6, x5, x4, x3, x2, x1, x0)                                 \
           ((x4) & ((x5) & ~(x2) ^ (x3) & ~(x6) ^ (x1) ^ (x6) ^ (x0)) ^ \
            (x3) & ((x1) & (x2) ^ (x5) ^ (x6)) ^                        \
            (x2) & (x6) ^ (x0))
*/

#define f_4(x6, x5, x4, x3, x2, x1, x0)          \
	((((~x2&x5)^(x3|x6)^x1^x0)&x4) ^ (((x1&x2)^x5^x6)&x3) ^ (x2&x6) ^ x0)


/*
#define f_5(x6, x5, x4, x3, x2, x1, x0)             \
           ((x0) & ((x1) & (x2) & (x3) ^ ~(x5)) ^   \
            (x1) & (x4) ^ (x2) & (x5) ^ (x3) & (x6))
*/

#define f_5(x6, x5, x4, x3, x2, x1, x0)          \
	((((x0&x2&x3)^x4)&x1) ^ ((x0^x2)&x5) ^ (x3&x6) ^ x0)

/*
 * Permutations phi_{i,j}, i=3,4,5, j=1,...,i.
 *
 * PASS = 3:
 *               6 5 4 3 2 1 0
 *               | | | | | | | (replaced by)
 *  phi_{3,1}:   1 0 3 5 6 2 4
 *  phi_{3,2}:   4 2 1 0 5 3 6
 *  phi_{3,3}:   6 1 2 3 4 5 0
 *
 * PASS = 4:
 *               6 5 4 3 2 1 0
 *               | | | | | | | (replaced by)
 *  phi_{4,1}:   2 6 1 4 5 3 0
 *  phi_{4,2}:   3 5 2 0 1 6 4
 *  phi_{4,3}:   1 4 3 6 0 2 5
 *  phi_{4,4}:   6 4 0 5 2 1 3
 *
 * PASS = 5:
 *               6 5 4 3 2 1 0
 *               | | | | | | | (replaced by)
 *  phi_{5,1}:   3 4 1 0 5 2 6
 *  phi_{5,2}:   6 2 1 0 3 4 5
 *  phi_{5,3}:   2 6 0 4 3 1 5
 *  phi_{5,4}:   1 5 3 2 0 4 6
 *  phi_{5,5}:   2 5 0 6 4 3 1
 */

#define Fphi_31(x6, x5, x4, x3, x2, x1, x0) \
            f_1(x1, x0, x3, x5, x6, x2, x4)

#define Fphi_41(x6, x5, x4, x3, x2, x1, x0) \
            f_1(x2, x6, x1, x4, x5, x3, x0)

#define Fphi_51(x6, x5, x4, x3, x2, x1, x0) \
            f_1(x3, x4, x1, x0, x5, x2, x6)

#define Fphi_32(x6, x5, x4, x3, x2, x1, x0) \
            f_2(x4, x2, x1, x0, x5, x3, x6)

#define Fphi_42(x6, x5, x4, x3, x2, x1, x0) \
            f_2(x3, x5, x2, x0, x1, x6, x4)

#define Fphi_52(x6, x5, x4, x3, x2, x1, x0) \
            f_2(x6, x2, x1, x0, x3, x4, x5)

#define Fphi_33(x6, x5, x4, x3, x2, x1, x0) \
            f_3(x6, x1, x2, x3, x4, x5, x0)

#define Fphi_43(x6, x5, x4, x3, x2, x1, x0) \
            f_3(x1, x4, x3, x6, x0, x2, x5)

#define Fphi_53(x6, x5, x4, x3, x2, x1, x0) \
            f_3(x2, x6, x0, x4, x3, x1, x5)

#define Fphi_44(x6, x5, x4, x3, x2, x1, x0) \
            f_4(x6, x4, x0, x5, x2, x1, x3)

#define Fphi_54(x6, x5, x4, x3, x2, x1, x0) \
            f_4(x1, x5, x3, x2, x0, x4, x6)

#define Fphi_55(x6, x5, x4, x3, x2, x1, x0) \
            f_5(x2, x5, x0, x6, x4, x3, x1)

#define FF_31(x7, x6, x5, x4, x3, x2, x1, x0, w) {                    \
      register word32 temp = Fphi_31(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w);}

#define FF_41(x7, x6, x5, x4, x3, x2, x1, x0, w) {                    \
      register word32 temp = Fphi_41(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w);}

#define FF_51(x7, x6, x5, x4, x3, x2, x1, x0, w) {                    \
      register word32 temp = Fphi_51(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w);}

#define FF_32(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_32(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_42(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_42(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_52(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_52(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_33(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_33(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_43(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_43(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_53(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_53(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_44(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_44(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_54(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_54(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

#define FF_55(x7, x6, x5, x4, x3, x2, x1, x0, w, c) {                 \
      register word32 temp = Fphi_55(x6, x5, x4, x3, x2, x1, x0);     \
      (x7) = rotr(temp, 7U) + rotr((x7), 11U) + (w) + (c);}

const unsigned int HAVAL::wi2[32] = { 5,14,26,18,11,28, 7,16, 0,23,20,22, 1,10, 4, 8,30, 3,21, 9,17,24,29, 6,19,12,15,13, 2,25,31,27};
const unsigned int HAVAL::wi3[32] = {19, 9, 4,20,28,17, 8,22,29,14,25,12,24,30,16,26,31,15, 7, 3, 1, 0,18,27,13, 6,21,10,23,11, 5, 2};
const unsigned int HAVAL::wi4[32] = {24, 4, 0,14, 2, 7,28,23,26, 6,30,20,18,25,19, 3,22,11,31,21, 8,27,12, 9, 1,29, 5,15,17,10,16,13};
const unsigned int HAVAL::wi5[32] = {27, 3,21,26,17,11,20,29,19, 0,12, 7,13, 8,31,10, 5, 9,14,30,18, 6,28,24, 2,23,16,22, 4, 1,25,15};

const word32 HAVAL::mc2[32] = {
  0x452821E6, 0x38D01377, 0xBE5466CF, 0x34E90C6C, 0xC0AC29B7, 0xC97C50DD, 0x3F84D5B5, 0xB5470917
, 0x9216D5D9, 0x8979FB1B, 0xD1310BA6, 0x98DFB5AC, 0x2FFD72DB, 0xD01ADFB7, 0xB8E1AFED, 0x6A267E96
, 0xBA7C9045, 0xF12C7F99, 0x24A19947, 0xB3916CF7, 0x0801F2E2, 0x858EFC16, 0x636920D8, 0x71574E69
, 0xA458FEA3, 0xF4933D7E, 0x0D95748F, 0x728EB658, 0x718BCD58, 0x82154AEE, 0x7B54A41D, 0xC25A59B5};

const word32 HAVAL::mc3[32] = {
0x9C30D539,0x2AF26013,0xC5D1B023,0x286085F0,0xCA417918,0xB8DB38EF,0x8E79DCB0,0x603A180E,
0x6C9E0E8B,0xB01E8A3E,0xD71577C1,0xBD314B27,0x78AF2FDA,0x55605C60,0xE65525F3,0xAA55AB94,
0x57489862,0x63E81440,0x55CA396A,0x2AAB10B6,0xB4CC5C34,0x1141E8CE,0xA15486AF,0x7C72E993,
0xB3EE1411,0x636FBC2A,0x2BA9C55D,0x741831F6,0xCE5C3E16,0x9B87931E,0xAFD6BA33,0x6C24CF5C};

const word32 HAVAL::mc4[32] = {
0x7A325381,0x28958677,0x3B8F4898,0x6B4BB9AF,0xC4BFE81B,0x66282193,0x61D809CC,0xFB21A991,
0x487CAC60,0x5DEC8032,0xEF845D5D,0xE98575B1,0xDC262302,0xEB651B88,0x23893E81,0xD396ACC5,
0x0F6D6FF3,0x83F44239,0x2E0B4482,0xA4842004,0x69C8F04A,0x9E1F9B5E,0x21C66842,0xF6E96C9A,
0x670C9C61,0xABD388F0,0x6A51A0D2,0xD8542F68,0x960FA728,0xAB5133A3,0x6EEF0B6C,0x137A3BE4};

const word32 HAVAL::mc5[32] = {
0xBA3BF050,0x7EFB2A98,0xA1F1651D,0x39AF0176,0x66CA593E,0x82430E88,0x8CEE8619,0x456F9FB4,
0x7D84A5C3,0x3B8B5EBE,0xE06F75D8,0x85C12073,0x401A449F,0x56C16AA6,0x4ED3AA62,0x363F7706,
0x1BFEDF72,0x429B023D,0x37D0D724,0xD00A1248,0xDB0FEAD3,0x49F1C09B,0x075372C9,0x80991B7B,
0x25D479D8,0xF6E8DEF7,0xE3FE501A,0xB6794C3B,0x976CE0BD,0x04C006BA,0xC1A94FB6,0x409F60C4};

void HAVAL3::Transform (word32 *digest, const word32 *w)
{
	register word32 t0 = digest[0],    // make use of
					t1 = digest[1],    // internal registers
					t2 = digest[2],
					t3 = digest[3],
					t4 = digest[4],
					t5 = digest[5],
					t6 = digest[6],
					t7 = digest[7];
	unsigned i;

	for (i=0; i<4; i++)
	{
		FF_31(t7, t6, t5, t4, t3, t2, t1, t0, w[8*i+0]);
		FF_31(t6, t5, t4, t3, t2, t1, t0, t7, w[8*i+1]);
		FF_31(t5, t4, t3, t2, t1, t0, t7, t6, w[8*i+2]);
		FF_31(t4, t3, t2, t1, t0, t7, t6, t5, w[8*i+3]);
		FF_31(t3, t2, t1, t0, t7, t6, t5, t4, w[8*i+4]);
		FF_31(t2, t1, t0, t7, t6, t5, t4, t3, w[8*i+5]);
		FF_31(t1, t0, t7, t6, t5, t4, t3, t2, w[8*i+6]);
		FF_31(t0, t7, t6, t5, t4, t3, t2, t1, w[8*i+7]);
	}

	/*
	// Pass 1
	FF_31(t7, t6, t5, t4, t3, t2, t1, t0, *(w   ));
	FF_31(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 1));
	FF_31(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 2));
	FF_31(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 3));
	FF_31(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 4));
	FF_31(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 5));
	FF_31(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 6));
	FF_31(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 7));

	FF_31(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 8));
	FF_31(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 9));
	FF_31(t5, t4, t3, t2, t1, t0, t7, t6, *(w+10));
	FF_31(t4, t3, t2, t1, t0, t7, t6, t5, *(w+11));
	FF_31(t3, t2, t1, t0, t7, t6, t5, t4, *(w+12));
	FF_31(t2, t1, t0, t7, t6, t5, t4, t3, *(w+13));
	FF_31(t1, t0, t7, t6, t5, t4, t3, t2, *(w+14));
	FF_31(t0, t7, t6, t5, t4, t3, t2, t1, *(w+15));

	FF_31(t7, t6, t5, t4, t3, t2, t1, t0, *(w+16));
	FF_31(t6, t5, t4, t3, t2, t1, t0, t7, *(w+17));
	FF_31(t5, t4, t3, t2, t1, t0, t7, t6, *(w+18));
	FF_31(t4, t3, t2, t1, t0, t7, t6, t5, *(w+19));
	FF_31(t3, t2, t1, t0, t7, t6, t5, t4, *(w+20));
	FF_31(t2, t1, t0, t7, t6, t5, t4, t3, *(w+21));
	FF_31(t1, t0, t7, t6, t5, t4, t3, t2, *(w+22));
	FF_31(t0, t7, t6, t5, t4, t3, t2, t1, *(w+23));

	FF_31(t7, t6, t5, t4, t3, t2, t1, t0, *(w+24));
	FF_31(t6, t5, t4, t3, t2, t1, t0, t7, *(w+25));
	FF_31(t5, t4, t3, t2, t1, t0, t7, t6, *(w+26));
	FF_31(t4, t3, t2, t1, t0, t7, t6, t5, *(w+27));
	FF_31(t3, t2, t1, t0, t7, t6, t5, t4, *(w+28));
	FF_31(t2, t1, t0, t7, t6, t5, t4, t3, *(w+29));
	FF_31(t1, t0, t7, t6, t5, t4, t3, t2, *(w+30));
	FF_31(t0, t7, t6, t5, t4, t3, t2, t1, *(w+31));
	*/

	/*
	for (i=0; i<4; i++)
	{
		FF_32(t7, t6, t5, t4, t3, t2, t1, t0, w[wi2[8*i+0]], mc2[8*i+0]);
		FF_32(t6, t5, t4, t3, t2, t1, t0, t7, w[wi2[8*i+1]], mc2[8*i+1]);
		FF_32(t5, t4, t3, t2, t1, t0, t7, t6, w[wi2[8*i+2]], mc2[8*i+2]);
		FF_32(t4, t3, t2, t1, t0, t7, t6, t5, w[wi2[8*i+3]], mc2[8*i+3]);
		FF_32(t3, t2, t1, t0, t7, t6, t5, t4, w[wi2[8*i+4]], mc2[8*i+4]);
		FF_32(t2, t1, t0, t7, t6, t5, t4, t3, w[wi2[8*i+5]], mc2[8*i+5]);
		FF_32(t1, t0, t7, t6, t5, t4, t3, t2, w[wi2[8*i+6]], mc2[8*i+6]);
		FF_32(t0, t7, t6, t5, t4, t3, t2, t1, w[wi2[8*i+7]], mc2[8*i+7]);
	}

	for (i=0; i<4; i++)
	{
		FF_33(t7, t6, t5, t4, t3, t2, t1, t0, w[wi3[8*i+0]], mc3[8*i+0]);
		FF_33(t6, t5, t4, t3, t2, t1, t0, t7, w[wi3[8*i+1]], mc3[8*i+1]);
		FF_33(t5, t4, t3, t2, t1, t0, t7, t6, w[wi3[8*i+2]], mc3[8*i+2]);
		FF_33(t4, t3, t2, t1, t0, t7, t6, t5, w[wi3[8*i+3]], mc3[8*i+3]);
		FF_33(t3, t2, t1, t0, t7, t6, t5, t4, w[wi3[8*i+4]], mc3[8*i+4]);
		FF_33(t2, t1, t0, t7, t6, t5, t4, t3, w[wi3[8*i+5]], mc3[8*i+5]);
		FF_33(t1, t0, t7, t6, t5, t4, t3, t2, w[wi3[8*i+6]], mc3[8*i+6]);
		FF_33(t0, t7, t6, t5, t4, t3, t2, t1, w[wi3[8*i+7]], mc3[8*i+7]);
	}
	*/

	// Pass 2
	FF_32(t7, t6, t5, t4, t3, t2, t1, t0, w[ 5], 0x452821E6);
	FF_32(t6, t5, t4, t3, t2, t1, t0, t7, w[14], 0x38D01377);
	FF_32(t5, t4, t3, t2, t1, t0, t7, t6, w[26], 0xBE5466CF);
	FF_32(t4, t3, t2, t1, t0, t7, t6, t5, w[18], 0x34E90C6C);
	FF_32(t3, t2, t1, t0, t7, t6, t5, t4, w[11], 0xC0AC29B7);
	FF_32(t2, t1, t0, t7, t6, t5, t4, t3, w[28], 0xC97C50DD);
	FF_32(t1, t0, t7, t6, t5, t4, t3, t2, w[ 7], 0x3F84D5B5);
	FF_32(t0, t7, t6, t5, t4, t3, t2, t1, w[16], 0xB5470917);

	FF_32(t7, t6, t5, t4, t3, t2, t1, t0, w[ 0], 0x9216D5D9);
	FF_32(t6, t5, t4, t3, t2, t1, t0, t7, w[23], 0x8979FB1B);
	FF_32(t5, t4, t3, t2, t1, t0, t7, t6, w[20], 0xD1310BA6);
	FF_32(t4, t3, t2, t1, t0, t7, t6, t5, w[22], 0x98DFB5AC);
	FF_32(t3, t2, t1, t0, t7, t6, t5, t4, w[ 1], 0x2FFD72DB);
	FF_32(t2, t1, t0, t7, t6, t5, t4, t3, w[10], 0xD01ADFB7);
	FF_32(t1, t0, t7, t6, t5, t4, t3, t2, w[ 4], 0xB8E1AFED);
	FF_32(t0, t7, t6, t5, t4, t3, t2, t1, w[ 8], 0x6A267E96);

	FF_32(t7, t6, t5, t4, t3, t2, t1, t0, w[30], 0xBA7C9045);
	FF_32(t6, t5, t4, t3, t2, t1, t0, t7, w[ 3], 0xF12C7F99);
	FF_32(t5, t4, t3, t2, t1, t0, t7, t6, w[21], 0x24A19947);
	FF_32(t4, t3, t2, t1, t0, t7, t6, t5, w[ 9], 0xB3916CF7);
	FF_32(t3, t2, t1, t0, t7, t6, t5, t4, w[17], 0x0801F2E2);
	FF_32(t2, t1, t0, t7, t6, t5, t4, t3, w[24], 0x858EFC16);
	FF_32(t1, t0, t7, t6, t5, t4, t3, t2, w[29], 0x636920D8);
	FF_32(t0, t7, t6, t5, t4, t3, t2, t1, w[ 6], 0x71574E69);

	FF_32(t7, t6, t5, t4, t3, t2, t1, t0, w[19], 0xA458FEA3);
	FF_32(t6, t5, t4, t3, t2, t1, t0, t7, w[12], 0xF4933D7E);
	FF_32(t5, t4, t3, t2, t1, t0, t7, t6, w[15], 0x0D95748F);
	FF_32(t4, t3, t2, t1, t0, t7, t6, t5, w[13], 0x728EB658);
	FF_32(t3, t2, t1, t0, t7, t6, t5, t4, w[ 2], 0x718BCD58);
	FF_32(t2, t1, t0, t7, t6, t5, t4, t3, w[25], 0x82154AEE);
	FF_32(t1, t0, t7, t6, t5, t4, t3, t2, w[31], 0x7B54A41D);
	FF_32(t0, t7, t6, t5, t4, t3, t2, t1, w[27], 0xC25A59B5);

	// Pass 3
	FF_33(t7, t6, t5, t4, t3, t2, t1, t0, w[19], 0x9C30D539);
	FF_33(t6, t5, t4, t3, t2, t1, t0, t7, w[ 9], 0x2AF26013);
	FF_33(t5, t4, t3, t2, t1, t0, t7, t6, w[ 4], 0xC5D1B023);
	FF_33(t4, t3, t2, t1, t0, t7, t6, t5, w[20], 0x286085F0);
	FF_33(t3, t2, t1, t0, t7, t6, t5, t4, w[28], 0xCA417918);
	FF_33(t2, t1, t0, t7, t6, t5, t4, t3, w[17], 0xB8DB38EF);
	FF_33(t1, t0, t7, t6, t5, t4, t3, t2, w[ 8], 0x8E79DCB0);
	FF_33(t0, t7, t6, t5, t4, t3, t2, t1, w[22], 0x603A180E);

	FF_33(t7, t6, t5, t4, t3, t2, t1, t0, w[29], 0x6C9E0E8B);
	FF_33(t6, t5, t4, t3, t2, t1, t0, t7, w[14], 0xB01E8A3E);
	FF_33(t5, t4, t3, t2, t1, t0, t7, t6, w[25], 0xD71577C1);
	FF_33(t4, t3, t2, t1, t0, t7, t6, t5, w[12], 0xBD314B27);
	FF_33(t3, t2, t1, t0, t7, t6, t5, t4, w[24], 0x78AF2FDA);
	FF_33(t2, t1, t0, t7, t6, t5, t4, t3, w[30], 0x55605C60);
	FF_33(t1, t0, t7, t6, t5, t4, t3, t2, w[16], 0xE65525F3);
	FF_33(t0, t7, t6, t5, t4, t3, t2, t1, w[26], 0xAA55AB94);

	FF_33(t7, t6, t5, t4, t3, t2, t1, t0, w[31], 0x57489862);
	FF_33(t6, t5, t4, t3, t2, t1, t0, t7, w[15], 0x63E81440);
	FF_33(t5, t4, t3, t2, t1, t0, t7, t6, w[ 7], 0x55CA396A);
	FF_33(t4, t3, t2, t1, t0, t7, t6, t5, w[ 3], 0x2AAB10B6);
	FF_33(t3, t2, t1, t0, t7, t6, t5, t4, w[ 1], 0xB4CC5C34);
	FF_33(t2, t1, t0, t7, t6, t5, t4, t3, w[ 0], 0x1141E8CE);
	FF_33(t1, t0, t7, t6, t5, t4, t3, t2, w[18], 0xA15486AF);
	FF_33(t0, t7, t6, t5, t4, t3, t2, t1, w[27], 0x7C72E993);

	FF_33(t7, t6, t5, t4, t3, t2, t1, t0, w[13], 0xB3EE1411);
	FF_33(t6, t5, t4, t3, t2, t1, t0, t7, w[ 6], 0x636FBC2A);
	FF_33(t5, t4, t3, t2, t1, t0, t7, t6, w[21], 0x2BA9C55D);
	FF_33(t4, t3, t2, t1, t0, t7, t6, t5, w[10], 0x741831F6);
	FF_33(t3, t2, t1, t0, t7, t6, t5, t4, w[23], 0xCE5C3E16);
	FF_33(t2, t1, t0, t7, t6, t5, t4, t3, w[11], 0x9B87931E);
	FF_33(t1, t0, t7, t6, t5, t4, t3, t2, w[ 5], 0xAFD6BA33);
	FF_33(t0, t7, t6, t5, t4, t3, t2, t1, w[ 2], 0x6C24CF5C);

	digest[0] += t0;
	digest[1] += t1;
	digest[2] += t2;
	digest[3] += t3;
	digest[4] += t4;
	digest[5] += t5;
	digest[6] += t6;
	digest[7] += t7;
}

void HAVAL4::Transform (word32 *digest, const word32 *w)
{
	register word32 t0 = digest[0],    // make use of
					t1 = digest[1],    // internal registers
					t2 = digest[2],
					t3 = digest[3],
					t4 = digest[4],
					t5 = digest[5],
					t6 = digest[6],
					t7 = digest[7];
	unsigned i;

	for (i=0; i<4; i++)
	{
		FF_41(t7, t6, t5, t4, t3, t2, t1, t0, w[8*i+0]);
		FF_41(t6, t5, t4, t3, t2, t1, t0, t7, w[8*i+1]);
		FF_41(t5, t4, t3, t2, t1, t0, t7, t6, w[8*i+2]);
		FF_41(t4, t3, t2, t1, t0, t7, t6, t5, w[8*i+3]);
		FF_41(t3, t2, t1, t0, t7, t6, t5, t4, w[8*i+4]);
		FF_41(t2, t1, t0, t7, t6, t5, t4, t3, w[8*i+5]);
		FF_41(t1, t0, t7, t6, t5, t4, t3, t2, w[8*i+6]);
		FF_41(t0, t7, t6, t5, t4, t3, t2, t1, w[8*i+7]);
	}

	for (i=0; i<4; i++)
	{
		FF_42(t7, t6, t5, t4, t3, t2, t1, t0, w[wi2[8*i+0]], mc2[8*i+0]);
		FF_42(t6, t5, t4, t3, t2, t1, t0, t7, w[wi2[8*i+1]], mc2[8*i+1]);
		FF_42(t5, t4, t3, t2, t1, t0, t7, t6, w[wi2[8*i+2]], mc2[8*i+2]);
		FF_42(t4, t3, t2, t1, t0, t7, t6, t5, w[wi2[8*i+3]], mc2[8*i+3]);
		FF_42(t3, t2, t1, t0, t7, t6, t5, t4, w[wi2[8*i+4]], mc2[8*i+4]);
		FF_42(t2, t1, t0, t7, t6, t5, t4, t3, w[wi2[8*i+5]], mc2[8*i+5]);
		FF_42(t1, t0, t7, t6, t5, t4, t3, t2, w[wi2[8*i+6]], mc2[8*i+6]);
		FF_42(t0, t7, t6, t5, t4, t3, t2, t1, w[wi2[8*i+7]], mc2[8*i+7]);
	}

	for (i=0; i<4; i++)
	{
		FF_43(t7, t6, t5, t4, t3, t2, t1, t0, w[wi3[8*i+0]], mc3[8*i+0]);
		FF_43(t6, t5, t4, t3, t2, t1, t0, t7, w[wi3[8*i+1]], mc3[8*i+1]);
		FF_43(t5, t4, t3, t2, t1, t0, t7, t6, w[wi3[8*i+2]], mc3[8*i+2]);
		FF_43(t4, t3, t2, t1, t0, t7, t6, t5, w[wi3[8*i+3]], mc3[8*i+3]);
		FF_43(t3, t2, t1, t0, t7, t6, t5, t4, w[wi3[8*i+4]], mc3[8*i+4]);
		FF_43(t2, t1, t0, t7, t6, t5, t4, t3, w[wi3[8*i+5]], mc3[8*i+5]);
		FF_43(t1, t0, t7, t6, t5, t4, t3, t2, w[wi3[8*i+6]], mc3[8*i+6]);
		FF_43(t0, t7, t6, t5, t4, t3, t2, t1, w[wi3[8*i+7]], mc3[8*i+7]);
	}
	
	for (i=0; i<4; i++)
	{
		FF_44(t7, t6, t5, t4, t3, t2, t1, t0, w[wi4[8*i+0]], mc4[8*i+0]);
		FF_44(t6, t5, t4, t3, t2, t1, t0, t7, w[wi4[8*i+1]], mc4[8*i+1]);
		FF_44(t5, t4, t3, t2, t1, t0, t7, t6, w[wi4[8*i+2]], mc4[8*i+2]);
		FF_44(t4, t3, t2, t1, t0, t7, t6, t5, w[wi4[8*i+3]], mc4[8*i+3]);
		FF_44(t3, t2, t1, t0, t7, t6, t5, t4, w[wi4[8*i+4]], mc4[8*i+4]);
		FF_44(t2, t1, t0, t7, t6, t5, t4, t3, w[wi4[8*i+5]], mc4[8*i+5]);
		FF_44(t1, t0, t7, t6, t5, t4, t3, t2, w[wi4[8*i+6]], mc4[8*i+6]);
		FF_44(t0, t7, t6, t5, t4, t3, t2, t1, w[wi4[8*i+7]], mc4[8*i+7]);
	}

	/*
	// Pass 1
	FF_41(t7, t6, t5, t4, t3, t2, t1, t0, *(w   ));
	FF_41(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 1));
	FF_41(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 2));
	FF_41(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 3));
	FF_41(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 4));
	FF_41(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 5));
	FF_41(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 6));
	FF_41(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 7));

	FF_41(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 8));
	FF_41(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 9));
	FF_41(t5, t4, t3, t2, t1, t0, t7, t6, *(w+10));
	FF_41(t4, t3, t2, t1, t0, t7, t6, t5, *(w+11));
	FF_41(t3, t2, t1, t0, t7, t6, t5, t4, *(w+12));
	FF_41(t2, t1, t0, t7, t6, t5, t4, t3, *(w+13));
	FF_41(t1, t0, t7, t6, t5, t4, t3, t2, *(w+14));
	FF_41(t0, t7, t6, t5, t4, t3, t2, t1, *(w+15));

	FF_41(t7, t6, t5, t4, t3, t2, t1, t0, *(w+16));
	FF_41(t6, t5, t4, t3, t2, t1, t0, t7, *(w+17));
	FF_41(t5, t4, t3, t2, t1, t0, t7, t6, *(w+18));
	FF_41(t4, t3, t2, t1, t0, t7, t6, t5, *(w+19));
	FF_41(t3, t2, t1, t0, t7, t6, t5, t4, *(w+20));
	FF_41(t2, t1, t0, t7, t6, t5, t4, t3, *(w+21));
	FF_41(t1, t0, t7, t6, t5, t4, t3, t2, *(w+22));
	FF_41(t0, t7, t6, t5, t4, t3, t2, t1, *(w+23));

	FF_41(t7, t6, t5, t4, t3, t2, t1, t0, *(w+24));
	FF_41(t6, t5, t4, t3, t2, t1, t0, t7, *(w+25));
	FF_41(t5, t4, t3, t2, t1, t0, t7, t6, *(w+26));
	FF_41(t4, t3, t2, t1, t0, t7, t6, t5, *(w+27));
	FF_41(t3, t2, t1, t0, t7, t6, t5, t4, *(w+28));
	FF_41(t2, t1, t0, t7, t6, t5, t4, t3, *(w+29));
	FF_41(t1, t0, t7, t6, t5, t4, t3, t2, *(w+30));
	FF_41(t0, t7, t6, t5, t4, t3, t2, t1, *(w+31));

	// Pass 2
	FF_42(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 5), 0x452821E6);
	FF_42(t6, t5, t4, t3, t2, t1, t0, t7, *(w+14), 0x38D01377);
	FF_42(t5, t4, t3, t2, t1, t0, t7, t6, *(w+26), 0xBE5466CF);
	FF_42(t4, t3, t2, t1, t0, t7, t6, t5, *(w+18), 0x34E90C6C);
	FF_42(t3, t2, t1, t0, t7, t6, t5, t4, *(w+11), 0xC0AC29B7);
	FF_42(t2, t1, t0, t7, t6, t5, t4, t3, *(w+28), 0xC97C50DD);
	FF_42(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 7), 0x3F84D5B5);
	FF_42(t0, t7, t6, t5, t4, t3, t2, t1, *(w+16), 0xB5470917);

	FF_42(t7, t6, t5, t4, t3, t2, t1, t0, *(w   ), 0x9216D5D9);
	FF_42(t6, t5, t4, t3, t2, t1, t0, t7, *(w+23), 0x8979FB1B);
	FF_42(t5, t4, t3, t2, t1, t0, t7, t6, *(w+20), 0xD1310BA6);
	FF_42(t4, t3, t2, t1, t0, t7, t6, t5, *(w+22), 0x98DFB5AC);
	FF_42(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 1), 0x2FFD72DB);
	FF_42(t2, t1, t0, t7, t6, t5, t4, t3, *(w+10), 0xD01ADFB7);
	FF_42(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 4), 0xB8E1AFED);
	FF_42(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 8), 0x6A267E96);

	FF_42(t7, t6, t5, t4, t3, t2, t1, t0, *(w+30), 0xBA7C9045);
	FF_42(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 3), 0xF12C7F99);
	FF_42(t5, t4, t3, t2, t1, t0, t7, t6, *(w+21), 0x24A19947);
	FF_42(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 9), 0xB3916CF7);
	FF_42(t3, t2, t1, t0, t7, t6, t5, t4, *(w+17), 0x0801F2E2);
	FF_42(t2, t1, t0, t7, t6, t5, t4, t3, *(w+24), 0x858EFC16);
	FF_42(t1, t0, t7, t6, t5, t4, t3, t2, *(w+29), 0x636920D8);
	FF_42(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 6), 0x71574E69);

	FF_42(t7, t6, t5, t4, t3, t2, t1, t0, *(w+19), 0xA458FEA3);
	FF_42(t6, t5, t4, t3, t2, t1, t0, t7, *(w+12), 0xF4933D7E);
	FF_42(t5, t4, t3, t2, t1, t0, t7, t6, *(w+15), 0x0D95748F);
	FF_42(t4, t3, t2, t1, t0, t7, t6, t5, *(w+13), 0x728EB658);
	FF_42(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 2), 0x718BCD58);
	FF_42(t2, t1, t0, t7, t6, t5, t4, t3, *(w+25), 0x82154AEE);
	FF_42(t1, t0, t7, t6, t5, t4, t3, t2, *(w+31), 0x7B54A41D);
	FF_42(t0, t7, t6, t5, t4, t3, t2, t1, *(w+27), 0xC25A59B5);

	// Pass 3
	FF_43(t7, t6, t5, t4, t3, t2, t1, t0, *(w+19), 0x9C30D539);
	FF_43(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 9), 0x2AF26013);
	FF_43(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 4), 0xC5D1B023);
	FF_43(t4, t3, t2, t1, t0, t7, t6, t5, *(w+20), 0x286085F0);
	FF_43(t3, t2, t1, t0, t7, t6, t5, t4, *(w+28), 0xCA417918);
	FF_43(t2, t1, t0, t7, t6, t5, t4, t3, *(w+17), 0xB8DB38EF);
	FF_43(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 8), 0x8E79DCB0);
	FF_43(t0, t7, t6, t5, t4, t3, t2, t1, *(w+22), 0x603A180E);

	FF_43(t7, t6, t5, t4, t3, t2, t1, t0, *(w+29), 0x6C9E0E8B);
	FF_43(t6, t5, t4, t3, t2, t1, t0, t7, *(w+14), 0xB01E8A3E);
	FF_43(t5, t4, t3, t2, t1, t0, t7, t6, *(w+25), 0xD71577C1);
	FF_43(t4, t3, t2, t1, t0, t7, t6, t5, *(w+12), 0xBD314B27);
	FF_43(t3, t2, t1, t0, t7, t6, t5, t4, *(w+24), 0x78AF2FDA);
	FF_43(t2, t1, t0, t7, t6, t5, t4, t3, *(w+30), 0x55605C60);
	FF_43(t1, t0, t7, t6, t5, t4, t3, t2, *(w+16), 0xE65525F3);
	FF_43(t0, t7, t6, t5, t4, t3, t2, t1, *(w+26), 0xAA55AB94);

	FF_43(t7, t6, t5, t4, t3, t2, t1, t0, *(w+31), 0x57489862);
	FF_43(t6, t5, t4, t3, t2, t1, t0, t7, *(w+15), 0x63E81440);
	FF_43(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 7), 0x55CA396A);
	FF_43(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 3), 0x2AAB10B6);
	FF_43(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 1), 0xB4CC5C34);
	FF_43(t2, t1, t0, t7, t6, t5, t4, t3, *(w   ), 0x1141E8CE);
	FF_43(t1, t0, t7, t6, t5, t4, t3, t2, *(w+18), 0xA15486AF);
	FF_43(t0, t7, t6, t5, t4, t3, t2, t1, *(w+27), 0x7C72E993);

	FF_43(t7, t6, t5, t4, t3, t2, t1, t0, *(w+13), 0xB3EE1411);
	FF_43(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 6), 0x636FBC2A);
	FF_43(t5, t4, t3, t2, t1, t0, t7, t6, *(w+21), 0x2BA9C55D);
	FF_43(t4, t3, t2, t1, t0, t7, t6, t5, *(w+10), 0x741831F6);
	FF_43(t3, t2, t1, t0, t7, t6, t5, t4, *(w+23), 0xCE5C3E16);
	FF_43(t2, t1, t0, t7, t6, t5, t4, t3, *(w+11), 0x9B87931E);
	FF_43(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 5), 0xAFD6BA33);
	FF_43(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 2), 0x6C24CF5C);

	// Pass 4. executed only when PASS =4 or 5
	FF_44(t7, t6, t5, t4, t3, t2, t1, t0, *(w+24), 0x7A325381);
	FF_44(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 4), 0x28958677);
	FF_44(t5, t4, t3, t2, t1, t0, t7, t6, *(w   ), 0x3B8F4898);
	FF_44(t4, t3, t2, t1, t0, t7, t6, t5, *(w+14), 0x6B4BB9AF);
	FF_44(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 2), 0xC4BFE81B);
	FF_44(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 7), 0x66282193);
	FF_44(t1, t0, t7, t6, t5, t4, t3, t2, *(w+28), 0x61D809CC);
	FF_44(t0, t7, t6, t5, t4, t3, t2, t1, *(w+23), 0xFB21A991);

	FF_44(t7, t6, t5, t4, t3, t2, t1, t0, *(w+26), 0x487CAC60);
	FF_44(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 6), 0x5DEC8032);
	FF_44(t5, t4, t3, t2, t1, t0, t7, t6, *(w+30), 0xEF845D5D);
	FF_44(t4, t3, t2, t1, t0, t7, t6, t5, *(w+20), 0xE98575B1);
	FF_44(t3, t2, t1, t0, t7, t6, t5, t4, *(w+18), 0xDC262302);
	FF_44(t2, t1, t0, t7, t6, t5, t4, t3, *(w+25), 0xEB651B88);
	FF_44(t1, t0, t7, t6, t5, t4, t3, t2, *(w+19), 0x23893E81);
	FF_44(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 3), 0xD396ACC5);

	FF_44(t7, t6, t5, t4, t3, t2, t1, t0, *(w+22), 0x0F6D6FF3);
	FF_44(t6, t5, t4, t3, t2, t1, t0, t7, *(w+11), 0x83F44239);
	FF_44(t5, t4, t3, t2, t1, t0, t7, t6, *(w+31), 0x2E0B4482);
	FF_44(t4, t3, t2, t1, t0, t7, t6, t5, *(w+21), 0xA4842004);
	FF_44(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 8), 0x69C8F04A);
	FF_44(t2, t1, t0, t7, t6, t5, t4, t3, *(w+27), 0x9E1F9B5E);
	FF_44(t1, t0, t7, t6, t5, t4, t3, t2, *(w+12), 0x21C66842);
	FF_44(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 9), 0xF6E96C9A);

	FF_44(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 1), 0x670C9C61);
	FF_44(t6, t5, t4, t3, t2, t1, t0, t7, *(w+29), 0xABD388F0);
	FF_44(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 5), 0x6A51A0D2);
	FF_44(t4, t3, t2, t1, t0, t7, t6, t5, *(w+15), 0xD8542F68);
	FF_44(t3, t2, t1, t0, t7, t6, t5, t4, *(w+17), 0x960FA728);
	FF_44(t2, t1, t0, t7, t6, t5, t4, t3, *(w+10), 0xAB5133A3);
	FF_44(t1, t0, t7, t6, t5, t4, t3, t2, *(w+16), 0x6EEF0B6C);
	FF_44(t0, t7, t6, t5, t4, t3, t2, t1, *(w+13), 0x137A3BE4);
	*/

	digest[0] += t0;
	digest[1] += t1;
	digest[2] += t2;
	digest[3] += t3;
	digest[4] += t4;
	digest[5] += t5;
	digest[6] += t6;
	digest[7] += t7;
}

void HAVAL5::Transform (word32 *digest, const word32 *w)
{
	register word32 t0 = digest[0],    // make use of
					t1 = digest[1],    // internal registers
					t2 = digest[2],
					t3 = digest[3],
					t4 = digest[4],
					t5 = digest[5],
					t6 = digest[6],
					t7 = digest[7];
	unsigned i;

	for (i=0; i<4; i++)
	{
		FF_51(t7, t6, t5, t4, t3, t2, t1, t0, w[8*i+0]);
		FF_51(t6, t5, t4, t3, t2, t1, t0, t7, w[8*i+1]);
		FF_51(t5, t4, t3, t2, t1, t0, t7, t6, w[8*i+2]);
		FF_51(t4, t3, t2, t1, t0, t7, t6, t5, w[8*i+3]);
		FF_51(t3, t2, t1, t0, t7, t6, t5, t4, w[8*i+4]);
		FF_51(t2, t1, t0, t7, t6, t5, t4, t3, w[8*i+5]);
		FF_51(t1, t0, t7, t6, t5, t4, t3, t2, w[8*i+6]);
		FF_51(t0, t7, t6, t5, t4, t3, t2, t1, w[8*i+7]);
	}
	
	for (i=0; i<4; i++)
	{
		FF_52(t7, t6, t5, t4, t3, t2, t1, t0, w[wi2[8*i+0]], mc2[8*i+0]);
		FF_52(t6, t5, t4, t3, t2, t1, t0, t7, w[wi2[8*i+1]], mc2[8*i+1]);
		FF_52(t5, t4, t3, t2, t1, t0, t7, t6, w[wi2[8*i+2]], mc2[8*i+2]);
		FF_52(t4, t3, t2, t1, t0, t7, t6, t5, w[wi2[8*i+3]], mc2[8*i+3]);
		FF_52(t3, t2, t1, t0, t7, t6, t5, t4, w[wi2[8*i+4]], mc2[8*i+4]);
		FF_52(t2, t1, t0, t7, t6, t5, t4, t3, w[wi2[8*i+5]], mc2[8*i+5]);
		FF_52(t1, t0, t7, t6, t5, t4, t3, t2, w[wi2[8*i+6]], mc2[8*i+6]);
		FF_52(t0, t7, t6, t5, t4, t3, t2, t1, w[wi2[8*i+7]], mc2[8*i+7]);
	}
	
	for (i=0; i<4; i++)
	{
		FF_53(t7, t6, t5, t4, t3, t2, t1, t0, w[wi3[8*i+0]], mc3[8*i+0]);
		FF_53(t6, t5, t4, t3, t2, t1, t0, t7, w[wi3[8*i+1]], mc3[8*i+1]);
		FF_53(t5, t4, t3, t2, t1, t0, t7, t6, w[wi3[8*i+2]], mc3[8*i+2]);
		FF_53(t4, t3, t2, t1, t0, t7, t6, t5, w[wi3[8*i+3]], mc3[8*i+3]);
		FF_53(t3, t2, t1, t0, t7, t6, t5, t4, w[wi3[8*i+4]], mc3[8*i+4]);
		FF_53(t2, t1, t0, t7, t6, t5, t4, t3, w[wi3[8*i+5]], mc3[8*i+5]);
		FF_53(t1, t0, t7, t6, t5, t4, t3, t2, w[wi3[8*i+6]], mc3[8*i+6]);
		FF_53(t0, t7, t6, t5, t4, t3, t2, t1, w[wi3[8*i+7]], mc3[8*i+7]);
	}
	
	for (i=0; i<4; i++)
	{
		FF_54(t7, t6, t5, t4, t3, t2, t1, t0, w[wi4[8*i+0]], mc4[8*i+0]);
		FF_54(t6, t5, t4, t3, t2, t1, t0, t7, w[wi4[8*i+1]], mc4[8*i+1]);
		FF_54(t5, t4, t3, t2, t1, t0, t7, t6, w[wi4[8*i+2]], mc4[8*i+2]);
		FF_54(t4, t3, t2, t1, t0, t7, t6, t5, w[wi4[8*i+3]], mc4[8*i+3]);
		FF_54(t3, t2, t1, t0, t7, t6, t5, t4, w[wi4[8*i+4]], mc4[8*i+4]);
		FF_54(t2, t1, t0, t7, t6, t5, t4, t3, w[wi4[8*i+5]], mc4[8*i+5]);
		FF_54(t1, t0, t7, t6, t5, t4, t3, t2, w[wi4[8*i+6]], mc4[8*i+6]);
		FF_54(t0, t7, t6, t5, t4, t3, t2, t1, w[wi4[8*i+7]], mc4[8*i+7]);
	}
	
	for (i=0; i<4; i++)
	{
		FF_55(t7, t6, t5, t4, t3, t2, t1, t0, w[wi5[8*i+0]], mc5[8*i+0]);
		FF_55(t6, t5, t4, t3, t2, t1, t0, t7, w[wi5[8*i+1]], mc5[8*i+1]);
		FF_55(t5, t4, t3, t2, t1, t0, t7, t6, w[wi5[8*i+2]], mc5[8*i+2]);
		FF_55(t4, t3, t2, t1, t0, t7, t6, t5, w[wi5[8*i+3]], mc5[8*i+3]);
		FF_55(t3, t2, t1, t0, t7, t6, t5, t4, w[wi5[8*i+4]], mc5[8*i+4]);
		FF_55(t2, t1, t0, t7, t6, t5, t4, t3, w[wi5[8*i+5]], mc5[8*i+5]);
		FF_55(t1, t0, t7, t6, t5, t4, t3, t2, w[wi5[8*i+6]], mc5[8*i+6]);
		FF_55(t0, t7, t6, t5, t4, t3, t2, t1, w[wi5[8*i+7]], mc5[8*i+7]);
	}
	

	// Pass 1
	/*
	FF_51(t7, t6, t5, t4, t3, t2, t1, t0, *(w   ));
	FF_51(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 1));
	FF_51(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 2));
	FF_51(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 3));
	FF_51(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 4));
	FF_51(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 5));
	FF_51(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 6));
	FF_51(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 7));

	FF_51(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 8));
	FF_51(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 9));
	FF_51(t5, t4, t3, t2, t1, t0, t7, t6, *(w+10));
	FF_51(t4, t3, t2, t1, t0, t7, t6, t5, *(w+11));
	FF_51(t3, t2, t1, t0, t7, t6, t5, t4, *(w+12));
	FF_51(t2, t1, t0, t7, t6, t5, t4, t3, *(w+13));
	FF_51(t1, t0, t7, t6, t5, t4, t3, t2, *(w+14));
	FF_51(t0, t7, t6, t5, t4, t3, t2, t1, *(w+15));

	FF_51(t7, t6, t5, t4, t3, t2, t1, t0, *(w+16));
	FF_51(t6, t5, t4, t3, t2, t1, t0, t7, *(w+17));
	FF_51(t5, t4, t3, t2, t1, t0, t7, t6, *(w+18));
	FF_51(t4, t3, t2, t1, t0, t7, t6, t5, *(w+19));
	FF_51(t3, t2, t1, t0, t7, t6, t5, t4, *(w+20));
	FF_51(t2, t1, t0, t7, t6, t5, t4, t3, *(w+21));
	FF_51(t1, t0, t7, t6, t5, t4, t3, t2, *(w+22));
	FF_51(t0, t7, t6, t5, t4, t3, t2, t1, *(w+23));

	FF_51(t7, t6, t5, t4, t3, t2, t1, t0, *(w+24));
	FF_51(t6, t5, t4, t3, t2, t1, t0, t7, *(w+25));
	FF_51(t5, t4, t3, t2, t1, t0, t7, t6, *(w+26));
	FF_51(t4, t3, t2, t1, t0, t7, t6, t5, *(w+27));
	FF_51(t3, t2, t1, t0, t7, t6, t5, t4, *(w+28));
	FF_51(t2, t1, t0, t7, t6, t5, t4, t3, *(w+29));
	FF_51(t1, t0, t7, t6, t5, t4, t3, t2, *(w+30));
	FF_51(t0, t7, t6, t5, t4, t3, t2, t1, *(w+31));

	// Pass 2
	FF_52(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 5), 0x452821E6);
	FF_52(t6, t5, t4, t3, t2, t1, t0, t7, *(w+14), 0x38D01377);
	FF_52(t5, t4, t3, t2, t1, t0, t7, t6, *(w+26), 0xBE5466CF);
	FF_52(t4, t3, t2, t1, t0, t7, t6, t5, *(w+18), 0x34E90C6C);
	FF_52(t3, t2, t1, t0, t7, t6, t5, t4, *(w+11), 0xC0AC29B7);
	FF_52(t2, t1, t0, t7, t6, t5, t4, t3, *(w+28), 0xC97C50DD);
	FF_52(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 7), 0x3F84D5B5);
	FF_52(t0, t7, t6, t5, t4, t3, t2, t1, *(w+16), 0xB5470917);

	FF_52(t7, t6, t5, t4, t3, t2, t1, t0, *(w   ), 0x9216D5D9);
	FF_52(t6, t5, t4, t3, t2, t1, t0, t7, *(w+23), 0x8979FB1B);
	FF_52(t5, t4, t3, t2, t1, t0, t7, t6, *(w+20), 0xD1310BA6);
	FF_52(t4, t3, t2, t1, t0, t7, t6, t5, *(w+22), 0x98DFB5AC);
	FF_52(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 1), 0x2FFD72DB);
	FF_52(t2, t1, t0, t7, t6, t5, t4, t3, *(w+10), 0xD01ADFB7);
	FF_52(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 4), 0xB8E1AFED);
	FF_52(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 8), 0x6A267E96);

	FF_52(t7, t6, t5, t4, t3, t2, t1, t0, *(w+30), 0xBA7C9045);
	FF_52(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 3), 0xF12C7F99);
	FF_52(t5, t4, t3, t2, t1, t0, t7, t6, *(w+21), 0x24A19947);
	FF_52(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 9), 0xB3916CF7);
	FF_52(t3, t2, t1, t0, t7, t6, t5, t4, *(w+17), 0x0801F2E2);
	FF_52(t2, t1, t0, t7, t6, t5, t4, t3, *(w+24), 0x858EFC16);
	FF_52(t1, t0, t7, t6, t5, t4, t3, t2, *(w+29), 0x636920D8);
	FF_52(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 6), 0x71574E69);

	FF_52(t7, t6, t5, t4, t3, t2, t1, t0, *(w+19), 0xA458FEA3);
	FF_52(t6, t5, t4, t3, t2, t1, t0, t7, *(w+12), 0xF4933D7E);
	FF_52(t5, t4, t3, t2, t1, t0, t7, t6, *(w+15), 0x0D95748F);
	FF_52(t4, t3, t2, t1, t0, t7, t6, t5, *(w+13), 0x728EB658);
	FF_52(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 2), 0x718BCD58);
	FF_52(t2, t1, t0, t7, t6, t5, t4, t3, *(w+25), 0x82154AEE);
	FF_52(t1, t0, t7, t6, t5, t4, t3, t2, *(w+31), 0x7B54A41D);
	FF_52(t0, t7, t6, t5, t4, t3, t2, t1, *(w+27), 0xC25A59B5);

	// Pass 3
	FF_53(t7, t6, t5, t4, t3, t2, t1, t0, *(w+19), 0x9C30D539);
	FF_53(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 9), 0x2AF26013);
	FF_53(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 4), 0xC5D1B023);
	FF_53(t4, t3, t2, t1, t0, t7, t6, t5, *(w+20), 0x286085F0);
	FF_53(t3, t2, t1, t0, t7, t6, t5, t4, *(w+28), 0xCA417918);
	FF_53(t2, t1, t0, t7, t6, t5, t4, t3, *(w+17), 0xB8DB38EF);
	FF_53(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 8), 0x8E79DCB0);
	FF_53(t0, t7, t6, t5, t4, t3, t2, t1, *(w+22), 0x603A180E);

	FF_53(t7, t6, t5, t4, t3, t2, t1, t0, *(w+29), 0x6C9E0E8B);
	FF_53(t6, t5, t4, t3, t2, t1, t0, t7, *(w+14), 0xB01E8A3E);
	FF_53(t5, t4, t3, t2, t1, t0, t7, t6, *(w+25), 0xD71577C1);
	FF_53(t4, t3, t2, t1, t0, t7, t6, t5, *(w+12), 0xBD314B27);
	FF_53(t3, t2, t1, t0, t7, t6, t5, t4, *(w+24), 0x78AF2FDA);
	FF_53(t2, t1, t0, t7, t6, t5, t4, t3, *(w+30), 0x55605C60);
	FF_53(t1, t0, t7, t6, t5, t4, t3, t2, *(w+16), 0xE65525F3);
	FF_53(t0, t7, t6, t5, t4, t3, t2, t1, *(w+26), 0xAA55AB94);

	FF_53(t7, t6, t5, t4, t3, t2, t1, t0, *(w+31), 0x57489862);
	FF_53(t6, t5, t4, t3, t2, t1, t0, t7, *(w+15), 0x63E81440);
	FF_53(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 7), 0x55CA396A);
	FF_53(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 3), 0x2AAB10B6);
	FF_53(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 1), 0xB4CC5C34);
	FF_53(t2, t1, t0, t7, t6, t5, t4, t3, *(w   ), 0x1141E8CE);
	FF_53(t1, t0, t7, t6, t5, t4, t3, t2, *(w+18), 0xA15486AF);
	FF_53(t0, t7, t6, t5, t4, t3, t2, t1, *(w+27), 0x7C72E993);

	FF_53(t7, t6, t5, t4, t3, t2, t1, t0, *(w+13), 0xB3EE1411);
	FF_53(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 6), 0x636FBC2A);
	FF_53(t5, t4, t3, t2, t1, t0, t7, t6, *(w+21), 0x2BA9C55D);
	FF_53(t4, t3, t2, t1, t0, t7, t6, t5, *(w+10), 0x741831F6);
	FF_53(t3, t2, t1, t0, t7, t6, t5, t4, *(w+23), 0xCE5C3E16);
	FF_53(t2, t1, t0, t7, t6, t5, t4, t3, *(w+11), 0x9B87931E);
	FF_53(t1, t0, t7, t6, t5, t4, t3, t2, *(w+ 5), 0xAFD6BA33);
	FF_53(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 2), 0x6C24CF5C);

	// Pass 4. executed only when PASS =4 or 5
	FF_54(t7, t6, t5, t4, t3, t2, t1, t0, *(w+24), 0x7A325381);
	FF_54(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 4), 0x28958677);
	FF_54(t5, t4, t3, t2, t1, t0, t7, t6, *(w   ), 0x3B8F4898);
	FF_54(t4, t3, t2, t1, t0, t7, t6, t5, *(w+14), 0x6B4BB9AF);
	FF_54(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 2), 0xC4BFE81B);
	FF_54(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 7), 0x66282193);
	FF_54(t1, t0, t7, t6, t5, t4, t3, t2, *(w+28), 0x61D809CC);
	FF_54(t0, t7, t6, t5, t4, t3, t2, t1, *(w+23), 0xFB21A991);

	FF_54(t7, t6, t5, t4, t3, t2, t1, t0, *(w+26), 0x487CAC60);
	FF_54(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 6), 0x5DEC8032);
	FF_54(t5, t4, t3, t2, t1, t0, t7, t6, *(w+30), 0xEF845D5D);
	FF_54(t4, t3, t2, t1, t0, t7, t6, t5, *(w+20), 0xE98575B1);
	FF_54(t3, t2, t1, t0, t7, t6, t5, t4, *(w+18), 0xDC262302);
	FF_54(t2, t1, t0, t7, t6, t5, t4, t3, *(w+25), 0xEB651B88);
	FF_54(t1, t0, t7, t6, t5, t4, t3, t2, *(w+19), 0x23893E81);
	FF_54(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 3), 0xD396ACC5);

	FF_54(t7, t6, t5, t4, t3, t2, t1, t0, *(w+22), 0x0F6D6FF3);
	FF_54(t6, t5, t4, t3, t2, t1, t0, t7, *(w+11), 0x83F44239);
	FF_54(t5, t4, t3, t2, t1, t0, t7, t6, *(w+31), 0x2E0B4482);
	FF_54(t4, t3, t2, t1, t0, t7, t6, t5, *(w+21), 0xA4842004);
	FF_54(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 8), 0x69C8F04A);
	FF_54(t2, t1, t0, t7, t6, t5, t4, t3, *(w+27), 0x9E1F9B5E);
	FF_54(t1, t0, t7, t6, t5, t4, t3, t2, *(w+12), 0x21C66842);
	FF_54(t0, t7, t6, t5, t4, t3, t2, t1, *(w+ 9), 0xF6E96C9A);

	FF_54(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 1), 0x670C9C61);
	FF_54(t6, t5, t4, t3, t2, t1, t0, t7, *(w+29), 0xABD388F0);
	FF_54(t5, t4, t3, t2, t1, t0, t7, t6, *(w+ 5), 0x6A51A0D2);
	FF_54(t4, t3, t2, t1, t0, t7, t6, t5, *(w+15), 0xD8542F68);
	FF_54(t3, t2, t1, t0, t7, t6, t5, t4, *(w+17), 0x960FA728);
	FF_54(t2, t1, t0, t7, t6, t5, t4, t3, *(w+10), 0xAB5133A3);
	FF_54(t1, t0, t7, t6, t5, t4, t3, t2, *(w+16), 0x6EEF0B6C);
	FF_54(t0, t7, t6, t5, t4, t3, t2, t1, *(w+13), 0x137A3BE4);

	FF_55(t7, t6, t5, t4, t3, t2, t1, t0, *(w+27), 0xBA3BF050);
	FF_55(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 3), 0x7EFB2A98);
	FF_55(t5, t4, t3, t2, t1, t0, t7, t6, *(w+21), 0xA1F1651D);
	FF_55(t4, t3, t2, t1, t0, t7, t6, t5, *(w+26), 0x39AF0176);
	FF_55(t3, t2, t1, t0, t7, t6, t5, t4, *(w+17), 0x66CA593E);
	FF_55(t2, t1, t0, t7, t6, t5, t4, t3, *(w+11), 0x82430E88);
	FF_55(t1, t0, t7, t6, t5, t4, t3, t2, *(w+20), 0x8CEE8619);
	FF_55(t0, t7, t6, t5, t4, t3, t2, t1, *(w+29), 0x456F9FB4);

	FF_55(t7, t6, t5, t4, t3, t2, t1, t0, *(w+19), 0x7D84A5C3);
	FF_55(t6, t5, t4, t3, t2, t1, t0, t7, *(w   ), 0x3B8B5EBE);
	FF_55(t5, t4, t3, t2, t1, t0, t7, t6, *(w+12), 0xE06F75D8);
	FF_55(t4, t3, t2, t1, t0, t7, t6, t5, *(w+ 7), 0x85C12073);
	FF_55(t3, t2, t1, t0, t7, t6, t5, t4, *(w+13), 0x401A449F);
	FF_55(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 8), 0x56C16AA6);
	FF_55(t1, t0, t7, t6, t5, t4, t3, t2, *(w+31), 0x4ED3AA62);
	FF_55(t0, t7, t6, t5, t4, t3, t2, t1, *(w+10), 0x363F7706);

	FF_55(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 5), 0x1BFEDF72);
	FF_55(t6, t5, t4, t3, t2, t1, t0, t7, *(w+ 9), 0x429B023D);
	FF_55(t5, t4, t3, t2, t1, t0, t7, t6, *(w+14), 0x37D0D724);
	FF_55(t4, t3, t2, t1, t0, t7, t6, t5, *(w+30), 0xD00A1248);
	FF_55(t3, t2, t1, t0, t7, t6, t5, t4, *(w+18), 0xDB0FEAD3);
	FF_55(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 6), 0x49F1C09B);
	FF_55(t1, t0, t7, t6, t5, t4, t3, t2, *(w+28), 0x075372C9);
	FF_55(t0, t7, t6, t5, t4, t3, t2, t1, *(w+24), 0x80991B7B);

	FF_55(t7, t6, t5, t4, t3, t2, t1, t0, *(w+ 2), 0x25D479D8);
	FF_55(t6, t5, t4, t3, t2, t1, t0, t7, *(w+23), 0xF6E8DEF7);
	FF_55(t5, t4, t3, t2, t1, t0, t7, t6, *(w+16), 0xE3FE501A);
	FF_55(t4, t3, t2, t1, t0, t7, t6, t5, *(w+22), 0xB6794C3B);
	FF_55(t3, t2, t1, t0, t7, t6, t5, t4, *(w+ 4), 0x976CE0BD);
	FF_55(t2, t1, t0, t7, t6, t5, t4, t3, *(w+ 1), 0x04C006BA);
	FF_55(t1, t0, t7, t6, t5, t4, t3, t2, *(w+25), 0xC1A94FB6);
	FF_55(t0, t7, t6, t5, t4, t3, t2, t1, *(w+15), 0x409F60C4);
	*/

	digest[0] += t0;
	digest[1] += t1;
	digest[2] += t2;
	digest[3] += t3;
	digest[4] += t4;
	digest[5] += t5;
	digest[6] += t6;
	digest[7] += t7;
}
