// ecbasic.cpp - written and placed in the public domain by Wei Dai

#include "nbtheory.h"
#include "ecbasic.h"

boolean EC2N::Equal(const EC2N::Point &P, const EC2N::Point &Q) const
{
	if (P.identity && Q.identity)
		return TRUE;

	if (P.identity && !Q.identity)
		return FALSE;

	if (!P.identity && Q.identity)
		return FALSE;

	return (field.Equal(P.x,Q.x) && field.Equal(P.y,Q.y));
}

EC2N::Point EC2N::Inverse(const EC2N::Point &P) const
{
    if (P.identity)
        return P;
    else
        return Point(P.x, field.Add(P.x, P.y));
}

EC2N::Point EC2N::Add(const EC2N::Point &P, const EC2N::Point &Q) const
{
	if (P.identity) return Q;
	if (Q.identity) return P;
	if (Equal(P, Q)) return Double(P);
	if (field.Equal(P.x, Q.x) && field.Equal(P.y, field.Add(Q.x, Q.y))) return Identity();

	Field::Element t = field.Divide(field.Add(P.y, Q.y), field.Add(P.x, Q.x));
	EC2N::Point R;
	R.identity = FALSE;
	R.x = field.Square(t);
	field.Accumulate(R.x, t);
	field.Accumulate(R.x, Q.x);
	field.Accumulate(R.x, a);
	R.y = field.Multiply(t, R.x);
	field.Accumulate(R.x, P.x);
	field.Accumulate(R.y, R.x);
	field.Accumulate(R.y, P.y);
	return R;
}

EC2N::Point EC2N::Double(const EC2N::Point &P) const
{
	if (P.identity) return P;
	if (field.Equal(P.x, field.Identity())) return Identity();

	Field::Element t = field.Add(P.x, field.Divide(P.y, P.x));
	EC2N::Point R;
	R.identity = FALSE;
	R.x = field.Square(t);
	field.Accumulate(R.x, t);
	field.Accumulate(R.x, a);
	R.y = field.Square(P.x);
	field.Accumulate(R.y, field.Multiply(t, R.x));
	field.Accumulate(R.y, R.x);
	return R;
}

EC2N::Point EC2N::Multiply(const Integer &k, const EC2N::Point &P) const
{
	return GeneralizedMultiplication(*this, P, k);
}

EC2N::Point EC2N::CascadeMultiply(const Integer &k1, const Point &P, const Integer &k2, const Point &Q) const
{
	return CascadeMultiplication(*this, P, k1, Q, k2);
}

Integer EC2N::Order(const EC2N::Point &P, const Integer &cardinality) const
{
	Point Q;
	Integer p = cardinality, n = cardinality;

	assert(100<=primeTableSize);

	for (unsigned int i=0; i<100; i++)
		if (p%primeTable[i]==0)
		{
			do
			{
				n /= primeTable[i];
				p /= primeTable[i];
			}
			while (p%primeTable[i]==0);

			Q = Multiply(n, P);
			while (!Q.identity)
			{
				Q = Multiply(primeTable[i], Q);
				n *= primeTable[i];
				assert(n<=cardinality);
			}
		}

	if (p==1)
		return n;

	assert(IsPrime(p));
	p = n/p;
	Q = Multiply(p, P);
	if (Q.identity)
		return p;
	else
		return n;
}

// ******************************************************************

boolean ECP::Equal(const ECP::Point &P, const ECP::Point &Q) const
{
	if (P.identity && Q.identity)
		return TRUE;

	if (P.identity && !Q.identity)
		return FALSE;

	if (!P.identity && Q.identity)
		return FALSE;

	return (field.Equal(P.x,Q.x) && field.Equal(P.y,Q.y));
}

ECP::Point ECP::Inverse(const ECP::Point &P) const
{
	if (P.identity)
        return P;
    else
        return Point(P.x, field.Inverse(P.y));
}

ECP::Point ECP::Add(const ECP::Point &P, const ECP::Point &Q) const
{
	if (P.identity) return Q;
	if (Q.identity) return P;
	if (Equal(P, Q)) return Double(P);
	if (field.Equal(P.x, Q.x) && field.Equal(P.y, field.Inverse(Q.y))) return Identity();

	GFP::Element t = field.Divide(field.Subtract(Q.y, P.y), field.Subtract(Q.x, P.x));
	ECP::Point R;
	R.identity = FALSE;
	R.x = field.Square(t);
	R.x = field.Subtract(R.x, P.x);
	R.x = field.Subtract(R.x, Q.x);
	R.y = field.Subtract(P.x, R.x);
	R.y = field.Multiply(t, R.y);
	R.y = field.Subtract(R.y, P.y);
	return R;
}

ECP::Point ECP::Double(const ECP::Point &P) const
{
	if (P.identity) return P;

	GFP::Element t = field.Square(P.x);
	field.Accumulate(t, field.Double(t));
	field.Accumulate(t, a);
	t = field.Divide(t, field.Double(P.y));
	ECP::Point R;
	R.identity = FALSE;
	R.x = field.Subtract(field.Square(t), field.Double(P.x));
	R.y = field.Subtract(P.x, R.x);
	R.y = field.Multiply(t, R.y);
	R.y = field.Subtract(R.y, P.y);
	return R;
}

ECP::Point ECP::Multiply(const Integer &k, const ECP::Point &P) const
{
	return GeneralizedMultiplication(*this, P, k);
}

ECP::Point ECP::CascadeMultiply(const Integer &k1, const Point &P, const Integer &k2, const Point &Q) const
{
	return CascadeMultiplication(*this, P, k1, Q, k2);
}

Integer ECP::Order(const ECP::Point &P, const Integer &cardinality) const
{
	Point Q;
	Integer p = cardinality, n = cardinality;

	assert(100<=primeTableSize);

	for (unsigned int i=0; i<100; i++)
		if (p%primeTable[i]==0)
		{
			do
			{
				n /= primeTable[i];
				p /= primeTable[i];
			}
			while (p%primeTable[i]==0);

			Q = Multiply(n, P);
			while (!Q.identity)
			{
				Q = Multiply(primeTable[i], Q);
				n *= primeTable[i];
				assert(n<=cardinality);
			}
		}

	if (p==1)
		return n;

	assert(IsPrime(p));
	p = n/p;
	Q = Multiply(p, P);
	if (Q.identity)
		return p;
	else
		return n;
}
