/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include "ObjectStore.h"
#include "stdlib.h"



JnhObjectStore::JnhObjectStore(const char * theDir, 
                               const char * theName, 
                               uint32 active_objects,
                               uint32 object_size) {
  state = OS_CLOSED;
  error = 0;
  mutex = PKI_MUTEX_INIT;
  pkiCreateMutex(&mutex, true);

  strncpy(dbDir, theDir, sizeof(dbDir));
  strncpy(dbName, theName, sizeof(dbName));
  
  db = new isam(dbDir);
  
  error = db->open(theName, false);
  if (error == ISM_FILE_NOT_FOUND) error = db->create(theName, active_objects, object_size);
  if (error == 0) state = OS_OPEN;
  else state = OS_ERROR;
  pkiUnlockMutex(&mutex);

}

os_state_t JnhObjectStore::getState(void) {
  return state;
}

uint32 JnhObjectStore::getError(void) {
  return error;
}

JnhObjectStore::~JnhObjectStore() {
  pkiLockMutex(&mutex);
  db->close();
  delete db;
  pkiDestroyMutex(&mutex);
}
  
uint32 JnhObjectStore::CreateObject(uint32 * key, 
                                    uint32 objClass, 
                                    r_buffer_t data) {
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  }
  pkiLockMutex(&mutex);
  *key = db->new_key();
  error = db->store(*key, objClass, data, false);
  pkiUnlockMutex(&mutex);
  return error;
}

uint32 JnhObjectStore::StoreObject(uint32 key, 
                                   uint32 objClass, 
                                   r_buffer_t data) {
  if (state != OS_OPEN) {
   error = ISM_DB_NOT_OPEN;
   return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->store(key, objClass, data, true);
  pkiUnlockMutex(&mutex);
  return error;
}

uint32 JnhObjectStore::FetchObject(uint32 key, 
                                   uint32 & objClass, 
                                   buffer_t & data) {
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->fetch(key, objClass, data);
  pkiUnlockMutex(&mutex);
  return error;
}

uint32 JnhObjectStore::DeleteObject(uint32 key) {
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->delete_record(key);
  pkiUnlockMutex(&mutex);
  return error;
}

uint32 JnhObjectStore::ListActiveObjects(void (* callback)(uint32 key, 
                                                           uint32 objClass)) {
  buffer_t keys;
  buffer_t flags;
  unsigned i;
  uint32 key;
  uint32 flag;
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->enumerate_bits(ObjClStActive, keys, &flags);
  pkiUnlockMutex(&mutex);
  if (error) return error;
  for (i=0; i<keys.data_len; i+=sizeof(uint32)) {
    key = *((uint32 *)(keys.data+i));
    flag = *((uint32 *)(flags.data+i));
    callback(key, flag);
//    error = callback(key, flag);
//    if (error) return error;
  };
  return error;
}


uint32 JnhObjectStore::ListSurrogateObjects(void (* callback)(uint32 key, 
                                                              uint32 objClass)) {
  buffer_t keys;
  buffer_t flags;
  unsigned i;
  uint32 key;
  uint32 flag;
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->enumerate_bits(ObjClStSurrogate, keys, &flags);
  pkiUnlockMutex(&mutex);
  if (error) return error;
  for (i=0; i<keys.data_len; i+=sizeof(uint32)) {
    key = *((uint32 *)(keys.data+i));
    flag = *((uint32 *)(flags.data+i));
    callback(key, flag);
//    error = callback(key, flag);
//    if (error) return error;
  };
  return error;
}


uint32 JnhObjectStore::ListObjectsByClass(uint32 objClass,
                                          void (* callback)(uint32 key, 
                                                            uint32 objClassSt)) {
  buffer_t keys;
  buffer_t flags;
  unsigned i;
  uint32 key;
  uint32 flag;
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->enumerate_bits(objClass, keys, &flags);
  pkiUnlockMutex(&mutex);
  if (error) return error;
  for (i=0; i<keys.data_len; i+=sizeof(uint32)) {
    key = *((uint32 *)(keys.data+i));
    flag = *((uint32 *)(flags.data+i));
    callback(key, flag);
//    error = callback(key, flag);
//    if (error) return error;
  };
  return error;
}


uint32 JnhObjectStore::ListObjectsByState(uint32 objState,
                                          void (* callback)(uint32 key, 
                                                            uint32 objClassSt)) {

  buffer_t keys;
  buffer_t flags;
  unsigned i;
  uint32 key;
  uint32 flag;
  objState = objState & ObjStAll;
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  pkiLockMutex(&mutex);
  error = db->enumerate_match(ObjStAll, objState, keys, &flags);
  pkiUnlockMutex(&mutex);
  if (error) return error;
  for (i=0; i<keys.data_len; i+=sizeof(uint32)) {
    key = *((uint32 *)(keys.data+i));
    flag = *((uint32 *)(flags.data+i));
    callback(key, flag);
//    error = callback(key, flag);
//    if (error) return error;
  };
  return error;
}


uint32 JnhObjectStore::ListKeysByMatch(uint32 value,
                                       uint32 mask,
                                       buffer_t & results) {

  value = value & mask;
  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };

  results.clear();
  pkiLockMutex(&mutex);
  error = db->enumerate_match(mask, value, results, NULL);
  pkiUnlockMutex(&mutex);
  return error;
}
 
uint32 JnhObjectStore::ListKeysByBits(uint32 mask,
                                      buffer_t & results) {

  if (state != OS_OPEN) {
    error = ISM_DB_NOT_OPEN;
    return ISM_DB_NOT_OPEN;
  };
  
  results.clear();
  
  pkiLockMutex(&mutex);
  error = db->enumerate_bits(mask, results, NULL);
  pkiUnlockMutex(&mutex);
  return error;
}

  
