/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include "ibmcylinkcsp.hpp"
#include "bytearraykey.hpp"

#include "exception.hpp"
#include "cssmerr.h"
#include "misc.hpp"

/**********************************************************
 *
 *	ByteArrayKey class
 *
 **********************************************************/
ByteArrayKey::ByteArrayKey(uint32 algId)
:KeyMaterial(algId, CSSM_KEYBLOB_RAW, CSSM_KEYBLOB_RAW_FORMAT_NONE, CSSM_KEYCLASS_SESSION_KEY)
{
	blob.setDestroy(CSSM_TRUE);
}

ByteArrayKey::ByteArrayKey(ByteArrayKey& keyMat) 
:KeyMaterial(keyMat.algId, keyMat.blobType, keyMat.format, CSSM_KEYCLASS_SESSION_KEY)
{
	blob.setData(keyMat.blob.getLength(), keyMat.blob.getData());
	blob.setDestroy(CSSM_TRUE);
}

ByteArrayKey::~ByteArrayKey() {
	blob.setLength(0);
}

void* ByteArrayKey::importCssmKey(CSSM_KEY_PTR key) {
	CSSM_GUID	tmpGuid = IBMCYLINKCSP_GUID;

	if (this->blob.getLength() != 0)
		//key is immutable
		throw InternalException(CSSM_CSP_KEY_PROTECTED);

	if (key == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY_POINTER);

	if (memcmp(&key->KeyHeader.CspId, &tmpGuid, sizeof(CSSM_GUID)) != 0)
		throw InputException(CSSM_CSP_CONVERSION_ERROR);

	if (key->KeyHeader.AlgorithmId != this->algId)
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);

	if (key->KeyHeader.BlobType != CSSM_KEYBLOB_RAW)
		throw InputException(CSSM_CSP_KEY_BLOBTYPE_INCORRECT);

	if (key->KeyHeader.Format != CSSM_KEYBLOB_RAW_FORMAT_NONE)
		throw InputException(CSSM_KEY_FORMAT_NOT_SUPPORTED);

	if (key->KeyHeader.KeyClass != CSSM_KEYCLASS_SESSION_KEY)
		throw InputException(CSSM_CSP_INVALID_KEYCLASS);

	if (key->KeyData.Data == NULL)
		throw InputException(CSSM_CSP_KEY_BAD_KEY);

	this->blob.setData(key->KeyData.Length, key->KeyData.Data);

	return(key->KeyData.Data + key->KeyData.Length);
}

CssmData& ByteArrayKey::exportCssmData() {
	CssmData *tmpKey;

	if (blob.getLength() == 0) {
		tmpKey = new CssmData((uint32)0);
	}
	else {
		tmpKey = new CssmData(blob.getLength(), blob.getData());
	}

	return(*tmpKey);
}

uint32 ByteArrayKey::getSizeInBits() {
	return(blob.getLength() * 8);
}

uint32 ByteArrayKey::getEffectiveSizeInBits() {
	return(blob.getLength() * 8);
}

uint8* ByteArrayKey::getBlob() {
	uint8	*tmpBlob;
	
	if (this->blob.getLength() == 0) {
		return(NULL);
	}

	tmpBlob = new uint8[this->blob.getLength()];

	memcpy(tmpBlob, this->blob.getData(), this->blob.getLength());
	return(tmpBlob);
}

void ByteArrayKey::setBlob(uint32 _length, uint8* _blob) {
	this->blob.setData(_length, _blob);
}

void ByteArrayKey::destroy() {
	blob.destroy();
}

KeyMaterial* ByteArrayKey::clone() {
	ByteArrayKey *material = new ByteArrayKey(algId);

	copy(material);
	return (material);
}


void ByteArrayKey::copy(ByteArrayKey *material) {
	if (material == NULL)
		throw InternalException(CSSM_CSP_INPUT_DATA_ERROR);

	if (material->getAlgId() != this->algId)
		throw InternalException(CSSM_CSP_KEY_ALGID_NOTMATCH);

	KeyMaterial::copy(material);
	
	material->cspGuid = cspGuid;
	material->blob.setData(blob.getLength(), blob.getData());
}
