/**
*  Class TripleDES
* <P>
*   The 3-Way cypher, inspired, inter alia, by
*	code from Schneier's <cite>Applied Cryptography</cite 2nd edition
*  <P>
*  Coded Mr. Tines &lt;tines@windsong.demon.co.uk&gt; 1998
*  and released into the public domain
*  <P>
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*  <P>
* @author Mr. Tines
* @version 1.0 23-Dec-1998
*/


package uk.co.demon.windsong.crypt.cea;

public class TripleDES implements CEA
{
    static DES_SPboxes sp = null;

    int[] ekeysched = null;
    int[] dkeysched = null;

    public TripleDES()
    {
        if(null == sp) sp = new DES_SPboxes();
    }

    /**
    * Initialise the object with one or three key blocks
    * @param key array of key bytes, 1 or 3 key block lengths
    * @param triple true if three keys for triple application
    */
    public void init(byte[] key, int offset, boolean triple)
    {
        if(triple) throw new RuntimeException("9-DES not supported");
        ekeysched = new int[3*DESkeys.LONGS];
        dkeysched = new int[3*DESkeys.LONGS];

        DESkeys.triple(key, offset, DESkeys.EN0, ekeysched, 0);
        DESkeys.triple(key, offset, DESkeys.DE1, dkeysched, 0);
    }

    /**
    * Transform one block in ecb mode
    * @param encrypt true if forwards transformation
    * @param in input block
    * @param offin offset into block of input data
    * @param out output block
    * @param offout offset into block of output data
    */
    public void ecb(boolean encrypt, byte[] in, int offin, byte[] out,
        int offout)
    {
        if(encrypt)
        {
            DESengine.triple(in, offin, out, offout,
                ekeysched, sp );
        }
        else
        {
            DESengine.triple(in, offin, out, offout,
                dkeysched, sp );
        }
    }

    /**
    * Wipe key schedule information
    */
    public void destroy()
    {
        if(ekeysched != null)
        {
            for(int e=0; e<ekeysched.length; ++e) ekeysched[e] = 0;
        }
        ekeysched = null;
        if(dkeysched != null)
        {
            for(int d=0; d<dkeysched.length; ++d) dkeysched[d] = 0;
        }
        dkeysched = null;
    }

    /**
    * Provide infomation of desired key size
    * @return byte length of key
    */
    public int getKeysize()
    {
        return 3*DESkeys.KEYSIZE;
    }

    /**
    * Provide infomation of algorithm block size
    * @return byte length of block
    */
    public int getBlocksize()
    {
        return DESengine.BLOCKSIZE;
    }

    /**
    * Drives test harness
    */
    public static void main(String[] args)
    {
        test();
    }

/* Validation set:
 *
 * Single-length key, single-length plaintext -
 * Key    : 0123 4567 89ab cdef
 * Plain  : 0123 4567 89ab cde7
 * Cipher : c957 4425 6a5e d31d
 *
 * Double-length key, single-length plaintext -
 * Key    : 0123 4567 89ab cdef fedc ba98 7654 3210
 * Plain  : 0123 4567 89ab cde7
 * Cipher : 7f1d 0a77 826b 8aff
 *
 * Triple-length key, single-length plaintext -
 * Key    : 0123 4567 89ab cdef fedc ba98 7654 3210 89ab cdef 0123 4567
 * Plain  : 0123 4567 89ab cde7
 * Cipher : de0b 7c06 ae5e 0ed5
 *
*/

    private static void printvec(String stub, byte [] out)
    {
        System.out.print(stub+": "+Integer.toHexString(out[0]&0xFF)+
                             Integer.toHexString(out[1]&0xFF));
        System.out.print(" "+Integer.toHexString(out[2]&0xFF)+
                             Integer.toHexString(out[3]&0xFF));
        System.out.print(" "+Integer.toHexString(out[4]&0xFF)+
                             Integer.toHexString(out[5]&0xFF));
        System.out.println(" "+Integer.toHexString(out[6]&0xFF)+
                             Integer.toHexString(out[7]&0xFF));
    }

    public static void test()
    {
        byte[] key1  = {0x01, 0x23, 0x45, 0x67,
            (byte)0x89, (byte)0xab, (byte)0xcd, (byte)0xef};
        byte[] plain = {0x01, 0x23, 0x45, 0x67,
            (byte)0x89, (byte)0xab, (byte)0xcd, (byte)0xe7};
        byte[] in = new byte[8];
        byte[] out = new byte[8];
        byte[] key2 = {0x01, 0x23, 0x45, 0x67,
            (byte)0x89, (byte)0xab, (byte)0xcd, (byte)0xef,
            (byte)0xfe, (byte)0xdc, (byte)0xba, (byte)0x98,
            0x76, 0x54, 0x32, 0x10};
        byte[] key3 = {0x01, 0x23, 0x45, 0x67,
            (byte)0x89, (byte)0xab, (byte)0xcd, (byte)0xef,
            (byte)0xfe, (byte)0xdc, (byte)0xba, (byte)0x98,
            0x76, 0x54, 0x32, 0x10,
            (byte)0x89, (byte)0xab, (byte)0xcd, (byte)0xef,
            0x01, 0x23, 0x45, 0x67};
        int[] ekey = new int[3*DESkeys.LONGS];
        int[] dkey = new int[3*DESkeys.LONGS];

        System.arraycopy(plain,0,in,0,plain.length);

        TripleDES d = new TripleDES();

	    DESkeys.single(key1, 0, DESkeys.EN0, ekey, 0);
	    DESengine.single(in, 0, out, 0, ekey, d.sp);
        System.out.println(  "Expect Cipher : c957 4425 6a5e d31d");
        printvec("Get    Cipher ",out);

        DESkeys.single(key1, 0, DESkeys.DE1, dkey, 0);
	    DESengine.single(out, 0, in, 0, dkey, d.sp);
        System.out.println(  "Expect Plain  : 0123 4567 89ab cde7");
        printvec("Get    Plain  ",in);
        System.out.println("");

	    DESkeys.dbl(key2, 0, DESkeys.EN0, ekey, 0);
	    DESengine.triple(in, 0, out, 0, ekey, d.sp);
        System.out.println(  "Expect Cipher : 7f1d 0a77 826b 8aff");
        printvec("Get    Cipher ",out);

	    DESkeys.dbl(key2, 0, DESkeys.DE1, dkey, 0);
	    DESengine.triple(out, 0, in, 0, dkey, d.sp);
        System.out.println(  "Expect Plain  : 0123 4567 89ab cde7");
        printvec("Get    Plain  ",in);
        System.out.println("");

        d.init(key3, 0, false);
	    d.ecb(true, in, 0, out, 0);
        System.out.println(  "Expect Cipher : de0b 7c06 ae5e 0ed5");
        printvec("Get    Cipher ",out);

	    d.ecb(false, out, 0, in, 0);
        System.out.println(  "Expect Plain  : 0123 4567 89ab cde7");
        printvec("Get    Plain  ",in);
    }

}