//<copyright>
// 
// Copyright (c) 1993,94,95
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
// 
//</copyright>

//<file>
//
// Name:        scenepick.C
//
// Purpose:     implementation of class SDFScene, as far as related to picking
//
// Created:     18 Mar 92   Michael Hofer and Michael Pichler
//
// Changed:     26 Apr 95   Michael Pichler
//
//
//</file>


#include "scene3d.h"
#include "memleak.h"


#include "geomobj.h"
#include "sdfcam.h"
#include "vecutil.h"

#include <ge3d/ge3d.h>

#include <string.h>
#include <iostream.h>
#include <stdio.h>
#include <math.h>



// pickObject
// return value: a pointer to the first geometric object hit by the ray from activecam->position
// through vppoint or NULL if no object is hit.
// tnear should be set to near clipping plane as fraction of distance between
// camera position and vppoint (0 means no near clipping).
// output parameters: hitpoint, normal vector, groups hit, hit time (unless pointers are NULL)

// change on 28 Jul 93:
// parameter "linksonly = 1" discarded
// for ordinary picks (argument 0 passed) no changes necessary
// for anchor picks (argument 1) the caller is responsible for determining
// whether the hit object is an anchor or not, e. g.: GeometricObjet* hitobj = pick_object (...);
// if (hitobj && hitobj->pickable ())  ... // hit an anchor

GeometricObject* SDFScene::pickObject (
  point3D& vppoint, float tnear,
  point3D* hitpoint, vector3D* normal,
  const StringArray** groups,
  float* hittime
)
{
  if (!activecam_)
    return 0;  // no scene

  point3D A;
  activecam_->getposition (A);  // start of ray, A
  vector3D b;
  sub3D (vppoint, A, b);  // direction of ray, b: vppoint - A
  float tmin = MAXFLOAT;  // hit time
  GeometricObject* hitobj = NULL;

/*
  // display ray for testing (only visible without double buffering)
  ge3d_setlinecolor (1.0, 0.0, 0.0);  
  ge3d_moveto (vppoint.x, vppoint.y, vppoint.z);  // line from eye to vppoint
  ge3d_lineto (A.x, A.y, A.z);  // projects to a single point on screen (hit point)
*/

//cerr << "ray from " << A << " to " << vppoint << endl;

  if (groups)
    *groups = 0;  // no group hit

  for (GeometricObject* gobjptr = (GeometricObject*) gobjlist_.first ();  gobjptr;
                        gobjptr = (GeometricObject*) gobjlist_.next ())
  { // if (!linksonly || gobjptr->pickable ())  // always all objects are considered
    if (gobjptr->ray_hits (A, b, tnear, tmin, normal, groups))  // updates tmin, computes normal (obj. coord.)
      hitobj = gobjptr;
  } // for all geometric objects

  if (hitobj && hitpoint)  // hitpoint unchanged if no object hit
    pol3D (A, tmin, b, *hitpoint);  // hitpoint = A + b*tmin

  if (hitobj && hittime)  // hittime undefined if no object hit
    *hittime = tmin;

  // transform normal from object to world coordinates (transposed inverse trf.mat.)
  if (hitobj && normal)
  {
    const matrix4D* im = hitobj->getInvTrfMat ();
    vector3D w_n;
    w_n.x = (*im) [0][0] * normal->x + (*im) [0][1] * normal->y + (*im) [0][2] * normal->z;
    w_n.y = (*im) [1][0] * normal->x + (*im) [1][1] * normal->y + (*im) [1][2] * normal->z;
    w_n.z = (*im) [2][0] * normal->x + (*im) [2][1] * normal->y + (*im) [2][2] * normal->z;
    float norm2 = dot3D (w_n, w_n);
    if (norm2 > 0.0)  // normalize
    { norm2 = 1 / sqrt (norm2);
      scl3D (w_n, norm2);
    }
    *normal = w_n;
  }

  return hitobj;

} // SDFScene::pickObject (point3D&, ...)



// pickObject
// as above but the point of interest is given in fractional screen coordinates
// fx (horizontal): 0 = left, 1 = right margin
// fy (vertical): 0 = bottom, 1 = top margin

GeometricObject* SDFScene::pickObject (
  float fx, float fy,
  point3D* hitpoint, vector3D* normal,
  const StringArray** groups,
  float* hittime
)
{
  if (!activecam_)
    return 0;  // no scene

  point3D cam_pos, cam_look;  // position and lookat point of the active camera
  activecam_->getposition (cam_pos);
  activecam_->getlookat (cam_look);

  vector3D cam_up;  // up vector of camera coordinate system, usually (0, 1, 0)
  activecam_->getupvector (cam_up);


  vector3D u, v, n;                     // camera coordinate system (orthonormal)
  point3D vppoint,                      // point of interest in world coordinates
          ref;                          // view reference point (midpoint of viewplane)
  float temp;

  // setup camera coordinate system (normalize later)
  sub3D (cam_look, cam_pos, n);         // n = look - pos; from eye to lookat, normal to viewplane
  crp3D (n, cam_up, u);                 // u = n X up;     "to the right" on viewplane
  crp3D (u, n, v);                      // v = u X n;      "upwards" on viewplane

//cerr << "u = " << u << ", v = " << v << ", n = " << n << endl;

  float focallen = activecam_->getfocallen ();

  // compute view reference point (midpoint of viewplane) 
  temp = focallen / sqrt (dot3D (n, n));
  pol3D (cam_pos, temp, n, ref);

  // compute vppoint (point on viewplane)
  float cam_aper = activecam_->getaper ();
  temp = (fx - 0.5) * cam_aper * scene_->getWinAspect () / sqrt (dot3D (u, u));
  pol3D (ref, temp, u, vppoint);
  temp = (fy - 0.5) * cam_aper / sqrt (dot3D (v, v));
  pol3D (vppoint, temp, v, vppoint);

  temp = activecam_->gethither () / focallen;  // near clipping plane

  return  pickObject (vppoint, temp, hitpoint, normal, groups, hittime);

}  // SDFScene::pickObject (float, float, ...)
