//<copyright>
// 
// Copyright (c) 1993,94,95
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
// 
//</copyright>

//<file>
//
// Name:        camera.C
//
// Purpose:     implementation of class Camera
//
// Created:      4 May 95   Michael Pichler (extracted from SDFCamera)
//
// Changed:      4 May 95   Michael Pichler
//
//
//</file>


#include "camera.h"

#include "vecutil.h"
#include <ge3d/ge3d.h>

#include <math.h>
#include <iostream.h>



Camera::Camera ()
{
  // defaults (choosen to produce an identity view transformation)
  init3D (position_, 0, 0, 0);
  init3D (lookat_, 0, 0, -1);
  init3D (up_, 0, 1, 0);  // untilted
  projtype_ = 'P';
  focallen_ = 1.0;
  aper_ = 1.0;  // to check
  aspect_ = 4.0/3.0;  // ignored (window aspect ratio used instead)
  hither_ = 0.1;
  yon_ = 1000.0;

  pos_look_set_ = 0;  // only relevant for SDFCamera:
  // use channel values for camera position/lookat by default
}



void Camera::setCamera (float winasp)
{
  // TODO: allow orthographic cameras too; possibly change up vector
  ge3d_setcamera (position_, lookat_, aper_, focallen_, winasp, hither_, yon_);
}



void Camera::print ()
{
  // print information about the camera
  cout << "  projtype: " << projtype_ << " focallen: " << focallen_
       << " aper: " << aper_ << " aspect: " << aspect_ << endl;
  cout << "  left: " << left_ << " top: " << top_ << " right: " << right_ << " bottom: " << bottom_
       << " hither: " << hither_ << " yon: " << yon_ << endl;

  cout << endl;
}



// translate (const point3D&)
// move whole camera by a vector

void Camera::translate (const vector3D& tran)
{
  inc3D (position_, tran);
  inc3D (lookat_, tran);
} // translate



// translate (float x, float y)
// move whole camera parallel to the viewplane
// x and y are fractions of window width and height
// corresponds to movement of plane at distance flen

void Camera::translate (float x, float y, float winasp, float flen)
{
  vector3D look_pos, u, v;  // pointing to the right/upwards on the viewplane

  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane

  // work with a window at distance flen instead of focallen
  // focallen is most often not related to the size of the scene

  float winheight = aper_ * flen / focallen_;

  x *= winasp * winheight / norm3D (u);  // x *= windowwidth / norm (u)
  y *= winheight / norm3D (v);  // y *= windowheight / norm (v)

  vector3D tran;  // translation
  lcm3D (x, u, y, v, tran);  

  inc3D (position_, tran);
  inc3D (lookat_, tran);
} // translate



// zoom_in
// move camera along straight line from position towards lookat
// (distance dist in world coordinates)

void Camera::zoom_in (float dist)
{
  vector3D look_pos;
  sub3D (lookat_, position_, look_pos);  // lookat - position

  float factor = dist / norm3D (look_pos);

  scl3D (look_pos, factor);

//   // collision detection (if enabled)
//   // currently only if flying forwards (otherwhise have to cast ray in opposite direction)
//   if (factor > 0 && scene && scene->collisionDetection ())
//   {
//     point3D hitpoint;

//     // cast a ray on line of sight (through middle of viewport)
//     // will need a pickObject routine that casts a ray from eye along an arbitrary vector
//     // (note: will have to handle rays going straightly vertical too!)
//     int hit = (scene->pickObject (0.5, 0.5, &hitpoint) != 0);

//     // test wheter near clipping plane is crossed (see do_trans of fly2_hold_button)
//     vector3D hit_pos = hitpoint_;
//     dec3D (hit_pos, position_);  // vector from eye pos. to hit point
//     if (<would move behind hitpoint or behind near clipping plane>)
//       return;
//   }

  // move position and lookat along ray through both
  inc3D (position_, look_pos);
  inc3D (lookat_, look_pos);

} // zoom_in



/* common to all rotations:
   to avoid accumulation of rounding errors
   distance of lookat from position is normalized to 1.0
*/


// rotate_camera_right
// rotate camera left to right (around position)
// (angle phi in radians)

void Camera::rotate_camera_right (float phi)
{
  double c = cos (phi), s = sin (phi);

  vector3D look_pos, u;

  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  c /= norm3D (look_pos);  // normalize
  s /= norm3D (u);
  lcm3D (c, look_pos, s, u, look_pos);

  double factor = 1 / norm3D (look_pos);
  // set distance between position and lookat to 1

  pol3D (position_, factor, look_pos, lookat_);  // set new lookat

} // rotate_camera_right



// rotate_camera_up
// rotate camera upwards (around position)
// (angle phi in radians)

void Camera::rotate_camera_up (float phi)
{
  double c = cos (phi), s = sin (phi);

  vector3D look_pos, u, v;

  // look-pos and v must be normalized!
  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane
  c /= norm3D (look_pos);  // normalize
  s /= norm3D (v);
  lcm3D (c, look_pos, s, v, look_pos);

  double factor = 1 / norm3D (look_pos);
  // set distance between position and lookat to 1

  pol3D (position_, factor, look_pos, lookat_);  // set new lookat

} // rotate_camera_up



// rotate
// rotate camera around an arbitrary center
// angle l2r left-to-right, b2t bottom-to-top (both in radians)

void Camera::rotate (float l2r, float b2t, const point3D& center)
{
  point3D possave = position_;
  point3D looksave = lookat_;

  // ignore anything on matrix stack
  ge3dPushIdentity ();

  // compute u and v
  vector3D look_pos, u, v;
  sub3D (lookat_, position_, look_pos);  // lookat - position
  crp3D (look_pos, up_, u);  // points to right on viewplane
  crp3D (u, look_pos, v);   // points upwards on viewplane

  if (l2r)
    ge3dRotate (&v, l2r);  // horicontal rotation
  if (b2t)
    ge3dRotate (&u, -b2t);  // vertical rotation (-u, b2t)

  vector3D pt_center;

  // position
  sub3D (position_, center, pt_center);  // position - center
  ge3dTransformVectorMcWc (&pt_center, &pt_center);
  add3D (pt_center, center, position_);  // new position

  // lookat
  sub3D (lookat_, center, pt_center);  // lookat - center
  ge3dTransformVectorMcWc (&pt_center, &pt_center);
  add3D (pt_center, center, lookat_);  // new lookat

  // set distance between position and lookat to 1
  sub3D (lookat_, position_, pt_center);  // lookat - position
  double factor = 1 / norm3D (pt_center);
  pol3D (position_, factor, pt_center, lookat_);  // set new lookat

  ge3d_pop_matrix ();

  // ignore vertical rotation when it causes the object to flip sides
  // because of the fixed up vector (0, 1, 0); not very pretty code,
  // but it's the exception, not the rule.
  // TODO: may introduce general camera with free orientation (tilt)

  vector3D u2;
  sub3D (lookat_, position_, look_pos);
  crp3D (look_pos, up_, u2);

  if (b2t && dot3D (u, u2) <= 0)  // orientation ("right") changed
  {
    // cerr << "ignore rotate";
    position_ = possave;
    lookat_ = looksave;
    // rotate (l2r, 0.0, center);
    // do not rotate the object at all (horicontal rotation unexpected fast)
  }

} // rotate (around center)



// rotate_
// rotate a vector vec around 'axis'
// by angle c = cos (phi), s = sin (phi), CCW


static void rotate_ (vector3D& vec, double c, double s, char axis)
{
  float* from,
       * to;

  switch (axis)
  { case 'x':  case 'X':
      from = &vec.y;  to = &vec.z;
    break;
    case 'y':  case 'Y':
      from = &vec.z;  to = &vec.x;
    break;
    case 'z':  case 'Z':
      from = &vec.x;  to = &vec.y;
    break;
  }

  float oldval = *from;

  *from = c * oldval - s * *to;
  *to   = c * *to    + s * oldval;

} // rotate_



// rotate
// rotate whole camera around point center
// along an axis, CCW

void Camera::rotate (float phi, const point3D& center, char axis)
{
  vector3D vec;
  double c = cos (phi),
         s = sin (phi);

  // rotate position
  sub3D (position_, center, vec);  // vec = position - center
  rotate_ (vec, c, s, axis);
  add3D (vec, center, position_);

  // rotate lookat
  sub3D (lookat_, center, vec);  // vec = lookat - center
  rotate_ (vec, c, s, axis);
  add3D (vec, center, lookat_);

} // rotate



// makeHoricontal
// rotate around position such that position and lookat are on same level

void Camera::makeHoricontal ()
{
  vector3D look_pos, new_look_pos;

  sub3D (lookat_, position_, look_pos);  // lookat - position

  new_look_pos = look_pos;
  new_look_pos.y = 0;

  double newlensquare = dot3D (new_look_pos, new_look_pos);

  if (newlensquare > 0.0)
  { double factor = sqrt (dot3D (look_pos, look_pos) / newlensquare);
    // maintain distance between position and lookat
    // set new lookat
    pol3D (position_, factor, new_look_pos, lookat_);
  }
  // else: line of sight was parallel to y-axis - no change

} // makehoricontal
