.packageName <- "reposTools"
saveDfRda <- function(reposDF, outFile="repdatadesc.rda") {
    ## DF can be of any type, but parameter "type" must define
    ## it properly for correct operations

    reposDF
    save(reposDF,file=outFile)
}

getPackDcfMatrix <- function(file, fields=c("Package","Version"),
                             mangSep=":v:") {
    ## Will retrieve the fields requested from the filename
    ## Will remove the first field and use that as the set of
    ## rownames, then return the result
    mtrx <- read.dcf(file,fields)

    if (ncol(mtrx) > 1) {
        mtrx <- mangleRowNames(mtrx,mangSep)
    }
    else if (ncol(mtrx) == 1) {
        rownames(mtrx) <- mtrx[,1]
    }
    else {
        mtrx <- NULL
    }

    return(mtrx)
}

getPackDcfList <- function(file, fields=c("Package","Version"),
                           mangSep=":v:") {
    ## Will retrieve the fields requested from the filename
    ## Will remove the first field and use that as the set of
    ## rownames, then return the result
    mtrx <- read.dcf(file,fields)

    if (ncol(mtrx) > 1) {
        mtrx <- mangleRowNames(mtrx,mangSep)
    }
    else if (ncol(mtrx) == 1) {
        rownames(mtrx) <- mtrx[,1]
    }

    dcfList <- split(mtrx, col(mtrx))
    dcfList[[length(dcfList)+1]] <- rownames(mtrx)
    names(dcfList) <- c(colnames(mtrx),"Mangled")

    return(dcfList)
}


reposDfFromList <- function(dcfList, fields=c("Package","Version")) {
    ## Creates a data frame where each column is constructed by
    ## an element of the passed in list.  Column names are from the
    ## list element names.  The last element of the list becomes the
    ## row.names of the data frame.

    ## Sanity check list
    if (length(dcfList) != (length(fields)+1)) {
        emsg <- "List does not contain the proper elements"
        stop(emsg)
    }

    reposDf <- data.frame(I(dcfList[[1]]))
    for (i in 2:(length(fields))) {
        reposDf <- cbind(reposDf,I(dcfList[[i]]))
    }

    colnames(reposDf) <- fields
    rownames(reposDf) <- dcfList[[length(dcfList)]]
    return(reposDf)
}

splitDFstrings <- function(df, fields, delim=",[[:space:]]*") {
    ## Will take a data frame and a set of field names.  For all those
    ## fields, will split the current string field into a vector of
    ## strings based on 'delim'.
    for (i in 1:length(fields)) {
        if (!all(is.na(df[,fields[i]]))) {
            tmp <- paste("df$",fields[i],
                         " <- I(strsplit(df$",
                         fields[i],", delim))",sep="")
            eval(parse(text=tmp))
        }
    }
    return(df)
}

buildVigDf <- function(VIGfile="Vignettes.in", infoDir = ".") {
    cDir <- getwd()
    on.exit(setwd(cDir))
    setwd(infoDir)
    vigFields <- c("VignetteIndexEntry","VignetteTitle", "VignettePackage",
                   "VignetteVersion", "VignetteDepends",
                   "VignetteKeywords", "PDFpath")
    vigMtrx <- read.dcf(VIGfile,vigFields)
    vigDcfList <- split(vigMtrx, col(vigMtrx))
    vDf <- data.frame(I(vigDcfList[[1]]))
    for (i in 2:length(vigFields)) {
        vDf <- cbind(vDf,I(vigDcfList[[i]]))
    }
    colnames(vDf) <- vigFields
    vDf <- splitDFstrings(vDf,c("VignetteDepends","VignetteKeywords"))
    class(vDf) <- c(class(vDf),"Vignette")
    return(vDf)
}

buildPkgDf <- function(PACKAGEfile="PACKAGES.in", infoDir = ".",
                       prefixPath=NULL, mangSep=":v:") {
    ## Builds the repository Rda file from various sources

    cDir <- getwd()
    on.exit(setwd(cDir))
    setwd(infoDir)

    ## Note the fields to be used in the Rda file from
    ## PACKAGE.in
    fields <- c("Package","Version", "Keywords", "Depends", "Title",
                "Suggests", "Uses", "Replaces", "Description", "Homepage",
                "Author", "Maintainer", "License", "Status",
                "Priority", "SystemRequirements")

    dcfList <- getPackDcfList(PACKAGEfile, fields, mangSep)
    ## Change the matrix into a data frame
    df <- reposDfFromList(dcfList,fields)
    ## Change any of the fields which are supposed to be vectors that
    ## are strings into vectors
    stringVex <- c("Keywords", "Depends", "Suggests", "Uses","Replaces")
    df <- splitDFstrings(df, stringVex)
    ## Replace version strings w/ VersionNumber objects
    df <- replaceDFversions(df)
    ## Create an empty matrix containing any fields that are not yet
    ## to be filled in, then attach it to the data frame
    xFields <- c("OSspecific","KeySearchOrder")
    xMtrx <- matrix(nrow=nrow(df),ncol=length(xFields))
    colnames(xMtrx) <- xFields
    df <- cbind(df,xMtrx)

    df <- addManifest(df)
    df <- addOsSpecific(df, prefixPath)
    class(df) <- c(class(df),"Pkg")
    return(df)
}

replaceDFversions <- function(df) {
    ## Will replace all of the df's version strings w/ VersionNumber
    ## objects

    tmpVers <- list()
    for (i in seq(along=row.names(df))) {
        tmpVers[[i]] <- buildVersionNumber(as.character(df[i,]$Version))
    }
    df$Version <- I(tmpVers)
    return(df)
}

addManifest <- function(df, ext=".manifest", mangSep=":v:") {
    ## Manifests provide for extra keywords to packages, beyond
    ## what the original package maintainer envisioned.
    manifs <- dir(pattern=paste("*",ext,sep=""))

    for (j in seq(along=manifs)) {
        keyword <- gsub(ext,"",manifs[j])
        manif <- getReposSpecialFile(df, manifs[j], mangSep)

        for (i in 1:nrow(manif)) {
            ## Get the row number in the data frame
            rowNum <- which(row.names(df)==rownames(manif)[i])
            ## The matrix has character 'NA', not real NAs
            if (df$Keywords[[rowNum]] == "NA") {
                df$Keywords[[rowNum]] <- keyword
            }
            else {
                df$Keywords[[rowNum]] <-
                    as.character(I(c(df$Keywords[[rowNum]],
                                     keyword)))
            }
            ## Create an one element list
            tmpList <- list(manif[i,"Priority"])
            names(tmpList) <- keyword
            if (!is.list(df$KeySearchOrder[[rowNum]])) {
                ## Empty, so just add it
                df$KeySearchOrder[[rowNum]] <- tmpList
            }
            else {
                ## Otherwise, need to attach it
                cur <- df$KeySearchOrder[[rowNum]]
                newList <- c(cur,tmpList)
                df$KeySearchOrder[[rowNum]] <- newList
            }
        }
    }
    return(df)
}

addOsSpecific <- function(df, prefixPath=NULL, ext=".status",mangSep=":v:") {
## Adds in specific package information
    oss <- dir(pattern=paste("*",ext,sep=""))
    fields <- c("File","Status","Rvers", "Date")

    df$OSspecific <- list()
    for (i in seq(along=row.names(df))) {
        df$OSspecific[[i]] <- list(NA)
    }

    for (j in seq(along=oss)) {
        curOS <- gsub(ext,"",oss[j])
        osMtrx <- getReposSpecialFile(df, oss[j], mangSep)

        ## Add in any prefixPath
        if (!is.null(prefixPath)) {
            osMtrx[,"File"] <- file.path(prefixPath, osMtrx[,"File"])
        }
        for (i in 1:nrow(osMtrx)) {
            mangRow <- rownames(osMtrx)[i]
            whichDFRow <- match(mangRow,row.names(df))

            osFields <- osMtrx[i,fields]
            tmpList <- as.list(osFields)

            if (is.null(df$OSspecific[[whichDFRow]][[1]]$File)) {
                df$OSspecific[[whichDFRow]] <- list(tmpList)
                names(df$OSspecific[[whichDFRow]]) <- curOS
            }
            else {
                tmpNames <- names(df$OSspecific[[whichDFRow]])
                df$OSspecific[[whichDFRow]][[j]] <- tmpList
                tmpNames <- c(tmpNames,curOS)
                names(df$OSspecific[[whichDFRow]]) <- tmpNames
            }
        }
    }
    return(df)
}

getReposSpecialFile <- function(df, fileName, mangSep) {
    ## Read it in via Dcf, mangle and create rownames,
    ## Determine which rows are in the dataframe, and then
    ## return the resultant matrix
    mtrx <- read.dcf(fileName)
    mtrx <- fillBlankVersions(mtrx,df)
    mtrx <- mangleRowNames(mtrx,mangSep)
    ## Determine which mangled names are in the dataframe
    found <- rownames(mtrx) %in% row.names(df)
    mtrx <- mtrx[found,,drop=FALSE]
    return(mtrx)
}

fillBlankVersions <- function(mtrx, df) {
    ## Will take the repos file dcf matrix and find situations where
    ## there is no version #.  Will then match it to the highest
    ## version # available for that package.  If no such package is
    ## found, will drop from matrix
    blanks <- which(mtrx[,"Version"]=="NA")
    for (blank in blanks) {
        blankPkg <- mtrx[blank,"Package"]
        pkgMatches <- which(df$Package == blankPkg)
        pkgVersions <- df[pkgMatches,]$Version
        ## !! see comments in getMaxVersion
        blankVers <- getMaxVersion(pkgVersions)
        ## We still want to use characters at this point ...
        mtrx[blank,"Version"] <- as.character(blankVers)
    }
    return(mtrx)
}

mangleRowNames <- function(mtrx, mangSep) {
    ## Given a matrix, will combine the first two fields
    ## separated by 'mangSep', and will make these the row names
    if (ncol(mtrx) < 2) {
        return(mtrx)
    }
    rownames(mtrx) <- paste(mtrx[,1],mtrx[,2],sep=mangSep)
    return(mtrx)
}

genRepos <- function (repName, urlBase, urlPath,
                      repType=c("package","vignette","data"), PACK=FALSE,
                      Status=FALSE, dir = ".", dataPath="", infoPath="") {

    repType <- match.arg(repType)
    outFile <- file.path(dir, "replisting")
    out <- paste("repname: ", repName, "\nreptype: ", repType,
                 "\nrepaddrBase: ", urlBase, "\nrepaddrPath: ",
                 urlPath, "\n\n", sep = "")
    cat(out, file = outFile)
    curDir <- getwd()
    on.exit(setwd(curDir), add=TRUE)
    if (!file.exists(dir)) {
        stop(paste("Directory parameter",dir,"does not exist"))
    }
    setwd(dir)
    infoDir <- file.path(dir,infoPath)
    ## Hack for Win32 (doesn't handle './' in file.exists)
    if (missing(infoPath)) {
        if (!file.exists(dir))
            stop(paste("Bad dir information:",dir))
    }
    else {
        if (!file.exists(infoDir)) {
            stop(paste("Bad infoPath parameter:",infoPath))
        }
    }
    dataDir <- file.path(dir,dataPath)
    ## Another Win32 hack
    if (missing(dataPath)) {
        if (!file.exists(dir))
            stop(paste("Bad dir information:",dir))
    }
    else {
        if (!file.exists(dataDir)) {
            stop(paste("Bad dataPath parameter:",dataPath))
        }
    }
    switch(repType,
           "package"=genPkgRepos(dataDir, infoDir, PACK, Status),
           "vignette"=genVigRepos(dataDir, infoDir, PACK),
           stop(paste("Repository type",repType,"not currently supported."))
           )


    return(list(repName=repName, repType = repType, repaddrBase =
                urlBase, repaddrPath=urlPath, repDir=dir))
}


genVigRepos <- function(dataDir, infoDir, VIG=FALSE) {
    ## Create a temporary directory for holding temp data
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)
    tmpInfo <- file.path(tmpDir,"info")
    dir.create(tmpInfo)
    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)
    ## Copy the manual VIG file if FALSE
    if (VIG==FALSE) {
        vigFile <- "Vignettes.in"
        if (file.exists(vigFile)) {
            file.copy(vigFile,tmpInfo)
        }
    }
    else {
        setwd(dataDir)
        ## Get list of vignettes to use
        vigList <- getPkgVigList(".",baseVigDesc,".",pkgVers=FALSE)
        vigList<- mergeVigListStrings(vigList)
        vigList <- filterVigList(vigList)
        ## Push this into a data frame
        tmpVigDf <- data.frame(I(vigList[[1]]))
        for (i in 2:length(vigList))
            tmpVigDf <- cbind(tmpVigDf,I(vigList[[i]]))
        tmpVigDf <- t(tmpVigDf)
        rownames(tmpVigDf) <- 1:nrow(tmpVigDf)
        tmpVigDf <- as.data.frame(tmpVigDf)
        ## !! Temporarily just remove those w/o a PDFpath.
        ## !!! Need to try to build their PDFs
        tmpVigDf <- tmpVigDf[!is.na(tmpVigDf$PDFpath),]
        ## Write to DCF file
        write.dcf(tmpVigDf,file=file.path(tmpInfo,"Vignettes.in"))
    }
    vigDf <- buildVigDf(infoDir=tmpInfo)
    setwd(curDir)
    saveDfRda(vigDf)
}

filterVigList <- function(vigList) {
    newVigList <- list()
    fields <- c("VignetteIndexEntry","VignettePackage", "VignetteTitle",
                   "VignetteVersion", "VignetteDepends",
                   "VignetteKeywords", "PDFpath")
    for (i in 1:length(vigList)) {
        tmp <- list()
        for (j in 1:length(fields)) {
            tmp[[j]] <- as.character(vigList[[i]][fields[j]])
        }
        names(tmp) <- fields
        newVigList[[i]] <- tmp
    }
    names(newVigList) <- names(vigList)
    return(newVigList)
}

genPkgRepos <- function(dataDir, infoDir, PACK=FALSE, Status=FALSE) {

    ## Create a temporary directory for unpacking packages
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)

    tmpInfo <- file.path(tmpDir,"info")
    dir.create(tmpInfo)

    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)
    ## Copy any manual files to the tmpInfo directory
    copyManualFiles(infoDir, tmpInfo, PACK, Status)
    ## check to make sure this works
    setwd(dataDir)
    ## Get list of packages to unpack
    pkgs <- dir(pattern="*\\.tar\\.gz|*\\.tgz|*\\.zip")
    PACKin <- SrcStat <- WinStat <- NULL
    for (pkg in pkgs) {
        ## Get extension/pkg split out
        ext <- strsplit(pkg,"\\.")[[1]]
        ext <- ext[length(ext)]

        DESC <-  unpackExtractPkgDESC(pkg)
        if (!is.null(DESC)) {
            samePack <- which(PACKin[,"Package"]==DESC[,"Package"])
            if ((!any(samePack)) ||
                (any(PACKin[samePack,"Version",drop=FALSE]!=DESC[,"Version"]))) {
                PACKin <- rbind(PACKin, DESC)
            }
            if (!is.na(DESC[,"Built"])) {
                parts <- strsplit(DESC[,"Built"],"; ")
                pkgRvers <- strsplit(parts[[1]][1]," ")[[1]][2]
                pkgDate <- parts[[1]][3]
            }
            else {
                pkgRvers <- ""
                pkgDate <- date()
            }
            newStatus <- c(DESC[,"Package"],
                           DESC[,"Version"],
                           file.path(dataDir,pkg),
                           "OK", pkgRvers, pkgDate)
            switch(ext,
                   "gz"=SrcStat<-rbind(SrcStat,newStatus),
                   "zip"=WinStat<-rbind(WinStat,newStatus)
                   )
        }
    }
    ##rbind seems to add unwanted rownames
    if( !is.null(SrcStat) ) row.names(SrcStat) <- NULL
    if( !is.null(WinStat) ) row.names(WinStat) <- NULL
    fields <- c("Package", "Version", "File", "Status", "Rvers","Date")
    if (PACK==TRUE) {
        write.dcf(PACKin,file.path(tmpInfo,"PACKAGES.in"))
    }

    if (Status==TRUE) {
        if (!is.null(SrcStat)) {
            colnames(SrcStat) <- fields
            write.dcf(SrcStat,file.path(tmpInfo,"Source.status"))
        }
        if (!is.null(WinStat)) {
            colnames(WinStat) <- fields
            write.dcf(WinStat,file.path(tmpInfo,"Win32.status"))
        }
    }

    df <- buildPkgDf(infoDir=tmpInfo)
    setwd(curDir)
    saveDfRda(df)
}

copyManualFiles <- function(infoDir, tmpInfo, PACK, Status) {
    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)
    setwd(infoDir)
    manifests <- dir(pattern="*.manifest")
    if (length(manifests) >= 1) {
        file.copy(manifests, tmpInfo)
    }

    if (PACK==FALSE) {
        packFile <- "PACKAGES.in"
        if (file.exists(packFile)) {
            file.copy(packFile,tmpInfo)
        }
    }

    if (Status == FALSE) {
        statusFiles <- dir(pattern="*.status")
        statusFiles <- statusFiles[file.exists(statusFiles)]
        if (length(statusFiles) >= 1) {
            file.copy(statusFiles, tmpInfo)
        }
    }
}

unpackExtractPkgDESC <- function(pkg) {
    ## Unpacks the package, extracts DESCRIPTION info
    cDir <- getwd()
    on.exit(setwd(cDir),add=TRUE)

    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)

    file.copy(pkg,tmpDir)
    setwd(tmpDir)
    ## Get extension/pkg split out
    ext <- strsplit(pkg,"\\.")[[1]]
    ext <- ext[length(ext)]
    ret <- switch(ext,
                  "gz"=unpackSourcePkg(pkg),
                  "tgz"=unpackSourcePkg(pkg),
                  "zip"=unpackZipPkg(pkg),
                  stop(paste("Don't know how to unpack package",pkg))
                  )
    if (ret != 0)
        return(NULL)

    pkgDir <- strsplit(pkg,"[_\\.]")[[1]][1]
    DESC <- file.path(pkgDir,"DESCRIPTION")
    if (!file.exists(DESC)) {
        return(NULL)
    }

    ## !!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ## !!! Thinking about replacing the structure here from matrix
    ## !!! to something more abstract.  Tagged list?
    ## !!!!!!!!!!!!!!!!!!!!!!!!!!!!
    fields <- c("Package","Version", "Keywords", "Depends", "Title",
                "Suggests", "Uses", "Replaces", "Description", "Homepage",
                "Author", "Maintainer", "License", "Status",
                "Priority", "Built","Bundle","BundleDescription",
                "Contains", "SystemRequirements")

    dFile <- read.dcf(DESC,fields)

    ## Detect if this is a package bundle
    if (!is.na(dFile[1,"Bundle"])) {
        ## If this is a source bundle we want to treat it like
        ## a normal package, but need to fill in the
        ## package/description fields w/ appropriate info
        dFile[1,"Package"] <- dFile[1,"Bundle"]
        dFile[1,"Description"] <- dFile[1,"BundleDescription"]
    }

    ## clean up from behind
    unlink(pkgDir,recursive=TRUE)
    return(dFile)
}

unpackSourcePkg <- function(pkg) {
    z <- try(system(paste("gzip -dc", pkg, "| tar -xf -")))
    z
}

unpackZipPkg <- function(pkg) {
    OST <- .Platform$OS.type
    if (OST == "unix") {
        zip <- getOption("unzip")
        system(paste(zip,"-q",pkg))
    }
    else {
        zip.unpack(pkg,".")
    }

    return(0)
}

mergeVigListStrings <- function(vList) {
    for (i in 1:length(vList)) {
        if ((!is.null(vList[[i]]$VignetteDepends))&&
            (!is.na(vList[[i]]$VignetteDepends))) {
            if (length(vList[[i]]$VignetteDepends > 1)) {
                vList[[i]]$VignetteDepends <-
                    paste(vList[[i]]$VignetteDepends, collapse=", ")
            }
        }
        else {
            vList[[i]]$VignetteDepends <- NA
        }
        if ((!is.null(vList[[i]]$VignetteKeywords))&&
            (!is.na(vList[[i]]$VignetteKeywords))) {
            if (length(vList[[i]]$VignetteKeywords > 1)) {
                vList[[i]]$VignetteKeywords <-
                    paste(vList[[i]]$VignetteKeywords, collapse=", ")
            }
        }
        else {
            vList[[i]]$VignetteKeywords <- NA
        }
    }
    return(vList)
}

getArgRepos <- function(rep) {
    ## Determine if rep is a directory or an online repository
    ## used for functions where one can pass in a rep or a local
    ## directory that is a rep (but might not be online)
    if (class(rep) == "character") {
        if ((!file.exists(file.path(rep,"replisting")))||
            (!file.exists(file.path(rep,"repdatadesc.rda")))) {
            stop(paste(rep,
                       "does not seem to be a valid repository directory"))
        }
        else {
            load(file.path(rep,"repdatadesc.rda"))
            repL <- read.dcf(file.path(rep,"replisting"))
            rep <- buildReposEntry(new("replisting",replisting=repL),
                                   new("repdatadesc",
                                       repdatadesc=reposDF))
        }
    }
    else if (!inherits(rep,"ReposEntry"))
        stop(paste("Don't know how to handle passed rep argument:",rep))

    return(rep)
}

genReposHtml <- function (rep, filename="index.html", outDir=".",
                          headerInfo="")
{
    rep <- getArgRepos(rep)
    if (!file.exists(outDir))
        dir.create(outDir)
    if(repType(rep) == "package") {
        GenPkgListingHTML(file.path(outDir,filename), rep,
                          headerInfo)
    }
    else if (repType(rep) == "vignette") {
        GenVigListingHTML(file.path(outDir,filename), rep, headerInfo)
    }
    else {
        stop(paste("Do not know how to construct HTML pages for",
                   "repositories of type:", repType(rep)))
    }
}

GenVigListingHTML <- function(filename, rep, headerInfo="", upFile) {
    if (missing(filename))
        filename <- "viglistingindex.html"
    outFile <- file(filename, "w")
    on.exit(close(outFile))

    rep <- getArgRepos(rep)

    reposDF <- repdataframe(rep)
    reposDF <- reposDF[sort(reposDF$VignettePackage,index.return=TRUE)$ix,]

    ## Write header
    t1 <- paste("<TITLE>",repName(rep),"</TITLE>")
    cat("<html>", "<head>", t1, headerInfo,
        "</head><body>", file = outFile, sep = "\n")
    cat("<h3 align=left>",repName(rep),"</h3>",file=outFile, sep="")
    cat(file=outFile, "<table border=0 align=left>",
        "<tr> <td><b>Vignette</b></td> <td><b>Package</b></td>",
        "</tr> <tr>", sep="\n")
    for (i in 1:nrow(reposDF)) {
        vigName <- reposDF$VignetteTitle[i]
        if (vigName == "Untitled")
            vigName <- reposDF$VignetteIndexEntry[i]
        vigPkg <- reposDF$VignettePackage[i]
        if (vigPkg == ".")
            vigPkg == "None"
        vigPath <- paste(repURL(rep),reposDF$PDFpath[i],sep="/")
        cat(file=outFile,"<tr>\n","<td><a href=\"",vigPath,
            "\"> ",vigName,"</a></td>\n<td>",vigPkg,
            "</td>\n</tr>\n",sep="")
    }
    if(!missing(upFile))
        cat(file=outFile, "<h3 align=left> <a href=", upFile,
            ">Return to Index</a></h3><br> \n", sep="")

    cat(file=outFile, "</table>", "</body>",  "</html>", sep="\n" )
}

##idea here is to write a function that creates the package listings
GenPkgListingHTML <- function(filename, rep, headerInfo="",
                              upFile)
{
  if( missing(filename) )
    filename <- "packagelistingindex.html"
  filename = file.path(outDir,filename)
  outFile <- file(filename, "w")
  on.exit(close(outFile))
  rep <- getArgRepos(rep)
  ## Write header
  t1 <- paste("<TITLE>",repName(rep),"</TITLE>")
  cat("<html>", "<head>", t1, headerInfo,
        "</head> <body>", file = outFile, sep = "\n")

  cat(file=outFile, "<table border=0 align=left>",
         "<tr> <td><b>Package</b></td> <td><b>Title</b></td>",
           "<td><b>Version</b></td> </tr>", sep="\n")
  reposDF <- repdataframe(rep)
  pfilenames <- vector("character", length=nrow(reposDF))
  pN <- rownames(reposDF)
  pNames <- reposDF$Package
  for(i in 1:nrow(reposDF) ) {
    cat(file=outFile, "<tr>")
    pfilenames[i] <- genPkgListing(pN[i], reposDF = reposDF,
                                   upFile=filename)
    cat(file=outFile, "<td><a href=\"" , pfilenames[i], "\"> ", pNames[i], "</a></td>\n",
      sep="")
    ## we need a short description
    cat(file=outFile,"<td>", as.character(reposDF[i,"Title"][[1]]),
         "</td>\n", sep="")
    cat(file=outFile, "<td>", as.character(reposDF[i,"Version"][[1]]),
         "</td>\n", sep="")
     cat(file=outFile, "</tr>")
  }
  if( !missing(upFile) )
    cat(file=outFile, "<h3 align=left> <a href=", upFile,
      ">Return to Index</a></h3><br> \n", sep="")

  cat(file=outFile, "</table>", "</body>",  "</html>", sep="\n" )
}

##should return some data -- such as the file name used?
genPkgListing <- function(pkgN, reposDF, filename, upFile)
{
    nVers <- strsplit(pkgN, ":v:")[[1]]
    repRow <- which(row.names(reposDF)==pkgN)
    if( missing(filename) )
       filename <- paste(paste(nVers[1], nVers[2], sep="v"), ".html", sep="")
    ##for now we overwrite
    outfile <- file(filepath, "w")
    on.exit(close(outfile))
    cat(file=outfile, "<html>",  "<title>", nVers[1], "</title>", "<body>",
      sep="\n")
    if( !missing(upFile) )
       cat(file=outfile, "<h3 align=left> <a href=", upFile,
         ">Return to package listing</a></h3><br> \n", sep="")
    cat(file=outfile, " <h1 align=left> Package: ", nVers[1],
          "</h1> <div align=center> </div> \n <br> \n", sep="")
    cat(file=outfile, "<b>Description:</b>", reposDF[pkgN, "Description"],
          "<br>\n", sep="")
    cat(file=outfile, "<b>Version:</b>", nVers[2], "<br>\n", sep="")
    cat(file=outfile,"<b>Author:</b>",reposDF[pkgN,"Author"],
        "<br\n",sep="")
    cat(file=outfile,"<b>Maintainer:</b>",reposDF[pkgN,"Maintainer"],
        "<br>\n",sep="")
    cat(file=outfile, "<b>Dependencies:</b>", reposDF[pkgN, "Depends"][[1]],
         "<br>\n",sep="")
    cat(file=outfile,"<b>License:</b>",reposDF[pkgN,"License"],
        "<br><br><br>\n",sep="")

    ## Set download links
    if (length(repRow) > 0) {
        osS <- reposDF$OSspecific[[repRow]]
        if (length(osS) > 0)
            for (i in 1:length(osS)) {
                curOS <- osS[[i]]
                if (curOS$Status == "OK")
                    cat(file=outfile,"<a href=\"",curOS$File,
                        "\">",names(osS)[i],
                        " package download</a><br>", sep="")
            }

    }
    cat(file=outfile, "</body> \n </html>")
    return(filename)
}



reposToolsCurrentInstalls <- list()

syncLocalLibList <- function(lib=reposToolsLibPaths()) {
    ## Need to see if there currently exists a 'liblisting.Rda' in lib
    ## if so, load it, otherwise set the object NULL
    cat("\nSynching your local package management information ...\n")

    ## Create a locale for any temporary lib dirs, if needed
    tmpLibDir <- file.path(tempdir(),"tempLibs")
    if (! file.exists(tmpLibDir))
        dir.create(tmpLibDir)

    ## Need to convert any old style local libraries to new style
    convert.locLib(lib)

    ## Now sync the local pkg info
    for (i in seq(along=lib)) {
        if (!file.exists(lib[i]))
            next

        ## Check to make sure we can write to this lib
        if (file.access(lib[i],mode=2) != 0) {
            cat("\n\t")
            note(paste("reposTools can not access ",lib[i],".\n\t",
                       "This will not affect your R session unless ",
                       "you wish \n\tto install/update/remove packages ",
                       "from this directory\n",sep=""))
            writeable <- FALSE
        }
        else
            writeable <- TRUE


        if (load.locLib(lib[i]) == FALSE) {
            locLibList <- NULL
        }

        if (length(dir(lib[i])) > 0) {
            ## Get listing of all packages installed in this library
            options(show.error.messages=FALSE)
            on.exit(options(show.error.messages=TRUE), add=TRUE)
            pkgs <- try(installed.packages(lib[i]))
            options(show.error.messages=TRUE)
            if (is(pkgs,"try-error")) {
                warning(paste(lib[i],"does not seem to be a valid",
                              "R package library, skipping ..."))
                next
            }

            ## Figure out which packages are already in the database,
            ## and which ones are new
            curLocPkgs <- unlist(lapply(locLibList, Package))
            oldPkgs <- pkgs[,"Package"] %in% curLocPkgs
            old <- pkgs[oldPkgs,,drop=FALSE]
            new <- pkgs[!oldPkgs,,drop=FALSE]
            gonePkgs <- which(!(curLocPkgs %in% pkgs[,"Package"]))

            if (nrow(old) > 0) {
                ## Need to find any updated packages ...
                remPkgs <- numeric()
                for (j in 1:nrow(old)) {
                    ## For each package, check to see if any of its
                    ## version has changed, if so, add it to the 'new' list
                    llPkg <- match(old[j,"Package"],curLocPkgs)
                    if (buildVersionNumber(old[j,"Version"])
                        != PkgVersion(locLibList[[llPkg]])) {
                        new <- rbind(new,old[j,])
                        remPkgs <- c(remPkgs, llPkg)
                    }
                }
                if (length(remPkgs) > 0)
                    locLibList <- locLibList[-remPkgs]
            }

            if (nrow(new) > 0) {
                locLibFields <- c("Package","Version","Keywords","Repos",
                                  "Depends", "Suggests", "Uses")
                ## Extract the information for any new packages
                locLibMtrx <- matrix(nrow=nrow(new),ncol=length(locLibFields))
                colnames(locLibMtrx) <- locLibFields
                pkgsFlds <- colnames(locLibMtrx)[colnames(locLibMtrx) %in% colnames(pkgs)]
                locLibMtrx[,pkgsFlds] <- new[,pkgsFlds]
                locLibMtrx[is.na(locLibMtrx)] <- "NA"
                ## Convert versions to VersionNumber
                vers <- locLibMtrx[,"Version"]
                versList <- list()
                for (k in 1:length(vers)) {
                    versList[[k]] <- buildVersionNumber(vers[k])
                }

                for (l in 1:nrow(locLibMtrx))
                    locLibList[[length(locLibList)+1]] <- new("localPkg",
                                                          Package=locLibMtrx[l,1],
                                                          PkgVersion=versList[[l]],
                                                          Keywords=locLibMtrx[l,3],
                                                          Repos=locLibMtrx[l,4],
                                                          Depends=locLibMtrx[l,5],
                                                          Suggests=locLibMtrx[l,6],
                                                          Uses=locLibMtrx[l,7])

                outNewPkgs <- paste(locLibMtrx[,"Package"], collapse="\n\t")
                ## If this is a non-writeable directory, packages
                ## there will show up as "new" but they shouldn't
                ## be reported.
                if (writeable)
                    cat("Packages which have been added/updated:",
                        outNewPkgs,"\n",sep="\n\t")
            }


            if (length(gonePkgs) > 0) {
                ## Also need to detect removed packages
                outPkgs <-
                    paste(curLocPkgs[gonePkgs],collapse="\n\t")
                cat("Packages which have been removed:",outPkgs,"\n",
                    sep="\n\t")
                locLibList <- locLibList[-(gonePkgs)]
            }

            if (writeable)
                save.locLib(locLibList, lib[i])
            else {
                ## !! write the locliblist to a temporary directory
                curDir <- tempfile(tmpdir=tmpLibDir)
                dir.create(curDir)
                save.locLib(locLibList, curDir)
            }
        }
    }
}

load.locLib <- function(lib=reposToolsLibPaths()[1]) {
    if (length(lib) != 1) {
        warning("load.locLib() does not accept vectorized inputs")
        return(FALSE)
    }

    libFile <- file.path(lib,"liblisting.Rda")
    if (file.exists(libFile)) {
        load(libFile,envir=parent.frame())
        return(TRUE)
    }
    else {
        return(FALSE)
    }
}

closeLocLib <- function() {
    ## Removes the locLibList w/o saving
    ## Probably unecessary in most cases, but can be used to prevent
    ## confusion.
    if ("locLibList" %in% ls(envir=parent.frame()))
        rm("locLibList",envir=parent.frame())
}

save.locLib <- function(locLibList, lib) {
    locLibList
    libFile <- file.path(lib,"liblisting.Rda")

    if (file.access(lib,mode=2) == 0) {
        save(locLibList, file=libFile)
    }
    else {
       warning(paste("Incorrect permissions to edit package database,",
                   libFile))
    }
}

convert.locLib <- function(libs=reposToolsLibPaths()) {
    ## Will convert a liblisting.Rda file from the old style
    ## (pre-1.1.39 reposTools) to the new structure

    for (lib in libs) {
        if (load.locLib(lib) == FALSE)
            next
        if ("locLibDf" %in% ls()) {
            cat("\nConverting",lib,"to new local library data format.\n")
            if (is.data.frame(locLibDf)) {
                locLibList <- list()
                for (i in 1:nrow(locLibDf)) {
                    ## The Depends field in the old locLibDf was often
                    ## messed up and included a lot of garbage.  We
                    ## need to filter out that garbage.
                    curDepends <- locLibDf[i,"Depends"][[1]]
                    curDepends <- gsub("(\\\\)|(\")|(\))|(c\\()","",curDepends)

                    locLibList[[length(locLibList)+1]] <-
                        new("localPkg", Package=as.character(locLibDf[i,"Package"]),
                            PkgVersion=locLibDf[i,"Version"][[1]],
                            Keywords=locLibDf[i,"Keywords"][[1]],
                            Repos=as.character(locLibDf[i,"Repos"]),
                            Depends=curDepends,
                            Suggests=locLibDf[i,"Suggests"][[1]],
                            Uses=locLibDf[i,"Uses"][[1]])
                }
                save.locLib(locLibList, lib)
            }
        }
    }
}

getLocalPkgs <- function(libs=reposToolsLibPaths()) {
    pkgInfos <- list()

    for (lib in libs) {
        if (load.locLib(lib) == FALSE) {
            next
        }

        pkgs <- lapply(locLibList, Package)
        vers <- lapply(locLibList, PkgVersion)

        for (i in seq(along=pkgs))
            pkgInfos[[i]] <-buildPkgInfo(as.character(pkgs[[i]]), vers[i][[1]])

        closeLocLib()
    }

    return(pkgInfos)
}

findInstallDepends <- function(pkg, vers, libList, depColName, remove=FALSE) {
    out <- NULL
    if (length(libList) == 0)
        return(out)

    depCol <- switch(depColName,
                     "Depends"=lapply(libList,Depends),
                     "Suggests"=lapply(libList,Suggests),
                     "Uses"=lapply(libList,Uses),
                     stop("Invalid depColName"))

    for (i in 1:length(depCol)) {
        curDeps <- depCol[[i]]
        if ((length(curDeps) == 1)&&(curDeps == "NA"))
            next
        if (length(curDeps) != 0) {
            depMtrx <- getDepMtrx(curDeps)
            if (is.null(depMtrx))
                next

            if (length(row <- grep(paste("^",pkg,"$",sep=""),
                                   depMtrx[,1])) > 0) {
                if (remove == TRUE) {
                    ## This is going to break a depency no matter what
                    out <- c(out,paste(Package(libList[[i]]),"version",
                                       stringRep(PkgVersion(libList[[i]])),
                                       "depends on",
                                       depMtrx[row,1],depMtrx[row,2]))
                }
                else if (depMtrx[row,2] != "") {
                    ## Specific version requirement.  If they're not
                    ## removing the file, and there's no version req
                    ## then no dep problem
                    if (checkVers(vers,getReqOper(depMtrx[row,2]),
                                  getReqVers(depMtrx[row,2])) == FALSE) {
                        ## Broken dependency
                        out <- c(out,paste(Package(libList[[i]]),"version",
                                           stringRep(PkgVersion(libList[[i]])),
                                           "depends on",
                                           depMtrx[row,1],depMtrx[row,2]))
                    }
                }
            }
        }
    }
    return(out)
}

is.installed <- function(pkg, vers=NULL,
                         verbose=TRUE, oper="==", libs) {
    ## Given a package name will check to see if it is installed
    ## if vers != NULL, will look for a specific version
    ## if oper != NULL and vers != NULL, won't look for a specific
    ## version, but rather will look to see if there's a version
    ## that satifies 'version oper vers'
    if ((!is.null(vers))&&(class(vers) != "VersionNumber"))
        vers <- buildVersionNumber(vers)

    if (missing(libs))
        libs <- reposToolsLibPaths()

    for (lib in libs) {
        if (load.locLib(lib)) {
            locPkgs <- unlist(lapply(locLibList, Package))
            whichPkg <- which(locPkgs == pkg)

            if (length(whichPkg) != 0) {
                ## Found package
                if (is.null(vers)) {
                    return(TRUE)
                }
                else {
                    pkgVers <- lapply(locLibList[whichPkg],PkgVersion)
                    if (any(unlist(lapply(pkgVers,checkVers, oper,vers))))
                        return(TRUE)
                }
            }
            closeLocLib()
        }
        else {
            if (verbose == TRUE)
                note(paste("No locliblisting in R library",lib))
        }
    }
    return(FALSE)
}

getPkgVers <- function(pkg, libs=reposToolsLibPaths(), verbose=TRUE) {
    ## Will get the version of an installed package
    ## returns a list of versions if multiple installs
    ## returns an empty list if not intalled
    vers <- list()
    for (lib in libs) {
        if (load.locLib(lib)) {
            locPkgs <- unlist(lapply(locLibList, Package))
            whichPkg <- which(locPkgs == pkg)
            if (length(whichPkg) > 0) {
                vers[length(vers)+1] <- lapply(locLibList, PkgVersion)[whichPkg]
            }
            closeLocLib()
        }
        else {
            if (verbose == TRUE)
                note(paste("No locliblisting in R library",lib))
        }
    }
    return(vers)
}

getLocLibURLs <- function(pkgs, lib) {
    ## match up any requested packages to
    ## the 'Repos' field of the locLib.

    if (load.locLib(lib) == FALSE) {
        return(NULL)
    }
    locPkgs <- unlist(lapply(locLibList, Package))
    urls <- unlist(lapply(locLibList,Repos))
    urls <- urls[match(pkgs,locPkgs)]
    if (all(is.na(urls)))
        return(NULL)

    closeLocLib()
    return(as.vector(urls))
}

updateLocLib <- function(lib, pkg, vers, repEntry) {
    if (load.locLib(lib) == FALSE) {
        syncLocalLibList(lib)
        load.locLib(lib)
    }

    if (!is(vers,"VersionNumber"))
        vers <- buildVersionNumber("0")

    pI <- buildPkgInfo(pkg, vers)

    newPkg <- new("localPkg", Package=pkg, PkgVersion=vers,
                  Keywords=keywords(repEntry, pI),
                  Depends=depends(repEntry, pI),
                  Suggests=suggests(repEntry, pI),
                  Uses=uses(repEntry, pI),
                  Repos=repURL(repEntry))

    listPkg <- length(locLibList)
    curPkgs <- unlist(lapply(locLibList, Package))
    if (is.na(oP <- match(pkg, curPkgs)))
        listPkg <- listPkg + 1
    else
       listPkg <- oP

        locLibList[[listPkg]] <- newPkg

    save.locLib(locLibList, lib)
    return(TRUE)
}

resolve.depends <- function(pkg, repEntry, force=FALSE,
                            forward=TRUE, libs=reposToolsLibPaths(),
                            remove=FALSE, depends=TRUE,
                            suggests=TRUE, uses=FALSE,
                            searchOptions=TRUE, getAllDeps=FALSE,
                            versForce=TRUE, method="auto", getNewest=FALSE) {
    ## Passed a pkgInfo object.  Will make sure this package does not
    ## break any current dependencies - it will error if trying a
    ## level w/ a FALSE value, and warn if TRUE.
    errors <- c(depends,suggests,uses)
    if (length(which(errors)) > 0) {
        levels <- c("Depends","Suggests","Uses")
        halt <- FALSE
        out <- NULL

        retEnv <- new.env()
        retPSlist <- new("pkgStatusList",statusList=list())

        for (lib in libs) {
            if (load.locLib(lib) == TRUE) {
                for (i in seq(along=levels)) {
                    ## Find any "reverse" depends - ie if this breaks anything
                    ## currently installed
                    out <- c(out,findInstallDepends(pkgName(pkg),pkgVersion(pkg),
                                                    locLibList, levels[i],
                                                    remove))

                    if (forward == TRUE) {
                        ## Now check for forward depends, if they need
                        ## to obtain any other packages
                        fdEnv <- solveForwardDepends(pkg, repEntry,
                                                     lib, levels[i],
                                                     searchOptions=searchOptions,
                                                     getAllDeps=getAllDeps,
                                                     method=method,
                                                     getNewest=getNewest)
                        out <- c(out,get("out",fdEnv))
                        statusList(retPSlist) <- get("pkgList",fdEnv)

                    }
                    if ((!is.null(out))&&(errors[i] == TRUE))
                        halt <- TRUE
                }
                closeLocLib()
                if (!is.null(out)) {
                    if ((halt == TRUE)&&(force==FALSE)) {
                        warning(paste("\n",paste(out,collapse="\n"),"\n"))
                        assign("halt",TRUE,retEnv)
                    }
                    else
                        assign("halt",FALSE,retEnv)
                }
                else {
                    assign("halt",FALSE,retEnv)
                }
            }
        }
    }
    assign("pkgList",retPSlist,retEnv)
    return(retEnv)
}

solveForwardDepends <- function(pkgInfo, repEntry, lib, depLevel,
                                searchOptions=TRUE,getAllDeps=FALSE,
                                versForce=TRUE, method="auto",
                                getNewest=getNewest) {
    pkg <- pkgName(pkgInfo)
    vers <- pkgVersion(pkgInfo)
    curPkgInfo <- new("pkgInfo", pkgName=pkg, pkgVersion=vers)

    ## Begin a hack to double check for circular references
    cIlen <- length(reposToolsCurrentInstalls)+1
    reposToolsCurrentInstalls[[cIlen]] <<- curPkgInfo

    out <- NULL

    retEnv <- new.env()
    retPSlist <- new("pkgStatusList",statusList=list())

    curMtrx <- switch(depLevel,
                      "Depends"=getDepMtrx(depends(repEntry, pkgInfo)),
                      "Suggests"=getDepMtrx(suggests(repEntry, pkgInfo)),
                      "Uses"=getDepMtrx(uses(repEntry, pkgInfo))
                      )
    ## Now have a dependency matrix for this package on the
    ## current level of depends
    for (i in seq(along=curMtrx[,1])) {
        depPkg <- curMtrx[i,1]
        ## If this is a R depend and it has a version (we already
        ## know they have *a* version of R), check it
        if (depPkg == "R") {
            if (curMtrx[i,2]!="") {
                ## Handle R depends differently
                curRvers <- getRversion()
                depVers <- getReqVers(curMtrx[i,2])
                depOper <- getReqOper(curMtrx[i,2])
                if (checkVers(curRvers, depOper, depVers) == FALSE)
                {
                    out <- c(out, paste("Package",pkg,"version",
                                        vers, "requires R version",
                                        depVers,"but you only have",curRvers,
                                        "- please see www.r-project.org to",
                                        "upgrade your R install.\n"))
                }
            }
        }
        else {
            ## Package dependency
            if (curMtrx[i,2] != "") {
                depVers <- getReqVers(curMtrx[i,2])
                depOper <- getReqOper(curMtrx[i,2])
            }
            else
                depOper <- depVers <- NULL

            if (havePackage(depPkg, depVers, depOper) == FALSE) {
                ## Broken dependency
                brk <- TRUE
                errMsg <- paste("Package",pkg,"version",vers,
                                "requires package",depPkg)
                if (!is.null(depVers))
                    errMsg <- paste(errMsg,"version",depOper,depVers)
                if (getAllDeps == FALSE) {
                    outErrMsg <- paste(errMsg,", would you like to try ",
                                       "to install this package?",
                                       sep="")
                    ans <- userQuery(outErrMsg,c("yes","no"))
                }
                else
                    ans <- "yes"

                if (ans == "yes") {
                    inOut<- install.packages2(depPkg, repEntry,
                                              lib, getNewest=getNewest,
                                              searchOptions=searchOptions,
                                              getAllDeps=getAllDeps, method=method)
                    updPkgs <- updatedPkgs(inOut)
                    if (length(updPkgs) > 0) {
                        ## Get the proper pkg in the status list
                        ## this should be the last, but just in case:
                        if (depPkg %in% updPkgs) {
                            pkgPos <- match(depPkg, packages(inOut))
                            newDepVers <- getPkgVersion(inOut, pkgPos)
                            ## need to verify version
                            if (!(is.null(depOper))&&!(is.null(depVers))) {
                                if (checkVers(newDepVers,depOper,
                                              depVers) == TRUE) {
                                    brk <- FALSE
                                }
                            }
                            else
                                brk <- FALSE
                        }
                        statusList(retPSlist) <- inOut
                    }
                }
                if (brk == TRUE) {
                    out <- c(out,errMsg)
                    note(paste("Package",depPkg,
                               "not found in any known repository."))
                }
            }
        }
    }

    assign("out",out,retEnv)
    assign("pkgList",retPSlist, retEnv)

    ## Remove this package from the circRef checker
    reposToolsCurrentInstalls <<- reposToolsCurrentInstalls[-cIlen]

    return(retEnv)
}

havePackage <- function(depPkg, depVers, depOper) {
    if (is.installed(depPkg,depVers, verbose=FALSE, depOper))
        return(TRUE)
    else {
        ## check the reposToolsCurrentInstalls list
        pkgNames <- unlist(lapply(reposToolsCurrentInstalls, pkgName))
        if (!is.na(match(depPkg, pkgNames))) {
            ## Package is in the reposToolsCurrentInstalls
            if (is.null(depVers))
                return(TRUE)
            else {
                pkgVers <- lapply(reposToolsCurrentInstalls, pkgVersion)
                if (any(unlist(lapply(pkgVers, checkVers, depOper,
                                      depVers))))
                    return(TRUE)
            }
        }
    }
    return(FALSE)
}


remove.packages2 <- function(pkgs, lib, force=FALSE) {
    if (missing(lib) || is.null(lib)) {
        lib <- reposToolsLibPaths()[1]
        note(paste("argument `lib' is missing: using", lib))
    }
    if (load.locLib(lib) == FALSE) {
        return(FALSE)
    }

    for (pkg in pkgs) {
        if (!is.installed(pkg)) {
            note(paste("Package",pkg,"is not installed"))
            next
        }
        vers <- getPkgVers(pkg, lib)[[1]]
        ## Check to see if removing this package will break any
        ## dependencies.
        rd <- resolve.depends(buildPkgInfo(pkg,vers),NULL,force,
                              libs=lib, forward=FALSE, remove=TRUE)

        if (get("halt",rd) == TRUE)
            stop(paste("Could not remove package",pkg))

        ## Remove R from the local install
        print(paste("Removing package",pkg,"from system ...."))
        remove.packages(pkg, lib)

        options(show.error.messages=FALSE)
        z <- try(.find.package(pkg, lib))
        options(show.error.messages=TRUE)
        if (! inherits(z,"try-error")) {
            warning("Could not remove package",pkg)
            next
        }

        ## Remove this package from the localLibList
        locPkgs <- unlist(lapply(locLibList, Package))
        whichPkg <- which(locPkgs == pkg)
        if (length(whichPkg) != 0)
            locLibList <- locLibList[-whichPkg]
        print("Removal complete")

    }
    save.locLib(locLibList,lib)
}

repositories <- function(recurse=TRUE) {
    repList <- getOptReposList(recurse)
    title <- "Available Repositories: Select By Number, 0 For None"
    repIndex <- menu(repNames(repList),title=title)
    if (repIndex == 0)
        return(NULL)
    repEntry <- getRepEntry(repList, repIndex)
    return(repEntry)
}

install.packages2 <- function(pkgs, repEntry, lib, recurse=TRUE,
                              type, getNewest=TRUE, force=FALSE,
                              syncLocal=TRUE, searchOptions=FALSE,
                              versForce=TRUE, getAllDeps=FALSE,
                              method="auto") {

    if (missing(repEntry) && missing(pkgs)) {
        repEntry <- repositories()
        if (is.null(repEntry))
            stop(paste("No valid repository was chosen.",
                       "\nCan not continue."))
    }

    if (missing(lib))
        lib <- reposToolsLibPaths()[1]

    retVal <- update.packages2(pkgs, repEntry, lib, recurse,
                               type, force=force, syncLocal=syncLocal,
                               versForce=versForce,
                               searchOptions=searchOptions,
                               install=TRUE, update=FALSE,
                               prevRepos=FALSE, getNewest=getNewest,
                               getAllDeps=getAllDeps, method=method)

    return(retVal)
}

download.packages2 <- function(pkgs, repEntry, destDir, recurse=TRUE,
                               type, getNewest=TRUE, searchOptions=FALSE,
                               versForce=TRUE, force=FALSE, method="auto") {
    if (missing(destDir)) {
        note("destDir parameter missing, using current directory")
        destDir <- getwd()
    }

    retVal <- update.packages2(pkgs, repEntry, libs=destDir,
                               recurse=recurse, type=type,
                               searchOptions=searchOptions,
                               versForce=versForce,
                               syncLocal=FALSE, install=FALSE,
                               prevRepos=FALSE, update=FALSE,
                               force=force, method=method)
    return(retVal)
}

update.packages2 <- function(pkgs=NULL, repEntry, libs=reposToolsLibPaths(),
                             recurse=TRUE, type, prevRepos=TRUE,
                             force=FALSE, fileSelect=baseFileSelect,
                             syncLocal=TRUE, versForce=TRUE,
                             searchOptions=FALSE, getNewest=TRUE,
                             install=TRUE, update=TRUE,
                             getAllDeps=FALSE, method="auto") {

    if (missing(libs) || is.null(libs)) {
        libs <- reposToolsLibPaths()
        note(paste("argument `lib' is missing: using", libs))
    }

    if (missing(pkgs))
        pkgs <- NULL

    repList <- new("ReposList", repList=list())
    if (!missing(repEntry)) {
        if (inherits(repEntry,"ReposEntry")) {
            if (repType(repEntry) != "package")
                stop("update.packages2 requires a package repository")
        }
        else {
            stop("Supplied repEntry argument does not point to a valid repository.")
        }

        repList(repList) <- repEntry
        if (recurse == TRUE) {
            repList(repList) <- getSubRepList(repEntry)
        }
        ## See if we're tacking on option based repositories
        if (searchOptions == TRUE) {
            repList(repList) <- getOptReposList(recurse)
        }
        ## If a repository is specified, we don't want to be using
        ## prevRepos
        prevRepos <- FALSE
    }
    else
        repList(repList) <- getOptReposList(recurse)

    ## Double check to make sure there are entries
    if (numReps(repList) == 0)
        stop("No repositories to search")

    if (is.null(pkgs)) {
        if (update==TRUE) {
            ## If we're updating and packages is blank, need
            ## to fill in w/ all that we currently have
            pkgs <- NULL
            for (i in 1:length(libs)) {
                if (load.locLib(libs[i]) == FALSE) {
                    if (syncLocal == TRUE) {
                        syncLocalLibList(libs[i])
                        load.locLib(libs[i])
                    }
                    else {
                        stop("No local library and syncLocal is FALSE")
                    }
                }
                pkgs <- c(pkgs,unlist(lapply(locLibList,Package)))
                closeLocLib()
            }
        }
        else {
            ## In this case they just want to get everything from
            ## the specified repository
            pkgs <- repPkgs(repEntry)
        }
    }

    if (missing(type) || is.null(type)) {
        bioCOpt <- getOption("BioC")
        type <- bioCOpt$reposEntry$type
        if (is.null(type))
            stop("Can not determine default download type")
        else
            note(paste("You did not specify a download type.",
                       " Using a default value of:",type,
                       "\nThis will be fine for almost all users\n"))
    }

    pStatList <- new("pkgStatusList",statusList=list())

    for (lib in libs) {
        if ((load.locLib(lib) == FALSE)&&(install==TRUE)) {
            if(syncLocal == TRUE) {
                note(paste("Couldn't find a local library database in library",lib,".  Synching"))
                syncLocalLibList(lib)
                load.locLib(lib)
            }
            else {
                next
            }
        }

        if (prevRepos == TRUE) {
            ## Now try to download anything that has a prevRepos
            cPRenv <- checkPrevRepos(pkgs, lib, type, versForce,
                                     force, install, searchOptions,
                                     getAllDeps=getAllDeps,
                                     method=method)
            pkgs <- get("newPkgs", cPRenv)
            statusList(pStatList) <- get("cprStatList", cPRenv)
        }

        if (length(pkgs) > 0) {
            ## Retrieve the pkgList object for this download
            pkgList <- buildPkgListing(repList, pkgs, type)

            ## If not found at prevRepos, or prevRepos is false, need to
            ## resolve which option to download
            statusList(pStatList) <- fileSelect(pkgList, lib, type,
                                                getNewest=getNewest,
                                                versForce=versForce, force=force,
                                                install=install,
                                                searchOptions=searchOptions,
                                                getAllDeps=getAllDeps,
                                                method=method)
        }

        closeLocLib()
    }
    return(invisible(pStatList))
}

checkPrevRepos <- function(pkgs, lib, type, versForce, force,
                           install, searchOptions=TRUE,
                           getAllDeps=FALSE, method="auto") {
    ## For every package already installed that is in pkgList
    ## Try to get it from its previous URL

    cprEnv <- new.env()
    newPSList <- list()

    urls <- getLocLibURLs(pkgs, lib)
    if (is.null(urls)) {
        assign("newPkgs", pkgs, cprEnv)
        assign("cprStatList", newPSList, cprEnv)
        return(cprEnv)
    }

    found <- NULL
    whichPkgs <- which(urls != "NA")
    for (whichPkg in whichPkgs) {
        repE <- getReposEntry(urls[whichPkg])
        if (!is.null(repE)) {
            pkg <- pkgs[whichPkg]
            repInfos <- repPkgInfos(repE, pkg, type)
            versions <- lapply(repInfos, pkgVersion)
            maxEle <- getMaxElement(versions)
            maxVers <- pkgVersion(repInfos[[maxEle]])
            if (maxVers > getPkgVers(pkg,lib)[[1]]) {
                ## Need to download/install
                pInf <- buildPkgInfo(pkg, maxVers)
                rd <- resolve.depends(pInf, repE, force,
                                      searchOptions=searchOptions,
                                      getAllDeps=getAllDeps)

                if (get("halt",rd) == FALSE) {
                    fileName <- downloadFile(repE, pkg, stringRep(maxVers),
                                             type, method=method)
                    handleDownloadedFile(fileName, pkg, maxVers, type, lib,
                                         install, versForce, repE)
                    newPSList[[length(newPSList)+1]] <-
                        new("pkgStatus", package=pkg, found=TRUE,
                            updated=TRUE, url=repURL(repE),
                            pkgVersion=maxVers)
                    found <- c(found,whichPkg)
                }
            }
        }
    }

    if (length(found) > 0)
        pkgs <- pkgs[-found]

    assign("newPkgs", pkgs, cprEnv)
    assign("cprStatList", newPSList, cprEnv)
    return(cprEnv)
}

baseFileSelect <- function(pkgList, lib, type, getNewest=TRUE,
                           versForce=TRUE, force=FALSE, install=TRUE,
                           searchOptions=FALSE, getAllDeps=FALSE,
                           method="auto") {

    pkgStats <- new("pkgStatusList",statusList=list())
    Packages <- pkgList(pkgList)
    if (length(Packages) == 0)
        return(pkgStats)
    pkgs <- packages(pkgList)

    ## gotPkgs denotes packages that were acquired as a side effect
    ## of dependency checking.  Some of these might be ones that we
    ## were already looking for, so need to detect this
    gotPkgs <- NULL


    for (i in 1:length(Packages)) {
        if (pkgs[i] %in% gotPkgs)
            next

        ok <- TRUE
        if (getNewest == TRUE) {
            ## Simply find the newest version of each package and
            ## download
            rep <- 1
            index <- 1
            for (repCheck in 1:length(Packages[[i]])) {
                curMax <- buildVersionNumber("0")
                idx <- getMaxElement(Packages[[i]][[rep]])
                if (Packages[[i]][[rep]][[idx]] >= curMax) {
                    index <- idx
                    rep <- repCheck
                }
            }
        }
        else {
            ## Provide a menu selection of the available downloads and
            ## get the one selected in each case
            pvl <- pkgVersionList(pkgList, pkgs[i], "names")
            if (length(pvl) == 0) {
                next()
            }
            choices <- vector()
            repNames <- names(pvl)
            choiceReps <- vector()
            for (curRN in 1:length(pvl)) {
                curPvl <- pvl[[curRN]]
                for (curVer in 1:length(curPvl)) {
                    choiceReps <- c(choiceReps, curRN)
                    choices <- c(choices,
                                 paste(repNames[curRN],
                                       "has version",
                                       stringRep(curPvl[[curVer]])))
                }
            }

            if (length(choices) > 1) {
                title <- paste("\nPlease select (by number) a download ",
                               "site for package ", pkgs[i],":",sep="")
                index <- menu(choices,title=title)
                if (index == 0) {
                    note(paste("Skippping package",pkgs[i]))
                    next()
                }
            }
            else {
                index <- 1
            }
            rep <- choiceReps[index]
        }

        pkg <- pkgs[i]
        pkgVer <- Packages[[i]][[rep]][[index]]

        pkgInfo <- buildPkgInfo(pkg, pkgVer)

        repIndex <- as.numeric(names(Packages[[i]][rep]))
        repEntry <- getRepEntry(pkgList, repIndex)

        curVers <- getPkgVers(pkg,lib)

        ## !! Couldn't get the commented version of the if to work the
        ## !! way I was hoping to, using this as a hack for onw
        if (length(curVers) == 0)
            curVers[[1]] <- buildVersionNumber("0")
##        if ((length(curVers) > 0)||(pkgVer > curVers[[1]])) {

        if (pkgVer > curVers[[1]]) {

            ## Check to see if changing this file will break any
            ## other package's dependencies on it
            rd <- resolve.depends(pkgInfo, repEntry, force,
                                  searchOptions=searchOptions,
                                  getAllDeps=getAllDeps,
                                  versForce=versForce,
                                  getNewest=getNewest)

            rdPkgs <- get("pkgList",rd)
            if (length(packages(rdPkgs)) > 0) {
                gotPkgs <- c(gotPkgs,packages(rdPkgs))
                statusList(pkgStats) <- rdPkgs
            }

            if (get("halt",rd) == TRUE) {
                ok <- FALSE
            }
            else {
                ## Check cur version of file vs. this one

                ## Only need the first ele of returned list here
                ## as we're using just one lib as arg and can't have
                ## multi-installs.
                curPkgVers <- getPkgVers(pkg, lib)
                if (length(curPkgVers) > 0) {
                    if (curPkgVers[[1]] >= pkgVer) {
                        ok <- FALSE
                    }
                }
            }

            if (ok == TRUE) {
                fileName <- downloadFile(repEntry, pkg, stringRep(pkgVer),
                                         type, method=method)

                ret <- handleDownloadedFile(fileName, pkg, pkgVer, type, lib,
                                            install, versForce, repEntry)
            }
            else {
                ret <- FALSE
            }
        }
        else
            ret <- FALSE

        ## Add this package to the pkgStatusList
        statusList(pkgStats) <- new("pkgStatus",package=pkg,
                                    found=TRUE, updated=ret,
                                    url=repURL(repEntry),
                                    pkgVersion=pkgVer)
    }
    return(pkgStats)
}

handleDownloadedFile <- function(fileName, pkg, pkgVer, type, lib,
                                 install, versForce, repEntry) {
    if (is.null(fileName)) {
        warning(paste("No package ",pkg," version ",
                      stringRep(pkgVers)," of type ",type,
                      " exists at ",
                      repURL(repEntry),",skipping",sep=""))
        return(FALSE)
    }

    if (install == TRUE) {

        if (installPkg(fileName, pkg, pkgVer, type, lib, repEntry,
                       versForce)==TRUE) {
            updateLocLib(lib, pkg, pkgVer, repEntry)
        }
    }
    else {
        ## Just copy the file to the real location
        file.copy(fileName,file.path(lib,basename(fileName)),
                  TRUE)
    }
    unlink(fileName)
    return(TRUE)
}

installPkg <- function(fileName, pkg, vers, type, lib, repEntry,
                       versForce=TRUE) {
    if ((!is.null(fileName))&&(file.exists(fileName))) {
        OST <- .Platform$OS.type
        print(paste("Installing",pkg))
        if (type == "Win32") {
            ## Check for version compat)
            Rvers <- getRversion()
            builtVers <- pkgRVersion(repEntry, pkg, vers, type)
            out <- paste("Running R version ",Rvers," and package ",
                         pkg," was built for R version ",
                         builtVers,"\n",sep="")
            if (builtVers < Rvers) {
                if (versForce==FALSE) {
                    out <- paste(out, ", skipping.\nPlease ",
                              "look at the option versForce if you ",
                              "would like to continue.",sep="")
                    warning(out)
                    return(FALSE)
                }
                else {
                    out <- paste(out,", installing anyway.\n")
                    note(out)
                }
            }
            if (OST != "windows") {
                warning(paste("Attempting to install Win32 version of",
                               pkg,"on a",OST,"system, skipping."))
                return(FALSE)
            }
            status <- zip.unpack(fileName, lib)
        }
        else if (type == "Source") {
            if (OST != "unix") {
                warning(paste("Attempting to install Unix version of",
                           pkg,"on a",OST,"system, skipping."))
                return(FALSE)
            }
            cmd <- paste(file.path(R.home(), "bin", "R"),
                         "CMD INSTALL -l", lib, fileName)
            status <- system(cmd)
        }
        else {
            warning(paste("Do not know how to install packages of type",
                          " ",type,", skipping.",sep=""))
            return(FALSE)
        }

        if (status != 0) {
            warning(paste("Installation of package", pkg,
                          "had non-zero exit status"))
        }
        else {
            print("Installation complete")
        }
    }
    return(TRUE)
}

getOptReposList <- function(recurse=TRUE) {
    ## Takes the option "repositories" and returns a list of
    ## reposEntry objects from those entries.
    reps <- as.character(getReposOption())
    if (length(reps) > 0) {
        repL <- lapply(reps,getReposEntry)
        ## remove NULL entries
        repL <- repL[!sapply(repL,is.null)]
        return(getReposList(repL, recurse))
    }
    else {
        return(NULL)
    }
}

reposToolsLibPaths <- function(baseLib=.libPaths(),
                               tmpLibPath=file.path(tempdir(),"tempLibs")) {
    ## Will add any temporarily maintained liblisting.Rda files to
    ## the search path
    newDirs <- character()

    if ((file.exists(tmpLibPath))&&(file.info(tmpLibPath)$isdir)) {
        newDirs <- dir(tmpLibPath, full.names=TRUE)
        if (length(newDirs) > 0) {

            ## Filter out any that aren't actually directories as these are
            ## obviously garbage
            areDirs <- unlist(lapply(lapply(newDirs,file.info),function(x){return(x$isdir)}))
            newDirs <- newDirs[areDirs]
        }
    }

    return(c(baseLib, newDirs))
}
buildPkgDepGraph <- function(pkg, repEntry, type,
                             curNodes=vector(mode="character")) {
    ## Given a repository will look first to see what packages it
    ## depends on, and what packages *those* depend on and so on,
    ## generating a complete dependency graph

    require("graph") || stop("buildPkgDepGraph needs package 'graph'")

    ## Create the 'root' of this graph
    dGraph <- new("graphNEL", nodes=pkg, edgemode="directed")


    if (length(pkg) != 1) {
        warning("pkg argument must be of length one")
        return(dGraph)
    }

    curRepList <- new("ReposList", repList=list())

    if (missing(type) || is.null(type)) {
        bioCOpt <- getOption("BioC")
        type <- bioCOpt$reposEntry$type
        if (is.null(type)) {
            warning("Can not determine package type")
            return(dGraph)
        }
    }

    if (!missing(repEntry)) {
        if (!inherits(repEntry,"ReposEntry")) {
            warning("Supplied repEntry argument is not a ReposEntry object")
            return(dGraph)
        }
        repList(curRepList) <- repEntry
        repList(curRepList) <- getSubRepList(repEntry)
    }
    else
        repList(curRepList) <- getOptReposList()

    curPkgList <- buildPkgListing(curRepList, pkg, type)
    Packages <- pkgList(curPkgList)
    if (length(Packages) == 0) {
        note(paste("Package",pkg,"not found in any online repositories"))
        return(dGraph)
    }

    rep <- 1
    index <- 1
    for (i in 1:length(Packages[[1]])) {
        curMax <- buildVersionNumber("0")
        idx <- getMaxElement(Packages[[1]][[rep]])
        if (Packages[[1]][[rep]][[idx]] >= curMax) {
            index <- idx
            rep <- i
        }
    }

    pkgVer <- Packages[[1]][[rep]][[index]]
    curPkgInfo <- buildPkgInfo(pkg, pkgVer)
    repIndex <- as.numeric(names(Packages[[1]][rep]))
    repEntry <- getRepEntry(curPkgList, repIndex)

    ## Get dependencies for this package, adding each one to the graph
    depMtrx <- getDepMtrx(depends(repEntry, curPkgInfo))

    for (i in seq(along=depMtrx[,1])) {
        depPkg <- depMtrx[i,1]
        if (depPkg == "R")
            next

        ## recursively call buildPkgDepGraph and take it's dGraph
        ## return and 'join' it with this graph, then create an
        ## edge between it's 'root' and this package
        newCurNodes <- unique(c(curNodes, nodes(dGraph)))

        if (!depPkg %in% curNodes) {
            newGraph <- buildPkgDepGraph(depPkg, curNodes=newCurNodes)
            dGraph <- join(dGraph, newGraph)
        }
        else {
            ## If the package is not in curnodes nor in the
            ## current graph, it represents a circular reference
            ## and the node needs to be in the current graph.
            ## If the node is in currnodes but already in the
            ## graph, it is simply a duplicate entry.
            if (!depPkg %in% nodes(dGraph))
                dGraph <- addNode(depPkg, dGraph)
        }

        dGraph <- addEdge(depPkg, pkgName(curPkgInfo), dGraph, 1)
        curNodes <- newCurNodes
    }
    dGraph
}
load.depends <- function(x,suggests=TRUE,uses=TRUE,Rversion=TRUE) {
    depList <- getObjectDeps(x)
    if (length(depList) == 0)
        return()
    else {
        if (!is.null(depList$Depends))
            check.depends(x, getDepMtrx(depList$Depends),failRDep=Rversion)
        if (!is.null(depList$Suggests))
            check.depends(x, getDepMtrx(depList$Suggests),failPkgDep=suggests,
                          failRDep=Rversion)
        if (!is.null(depList$Uses))
            check.depends(x, getDepMtrx(depList$Uses),failPkgDep=uses,
                          failRDep=Rversion)
    }
}

unresolved.depends <- function(x,suggests=TRUE,uses=TRUE,Rversion=TRUE) {
    depList <- getObjectDeps(x,suggests,uses)
    outList <- list()
    if (length(depList) == 0)
        return()
    else {
        if (!is.null(depList$Depends)) {
            deps <- check.depends(x, getDepMtrx(depList$Depends),
                                  failRDep=Rversion, load=FALSE)
            if (length(deps) > 0)
                outList$Depends <- deps
        }
        if (!is.null(depList$Suggests)) {
            sugs<- check.depends(x, getDepMtrx(depList$Suggests), load=FALSE,
                                 failPkgDep=suggests,failRDep=Rversion)
            if (length(sugs) > 0)
                outList$Suggests <- sugs
        }
        if (!is.null(depList$Uses)) {
            uses <- check.depends(x, getDepMtrx(depList$Uses),
                                  failPkgDep=uses, load=FALSE,
                                  failRDep=Rversion)
            if (length(uses) > 0)
                outList$Uses <- uses
        }
    }
    return(outList)
}

check.depends <- function(x, depMtrx, failPkgDep=TRUE,failRDep=TRUE,
                          load=TRUE) {
    outLine <- vector()
    for (i in 1:nrow(depMtrx)) {
        if (depMtrx[i,1] == "R") {
            ## If the R version doesn't match the requested dependency
            ## stop the function w/ an error
            ret <- checkRVersion(x,depMtrx[i,,drop=FALSE])
            if (ret == FALSE) {
                outLine <- c(outLine,paste(depMtrx[i,1]," (",
                                           depMtrx[i,2],")",sep=""))
                if (load == TRUE) {
                    outString <- paste(" package ",x," requires R version ",
                                       getReqVers(depMtrx[i,2]),
                                       ".  Please see http://http://cran.r-project.org/ for information on how to get the proper version.", sep="")
                    if (failRDep == TRUE) {
                        stop(paste("Can not continue:",outString))
                    }
                    else {
                        warning(outString)
                    }
                }
            }
        }
        else {
            outString <- paste("Package",x,"requires version",
                               depMtrx[i,2],
                               "of package",depMtrx[i,1],
                               ", which is not currently installed.",
                               "Please see help(\"install.packages2\")",
                               "for instructions on how to install",
                               "this package")
            if (depMtrx[i,2] != "") {
                ## need to check pkg version
                pkgVers <- getPkgVers(x,verbose=FALSE)
                rets <- lapply(pkgVers,checkVers,
                               getReqOper(depMtrx[i,2]),
                               getReqVers(depMtrx[i,2]))
                if (!any(unlist(rets))) {
                    outLine <- c(outLine,paste(depMtrx[i,1]," (",
                                               depMtrx[i,2],")",sep=""))
                    if (load == TRUE) {
                        if (failPkgDep == TRUE)
                            stop(paste("Can not continue:",outString))
                        else
                            warning(outString)
                    }
                }
            }
            if (load == TRUE) {
                ## !!!! If requiring a version, should eventually make
                ## !! sure we are getting the proper one.  'require'
                ## !! does not yet
                ## !! handles this.  In the shorter term, should get the highest
                ## !! version that matches criterion if there are multiples
                curWarn <- getOption("warn")
                options(warn = -1)
                on.exit(options(warn=curWarn),add=TRUE)
                ret <- do.call("require",list(depMtrx[i,1]))
                options(warn=curWarn)
                if (!ret) {
                    if (failPkgDep == TRUE)
                        stop(paste("Can not continue:",outString))
                    else
                        warning(outString)
                }
            }
            else {
                ## Only need to check to see if the pkg is installed

                if (!is.installed(depMtrx[i,1],verbose=FALSE))
                    outLine <- c(outLine,paste(depMtrx[i,1]))
            }
        }
    }
    invisible(outLine)
    return(outLine)
}

checkRVersion <- function(x,depMtrx) {
    ## Passed a row of a dependency matrix, will check to see if the R
    ## version is a dependency, and if so if the current R version is
    ## acceptable.  Returns TRUE if either the R version is fine or
    ## this row does not correspond to a R dependency, and FALSE if it
    ## does and the version is not correct
    if (depMtrx[1,1] == "R") {
        if (depMtrx[1,2] != "") {
            ## need to check R version
            rVers <- getRversion()
            ret <- checkVers(rVers,getReqOper(depMtrx[1,2]),
                             getReqVers(depMtrx[1,2]))
            if (ret == FALSE)
                return(FALSE)
        }
    }
    return(TRUE)
}

getVigDeps <- function(x,suggests=TRUE,uses=TRUE) {
    if (!file.exists(x))
        stop(paste("Vignette file",x,"not found."))

    vigList <- getVigInfo(x)
    depList <- list()
    vigDeps <- vigList$VignetteDepends
    if (!is.na(vigDeps))
        depList$Depends <- vigDeps

    if (suggests) {
        vigSugs <- vigList$VignetteSuggests
        if (!is.na(vigSugs))
            depList$Suggests <- vigSugs
    }

    if(uses) {
        vigUses <- vigList$VignetteUses
        if (!is.na(vigUses))
            depList$Uses <- vigUses
    }

    return(depList)
}

getObjectDeps <- function(x,suggests=TRUE,uses=TRUE) {
    ## First determien if x is a package or vignette
    if (length(grep("\\.(Rnw|Snw|rnw|snw)$",x)) == 0)
        depList <- getPkgDeps(x,suggests,uses)
    else
        depList <- getVigDeps(x,suggests,uses)
    if (length(depList) == 0)
        return()

    return(depList)
}

getPkgDeps <- function(x,suggests=TRUE,uses=TRUE) {
    depList <- list()
    if (!is.na(desc <- package.description(x,fields="Depends")))
        depList$Depends <- strsplit(desc,",[[:space:]]*")[[1]]

    if(suggests==TRUE) {
        if (!is.na(sugs <- package.description(x,fields="Suggests"))) {
            sugs <- strsplit(sugs,",[[:space:]]*")[[1]]
            depList$Suggests <- sugs
        }
    }

    if (uses==TRUE) {
        if (!is.na(use <- package.description(x,fields="Uses"))) {
            use <- strsplit(use,",[[:space:]]*")[[1]]
            depList$Uses <- use
        }
    }
    return(depList)
}

getDepMtrx <- function(deps) {
    ## Takes in a vector of dependencies, separates any version
    ## requirements.  Returns a matrix, col1 is the packs, col2
    ## is any algebraic requirements
    if ((is.null(deps))||(length(deps)==0)||(deps==""))
        return(NULL)

    tmp <- strsplit(deps,",")
    if (!is.null(tmp))
        deps <- unlist(tmp)

    deps <- gsub("\\)","",deps)
    deps <- strsplit(deps,"\\(")
    ## Now have a list, some w/ 1 els some w/ more (those w/ reqs)
    deps <- lapply(deps, function(x){
                             if (length(x) == 1)
                                 return(c(x, ""))
                             else
                                 return(x)
                         }
                   )
    pkgs <- lapply(deps,function(x){return(x[1])})
    reqs <- lapply(deps,function(x){return(x[2])})
    depMtrx <- cbind(matrix(unlist(pkgs)),matrix(unlist(reqs)))
    ## Kill any trailing whitespace on entries
    for (i in 1:2)
        depMtrx[,i] <- gsub("[[:space:]]$","",depMtrx[,i])

    if (depMtrx[1,1] == "NA")
        depMtrx <- NULL

    return(depMtrx)
}


getReqVers <- function(depReq) {
    return(buildVersionNumber(gsub("(>)|(<)|(=)|([[:space:]])","",depReq)))
}

getReqOper <- function(depReq) {
    return(gsub("[^><=]","",depReq))
}





## Misc small utility functions

note <- function(...) {
    ## A "less drastic" version of warning()
    if (nargs() > 0) {
        message <- paste("Note:",...,"\n")
        cat(message)
    }
}

userQuery <- function(msg, allowed=c("yes","y","no","n")) {
    ## Prompts the user with a string and for an answer
    ## repeats until it gets allowable input
    repeat {
        allowMsg <- paste("[",paste(allowed,collapse="/"),
                          "] ", sep="")
        outMsg <- paste(msg,allowMsg)
        cat(outMsg)
        ans <- readLines(n=1)
        if (ans %in% allowed)
            break
        else
            cat(paste(ans,"is not a valid response, try again.\n"))
    }
    ans
}
### pkgInfo object system

    setClass("pkgInfo", representation(pkgName="character",
                                       pkgVersion="VersionNumber",
                                       pkgPath="character"))

    if (is.null(getGeneric("pkgName")))
        setGeneric("pkgName", function(object)
                   standardGeneric("pkgName"))

    if (is.null(getGeneric("pkgVersion")))
        setGeneric("pkgVersion", function(object)
                   standardGeneric("pkgVersion"))

    if (is.null(getGeneric("pkgPath")))
        setGeneric("pkgPath", function(object)
                   standardGeneric("pkgPath"))

    if (is.null(getGeneric("==")))
        setGeneric("==", function(e1, e2)
                   standardGeneric("=="))

    setMethod("pkgName", "pkgInfo", function(object)
              object@pkgName)
    setMethod("pkgVersion", "pkgInfo", function(object)
              object@pkgVersion)
    setMethod("pkgPath", "pkgInfo", function(object)
              object@pkgPath)

    setMethod("==","pkgInfo", function(e1, e2) {
        if (!is(e2, "pkgInfo"))
            stop("Right hand argument is not of class 'pkgInfo'")

        if (pkgName(e1) != pkgName(e2))
            return(FALSE)
        if (pkgVersion(e1) != pkgVersion(e2))
            return(FALSE)
        pathA <- pkgPath(e1)
        pathB <- pkgPath(e2)
        if (length(pathA) != length(pathB))
            return(FALSE)
        if ((length(pathA) > 0)&&(pkgPath(e1) != pkgPath(e2)))
            return(FALSE)
        TRUE
    })

    setMethod("show","pkgInfo",function(object) {
        print(paste(pkgName(object),": ",
                    stringRep(pkgVersion(object)),sep=""))
    })


buildPkgInfo <- function(name, vers, path="") {
    if (missing(name))
        stop("Need a package name")

     if (missing(vers)) {
        if (missing(path))
            return(new("pkgInfo",pkgName=name))
        else
            return(new("pkgInfo",pkgName=name, pkgPath=path))
    }
    else {
        if (is.character(vers)) {
            vers <- buildVersionNumber(vers)
        }
        if (missing(path))
            return(new("pkgInfo",pkgName=name,pkgVersion=vers))
        else
           return(new("pkgInfo",pkgName=name,pkgVersion=vers,pkgPath=path))
    }
}

    setClass("localPkg", representation(Package="character",
                                        PkgVersion="VersionNumber",
                                        Keywords="character",
                                        Depends="character",
                                        Suggests="character",
                                        Uses="character",
                                        Repos="character"))


    if (is.null(getGeneric("Package")))
        setGeneric("Package", function(object, ...)
                   standardGeneric("Package"))
    setMethod("Package", "localPkg", function(object, ...)
              object@Package)

    if (is.null(getGeneric("PkgVersion")))
        setGeneric("PkgVersion", function(object, ...)
                   standardGeneric("PkgVersion"))
    setMethod("PkgVersion", "localPkg", function(object, ...)
              object@PkgVersion)

    if (is.null(getGeneric("Keywords")))
        setGeneric("Keywords", function(object, ...)
                   standardGeneric("Keywords"))
    setMethod("Keywords", "localPkg", function(object, ...)
              object@Keywords)

    if (is.null(getGeneric("Depends")))
        setGeneric("Depends", function(object, ...)
                   standardGeneric("Depends"))
    setMethod("Depends", "localPkg", function(object, ...)
              object@Depends)


    if (is.null(getGeneric("Suggests")))
        setGeneric("Suggests", function(object, ...)
                   standardGeneric("Suggests"))
    setMethod("Suggests", "localPkg", function(object, ...)
              object@Suggests)

    if (is.null(getGeneric("Uses")))
        setGeneric("Uses", function(object, ...)
                   standardGeneric("Uses"))
    setMethod("Uses", "localPkg", function(object, ...)
              object@Uses)

    if (is.null(getGeneric("Repos")))
        setGeneric("Repos", function(object, ...)
                   standardGeneric("Repos"))
    setMethod("Repos", "localPkg", function(object, ...)
              object@Repos)

    setClass("pkgStatusList", representation(statusList="list"))

    setMethod("show", "pkgStatusList", function(object) {
        if (length(statusList(object)) == 0) {
            cat("No package information available\n")
        }
        else {
            found <- found(object)
            out <- NULL
            if (length(found) > 0) {
                urls <- urls(object)
                pkgs <- packages(object)
                vers <- pkgVersions(object)
                updated <- updated(object)
                foundUrls <- urls[found]

                for (rep in unique(foundUrls)) {
                    eles <- which(urls==rep)
                    ## Now see which of these were actually updated
                    upPos <- eles[match(updated,eles)]
                    upPos <- upPos[!is.na(upPos)]
                    if ((length(upPos) > 0) && !(is.na(upPos))) {
                        versStrs <- unlist(lapply(vers,stringRep))[upPos]
                        pkgOuts <- paste(pkgs[upPos],versStrs,
                                         sep=" version ", collapse="\n\t")
                        out <- paste(out,"From URL:  ",rep,
                                     "\n\t",pkgOuts,"\n\n",sep="")
                    }
                }
            }
            ## Fill out the not found info
            nfPkgs <- pkgs[notFound(object)]
            if (length(nfPkgs) > 0) {
                out <- paste(out,"Packages that were not found:\n\t",
                             paste(nfPkgs,collapse="\n\t"),"\n\n",sep="")
            }
            ## Fill out the not updated info
            nuPkgs <- pkgs[notUpdated(object)]
            if (length(nuPkgs) > 0) {
                out <- paste(out,"Packages that were not updated:\n\t",
                             paste(nuPkgs,collapse="\n\t"),"\n\n",sep="")
            }
            cat(out)
        }
    })

    if (is.null(getGeneric("statusList")))
        setGeneric("statusList", function(object)
                   standardGeneric("statusList"))
    setMethod("statusList","pkgStatusList", function(object)
              object@statusList)
    if (is.null(getGeneric("statusList<-")))
        setGeneric("statusList<-", function(object, value)
                   standardGeneric("statusList<-"))
    setMethod("statusList<-", "pkgStatusList", function(object,value)
          {
              if (inherits(value,"pkgStatusList")) {
                  newList <- statusList(value)
                  object@statusList <- c(object@statusList, newList)
              }
              else if (inherits(value,"pkgStatus")) {
                  object@statusList[[length(object@statusList)+1]] <-
                      value
              }
              else if (is.list(value)) {
                  object@statusList <- c(object@statusList,value)
              }
              object
          })

    if (is.null(getGeneric("packages")))
        setGeneric("packages", function(object)
                   standardGeneric("packages"))
    setMethod("packages", "pkgStatusList", function(object) {
        z <- unlist(lapply(statusList(object),package))
        z
    })

    if (is.null(getGeneric("found")))
        setGeneric("found", function(object)
                   standardGeneric("found"))
    setMethod("found", "pkgStatusList", function(object) {
        z <- which(unlist(lapply(statusList(object),found)))
        z
    })

    if (is.null(getGeneric("notFound")))
        setGeneric("notFound", function(object)
                   standardGeneric("notFound"))
    setMethod("notFound", "pkgStatusList", function(object) {
        z <- unlist(lapply(statusList(object),found))
        z <- which(!z)
        z
    })

    if (is.null(getGeneric("updated")))
        setGeneric("updated", function(object)
                   standardGeneric("updated"))
    setMethod("updated", "pkgStatusList", function(object) {
        z <- unlist(lapply(statusList(object),updated))
        if (!(is.null(z)))
            z <- which(z)
        z
    })

    if (is.null(getGeneric("notUpdated")))
        setGeneric("notUpdated", function(object)
                   standardGeneric("notUpdated"))
    setMethod("notUpdated", "pkgStatusList", function(object) {
        z <- unlist(lapply(statusList(object),updated))
        z <- which(!z)
        z
    })

    if (is.null(getGeneric("urls")))
        setGeneric("urls", function(object)
                   standardGeneric("urls"))
    setMethod("urls", "pkgStatusList", function(object) {
        z <- unlist(lapply(statusList(object),URL))
        z
    })

    if (is.null(getGeneric("pkgVersions")))
        setGeneric("pkgVersions", function(object)
                   standardGeneric("pkgVersions"))
    setMethod("pkgVersions", "pkgStatusList", function(object) {
        z <- lapply(statusList(object),pkgVersion)
        z
    })



    if (is.null(getGeneric("foundPkgs")))
        setGeneric("foundPkgs", function(object)
                   standardGeneric("foundPkgs"))
    setMethod("foundPkgs", "pkgStatusList", function(object) {
        z <- found(object)
        pkgs <- packages(object)[z]
        pkgs
    })

    if (is.null(getGeneric("notFoundPkgs")))
        setGeneric("notFoundPkgs", function(object)
                   standardGeneric("notFoundPkgs"))
    setMethod("notFoundPkgs", "pkgStatusList", function(object) {
        z <- notFound(object)
        pkgs <- packages(object)[z]
              pkgs
    })

    if (is.null(getGeneric("updatedPkgs")))
        setGeneric("updatedPkgs", function(object)
                   standardGeneric("updatedPkgs"))
    setMethod("updatedPkgs", "pkgStatusList", function(object) {
        z <- updated(object)
        if (!(is.null(z)))
            pkgs <- packages(object)[z]
        else
            pkgs <- NULL
        pkgs
    })

    if (is.null(getGeneric("getPackage")))
        setGeneric("getPackage", function(object,pos)
                   standardGeneric("getPackage"))
    setMethod("getPackage","pkgStatusList",function(object,pos)
              package(object@statusList[[pos]]))
    if (is.null(getGeneric("getFound")))
        setGeneric("getFound", function(object,pos)
                   standardGeneric("getFound"))
    setMethod("getFound","pkgStatusList",function(object,pos)
              found(object@statusList[[pos]]))
    if (is.null(getGeneric("getUpdated")))
        setGeneric("getUpdated", function(object,pos)
                   standardGeneric("getUpdated"))
    setMethod("getUpdated","pkgStatusList",function(object,pos)
              updated(object@statusList[[pos]]))
    if (is.null(getGeneric("getUrl")))
        setGeneric("getUrl", function(object,pos)
                   standardGeneric("getUrl"))
    setMethod("getUrl","pkgStatusList",function(object,pos)
              URL(object@statusList[[pos]]))
    if (is.null(getGeneric("getPkgVersion")))
        setGeneric("getPkgVersion", function(object,pos)
                   standardGeneric("getPkgVersion"))
    setMethod("getPkgVersion","pkgStatusList",function(object,pos)
              pkgVersion(object@statusList[[pos]]))



    setClass("pkgStatus", representation(package="character",
                                         found="logical",
                                         updated="logical",
                                         url="character",
                                         pkgVersion="VersionNumber"))
    if (is.null(getGeneric("package")))
        setGeneric("package", function(object)
                   standardGeneric("package"))
    setMethod("package","pkgStatus",function(object)
              object@package)
    if (is.null(getGeneric("found")))
        setGeneric("found", function(object)
                   standardGeneric("found"))
    setMethod("found","pkgStatus",function(object)
              object@found)
    if (is.null(getGeneric("updated")))
        setGeneric("updated", function(object)
                   standardGeneric("updated"))
    setMethod("updated","pkgStatus",function(object)
              object@updated)
    if (is.null(getGeneric("URL")))
        setGeneric("URL", function(object)
                   standardGeneric("URL"))
    setMethod("URL","pkgStatus",function(object)
              object@url)
    if (is.null(getGeneric("pkgVersion")))
        setGeneric("pkgVersion", function(object)
                   standardGeneric("pkgVersion"))
    setMethod("pkgVersion","pkgStatus",function(object)
              object@pkgVersion)

    setMethod("show","pkgStatus",function(object) {
        out <- paste("Package ",package(object))
        if (found(object) == TRUE) {
            out <- paste(out,"version",pkgVersion(object),"was found at",
                         URL(object), "and")
            if (updated(object) == FALSE)
                out <- paste(out,"not")
            out <- paste(out,"updated.")
        }
        else
            out <- paste(out,"was not found.")
        print(out)
    })


getReposOption <- function() {
    out <- getOption("repositories2")
    out
}

getMangledName <- function(pkg, vers, mangSep=":v:") {
    if (inherits(vers,"VersionNumber"))
        vers <- stringRep(vers)
    return(paste(pkg,mangSep,vers,sep=""))
}

getReposEntry <- function(repURL, repFile="replisting",
                          rddFile="repdatadesc.rda") {
    ## First check to see if repURL is symbolic
    optReps <- getReposOption()
    if (repURL %in% names(optReps))
        repURL <- as.character(optReps[repURL])

    ## Get the replisting from the remote repository
    repListing <- getReplisting(repURL, repFile)
    if (is.null(repListing)) {
        note(paste(repURL,
                   "does not seem to have a valid repository, skipping"))
        return(NULL)
    }
    repDf <- loadRepDD(repURL, rddFile)
    ## repDf is supposed to be able to be NULL, but need something of
    ## class 'data.frame' for the reposEntry object.
    if (is.null(repDf))
        repDf <- data.frame()
    repDf <- new("repdatadesc", repdatadesc=repDf)
    return(buildReposEntry(repListing, repDf))
}

getReplisting <- function(repURL,repFile="replisting") {
    ## Passed a repository URL, will retrieve the replisting (if any)
    fileURL <- paste(repURL, repFile, sep="/")
    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages = TRUE))
    repCon <- try(url(fileURL))
    if (inherits(repCon,"try-error") == TRUE) {
        warning(paste("Failed to read replisting at",
                      repURL))
        return(NULL)
    }
    on.exit(close(repCon),add=TRUE)
    tryRes <- try(read.dcf(file=repCon))
    options(show.error.messages=TRUE)
    if (inherits(tryRes,"try-error") == TRUE) {
        return(NULL)
    }
    if (length(tryRes) == 0) {
        warning("Empty replisting file at ", repURL)
        return(NULL)
    }

    repL <- new("replisting", replisting=tryRes)
    return(repL)
}

loadRepDD <- function(repURL, repDD="repdatadesc.rda") {
    ## Will load the remote repository's repdatadesc RDA file, if
    ## available
    ddURL <- paste(repURL,repDD,sep="/")

    options(show.error.messages=FALSE)
    retVal <- try(loadURL(ddURL, quiet=TRUE, mode="wb"))
    options(show.error.messages=TRUE)

    if (inherits(retVal, "try-error")) {
        warning(paste("Failed to read repdatadesc at",
                      repURL))
        return(NULL)
    }

    return(reposDF)
}

    setClass("replisting", representation(replisting="matrix"))


    if (is.null(getGeneric("replisting")))
        setGeneric("replisting", function(object)
                   standardGeneric("replisting"))

    if (is.null(getGeneric("repURL")))
        setGeneric("repURL", function(object, pos)
                   standardGeneric("repURL"))

    setMethod("summary","replisting",function(object, ...) {
        repL <- replisting(object)
        rem <- nrow(repL)-1
        repL <- repL[1,,drop=FALSE]
        print.matrix(repL,quote=FALSE,collab=c("Repository Name   ",
                                      "Repository Type   ","Repository Base URL   ",
                                      "Repository URL Path"),
                     rowlab=paste(as.character(1:nrow(repL)),":",sep=""))
        if (rem > 1) {
            print.noquote(paste(rem,"other repositories known.  Use verbose=TRUE to see their information"))
        }
    })


    setMethod("show","replisting",function(object) {
        rL <- replisting(object)
        nreps <- nrow(rL)
        cat("Repository Listing:\n")
        for(i in seq(along=nreps)) {
            cat("\tRepository: ", rL[i, "repname"], "\n")
            cat("\t\t Type: ", rL[i, "reptype"], "\n")
        }
        return(NULL)})


    setMethod("replisting", "replisting", function(object)
              object@replisting)

    setMethod("repURL","replisting",
              function(object, pos) {
                  if (missing(pos))
                      pos <- 1
                  base <- object@replisting[pos,"repaddrBase"]
                  path <- object@replisting[pos,"repaddrPath"]

                  reps <- getReposOption()
                  if (!is.na(reps[base]))
                      base <- as.character(reps[base])

                  url <- paste(base, path, sep="/")
                  return(url)
              })

    if (is.null(getGeneric("numSubReps")))
        setGeneric("numSubReps", function(object)
                   standardGeneric("numSubReps"))
    setMethod("numSubReps", "replisting", function(object)
              (nrow(object@replisting)-1))

    setClass("repdatadesc", representation(repdatadesc="data.frame"))

if (is.null(getGeneric("repdatadesc")))
    setGeneric("repdatadesc", function(object)
               standardGeneric("repdatadesc"))

    setMethod("repdatadesc", "repdatadesc", function(object)
              object@repdatadesc)
    setMethod("summary","repdatadesc",function(object, ...) {
            repD <- repdatadesc(object)
            print(noquote(paste(as.character(nrow(repD)),"packages available")))
        })

    if (is.null(getGeneric("depends")))
        setGeneric("depends", function(object, pkgInfo)
            standardGeneric("depends"))
    setMethod("depends","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Depends[[1]]
            out
        }
        else
            return(NULL)
    })

    if (is.null(getGeneric("suggests")))
        setGeneric("suggests", function(object, pkgInfo)
            standardGeneric("suggests"))
    setMethod("suggests","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Suggests[[1]]
            out
        }
        else
            return(NULL)
    })

    if (is.null(getGeneric("uses")))
        setGeneric("uses", function(object, pkgInfo)
            standardGeneric("uses"))
    setMethod("uses","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Uses[[1]]
            out
        }
        else
            return(NULL)
    })

    if (is.null(getGeneric("keywords")))
        setGeneric("keywords", function(object, pkgInfo)
            standardGeneric("keywords"))
    setMethod("keywords","repdatadesc",function(object, pkgInfo){
        mangName <- getMangledName(pkgName(pkgInfo), pkgVersion(pkgInfo))
        if (mangName %in% rownames(object@repdatadesc)) {
            out <- object@repdatadesc[mangName,]$Keywords[[1]]
            out
        }
        else
            return(NULL)
    })

    setMethod("show","repdatadesc",function(object) {
        repD <- repdatadesc(object)
        if (inherits(repD,"Pkg")) {
            pkgs <- repD$Package
            vers <- unlist(lapply(repD$Version,as.character))
            OSspec <- repD$OSspecific
            types <- unlist(lapply(lapply(OSspec,names),paste,collapse=", "))
            mat <- t(rbind(pkgs,vers,types))
            print.matrix(mat,quote=FALSE,collab=c("Package   ",
                                         "Version    ","Types Available"),
                         rowlab=paste(as.character(1:nrow(repD)),":",sep=""))
        }
        else if (inherits(repD,"Vignette")) {
            vigs <- repD$VignetteIndexEntry
            vigMtrx <- matrix(vigs)
            print.matrix(vigMtrx,quote=FALSE,
                         collab=c("Vignette Name"),
                         rowlab=paste(as.character(1:nrow(repD)),":",sep=""))
        }
    })


    setClass("pkgListing", representation(pkgList="list",
                                          repList="ReposList"))

    if (is.null(getGeneric("pkgList")))
        setGeneric("pkgList", function(object)
                   standardGeneric("pkgList"))
    setMethod("pkgList", "pkgListing", function(object)
              object@pkgList)

    if (is.null(getGeneric("repList")))
        setGeneric("repList", function(object)
                   standardGeneric("repList"))
    setMethod("repList", "pkgListing", function(object)
              object@repList)

    if (is.null(getGeneric("repListing")))
        setGeneric("repListing", function(object)
                   standardGeneric("repListing"))
    setMethod("repListing", "pkgListing", function(object)
              repList(object@repList))

    if (is.null(getGeneric("getRepEntry")))
        setGeneric("getRepEntry", function(object, pos)
                   standardGeneric("getRepEntry"))
    setMethod("getRepEntry", "pkgListing", function(object, pos)
              getRepEntry(object@repList,pos))

    if (is.null(getGeneric("packages")))
        setGeneric("packages", function(object)
                   standardGeneric("packages"))
    setMethod("packages", "pkgListing", function(object)
              names(object@pkgList))

    if (is.null(getGeneric("downloadRepFile")))
        setGeneric("downloadRepFile", function(object, pos, file, vers, type,
                                               dest=NULL)
                   standardGeneric("downloadRepFile"))
    setMethod("downloadRepFile", "pkgListing", function(object, pos,
                                                        file, vers,
                                                        type,
                                                        dest=NULL) {
        z <- downloadRepFile(object@repList, pos, file, vers, type,
                             dest)
        z
    })



    if (is.null(getGeneric("pkgVersionList")))
        setGeneric("pkgVersionList", function(object, pkg, names)
                   standardGeneric("pkgVersionList"))
    setMethod("pkgVersionList", "pkgListing", function(object, pkg,
                                                       names) {
        if (missing(names))
            names <- "index"

        pList <- object@pkgList[[pkg]]
        if (length(pList) == 0)
            return(pList)

        curNames <-  names(pList)
        names(pList) <- switch(names,
                              "names"=repNames(repList(object),as.numeric(curNames)),
                              "urls"=repURLs(repList(object),as.numeric(curNames)),
                              "index"=curNames,
                               stop(paste(names,
                                          "is not valid for option names:",
                                          "please choose between\n 'names',",
                                          "'urls', and 'index'."))
                               )
        pList
    })


    setMethod("summary","pkgListing",function(object, ...){
        cat("A package listing with the following packages:\n")
        print(packages(object))
        cat("\nFrom:\n")
        show(repList(object))
    })

    setMethod("show","pkgListing",function(object) {
        pL <- pkgList(object)
        if (length(pL) == 0)
            return(NULL)
        out <- matrix(ncol=3)
        pkgs <- names(pL)
        for (i in 1:length(pL)) {
            if (length(pL[[i]]) > 0) {
                whichREs <- as.numeric(names(pL[[i]]))
                outURLs <- repURLs(repList(object), whichREs)
                for (j in 1:length(pL[[i]])) {
                    if (length(pL[[i]][[j]]) > 0) {
                        for (k in 1:length(pL[[i]][[j]])) {
                            out <- rbind(out,c(pkgs[i],
                                               stringRep(pL[[i]][[j]][[k]]),
                                               outURLs[j]))
                        }
                    }
                }
            }
        }
        print.matrix(out[2:nrow(out),,drop=FALSE],quote=FALSE,
                     collab=c("Package   ", "Version   ","Repository URL"),
                     rowlab=paste(as.character(1:nrow(out)),":",sep=""))
    })

buildPkgListing <- function(repList, pkgs=NULL, type=NULL) {
    if (!(inherits(repList,"ReposList")) && (is.list(repList)))
        repList <- buildReposList(repList)

    if (!inherits(repList,"ReposList"))
        return(NULL)

    ## pkgs==NULL implies "all packages"
    if (numReps(repList)<1)
        return(NULL)

    if (is.null(pkgs)) {
        pkgs <- repPkgs(repList)
        if (is.null(pkgs))
            return(NULL)
    }

    ## We can have either a vector of pkg names, a list of pkg names,
    ## or a list of pkgInfos.  In any case, make sure that we have a
    ## vector of pkg names
    if (is.list(pkgs)) {
        pkgs <- lapply(pkgs, function(x) {
            if (is(x,"pkgInfo"))
                return(pkgName(x))
            else if (is.character(x))
                return(x)
            else
                stop("Malforemd 'pkgs' parameter")
        })
    }

    ## Create skeleton for pkgList
    pkgList <- vector(mode="list",length=length(pkgs))
    names(pkgList) <- pkgs
    for (i in 1:length(pkgList))
        pkgList[[i]] <- list()

    ## Now for each repository, get any appropriate pkg info
    repPIs <- repPkgInfoList(repList, pkgs, type)

    ## !! There has to be a better way here then a double for loop
    for (j in 1:numReps(repList)) {
        if (!is.null(repPIs[[j]])) {
            pkgN <- unlist(lapply(repPIs[[j]],pkgName))
            pkgV <- lapply(repPIs[[j]],pkgVersion)
            for (k in which(pkgs %in% pkgN)) {
                len <- length(pkgList[[k]])
                pkgList[[k]][[len+1]] <- pkgV[pkgN == pkgs[k]]
                names(pkgList[[k]])[len+1] <- as.character(j)
            }
        }
    }

    pkgList <- pkgList[sapply(pkgList,
                              function(x){if (length(x) > 0){
                                  TRUE
                              }
                              else {
                                  FALSE
                              }})]
    return(new("pkgListing",pkgList=pkgList,repList=repList))
}

    setClass("ReposList", representation(repList="list"))

    if (is.null(getGeneric("repList")))
        setGeneric("repList", function(object)
                   standardGeneric("repList"))
    setMethod("repList", "ReposList", function(object)
              object@repList)

    if (is.null(getGeneric("repList<-")))
        setGeneric("repList<-", function(object, value)
                   standardGeneric("repList<-"))
    setMethod("repList<-", "ReposList", function(object, value) {
        curURLs <- repURLs(object)
        if (inherits(value,"ReposList")) {
            newURLs <- repURLs(value)
            goodURLs <- which(!(newURLs %in% curURLs))
            if (length(goodURLs) > 0) {
                newList <- repList(value)
                object@repList <- c(object@repList,newList[goodURLs])
            }
        }
        else if (inherits(value,"ReposEntry")){
            if (!(repURL(value) %in% curURLs))
                object@repList[[length(object@repList)+1]] <- value
        }
        else if (is.list(value)){
            newURLs <- repURLs(value)
            goodURLs <- which(!(newURLs %in% curURLs))
            if (length(goodURLs) > 0) {
                object@repList <- c(object@repList,value[goodURLs])
            }
        }
        object
    })

    if (is.null(getGeneric("numReps")))
        setGeneric("numReps", function(object)
                   standardGeneric("numReps"))
    setMethod("numReps", "ReposList", function(object)
              length(object@repList))

    if (is.null(getGeneric("getRepEntry")))
        setGeneric("getRepEntry", function(object, pos)
                   standardGeneric("getRepEntry"))
    setMethod("getRepEntry", "ReposList", function(object, pos)
              object@repList[[pos]])

    if (is.null(getGeneric("repPkgs")))
        setGeneric("repPkgs", function(object)
                   standardGeneric("repPkgs"))
    setMethod("repPkgs", "ReposList", function(object) {
        unique(unlist(lapply(repList(object), repPkgs)))
    })

    if (is.null(getGeneric("repPkgInfoList")))
        setGeneric("repPkgInfoList", function(object, pkgs, type)
                   standardGeneric("repPkgInfoList"))

    setMethod("repPkgInfoList", "ReposList", function(object,pkgs,type) {
        if (missing(type))
            type <- NULL
        if (missing(pkgs))
            pkgs <- NULL
        x <- lapply(repList(object),repPkgInfos,pkgs,type)
        x
    })

    if (is.null(getGeneric("repNames")))
        setGeneric("repNames", function(object,which)
            standardGeneric("repNames"))
    setMethod("repNames", "ReposList", function(object,which) {
        x <- unlist(lapply(repList(object), repName))
        if (!missing(which))
            x <- x[which]
        x
    })

    if (is.null(getGeneric("repURLs")))
        setGeneric("repURLs", function(object,which)
            standardGeneric("repURLs"))
    setMethod("repURLs", "ReposList", function(object,which) {
        x <- unlist(lapply(repList(object), repURL))
        if (!missing(which))
            x <- x[which]
        x
    })

    if (is.null(getGeneric("downloadRepFile")))
        setGeneric("downloadRepFile", function(object, pos, file, vers, type,
                                               dest=NULL)
                   standardGeneric("downloadRepFile"))

    setMethod("downloadRepFile", "ReposList", function(object, pos, file,
                                                    vers, type, dest=NULL){
        z <- downloadFile(object@repList[[pos]], file, vers, type,
                                                    dest)
        z
    })


    setMethod("show", "ReposList", function(object)
              print(paste("A ReposList with",numReps(object),
                          "repositories")))

buildReposList <- function(reps, recurse=TRUE) {
    repList <- getReposList(reps, recurse=TRUE)
    return(new("ReposList",repList=repList))
}

getReposList <- function(reps, recurse=TRUE) {
    ## Will generate a ReposList object out of a set of reposiotires
    outList <- new("ReposList", repList=list())
    if (inherits(reps,"ReposList"))
        reps <- repList(reps)

    for (rep in reps) {
        if (hasFiles(rep))
            repList(outList) <- rep
        if (recurse == TRUE) {
            subs <- numSubReps(rep)
            if (subs > 0)
                repList(outList) <- getReposList(getSubRepList(rep),
                                                 recurse)
        }
    }
    return(outList)
}


    setClass("ReposEntry",representation(replisting="replisting",
                                         repdatadesc="repdatadesc"))

    if (is.null(getGeneric("replisting")))
        setGeneric("replisting", function(object)
                   standardGeneric("replisting"))

    if (is.null(getGeneric("repdatadesc")))
        setGeneric("repdatadesc", function(object)
                   standardGeneric("repdatadesc"))
    if (is.null(getGeneric("repURL")))
        setGeneric("repURL", function(object,pos)
            standardGeneric("repURL"))

    if (is.null(getGeneric("repType")))
        setGeneric("repType", function(object)
                   standardGeneric("repType"))
    if (is.null(getGeneric("repName")))
        setGeneric("repName", function(object)
                   standardGeneric("repName"))

    ################
    ## Basic slot gets, and info straight from replisting
    ################

    setMethod("repType", "ReposEntry", function(object)
              (object@replisting)@replisting[1,"reptype"])

    setMethod("replisting", "ReposEntry", function(object)
              object@replisting)
    setMethod("repdatadesc", "ReposEntry", function(object)
              object@repdatadesc)

    setMethod("repName", "ReposEntry", function(object)
              object@replisting@replisting[1,"repname"])

    setMethod("repURL", "ReposEntry", function(object,pos) {
        if (missing(pos))
            pos <- 1
        repURL(replisting(object),pos)
        })

    setGeneric("repdataframe", function(object)
         standardGeneric("repdataframe"))

    setMethod("repdataframe", "ReposEntry", function(object)
        repdatadesc(repdatadesc(object)))

    ####################
    ####################

    ############
    ## Subrepository management
    ############
    if(is.null(getGeneric("numSubReps")))
        setGeneric("numSubReps", function(object)
                   standardGeneric("numSubReps"))
    setMethod("numSubReps", "ReposEntry", function(object)
              numSubReps(replisting(object)))

    if(is.null(getGeneric("getSubRep")))
        setGeneric("getSubRep", function(object,pos)
                   standardGeneric("getSubRep"))
    setMethod("getSubRep", "ReposEntry", function(object,pos) {
        if (pos > numSubReps(object))
            return(NULL)
        else
            return(getReposEntry(repURL(object,pos+1)))
        })
    if (is.null(getGeneric("getSubRepList")))
        setGeneric("getSubRepList", function(object)
                   standardGeneric("getSubRepList"))
    setMethod("getSubRepList", "ReposEntry", function(object) {
        num <- numSubReps(object)
        out <- list()
        if (num > 0) {
            for (i in 1:num) {
                out[[length(out)+1]] <-
                    getReposEntry(repURL(object,i+1))
            }
        }
        outObj <- new("ReposList",repList=out)
        return(outObj)
    })
    #####################
    #####################

    #####################
    ## Dataframe information
    ######################
        if (is.null(getGeneric("depends")))
        setGeneric("depends", function(object, pkgInfo)
            standardGeneric("depends"))
    setMethod("depends","ReposEntry",function(object, pkgInfo){
        out <- depends(object@repdatadesc, pkgInfo)
        out
    })

    if (is.null(getGeneric("suggests")))
        setGeneric("suggests", function(object, pkgInfo)
            standardGeneric("suggests"))
    setMethod("suggests","ReposEntry",function(object, pkgInfo){
        out <- suggests(object@repdatadesc, pkgInfo)
        out
    })

    if (is.null(getGeneric("uses")))
        setGeneric("uses", function(object, pkgInfo)
            standardGeneric("uses"))
    setMethod("uses","ReposEntry",function(object, pkgInfo){
         out <- uses(object@repdatadesc, pkgInfo)
        out
    })

    if (is.null(getGeneric("keywords")))
        setGeneric("keywords", function(object, pkgInfo)
            standardGeneric("keywords"))
    setMethod("keywords","ReposEntry",function(object, pkgInfo){
         out <- keywords(object@repdatadesc, pkgInfo)
         out
     })

    ######################
    ## Methods to for file manipulation & file information
    ######################
    if(is.null(getGeneric("hasFiles")))
        setGeneric("hasFiles", function(object)
                   standardGeneric("hasFiles"))
    ## hasFiles will determine if the repository has any files for download
    setMethod("hasFiles","ReposEntry", function(object) {
        if (nrow(repdataframe(object)) < 1)
            return(FALSE)
        else
            return(TRUE)
    })

    if (is.null(getGeneric("pkgRVersion")))
        setGeneric("pkgRVersion", function(object, pkg, vers, type)
                   standardGeneric("pkgRVersion"))
    setMethod("pkgRVersion", "ReposEntry", function(object, pkg, vers,
                                                    type)
          {
              z <- repdataframe(object)
              rowN <- getMangledName(pkg, vers)
              if (rowN %in% rownames(z)) {
                  os <- z[rowN,]$OSspecific[[1]][[type]]$Rvers
                  if ((is.null(os))||(os==""))
                      return(NULL)
                  else
                      return(buildVersionNumber(os))
              }
              else
                  return(NULL)
          })

    if (is.null(getGeneric("repPkgs")))
        setGeneric("repPkgs", function(object)
                   standardGeneric("repPkgs"))
    setMethod("repPkgs", "ReposEntry", function(object) {
        repD <- repdataframe(object)
        repD$Package
    })

    if( is.null(getGeneric("repObjects")))
        setGeneric("repObjects", function(object,pkgs,type,files)
            standardGeneric("repObjects"))
    setMethod("repObjects", "ReposEntry", function(object,pkgs=NULL,
                                                   type,files){
	repD <- repdataframe(object)
        if (nrow(repD) == 0)
            return(NULL)
        if (missing(pkgs))
            pkgs <- NULL
        if (!is.null(pkgs))
            whichPkgs <- which(repD$Package %in% pkgs)
        else
            whichPkgs <- 1:nrow(repD)

        if (length(whichPkgs) > 0) {
            repD <- repD[whichPkgs,,drop=FALSE]

            pkg <- repD$Package
            vers <- unlist(lapply(repD$Version,as.character))
            mat <- t(rbind(pkg,vers))
            if ((!missing(type))&&(!is.null(type))) {
                OSspec <- repD$OSspecific
                types <-
                    unlist(lapply(lapply(OSspec,names),paste,collapse=", "))
                types <- strsplit(types,", ")
                rightType <- NULL
                for (i in 1:length(types)) {
                    if (type %in% types[[i]])
                        rightType <- c(rightType,i)
                }
                mat <- mat[rightType,,drop=FALSE]
                if (nrow(mat) == 0)
                    mat <- NULL
                else {
                    if (missing(files))
                        files <- FALSE
                    if (files == TRUE) {
                        ## Get file names
                        OSspec <- OSspec[rightType]
                        tmpMat <- matrix(nrow=nrow(mat),ncol=1)
                        for (i in 1:nrow(mat)) {
                            tmpMat[i,1] <-
                                basename(OSspec[[i]][type][[1]]$File)
                        }
                        mat <- cbind(mat, tmpMat)
                        colnames(mat)[3] <- "filenames"
                    }
                }
            }
            return(mat)
        }
        else return(NULL)
    })

    if (is.null(getGeneric("repPkgInfos")))
        setGeneric("repPkgInfos", function(object,pkgs,type)
                   standardGeneric("repPkgInfos"))
    setMethod("repPkgInfos", "ReposEntry", function(object,pkgs=NULL,type){
        if (missing(type))
            type <- NULL
        if (missing(pkgs))
            pkgs <- NULL

        ro <- repObjects(object,pkgs=pkgs,type=type)
        if (is.null(ro))
            return(ro)
        out <- list()
        if (nrow(ro) > 0) {
            for (i in 1:nrow(ro)) {
                out[[i]] <- buildPkgInfo(ro[i,"pkg"],ro[i,"vers"])
            }
        }
        return(out)
    })

    if (is.null(getGeneric("downloadFile")))
        setGeneric("downloadFile", function(object, file, vers, type,
                                            dest=NULL,method="auto")
                   standardGeneric("downloadFile"))
    setMethod("downloadFile", "ReposEntry", function(object, file,
                                                     vers, type,
                                                     dest=NULL,
                                                     method="auto") {

        if (missing(file)) {
            warning("Can not continue without a filename to download")
            return(NULL)
        }
        if (missing(type)) {
            warning("Can not continue without a 'type' to download")
            return(NULL)
        }
        if ((missing(vers)) && (type != "vignette")) {
            warning("Can not continue without a version of the file")
            return(NULL)
        }
        errMsg <- paste("There is no",type,"version of package",
                        file,"available at",repURL(object))
        ## Look up pkg/vers in DF
        rd <- repdataframe(object)

        print(paste("Attempting to download",file,"from",repURL(object)))

        if (repType(object) == "package") {
            rowName <- getMangledName(file,vers)
            ## try() doesn't seem to stop it from failing when
            ## indexing the rowname, so check first
            if (! rowName %in% rownames(rd))
                return(NULL)
            row <- rd[rowName,]
            ## get type
            osS <- row$OSspecific[[1]]
            if (is.null(osS))
                return(NULL)
            curOS <- which(names(osS) %in% type)
            if (length(curOS) > 0) {
                filePath <- osS[[curOS]]$File
                if (is.null(filePath)) {
                    warning(errMsg)
                    return(NULL)
                }
            }
            else {
                warning(errMsg)
                return(NULL)
            }
        }
        else if (repType(object) == "vignette") {
            vigRow <- which(rd$VignetteIndexEntry %in% vig)[1]
            filePath <- rd[vigRow,]$PDFpath
            if ((is.na(filePath))||(is.null(filePath))) {
                warning(paste("Vignette",file,"not found at",
                        repURL(object)))
                return(NULL)
            }
        }
        ## download to dest (NULL is temp)
        fileURL <- paste(repURL(object),filePath, sep="/")
        if (is.null(dest))
            dest <- tempdir()

        destFile <- file.path(dest,basename(filePath))
        download.file(fileURL, destFile, mode="wb", quiet=TRUE,
                      method=method)
        print("Download complete.")
        ## return filename (or NULL on error)
        return(destFile)
    })

    #########################
    #########################

    ############
    ## Display Methods
    ############
    setMethod("summary","ReposEntry",function(object, ...){
        cat("Repository Information:\n")
        summary(replisting(object))
        cat("\n")
        summary(repdatadesc(object))
    })
    setMethod("show","ReposEntry",function(object){
        cat("Repository Information:\n")
        show(replisting(object))
        cat("\n")
        show(repdatadesc(object))
    })
    #############

is.ReposEntry <- function(x) {
    if (class(x) == "ReposEntry")
        return(TRUE)
    else
        return(FALSE)
}

buildReposEntry <- function(replisting, repdatadesc) {
    a <- new("ReposEntry", replisting=replisting,
             repdatadesc=repdatadesc)
    if (is.null(a))
        warning("NULL ReposEntry object.  Check URL")
    return(a)
}


### Version Number System

## Defines the VersionNumber class

## Define the class structure of the object
setClass("VersionNumber",representation(stringRep="character"),
         prototype=list(stringRep=as.character(NULL)))

## Define the accessors
if (is.null(getGeneric("stringRep")))
    setGeneric("stringRep", function(object)
               standardGeneric("stringRep"))
if (is.null(getGeneric("major")))
    setGeneric("major", function(object)
               standardGeneric("major"))
if (is.null(getGeneric("minor")))
    setGeneric("minor", function(object)
               standardGeneric("minor"))
    if(is.null(getGeneric("revision")))
    setGeneric("revision", function(object)
               standardGeneric("revision"))

setMethod("stringRep", "VersionNumber", function(object)
          object@stringRep)
setMethod("major","VersionNumber", function(object) {
    major <- getVersionPos(stringRep(object), 1)
    return(major)})
setMethod("minor", "VersionNumber", function(object) {
    minor <- getVersionPos(stringRep(object),2)
    return(minor)})
setMethod("revision", "VersionNumber", function(object) {
    revision <- getVersionPos(stringRep(object),3)
    return(revision)})
setMethod("show","VersionNumber", function(object) {
    print(stringRep(object))
    invisible(object)
})

.initComparisonMethods <- function(where) {
    setMethod("==","VersionNumber", function(e1,e2) {
        versVals <- getVersVals(e1,e2)
        e1 <- versVals[[1]]
        e2 <- versVals[[2]]
        for (i in seq(along=e1)) {
            if (e1[i] != e2[i])
                return(FALSE)
        }
        return(TRUE)
    }, where=where)

    setMethod("!=","VersionNumber", function(e1,e2) {
        versVals <- getVersVals(e1,e2)
        e1 <- versVals[[1]]
        e2 <- versVals[[2]]
        for (i in seq(along=e1)) {
            if (e1[i] != e2[i])
                return(TRUE)
        }
        return(FALSE)
    }, where=where)

    setMethod(">","VersionNumber",function(e1,e2) {
        versVals <- getVersVals(e1,e2)
        e1 <- versVals[[1]]
        e2 <- versVals[[2]]
        for (i in seq(along=e1)) {
            if (e1[i] > e2[i])
                return(TRUE)
            else if (e2[i] > e1[i])
                return(FALSE)
        }
        return(FALSE)
    }, where=where)

    setMethod("<","VersionNumber",function(e1,e2) {
        versVals <- getVersVals(e1,e2)
        e1 <- versVals[[1]]
        e2 <- versVals[[2]]
        for (i in seq(along=e1)) {
            if (e1[i] < e2[i])
                return(TRUE)
            else if (e2[i] < e1[i])
                return(FALSE)
        }
        return(FALSE)
    }, where=where)

    setMethod(">=", "VersionNumber",function(e1,e2) {
        versVals <- getVersVals(e1,e2)
        e1 <- versVals[[1]]
        e2 <- versVals[[2]]
        for (i in seq(along=e1)) {
            if (e1[i] > e2[i])
                return(TRUE)
            if (e1[i] < e2[i])
                return(FALSE)
        }
        return(TRUE)
    }, where=where)

    setMethod("<=","VersionNumber", function(e1,e2) {
        versVals <- getVersVals(e1,e2)
        e1 <- versVals[[1]]
        e2 <- versVals[[2]]
        for (i in seq(along=e1)) {
            if (e1[i] < e2[i])
                return(TRUE)
            if (e1[i] > e2[i])
                return(FALSE)
        }
        return(TRUE)
    }, where=where)
}

setMethod("as.character","VersionNumber",function(x){
    return(stringRep(x))})

buildVersionNumber <- function(vers) {
    if (missing(vers))
        return(new("VersionNumber"))

    ## Check to make sure that this doesn't contain characters
    if (length(grep("[[:alpha:]]",vers)) > 0)
        vers <- "0"

    ## If the first char is a "." or "-", place a 0 in front
    if (length(grep("\\.|-",substr(vers,1,1))) > 0)
        vers <- paste("0",vers,sep="")

    ## Weed out any bad digits
    if (is.null(getDigits(vers))) {
        return(NULL)
    }
    else {
        a <- new("VersionNumber",stringRep=vers)
        if (!is(a,"VersionNumber"))
            a <- buildVersionNumber("0")
        return(a)
    }
}

getVersionPos <- function(version,pos) {
    ## Returns the digit in position pos from the version
    vers <- strsplit(version,"\\.|-")[[1]][pos]
    if ((is.na(vers))||(vers==""))
        return(0)
    else
        return(as.numeric(vers))
}

max.VersionNumber <- function(..., na.rm=FALSE) {
    versNums <- list(...)
    maxVers <- versNums[[1]]
    if (length(versNums) > 1) {
        for (i in 2:length(versNums)) {
            if (versNums[[i]] > maxVers)
                maxVers <- versNums[[i]]
        }
    }
    return(maxVers)
}

min.VersionNumber <- function(..., na.rm=FALSE) {
    versNums <- list(...)
    minVers <- versNums[[1]]
    if (length(versNums) > 1) {
        for (i in 2:length(versNums)) {
            if (versNums[[i]] < minVers)
                minVers <- versNums[[i]]
        }
    }
    return(minVers)
}


getMaxVersion <- function(x) {
    ## !!! I can't seem to find a way to get a list of VersionNumbers
    ## to be fed into max.VersionNumber, so for now ....

    maxVers <- x[[1]]
    len <- length(x)
    if (len > 1) {
        for (i in 2:len) {
            if (x[[i]] > maxVers)
                maxVers <- x[[i]]
        }
    }
    return(maxVers)
}

getMaxElement <- function(x) {
    ## Corrollary to getMaxVersion except it returns the position in
    ## the list that contains the max version
    maxPos <- 1
    len <- length(x)

    if (len > 1) {
        for (i in 2:len) {
            if (x[[i]] > x[[maxPos]])
                maxPos <- i
        }
    }
    return(maxPos)
}

getMinElement <- function(x) {
    minPos <- 1
    len <- length(x)

    if (len > 1) {
        for (i in 2:len) {
            if (x[[i]] < x[[minPos]])
                minPos <- i
        }
    }
    return(minPos)
}

getDigits <- function(vers) {
    ## Removes any . and - characters
    ## returns NULL nothing left
    xVals <- strsplit(vers,"\\.|-")[[1]]
    if (any(xVals=="")) {
        return(NULL)
    }

    xVals <- as.numeric(xVals)
    return(xVals)
}

getVersVals <- function(x,y) {
    ## Pads the shorter version with 0s, and removes any punctuation
    ## returns a list with both manipulated version numbers
    xVals <- getDigits(stringRep(x))
    yVals <- getDigits(stringRep(y))
    valDiff <- length(xVals) - length(yVals)
    if (valDiff < 0)
        xVals <- c(xVals,rep(0,abs(valDiff)))
    else if (valDiff > 0)
        yVals <- c(yVals,rep(0,valDiff))

    return(list(xVals,yVals))
}

checkVers <- function(x,op,y) {
    ## Passed two VersionNumbers x & y
    ## and a character string 'operator',
    ## will check to see if 'x op y' is TRUE or FALSE
    ## where op is coerced into the real operator.
    ret <- switch(op,
                  "==" = x==y,
                  ">" = x>y,
                  "<" = x<y,
                  ">=" = x >= y,
                  "<=" = x <= y,
                  "!=" = x != y
                  )
    return(ret)
}

getRversion <- function() {
    return(buildVersionNumber(strsplit(R.version.string,"( )|(,)")[[1]][3]))
}
winConvertSourceRepos <- function(repEntry, pkgs, destDir, recurse=TRUE,
                                  searchOptions=TRUE,
                                  errorLog="error.log") {
    if (.Platform$OS.type != "windows")
        stop("This function is for Windows systems only")
    if (missing(destDir))
        stop("no destination directory specified")
    if (missing(repEntry) && missing(pkgs))
        stop("Need to provide either/both of repEntry or pkgs")
    if(!missing(repEntry) && (repType(repEntry) != "package"))
        stop("winConvertSourceRepos requires a package repository")

    if (missing(pkgs) || is.null(pkgs) )
	pkgs <- repPkgInfos(repEntry)

    ret <- try(setwd(destDir))
    if (inherits(ret,"try-error"))
        stop(paste(destDir,"does not seem to be a valid directory"))

    cDir <- getwd()
    on.exit(setwd(cDir),add=TRUE)
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)


    ## Error log
    errorLog <- file.path(cDir, errorLog)

    ## Download package files to tmp directory
    ret <- lapply(pkgs, function(x) {downloadFile(repEntry, pkgName(x),
                                                  stringRep(pkgVersion(x)),
                                                  "Source", dest=tmpDir)})
    if (length(ret) != length(pkgs))
        stop("Error downloading requested files")
    gc()
    setwd(tmpDir)
    pkgs <- dir(tmpDir)
    repObjs <- repObjects(repEntry, type="Source", files=TRUE)
    repFiles <- repObjs[,"filenames"]
    ## Filter out any packages that weren't downloaded for some reason
    repObjs <- repObjs[match(pkgs,repFiles),]
    if (length(repObjs) == 0) {
        return(NULL)
    }
    for (i in 1:nrow(repObjs)) {
        pkg <- repObjs[i,"pkg"]
        vers <- repObjs[i,"vers"]
        file <- repObjs[i,"filenames"]

        system(paste("Rcmd install --build",file))

        outFile <- paste(pkg,"_", vers, ".zip",sep="")
        file.copy(outFile,file.path(cDir,outFile),overwrite=TRUE)
    }
}



.buildReposToolsOpts <- function(OST) {
    if (is.null(getOption("BioC"))) {
        BioC <- list()
        class(BioC) <- "BioCOptions"
        options("BioC"=BioC)
    }
    ReposEntry <- list()
    class(ReposEntry) <- "BioCPkg"

    ## Set the default download type
    if (OST == "windows")
        ReposEntry$type <- "Win32"
    else
        ReposEntry$type <- "Source"

    BioC <- getOption("BioC")
    BioC$reposEntry <- ReposEntry
    options("BioC"=BioC)

    ## Set up the repositories2 option
    if (!("CRAN" %in% names(getOption("repositories2")))) {
        bioCOpt <- "http://www.bioconductor.org/CRANrepository"
        names(bioCOpt) <- "CRAN"
        options("repositories2"=c(getOption("repositories2"),bioCOpt))
    }

    if (!("BIOCRel1.3" %in% names(getOption("repositories2")))) {
        bioROpt <- "http://www.bioconductor.org/repository/release1.3/package"
        names(bioROpt) <- "BIOCRel1.3"
        options("repositories2"=c(getOption("repositories2"),bioROpt))
    }

    if (!("BIOCDevel" %in% names(getOption("repositories2")))) {
        bioDOpt <- "http://www.bioconductor.org/repository/devel/package"
        names(bioDOpt) <- "BIOCDevel"
        options("repositories2"=c(getOption("repositories2"),bioDOpt))
    }

    if (!("BIOCData" %in% names(getOption("repositories2")))) {
        bioDaDopt <- "http://www.bioconductor.org/data/metaData"
        names(bioDaDopt) <- "BIOCData"
        options("repositories2"=c(getOption("repositories2"),
        bioDaDopt))
    }
    if (!("BIOCCourses" %in% names(getOption("repositories2")))) {
        bioCCourse <- "http://www.bioconductor.org/repository/Courses"
        names(bioCCourse) <- "BIOCCourses"
        options("repositories2"=c(getOption("repositories2"),
                bioCCourse))
    }
    if (!("BIOCcdf" %in% names(getOption("repositories2")))) {
        bioCcdf <- "http://www.bioconductor.org/data/cdfenvs/repos"
        names(bioCcdf) <- "BIOCcdf"
        options("repositories2"=c(getOption("repositories2"),
                bioCcdf))
    }
    if (!("BIOCprobes" %in% names(getOption("repositories2")))) {
        bioCProbe <- "http://www.bioconductor.org/data/probes/Packages"
        names(bioCProbe) <- "BIOCprobes"
        options("repositories2"=c(getOption("repositories2"),
                bioCProbe))
    }
    if (!("BIOCOmegahat" %in% names(getOption("repositories2")))) {
        bioCOmegahat <-
            "http://www.bioconductor.org/repository/Omegahat"
        names(bioCOmegahat) <- "BIOCOmegahat"
        options("repositories2"=c(getOption("repositories2"),
                bioCOmegahat))
    }
}

.First.lib <- function(libname, pkgname, where) {
    ## Bump up the internet.info - the current model for internet
    ## connections is extremely chatty.  Ugh.
    options("internet.info"=3)

     if (missing(where)) {
        where <- match(paste("package:", pkgname, sep=""), search())
        if(is.na(where)) {
            warning(paste("Not a package name: ",pkgname))
            return()
        }
        where <- pos.to.env(where)
    }

    .initComparisonMethods(where)
    OST <- .Platform$OS.type
    .buildReposToolsOpts(OST)
    if( interactive() )
        syncLocalLibList()
}

