.packageName <- "affy"
# A function that takes user inputs for correction methods for
# expresso (affy). Default values can be missing, in which case the
# first element will be chosen as the default.

expressoWidget <- function(BGMethods, normMethods, PMMethods, expMethods,
                           BGDefault, normDefault, PMDefault, expDefault){
    methodList <- list()
    END <- FALSE

    if(any(missing(BGMethods), missing(normMethods),
           missing(PMMethods), missing(expMethods))){
        stop("At least one of the method arguments is missing")
    }
    if(any(c(length(BGMethods), length(normMethods),
             length(PMMethods), length(expMethods)) == 0)){
        stop("At least one of the method argument is of length 1")
    }

    if(missing(BGDefault)){
        BGM <- tclVar(BGMethods[1])
    }else{
        BGM <- tclVar(BGDefault)
    }
    if(missing(normDefault)){
        NMM <- tclVar(normMethods[1])
    }else{
        NMM <- tclVar(normDefault)
    }
    if(missing(PMDefault)){
        PMM <- tclVar(PMMethods[1])
    }else{
        PMM <- tclVar(PMDefault)
    }
    if(missing(expDefault)){
        EXM <- tclVar(expMethods[1])
    }else{
        EXM <- tclVar(expDefault)
    }


    quit <- function(){
        tkdestroy(base)
    }
    end <- function(){
        END <<- TRUE
        methodList[["BG"]] <<- tclvalue(BGM)
        methodList[["NORM"]] <<- tclvalue(NMM)
        methodList[["PM"]] <<- tclvalue(PMM)
        methodList[["EXP"]] <<- tclvalue(EXM)
        quit()
    }

    base <- tktoplevel()
    ## post -- hook
    on.exit(tkdestroy(base))

    tktitle(base) <- "Expresso methods selection"
    ## Description text
    tkpack(tklabel(base, text = "Welcome to Expresso methods selection"),
           expand = FALSE, fill = "x", padx = 5, pady = 5)
    tkpack(tklabel(base, text = paste("You need to choose correction",
                         "methods or go with the defaults")),
           expand = FALSE, fill = "x", padx = 5)

    ## Selections for correction methods
    methodFrame <- tkframe(base)
    ## Background selection
    BGLabel <- tklabel(methodFrame, text = "Background correction")
    BGDropdown <- tkframe(methodFrame)
    dropdownList(BGDropdown, BGMethods, BGM, 20,
                                      tclvalue(BGM), TRUE)
    tkgrid(BGLabel, BGDropdown)
    tkgrid.configure(BGLabel, sticky = "e")
    tkgrid.configure(BGDropdown, sticky = "w")

    ## Normlization
    NMLabel <- tklabel(methodFrame, text = "Normalization")
    NMDropdown <- tkframe(methodFrame)
    dropdownList(NMDropdown,normMethods, NMM, 20,
                                      tclvalue(NMM), TRUE)
    tkgrid(NMLabel, NMDropdown)
    tkgrid.configure(NMLabel, sticky = "e")
    tkgrid.configure(NMDropdown, sticky = "w")

    ## PM correction
    PMLabel <- tklabel(methodFrame, text = "PM correction")
    PMDropdown <- tkframe(methodFrame)
    dropdownList(PMDropdown, PMMethods, PMM, 20,
                                      tclvalue(PMM), TRUE)
    tkgrid(PMLabel, PMDropdown)
    tkgrid.configure(PMLabel, sticky = "e")
    tkgrid.configure(PMDropdown, sticky = "w")

    ## PM correction
    EXLabel <- tklabel(methodFrame, text = "Expression")
    EXDropdown <- tkframe(methodFrame)
    dropdownList(EXDropdown, expMethods, EXM, 20,
                                      tclvalue(EXM), TRUE)
    tkgrid(EXLabel, EXDropdown)
    tkgrid.configure(EXLabel, sticky = "e")
    tkgrid.configure(EXDropdown, sticky = "w")

    tkpack(methodFrame, expand = TRUE, fill = "both", padx = 5,
           pady = 10)

    butFrame <- tkframe(base)
    quitBut <- tkbutton(butFrame, text = "Quit", width = 7, command = quit)
    endBut <- tkbutton(butFrame, text = "Select", width = 7, command = end)
    tkgrid(quitBut, endBut, padx = 5)
    tkpack(butFrame, expand = FALSE, fill = "x", pady = 5)

    tkwait.window(base)

    if(END){
        return(methodList)
    }else{
        return(NULL)
    }
}
getCdfInfo <- function(object,  how=getOption("BioC")$affy$probesloc, verbose=FALSE) {
    ## cdfname is the cdf environment
    ## methods is a vector detailing how to get the file - one of
    ## 'environment', 'data', 'library', 'bioC'
    ## where is used for specific information corresponding to method

    if (length(how) == 0)
        stop("No available method to obtain CDF file")

    cdfname <- cdfName(object)

    badOut <- list()
    for (i in 1:length(how)) {
        cur <- how[[i]]
        out <- switch(cur$what,
                      "environment" = cdfFromEnvironment(cdfname,
                      cur$where, verbose),
                      "data" = cdfFromData(cdfname, cur$where, verbose),
                      "libPath" = cdfFromLibPath(cdfname, cur$where,
                      verbose=verbose),
                      "bioC" = cdfFromBioC(cdfname, cur$where,
                      verbose)
                      )

        if (is.environment(out))
            return(out)
        else
            badOut <- c(badOut, out)
    }

    stop(paste("Could not obtain CDF environment, problems encountered:",
               paste(unlist(badOut),collapse="\n"),sep="\n"))
}

cdfFromData <- function(cdfname, pkg, verbose=TRUE) {
    if (verbose)
        print(paste("Attempting to locate",cdfname,
                    "in the data directory of package", pkg))

    if(cdfname %in% do.call("data", list(package=pkg))$results[, 3]) {
        where.env <- pos.to.env(match(paste("package:", pkg, sep = ""), search()))

        ## check if the cdfenv is already loaded. If not load it *in* the environment
        ## of the package (where.env)
        if(!exists(cdfname, where = where.env, inherits = FALSE)) {
            path <- .path.package(pkg)
            filename <- paste(cdfname, ".rda", sep="")
            load(file.path(path, "data", filename) ,
                 envir = where.env)
        }
        cdfenv <- get(cdfname, envir=where.env)
        return(cdfenv)
    }
    else
        return(list(paste("Data for package",pkg,"did not contain",cdfname)))
}

cdfFromEnvironment <- function(cdfname, where, verbose=TRUE) {
    if (verbose)
        print(paste("Attempting to locate",cdfname,"in specified environment"))

    if (exists(cdfname, inherits=FALSE, where=where))
        return(as.environment(get(cdfname,inherits=FALSE,envir=where)))
    else {
        if (verbose)
            print(paste("Specified environment does not contain",cdfname))
        return(list(paste("Specified environment specified did not contain",cdfname)))
    }
}

cdfFromBioC <- function(cdfname, lib=.libPaths()[1], verbose=TRUE) {
  cdfname <- cleancdfname(cdfname)
  require(reposTools) || stop("Package 'reposTools' is required",
                                " for this operation.")

    if (verbose)
        print(paste("Attempting to obtain",cdfname,"from Bioconductor website"))

    ## First search the user's libPaths to see if it is installed
    if (verbose)
        print(paste("Checking to see if the environment",
                    cdfname,"is already installed ..."))
    if (is.installed(cdfname)) {
        if (verbose)
            print(paste("The environment",cdfname,"is already installed."))
    }
    else {
        if (verbose)
            print(paste("The environment ",cdfname," was not found in",
                        " these directories: ",
                        paste(.libPaths(), collapse=", "),
                        ".  Now searching the internet repository.",
                        sep=""))
        if (verbose)
            print(paste("Checking to see if your internet",
                        "connection works ..."))
        if (testBioCConnection()) {
            ## Check for file permissions
            if (file.access(lib, mode=0) < 0) {
                if (verbose) {
                    print(paste("Directory",lib,"does not seem to exist.\n",
                                "Please check your 'lib' parameter and try again"))
                    return(list("Bioconductor - lib does not exist"))
                }
            }

            if (file.access(lib,mode=2) < 0) {
                if (verbose) {
                    print(paste("You do not have write access to",lib,
                               "\nPlease check your permissions or provide",
                               "a different 'lib' parameter"))
                    return(list("Bioconductor - lib is not writeable"))
                }
            }

            z <- install.packages2(cdfname, lib=lib)
            if(! cdfname %in% updatedPkgs(z)) {
                if (verbose)
                    print(paste("Environment",cdfname,
                                "was not found in the Bioconductor",
                                "repository."))
                return(list(paste("Bioconductor -",cdfname,"not available")))
            }
            else
                if (verbose)
                    print(paste("Installation of environment",
                                cdfname, "was succesful."))
        }
        else {
            if (verbose)
                print(paste("The current operation could not access",
                            "the Bioconductor repository.  Please",
                            "check your internet connection, and",
                            "report further problems to",
                            "bioconductor@stat.math.ethz.ch"))
            return(list("Bioconductor - could not connect"))
        }
    }

    ## Now load the library and return the environment
    do.call("library", list(cdfname, lib.loc=lib))
    ## !!! Double check that library is actually loaded
    if (! cdfname %in% .packages()) {
        ## package was not properly loaded
        if (verbose)
            print(paste("The package", cdfname,
                        "could not be loaded"))
        return(list("Bioconductor - package downloaded but not loadable"))
    }
    else
        return(get(cdfname,
                   envir=as.environment(paste("package:", cdfname, sep=""))))
}

cdfFromLibPath <- function(cdfname, lib = NULL, verbose=TRUE) {
  cdfname <- cleancdfname(cdfname)
  ## First check to see if package is installed
  if (verbose)
    print(paste("Checking to see if package",cdfname,
                "is already installed"))
  
  if (length(.find.package(cdfname, lib.loc=lib, quiet=TRUE)) == 0)
    return(list(paste("Library - package",cdfname,"not installed")))
  
  ## See if package is already loaded
  if (cdfname %in% .packages()) {
    if (verbose)
      print(paste("The package", cdfname, "is already loaded"))
  }
  else {
    if (verbose)
      print(paste("Attempting to load package", cdfname))
    ## Attempt to load the library requested
    do.call("library", list(cdfname, lib.loc=lib))
    
    ## Check to see if it got loaded
    if (! cdfname %in% .packages()) {
      ## package didn't get loaded
      if (verbose)
        print(paste("The package", cdfname, "could not be loaded"))
      return(list(paste("Library - package",cdfname,"is not loadable")))
    }
  }
  
    return(get(cdfname, envir=as.environment(paste("package:", cdfname, sep=""))))
}
"AffyRNAdeg" <-
function (abatch,log.it=TRUE) 
{
    {
        data <- pm(abatch, LIST = TRUE)
        if(log.it==TRUE) data <- lapply(data,log2)
        names <- colnames(exprs(abatch))
        probe.set.size <- function(x) {
            size <- dim(x)[1]
            return(size)
        }
        max.num <- sapply(data, probe.set.size)
        tab <- (table(max.num))
        ord <- order(-as.numeric(tab))
        K <- as.numeric(names(tab))[ord[1]]
        data <- data[max.num == K]
    }
    get.row <- function(x, i = 1) {
        return(x[i, ])
    }
    get.col <- function(x, i = 1) {
        return(x[, i])
    }
    rowstack <- function(x, i = 1) {
        return(t(sapply(x, get.row, i)))
    }
    colstack <- function(x, i = 1) {
        return(t(sapply(x, get.col, i)))
    }
    N <- length(data)
    
    n <- dim(data[[1]])[2]
    mns <- matrix(nrow = n, ncol = K)
    sds <- mns
    for (i in 1:K) {
        data.stack <- rowstack(data, i)
        if(dim(data[[1]])[2]==1) data.stack <- t(data.stack)
        mns[, i] <- apply(data.stack, 2, mean)
        sds[, i] <- apply(data.stack, 2, sd)
    }
    mns.orig <- mns
    mn <- mns[, 1]
    mns <- sweep(mns, 1, mn)
    mns <- mns/(sds/sqrt(N))
    lm.stats <- function(x) {
        index <- 0:(length(x) - 1)
        ans <- summary(lm(x ~ index))$coefficients[2, c(1, 4)]
        return(ans)
    }
    stats <- apply(mns, 1, lm.stats)
    answer <- list(N, names, mns.orig, sds/sqrt(N), stats[1, 
        ], stats[2, ])
    names(answer) <- c("N", "sample.names", "means.by.number", 
        "ses", "slope", "pvalue")
    return(answer)
}
"summaryAffyRNAdeg" <-
function (rna.deg.obj, signif.digits = 3) 
{
    temp.table <- rbind(signif(rna.deg.obj$slope, signif.digits), 
        signif(rna.deg.obj$pvalue, signif.digits))
    colnames(temp.table) <- rna.deg.obj$sample.names
    rownames(temp.table) <- c("slope", "pvalue")
    write.table(temp.table, file = "", quote = FALSE)
}
"plotAffyRNAdeg" <-
function (rna.deg.obj,transform="shift.scale",cols=NULL, ...) 
{
    if(!is.element(transform,c("shift.scale","shift.only","neither"))) stop("Tranform must be 'shift.scale','shift.only', or 'neither'")
    mns <- rna.deg.obj$means.by.number
    if(is.null(cols)) cols=rep(4,dim(mns)[1])
    ylab="Mean Intensity"
    if(transform=="shift.scale"){
    sds <- rna.deg.obj$ses
    mn <- mns[, 1]
    mns <- sweep(mns, 1, mn)
    mns <- mns/(sds)
    mns <- sweep(mns, 1, 1:(dim(mns)[1]), "+")
    ylab <- paste(ylab,": shifted and scaled")
   }else if(transform=="shift.only"){
    mn <- mns[, 1]
    mns <- sweep(mns, 1, mn)
    mns <- sweep(mns, 1, 1:(dim(mns)[1]), "+")
    ylab <- paste(ylab,": shifted")
   }
    plot(-2, -1, pch = "", xlim = range(-1, (dim(mns)[2])), 
        ylim = range(min(as.vector(mns)) - 1, max(as.vector(mns)) + 1), xlab = "5' <-----> 3'\n Probe Number ", 
        ylab = ylab, axes = FALSE, main = "RNA digestion plot", 
        ...)
    axis(1)
    axis(2)
    for (i in 1:dim(mns)[1]) lines(0:((dim(mns)[2]-1)), mns[i, ],col=cols[i])
}




avdiff <- function(x,verbose=FALSE){
  if(missing(x)) stop("Argument x missing, with no default\n")
  cat("Computing average difference for",dim(x$pm)[2],"columns")
  avdiff <- apply(x$pm-x$mm,2,function(y){
    cat(".")
    tapply(y,x$name,function(z){
      o <- order(z)
      zz <- z[-c(o[1],o[length(z)])] #take out biggest and smallest
      mean(z[abs(z-mean(zz))<3*sd(zz)])
    })
  })
  colnames(avdiff) <- x$chip.names
  return(avdiff)
}
barplot.ProbeSet <- function(height,
                             xlab="Probe pair",ylab="Intensity",
                             main=NA,
                             col.pm="red", col.mm="blue",
                             beside=TRUE, names.arg="pp",
                             ask = TRUE, scale = TRUE,
                             ...)
{

  opar <- par()$ask
  par(ask=ask)
  on.exit(par(ask=opar))

  if (names.arg == "pp") {
    names.arg <- seq(1, nrow(pm(height)))
  }

  col <- c(col.pm, col.mm)

  if (scale) {
    ylim <- range(c(pm(height), mm(height)), na.rm=TRUE)
  } else {
    ylim <- NULL
  }

  if (is.na(main)) {
    main <- paste(height@id, "( sample", 1:ncol(pm(height)), ")")
  } else {
    main <- rep(main, length=ncol(pm(height)))
  }
  
  for (i in 1:ncol(pm(height))) {
    
    hh <- rbind(pm(height)[, i], mm(height)[, i])
      
    barplot(hh, xlab=xlab, ylab=ylab,
            main=main[i],
            col=col,
            beside=beside,
            names.arg=names.arg,
            ylim = ylim,
            ...)
  }
}
bg.correct.mas <- function(object, griddim=16)
{
   nchips <- length(object)

   pm.index <- unique(unlist(indexProbes(object, "pm")))
   mm.index <- unique(unlist(indexProbes(object, "mm")))
   
   rows <- nrow(object)
   cols <- ncol(object)
   
   allintensities <- intensity(object)[c(pm.index, mm.index), ]

   # note that the indexing is +1 more than you'd expect because
   # the c code expects it that way
   ## (note about the remark above: R indexing starts at 1 and not at 0,
   ## that's why the indexing is done this way. The package is primarily done to
   ## be used with R...)
   
   allx <- c(pm.index, mm.index) %% nrow(object) 
   allx[allx == 0] <- rows
   ally <- c(pm.index, mm.index) %/% nrow(object) + 1
   
   nprobes <- length(allx)
   
   corrected <- matrix(.C("affy_background_adjust_R",
                          as.double(as.vector(allintensities)), as.integer(allx), as.integer(ally),
                          as.integer(nprobes), as.integer(nchips), as.integer(rows), as.integer(cols),
                          as.integer(griddim))[[1]], nprobes, nchips)
   
   intensity(object)[c(pm.index, mm.index), ] <- corrected
   ## and what with the 'non pm or mm' probes ?
   ## answer: they are not used per Affymetrix Statistical Algorithms Description Document.
   
   return(object)
   
 }
####These functions take an AffyBatch object "background correct"
####the pms and return an AffyBatch with the background corrected PMs
###
bg.parameters <- function(pm, n.pts=2^14){
  
  max.density <- function(x, n.pts){
    aux <- density(x, kernel="epanechnikov", n=n.pts, na.rm=TRUE)
    aux$x[order(-aux$y)[1]] 
  }
  
  pmbg <- max.density(pm,n.pts) ##Log helps detect mode
  bg.data <- pm[pm < pmbg]
  ##do it again to really get the mode
  pmbg <- max.density(bg.data,n.pts) 
  bg.data <- pm[pm < pmbg]
  bg.data <- bg.data - pmbg
  
  bgsd <- sqrt(sum(bg.data^2)/(length(bg.data)-1))*sqrt(2)#/.85
  
  sig.data <- pm[pm > pmbg]
  sig.data <- sig.data-pmbg
  
  expmean <- max.density(sig.data,n.pts)
  alpha <- 1/expmean
  mubg <- pmbg
  list(alpha=alpha,mu=mubg,sigma=bgsd)  
}

bg.adjust <- function(pm, n.pts=2^14, ...){
  param <- bg.parameters(pm,n.pts)
  b <- param$sigma
  pm <- pm - param$mu - param$alpha*b^2
  pm + b*((1./sqrt(2*pi))*exp((-1./2.)*((pm/b)^2)))/pnorm(pm/b)
}

bg.correct.none <- function(object, ...)
  object

##bg.correct.subtractmm <- function(object){
##  pm(object) <- pm(object) - mm(object)
##  return(object)
##}

bg.correct.rma <- function(object, ...){
  pm(object) <- apply(pm(object),2,bg.adjust)
  return(object)
}

##
## this function calls the c code as an alternative to the R code above.
## it should help end the disagreement between rma() and expresso()
##

bg.correct.rma2 <- function(object,bgtype=1){
  
        bg.dens <- function(x){density(x,kernel="epanechnikov",n=2^14)}
        pm(object) <- .Call("bg_correct_c_copy",pm(object),mm(object),body(bg.dens),new.env(),bgtype,PACKAGE="affy")
        return(object)
}

## --- pmcorrect things

pmcorrect.subtractmm <- function(object){
  pm.corrected <- pm(object) - mm(object)
  return(pm.corrected)
}

pmcorrect.pmonly <- function(object) {
  return(pm(object))
}







expresso <- function(afbatch,
                     ## --
                     bg.correct=TRUE,
                     bgcorrect.method = NULL,
                     bgcorrect.param = list(),
                     ## --
                     normalize = TRUE,
                     normalize.method = NULL,
                     normalize.param=list(),
                     ## --
                     pmcorrect.method = NULL,
                     pmcorrect.param = list(),
                     ## --
                     summary.method = NULL,
                     summary.param = list(),
                     summary.subset = NULL,
                     ## ---
                     verbose = TRUE,
                     widget = FALSE
                     ) {
    # JZ added this function
    setCorrections <- function(){
        bioc.opt <- getOption("BioC")
        if(bg.correct){
            if(is.null(bgcorrect.method)){
                BGMethods <- bgcorrect.methods
            }else{
                BGMethods <- bgcorrect.method
            }
        }else{
            BGMethods <- "None"
        }
        if(normalize){
            if(is.null(normalize.method)){
                normMethods <- normalize.methods(afbatch)
            }else{
                normMethods <- normalizer.method
            }
        }else{
            normMethods <- "None"
        }
        # Default for this one may not be correct
        if(is.null(pmcorrect.method)){
            PMMethods <- pmcorrect.methods
        }else{
            PMMethods <- pmcorrect.method
        }
        # Default for this one may not be correct
        if(is.null(summary.method)){
            expMethods <- generateExprSet.methods

        }else{
            expMethods <- summary.method
        }

        corrections <- expressoWidget(BGMethods, normMethods, PMMethods,
                           expMethods, bioc.opt$affy$bgcorrect.method,
                                      bioc.opt$affy$normalize.method,
                                      bioc.opt$affy$pmcorrect.method,
                                      bioc.opt$affy$summary.method)
        if(!is.null(corrections)){
            if(corrections[["BG"]] != "None"){
                bgcorrect.method <<- corrections[["BG"]]
            }
            if(corrections[["NORM"]] != "None"){
                normalize.method <<- corrections[["NORM"]]
            }
            if(corrections[["PM"]] != "None"){
                pmcorrect.method <<- corrections[["PM"]]
            }
            if(corrections[["EXP"]] != "None"){
                summary.method <<- corrections[["EXP"]]
            }
        }else{
            stop("Aborted by user")
        }
    }

  if (widget) {
    require(tkWidgets) || stop("library tkWidgets could not be found !")
  }

  nchips <- length(afbatch)

  ###background stuff must be added before normalization!

  if(widget){
      setCorrections()
  }

  ## -- summary of what will be done
  if (verbose) {
    if (bg.correct){
      cat("background correction:", bgcorrect.method, "\n")
    }
    if (normalize) {
      cat("normalization:", normalize.method, "\n")
    }
    cat("PM/MM correction :", pmcorrect.method, "\n")
    cat("expression values:", summary.method, "\n")
  }


  ## -- background correct (if needed)
  if (bg.correct) {

    if (verbose)
      cat("background correcting...")

    afbatch <- do.call("bg.correct", c(alist(afbatch, method=bgcorrect.method), bgcorrect.param))

    if (verbose)
      cat("done.\n")
  }

  ## -- normalize (if wished)
  if (normalize) {

    if (verbose)
      cat("normalizing...")

    afbatch <- do.call("normalize",
                       c(alist(afbatch, normalize.method), normalize.param))

    if (verbose)
      cat("done.\n")
  }

  eset <- computeExprSet(afbatch,
                         summary.method=summary.method, pmcorrect.method= pmcorrect.method,
                         ids=summary.subset,
                         summary.param=summary.param, pmcorrect.param=pmcorrect.param)

  return(eset)
}



fit.li.wong <- function(data.matrix, remove.outliers=TRUE,
                        normal.array.quantile=0.5,
                        normal.resid.quantile=0.9,
                        large.threshold=3,
                        large.variation=0.8,
                        outlier.fraction=0.14,
                        delta = 1e-06,maxit=50,outer.maxit=50, verbose=FALSE, ...){
  if(missing(data.matrix)) stop("Argument data.matrix missing, with no default")
  II <- dim(data.matrix)[1] ##II instrad of I cause I is a fuction in R
  J <- dim(data.matrix)[2]
  if(J==1){
    warning("Li and Wong's algorithm is not suitable when only one probe pair")
    return(list(theta = as.vector(data.matrix), phi = 1, sigma.eps = NA, sigma.theta = NA, sigma.phi=NA, theta.outliers=NA, phi.outliers=NA, single.outliers=NA,convergence1=NA,convergence2=NA,iter = NA, delta = NA)) 
  }
  cI <- II ##current I
  cJ <- J ##current J
  theta.outliers.old <- rep(FALSE, II) ##ith entry will be true if theta_i is an outlier
  phi.outliers.old <- rep(FALSE, J) ##jth entry will be true if phi_j is an outlier
  single.outliers.old <- matrix(FALSE, II, J) ##ij entry will be true if y_is an outlier
  theta.outliers <- theta.outliers.old ##need this to now if change ocurred in outliers
  phi.outliers <- phi.outliers.old ##need this to know if chages occured in outlies
  single.outliers <- single.outliers.old
  flag1 <- NA ##these will be false if convergence not reacher,
  flag2 <- NA ## this will be false if outliers respectively cuase iter to stop
  if(remove.outliers){
    flag1 <- TRUE; flag2<-TRUE
    original.data.matrix <- data.matrix ##so we can get it back after outlier removal
    change.theta <- 1 #start with 1 
    change.phi <- 1
    change.single <- 1
    outer.iter <- 0 
    while(flag1 & flag2 & change.theta+change.phi+change.single >0 & outer.iter < outer.maxit) {
      outer.iter <- outer.iter + 1
      
      if((outer.iter%%3==0 & change.theta>0)  |
         (outer.iter%%3==1 & change.phi>0)){ #something has to change
        ##starting values
        phi <- apply(data.matrix, 2, mean)
        c <- sqrt(cJ/sum(phi[!phi.outliers]^2))
        phi <- c * phi
        theta <- (data.matrix[, !phi.outliers, drop=FALSE] %*% phi[!phi.outliers, drop=FALSE])/cJ
        iter <- 0
        change <- 1 #start with one
        theta.old <- rep(0, II)
        while(change > delta & iter < maxit) {
          iter <- iter + 1
          phi <- t(data.matrix[!theta.outliers, ,drop=FALSE]) %*% theta[!theta.outliers, drop=FALSE] ##ignore the outliers
          c <- sqrt(cJ/sum(phi[!phi.outliers, drop=FALSE]^2))
          phi <- c * phi
          theta <- (data.matrix[,!phi.outliers, drop=FALSE] %*% phi[!phi.outliers, drop=FALSE])/cJ
          change <- max(abs(theta[!theta.outliers] - theta.old[!theta.outliers]))
          if(verbose) cat(paste("Outlier iteration:",outer.iter,"estimation iteration:",iter,"chage=",change,"\n"))
          theta.old <- theta
        }
        if(iter>=maxit){ ##convergence not reached. might as well get out
          warning(paste("No convergence in inner loop after",iter,"in outerler tieration",outer.iter,"\n"))
          flag1 <- FALSE
        }
        if(mean(phi[!phi.outliers]<0)>.5){ ##for identifiability.. theta*phi = (-theta)*(-phi), i require that most phis are positive
          theta <- -theta
          phi <- -phi
        }
        theta <- as.vector(theta)
        phi <- as.vector(phi)
                
        data.matrixhat <- outer(theta, phi)
        resid <- data.matrix-data.matrixhat
      }
      ##DEALING WITH OUTLIERS
      ##we alternate removal of outliers
      ##if even iteration take out thetas that are outliers (as defined by Li and Wong).
      if(outer.iter%%3==1){ ## we start with single outliers
        single.outliers <- resid > large.threshold*quantile(abs(resid),normal.resid.quantile)
        single.outliers[apply(single.outliers,1,sum) > outlier.fraction*cJ,]<-rep(FALSE,J)
        ##probably chip oulier, defer calling outlier
        single.outliers[,apply(single.outliers,2,sum) > outlier.fraction*cI]<-rep(FALSE,II)
        ##probably probe outlier, defer calling outlier
        data.matrix[single.outliers] <- data.matrixhat[single.outliers]
        data.matrix[!single.outliers] <- original.data.matrix[!single.outliers]
        change.single <- sum(abs(single.outliers.old-single.outliers)) #sum will be total of changes
        single.outliers.old <- single.outliers
      }
      else{
        sigma.theta <- sqrt(apply(resid[, !phi.outliers, drop=FALSE]^2, 1, sum)/(cJ - 1))
        sigma.phi <- sqrt(apply(resid[!theta.outliers, , drop=FALSE]^2, 2, sum)/(cI - 1))
        ###THETA OUTLIERS
        if(outer.iter%%3==2){
          theta.outliers <- sigma.theta > large.threshold*quantile(sigma.theta,normal.array.quantile) | theta^2/sum(theta^2) > large.variation
          cI <- sum(!theta.outliers)
          if(cI<3) {
            warning("No convergence achieved, too many outliers")
            flag2 <- FALSE
          }
          ##single outliers in outlier chips are not longer single outliers
          single.outliers[theta.outliers,] <- rep(FALSE,J)
          data.matrix[single.outliers] <- data.matrixhat[single.outliers]
          data.matrix[!single.outliers]<-original.data.matrix[!single.outliers]
          change.theta <- sum(abs(theta.outliers.old-theta.outliers)) #sum will be total of changes
          change.single <- sum(abs(single.outliers.old-single.outliers)) #sum will be total of changes
          theta.outliers.old <- theta.outliers
        }
        ##PHI OUTLIERS
        else{
          phi.outliers <- sigma.phi > large.threshold*quantile(sigma.phi,normal.array.quantile) | phi^2/sum(phi^2) > large.variation | phi <0
          cJ <- sum(!phi.outliers)
          if(cJ<3) {
            warning("No convergence achieved, too many outliers")
            flag2 <- FALSE
          }
          single.outliers[,phi.outliers] <- rep(FALSE,II)
          data.matrix[single.outliers] <- data.matrixhat[single.outliers]
          data.matrix[!single.outliers]<-original.data.matrix[!single.outliers]
          change.phi <- sum(abs(phi.outliers.old-phi.outliers))
          change.single <- sum(abs(single.outliers.old-single.outliers)) #sum will be total of changes
          phi.outliers.old <- phi.outliers
        }
      }
      if(verbose){
        cat("chips used=",cI,", probes used=",cJ,", single outler=",sum(single.outliers),"\n")
        cat("Number of changes: single=",change.single,", theta=",change.theta,", phi=",change.phi,"\n",sep="")
      }
    }
    if(outer.iter>=outer.maxit){
      warning("No convergence achieved in outlier loop\n")
      flag2 <- FALSE
    }
    all.outliers <- outer(theta.outliers,phi.outliers,FUN="|") | single.outliers
    sigma <- sqrt(sum(resid[!all.outliers]^2)/sum(!all.outliers))
    ##in case we leave iteration and these havent been defined
    sigma.theta <- sqrt(apply(resid[,!phi.outliers, drop=FALSE]^2, 1, sum)/(cJ - 1))
    sigma.phi <- sqrt(apply(resid[!theta.outliers, ,drop=FALSE]^2, 2, sum)/(cI - 1))
  }
  ###code for NO OUTLIER REMOVAL
  else{
    flag1 <- TRUE
    phi <- apply(data.matrix, 2, mean)
    c <- sqrt(J/sum(phi^2))
    phi <- c * phi
    theta <- (data.matrix %*% phi)/J

    iter <- 0
    change <- 1
    theta.old <- rep(0, II)
    while(change > delta & iter < maxit) {
      iter <- iter + 1
      phi <- t(data.matrix) %*% theta
      c <- sqrt(J/sum(phi^2))
      phi <- c * phi
      theta <- (data.matrix %*% phi)/J
      change <- max(abs(theta - theta.old))
      if(verbose) cat(paste("Iteration:",iter,"chage=",change,"\n"))
      theta.old <- theta
    }
    if(iter>=maxit){
      warning(paste("No convergence after",iter,"iterations.\n"))
      flag1 <- FALSE
    }
    if(mean(phi[!phi.outliers]<0)>.5){
      ##for identifiability.. theta*phi = (-theta)*(-phi), i require that most phis are positive
      theta <- -theta
      phi <- -phi
    }
    theta <- as.vector(theta)
    phi <- as.vector(phi)
    data.matrixhat <- outer(theta, phi)
    sigma.theta <- sqrt(apply((data.matrix - data.matrixhat)^2, 1, sum)/(J - 1))
    sigma.phi <- sqrt(apply((data.matrix - data.matrixhat)^2, 2, sum)/(II - 1))
    sigma <- sqrt(sum((data.matrix - data.matrixhat)^2)/(II * J))
  }
  return(list(theta = theta, phi = phi, sigma.eps = sigma, sigma.theta = sigma.theta, sigma.phi=sigma.phi,theta.outliers=theta.outliers,phi.outliers=phi.outliers,single.outliers=single.outliers,convergence1=flag1,convergence2=flag2,iter = iter, delta = change)) 
}
## Currently, the input is a 2 matrices a pm and a mm

##avdiff is more like median than mean, it would be nice to actually have
##avfif
##added typical se of the mean as returned se
generateExprVal.method.avgdiff <- function(probes, ...) {
  list(exprs=apply(probes, 2, median),se.exprs=apply(probes,2,sd)/sqrt(nrow(probes)))
}
generateExprVal.method.liwong <- function(probes, ...) {
  probes <- t(probes)
  if (ncol(probes) == 1) {
    warning("method liwong unsuitable when only one probe pair")
    list(exprs=as.vector(probes),se.exprs=rep(NA,length(probes)))
  }
  else {
    tmp <- fit.li.wong(probes, ...)
    list(exprs=tmp$theta,se.exprs=tmp$sigma.theta)
  }
}
generateExprVal.method.mas <- function(probes, ...)
{
  
  probes <- log2(probes)
  M <-  ncol(probes)
  slg <- rep(NA,M)
  
  for (i in 1:ncol(probes)) {
    
    slg[i] <- tukey.biweight(probes[ ,i], ...)
    
  }
  
  return(list(exprs=2^slg,se.exprs=rep(NA,M)))
    
}

affy.scalevalue.exprSet <- function(eset, sc=500, analysis="absolute")
{
  
  analysis <- match(analysis, c("absolute", "comparison"))
  
  if(analysis == 1)
    nf <- 1
  else
    stop("sorry! comparison not implemented.")
  for (i in 1:ncol(exprs(eset))) {
    slg <- exprs(eset)[, i]
    sf <- sc / mean(slg, trim=0.02)  
    reported.value <- nf * sf * slg
    eset@exprs[, i] <- reported.value
  }
  
  return(eset)
}
generateExprVal.method.medianpolish <- function(probes, ...) 
  medianpolish(probes, ...)

generateExprVal.method.playerout <- function(probes, weights=FALSE, optim.method="L-BFGS-B"){
  
  probes <- t(probes)
  nprobes <- ncol(probes)
  
  ## skip if only one probe
  if (nprobes == 1) return(t(probes))
  
  ## I do not know to which extend the use of optim
  ## is really equivalent to the use of nlminb in S-plus
  S1 <- optim(runif(nprobes),
              playerout.costfunction,
              method=optim.method,
              control=list(maxit=500),
              y=probes)
  ##S1 <- nlm(playerout,runif(20),iterlim=500,y=t(y))
  r <- c(probes %*% S1$par / sum(S1$par))
  if (weights)
    attr(r,"weights") <- S1$par
  return(list(exprs=r,se.exprs=rep(NA,length(r))))
}


## The loss function:

playerout.costfunction <- function(w, y) {
  N <- length(w)        # Number of players
  J <- length(y)/N      # Number of games (the number of games is the number of chips used)
  sumw <- sum(w)
  
  tx <- y %*% w    # Full weighted score at each game  
  pl <- matrix(0,J,N)    # Loss at each game due to each player
  
  for(j in 1:J)
    pl[j,] <- w * y[j,] - (tx[j] - w * y[j,]) / (sumw - w)
  
  sum(pl^2)         # Loss
}


   
hlog <- function(x,constant=1){ #constant is where the change occurs
  if(constant<=0){
    warning("constant less than or equal to 0. Returning log(x)\n")
    return(log(x))
  }
  else{
    if(constant==Inf)
      return(x)
    else{
      aux <- (abs(x)<constant)*(x) +
        (abs(x)>=constant)*(sign(x)*(constant*log(abs(x/constant))+constant))
      aux[x==0] <- 0
      return(aux)
    }
  }
}
## Sept 11, 2003 - justRMA calls just.rma2
### A user friendly wrapper for just.rma
justRMA <- function(..., filenames=character(0),
                     widget=getOption("BioC")$affy$use.widgets,
                     compress=getOption("BioC")$affy$compress.cel,
                     celfile.path=getwd(),
                     sampleNames=NULL,
                     phenoData=NULL,
                     description=NULL,
                     notes="",
                     rm.mask=FALSE, rm.outliers=FALSE, rm.extra=FALSE,
                     hdf5=FALSE, hdf5FilePath=NULL,verbose=FALSE,
                     normalize=TRUE, background=TRUE, 
                     bgversion=2, destructive=FALSE){
  ##first figure out filenames
  auxnames <- unlist(as.list(substitute(list(...)))[-1])
  
  if (widget){
    require(tkWidgets)
    widgetfiles <- fileBrowser(textToShow="Choose CEL files",
                               testFun=hasSuffix("[cC][eE][lL]"))
  }
  else
    widgetfiles <- character(0)
  
  filenames <- .Primitive("c")(filenames, auxnames, widgetfiles)
  
  if(length(filenames)==0) filenames <- list.celfiles(celfile.path,full.names=TRUE)
  
  if(length(filenames)==0) stop("No cel filenames specified and no cel files in specified directory:",celfile.path,"\n")
  
  
  ##now assign sampleNames if phenoData not given
  if(is.null(phenoData)){
    if(is.null(sampleNames)){
      if(widget){
        require(tkWidgets)
        tksn <- tkSampleNames(filenames=filenames)
        sampleNames <- tksn[,1]
        ##notice that a description of the files is ingored for now
        ##soon to go into MIAME
      }
      else{
        sampleNames <- sub("^/?([^/]*/)*", "", filenames, extended=TRUE)
      }
    }
    else{
      if(length(sampleNames)!=length(filenames)){
        warning("sampleNames not same length as filenames. Using filenames as sampleNames instead\n")
        sampleNames <- sub("^/?([^/]*/)*", "", filenames, extended=TRUE)
      }
    }
  }
  
  ##now get phenoData
  if(is.character(phenoData)) ##if character read file
    phenoData <- read.phenoData(filename=phenoData)
  else{
    if(class(phenoData)!="phenoData"){
      if(widget){
        require(tkWidgets)
        phenoData <- read.phenoData(sampleNames=sampleNames,widget=TRUE)
      }
      else
        phenoData <- read.phenoData(sampleNames=sampleNames,widget=FALSE)
    }
  }
  
  ##get MIAME information
  if(is.character(description)){
    description <- read.MIAME(filename=description,widget=FALSE)
  }
  else{
    if(class(description)!="MIAME"){
      if(widget){
        require(tkWidgets)
        description <- read.MIAME(widget=TRUE)
      }
      else
        description <- new("MIAME")
    }
  }
  
  ##MIAME stuff
  description@preprocessing$filenames <- filenames
  if(exists("tksn")) description@samples$description <- tksn[,2]
  description@preprocessing$affyversion <- library(help=affy)$info[[2]][[2]][2]

  ##and now we are ready to read cel files
  return(just.rma(filenames=filenames,
                        phenoData=phenoData,
                        description=description,
                        notes=notes,
                        compress=compress,
                        rm.mask=rm.mask,
                        rm.outliers=rm.outliers,
                        rm.extra=rm.extra,
                        #hdf5=hdf5, ## took these two out b/c I am not sure if hdf5 should be used
                        #hdf5FilePath=hdf5FilePath,
                        verbose=verbose,
                        normalize=normalize,
                        background=background,
                        bgversion=bgversion,
				destructive=destructive))
}




###########################################################################################
#
# this function uses a different parsing routine
# It was added Jul 7, 2003 by B. M. Bolstad
#
###########################################################################################

just.rma <- function(..., filenames=character(0),
                     phenoData=new("phenoData"),
                     description=NULL,
                     notes="",
                     compress=getOption("BioC")$affy$compress.cel,
                     rm.mask=FALSE, rm.outliers=FALSE, rm.extra=FALSE,
                     verbose=FALSE, background=TRUE, normalize=TRUE,
                     bgversion=2, destructive=FALSE) {
  
  auxnames <- as.list(substitute(list(...)))[-1]
  filenames <- .Primitive("c")(filenames, auxnames)
  
  n <- length(filenames)
  
  ## error if no file name !
  if (n == 0)
    stop("No file name given !")
  
  pdata <- pData(phenoData)
  ##try to read sample names form phenoData. if not there use CEL filenames
  if(dim(pdata)[1]!=n){#if empty pdata filename are samplenames
    warning("Incompatible phenoData object. Created a new one.\n")
    
    samplenames <- gsub("^/?([^/]*/)*", "", unlist(filenames), extended=TRUE	)
    pdata <- data.frame(sample=1:n,row.names=samplenames)
    phenoData <- new("phenoData",pData=pdata,varLabels=list(sample="arbitrary numbering"))
  }
  else samplenames <- rownames(pdata)
  
  if (is.null(description))
    {
      description <- new("MIAME")
      description@preprocessing$filenames <- filenames
      description@preprocessing$affyversion <- library(help=affy)$info[[2]][[2]][2]
    }
  ## read the first file to see what we have
  ##if (verbose) cat(1, "reading",filenames[[1]],"...")
  
  ## get information from cdf environment

  headdetails <- .Call("ReadHeader", filenames[[1]], compress)
  dim.intensity <- headdetails[[2]]
  cdfname <- headdetails[[1]]
  
  tmp <- new("AffyBatch",
             cdfName=cdfname,
             annotation=cleancdfname(cdfname, addcdf=FALSE))
  pmIndex <- pmindex(tmp)
  probenames <- rep(names(pmIndex), unlist(lapply(pmIndex,length)))
  pmIndex <- unlist(pmIndex)
  
  ## read pm data into matrix
  
  probeintensities <- read.probematrix(filenames=filenames)

  ##pass matrix of pm values to rma
  
  ngenes <- length(geneNames(tmp))
  
  ##background correction
  bg.dens <- function(x){density(x,kernel="epanechnikov",n=2^14)}
  
  #if(destructive){
    exprs <- .Call("rma_c_complete",probeintensities$pm,probeintensities$pm,probenames,ngenes,body(bg.dens),new.env(),normalize,background,bgversion)
  #}else{
  #  exprs <- .Call("rma_c_complete_copy",probeintensities$pm,probeintensities$pm,probenames,ngenes,body(bg.dens),new.env(),normalize,background,bgversion)
  #}
  colnames(exprs) <- samplenames
  se.exprs <- array(NA, dim(exprs))
  
  annotation <- annotation(tmp)
  
  new("exprSet", exprs = exprs, se.exprs = se.exprs, phenoData = phenoData, 
      annotation = annotation, description = description, notes = notes)    
}





loess.normalize <- function(mat,subset=sample(1:(dim(mat)[2]),5000),
                      epsilon=10^-2,maxit=1,log.it=TRUE,verbose=TRUE,span=2/3,
                      family.loess="symmetric"){
  J <- dim(mat)[2]
  II <- dim(mat)[1]
  newData <- mat
  if(log.it){
    mat <- log2(mat)
    newData <- log2(newData)
  }
  change <- epsilon +1
  fs <- matrix(0,II,J)##contains what we substract
  iter <- 0
  w <- c(0,rep(1,length(subset)),0) ##this way we give 0 weight to the
  ##extremes added so that we can interpolate
  while(iter < maxit){
    iter <- iter+1
    means <- matrix(0,II,J) ##contains temp of what we substract
    for(j in 1:(J-1)){
      for(k in (j+1):J){
        y <- newData[,j]-newData[,k]
        x <-(newData[,j]+newData[,k])/2
        index <- c(order(x)[1],subset,order(-x)[1])
        ##put endpoints in so we can interpolate
        xx <- x[index]
        yy <- y[index]
        aux <-loess(yy~xx,span=span,degree=1,weights=w,family=family.loess)
        aux <- predict(aux,data.frame(xx=x))/J
        means[,j] <- means[,j] + aux 
        means[,k] <- means[,k] - aux
        if(verbose) cat("Done with",j,"vs",k," in iteration ",iter,"\n")
      }
    }
    fs <- fs+means
    newData <- mat-fs
    change <- max(apply((means[subset,])^2,2,mean))
    if(verbose) cat(iter,change,"\n")
    oldfs <- fs
  }
  if(change>epsilon & maxit>1) warning(paste("No convergence after",maxit,"iterations.\n"))
  if(log.it) return(2^newData)
  else return(newData)
}


library(modreg)
##*******************************************************************************************
#**********  maffy.normalize     *****
maffy.normalize <- function(data,subset,verbose=FALSE,span=0.25,family="symmetric",log.it=TRUE){

k <- dim(data)[2]   ### Number of chips

####   Create the transformation matrix 
t1 <- 1/sqrt(k)
t2 <- (k-2-t1)/(k-1)
t3 <- -(1+t1)/(k-1)

transmat <- matrix(t3,k,k)
for(i in 1:k){ 
     transmat[1,i]<-t1
     transmat[i,1]<-t1
} 
for(i in 2:k) transmat[i,i]<-t2 

#### Find normalizing curve   

if(verbose) cat("Fitting normalizing curve\n")
n<- length(subset)
data.subset <- data[subset,]

data.subset <- log(data.subset)%*%t(transmat)
index <- order(data.subset[,1])
data.subset <- data.subset[index,]

if( k>2) curve <- multiloess(data.subset[,2:k]~data.subset[,1],span=span,family=family,surface="direct")
else     curve <-      loess(data.subset[,2:k]~data.subset[,1],span=span,family=family,surface="direct")

### Transform the normalizing curve before and after normalization
scaled   <- cbind(data.subset[,1],matrix(0,n,k-1)) %*%(transmat)
unscaled <- cbind(data.subset[,1],curve$fitted)      %*%(transmat)

w <-c(0,rep(1,n,n),0)

data.scaled <- NULL

### Normalize each array
for(i in 1:k){
    if(verbose) cat("Normalizing chip ",i,"\n")
    if(log.it){
        mini  <- log(min(data[,i]))
        maxi  <- log(max(data[,i]))
    }
    else{
        mini  <- min(data[,i])
        maxi  <- max(data[,i])
    }

    curve <- loess(c(mini,scaled[,i],maxi)~c(mini,unscaled[,i],maxi),weights=w,span=span)

    if(log.it) 
         temp <-  exp(predict(curve,log(data[,i])))
    else
         temp <-      predict(curve,data[,i])

    data.scaled <- cbind(data.scaled,temp) 
}

data.scaled

}


##*******************************************************************************************
#**********  Select A subset with small rank-range over arrays  *****

maffy.subset <- function(data,subset.size=5000,maxit=100,subset.delta=max(round(subset.size/100),25),verbose=FALSE){


k     <- dim(data)[2]   ### Number of chips
n     <- dim(data)[1]   ## Size of starting subset, i.e. all rows

if(verbose)
      cat("Data size",n,"x",k,"Desired subset size",subset.size,"+-",subset.delta,"\n")       

means <- data%*%(rep(1,k,k)/k)

index0 <- order(means)

data.sorted <- data[index0,]

## Init
set <- rep(TRUE,n,n)      ## Set-indicator
index.set <- 1:n       ## Indexes for subset 
nprev <- n+1           
iter  <- 1
part.of.n <- 1

## loop
while(nprev>n & n>(subset.size+subset.delta) & iter <maxit){
    if(verbose)
      cat("Comuting ranks of old subset....")       
    ranks <-apply(data.sorted[index.set,],2,rank)              ## Compute ranks, chip by chip.
    ranks.range <- apply(ranks,1,function(r) max(r)-min(r) )   ## Range of ranks over chips

    q <-min((n*part.of.n+subset.size)/((1+part.of.n)*n),1)     ## Select quantiles
    low <- quantile(ranks.range[1:(n*0.2)+n*0.0],probs=q,names=FALSE)/n  
    high <-quantile(ranks.range[n+1-(1:(n*0.2))],probs=q,names=FALSE)/n
    
    newset <-  ranks.range < (low*n+(0:n-1)*(high-low))        ## Set-indicator of new set

    if(sum(newset)<subset.size-subset.delta){                  ## To small?
       part.of.n <- 1+part.of.n
       if(verbose)
         cat("\nSize of newset to small (",sum(newset),"). Increasing part.of.n.\n")
    }
    else{                                                      ## New set OK
       set <- newset
       index.set <- subset(index.set,set)
       index.set <- index.set[!is.na(index.set)] 
       nprev <- n
       n <- length(index.set)
       if(verbose)
          cat("Size of new subset: ",n,"\n")       
   }

   iter <- iter+1
}
##end loop

if(!iter <maxit) warning("Maximum number of iterations reached, result my not be correct\n")

list(subset=index0[index.set])

}




##*******************************************************************************************
multiloess <-
function(formula, data=NULL, weights, subset, na.action, model = FALSE,
	 span = 0.75, enp.target, degree = 2,
	 normalize = TRUE,
	 family = c("gaussian", "symmetric"),
	 method = c("loess", "model.frame"),
	 control = loess.control(...), ...)
{
    parametric <- FALSE
    drop.square <- FALSE

    mt <- terms(formula, data = data)
    mf <- match.call(expand.dots=FALSE)
    mf$model <- mf$span <- mf$enp.target <- mf$degree <-
	mf$parametric <- mf$drop.square <- mf$normalize <- mf$family <-
	    mf$control <- mf$... <- NULL
    mf[[1]] <- as.name("model.frame")
    mf <- eval(mf, parent.frame())
    if (match.arg(method) == "model.frame") return(mf)
    y <- model.response(mf, "numeric")

    if(is.vector(y))  stop("The respons variable is not a matrix, use loess")

    w <- model.weights(mf)
    if(is.null(w)) w <- rep(1, NROW(y))
    nmx <- as.character(attr(mt, "variables"))[-(1:2)]
    x <- mf[, nmx, drop=FALSE]
    if(any(sapply(x, is.factor))) stop("predictors must all be numeric")
    x <- as.matrix(x)
    D <- ncol(x)
    nmx <- colnames(x)
    names(nmx) <- nmx
    drop.square <- match(nmx, nmx[drop.square], 0) > 0
    parametric <- match(nmx, nmx[parametric], 0) > 0
    if(!match(degree, 0:2, 0)) stop("degree must be 0, 1 or 2")
    iterations <- if(family=="gaussian") 1 else control$iterations
    if(!missing(enp.target))
	if(!missing(span))
	    warning("both span and enp.target specified: span will be used")
	else {				# White book p.321
	    tau <- switch(degree+1, 1, D+1, (D+1)*(D+2)/2) - sum(drop.square)
	    span <- 1.2 * tau/enp.target
	}
    fit <- simplemultiLoess(y, x, w, span, degree,
		       normalize, control$statistics, control$surface,
		       control$cell, iterations, control$trace.hat)
    fit$call <- match.call()
    fit$terms <- mt
    fit$xnames <- nmx
    fit$x <- x
    fit$y <- y
    fit$weights <- w
    if(model) fit$model <- mf
    fit
}



##*******************************************************************************************
simplemultiLoess <- function(y, x, weights, span = 0.75, degree = 2,
	   normalize = TRUE,
	   statistics = "approximate", surface = "interpolate",
	   cell = 0.2, iterations = 1, trace.hat = "exact")
{
 
    ## Extra init 
    parametric <- FALSE
    drop.square <- FALSE
    
    M <- NCOL(y)
    A <- rep(1,M,M)

    D <- NCOL(x)
    N <- NROW(x)

    fitted.all <- matrix(1,N,M)
    fitted.residuals <- matrix(1,N,M)
    pseudo.resid.all <-  matrix(1,N,M)

    if(!N || !D)	stop("invalid `x'")
    if(!length(y))	stop("invalid `y'")
    x <- as.matrix(x)
    max.kd <-  max(N, 200)
    robust <- rep(1, N)
    divisor<- rep(1, D)
    if(normalize && D > 1) {
	trim <- ceiling(0.1 * N)
	divisor <-
	    sqrt(apply(apply(x, 2, sort)[seq(trim+1, N-trim), , drop = FALSE],
		       2, var))
	x <- x/rep(divisor, rep(N, D))
    }
    sum.drop.sqr <- sum(drop.square)
    sum.parametric <- sum(parametric)
    nonparametric <- sum(!parametric)
    order.parametric <- order(parametric)
    x <- x[, order.parametric]
    order.drop.sqr <- (2 - drop.square)[order.parametric]
    if(degree==1 && sum.drop.sqr)
	stop("Specified the square of a factor predictor to be dropped when degree = 1")
    if(D == 1 && sum.drop.sqr)
	stop("Specified the square of a predictor to be dropped with only one numeric predictor")
    if(sum.parametric == D) stop("Specified parametric for all predictors")

    if(iterations)
    for(j in 1:iterations) {
	robust <- weights * robust
	if(j > 1) statistics <- "none"
	if(surface == "interpolate" && statistics == "approximate")
	    statistics <- if(trace.hat == "approximate") "2.approx"
	    else if(trace.hat == "exact") "1.approx"
	surf.stat <- paste(surface, statistics, sep="/")
        for(k in 1:M) {        
        	z <- .C("loess_raw",
			as.double(y[,k]),
			as.double(x),
			as.double(weights),
			as.double(robust),
			as.integer(D),
			as.integer(N),
			as.double(span),
			as.integer(degree),
			as.integer(nonparametric),
			as.integer(order.drop.sqr),
			as.integer(sum.drop.sqr),
			as.double(span*cell),
			as.character(surf.stat),
			fitted.values = double(N),
			parameter = integer(7),
			a = integer(max.kd),
			xi = double(max.kd),
			vert = double(2*D),
			vval = double((D+1)*max.kd),
			diagonal = double(N),
			trL = double(1),
			delta1 = double(1),
			delta2 = double(1),
			as.integer(surf.stat == "interpolate/exact"),
			PACKAGE="modreg")
			fitted.all[,k] <- z$fitted.values
	}

	if(j==1) {
	    trace.hat.out <- z$trL
	    one.delta <- z$delta1
	    two.delta <- z$delta2
	}

	residuals.all <- (y-fitted.all)
	fitted.residuals <- sqrt((residuals.all^2)%*%A)

	if(j < iterations)
	    robust <- .Fortran("lowesw",
			       as.double(fitted.residuals),
			       as.integer(N),
			       robust = double(N),
			       double(N),
			       PACKAGE="modreg")$robust
    }
    if(surface == "interpolate")
    {
	pars <- z$parameter
	names(pars) <- c("d", "n", "vc", "nc", "nv", "liv", "lv")
	enough <- (D + 1) * pars["nv"]
	fit.kd <- list(parameter=pars, a=z$a[1:pars[4]], xi=z$xi[1:pars[4]],
		       vert=z$vert, vval=z$vval[1:enough])
    }
    if(iterations > 1) {
        for(k in 1:M) {        
		pseudovalues <- .Fortran("lowesp",
					 as.integer(N),
					 as.double(y[,k]),
					 as.double(fitted.all[,k]),
					 as.double(weights),
					 as.double(robust),
					 double(N),
					 pseudovalues = double(N),
					 PACKAGE="modreg")$pseudovalues
		zz <- .C("loess_raw",
			as.double(pseudovalues),
			as.double(x),
			as.double(weights),
			as.double(weights),
			as.integer(D),
			as.integer(N),
			as.double(span),
			as.integer(degree),
			as.integer(nonparametric),
			as.integer(order.drop.sqr),
			as.integer(sum.drop.sqr),
			as.integer(span*cell),
			as.character(surf.stat),
			temp = double(N),
			parameter = integer(7),
			a = integer(max.kd),
			xi = double(max.kd),
			vert = double(2*D),
			vval = double((D+1)*max.kd),
			diagonal = double(N),
			trL = double(1),
			delta1 = double(1),
			delta2 = double(1),
			as.integer(0),
			PACKAGE="modreg")
		pseudo.resid.all[,k] <- pseudovalues-zz$temp
	}

	pseudo.resid <- sqrt((pseudo.resid.all^2)%*%A)

    }
    sum.squares <- if(iterations <= 1) sum(weights * fitted.residuals^2)
    else sum(weights * pseudo.resid^2)
    enp <- one.delta + 2*trace.hat.out - N
    s <- sqrt(sum.squares/one.delta)
    pars <- list(robust=robust, span=span, degree=degree, normalize=normalize,
		 parametric=parametric, drop.square=drop.square,
		 surface=surface, cell=cell, family=
		 if(iterations <= 1) "gaussian" else "symmetric",
		 iterations=iterations)
    fit <- list(n=N, fitted=fitted.all, residuals=residuals.all,
		enp=enp, s=s, one.delta=one.delta, two.delta=two.delta,
		trace.hat=trace.hat.out, divisor=divisor)
    fit$pars <- pars
    if(surface == "interpolate") fit$kd <- fit.kd
    class(fit) <- "loess"
    fit

}


##*******************************************************************************************





mas5 <- function(object,normalize=TRUE,sc = 500, analysis = "absolute",...){
  res <- expresso(object,bgcorrect.method="mas",pmcorrect.method="mas",normalize=FALSE,summary.method="mas",...) 
  if(normalize) 
    res <- affy.scalevalue.exprSet(res,sc=sc,analysis=analysis)
  return(res)
}

mas5calls.ProbeSet <- function(object, 
                               tau=0.015, alpha1=0.04, alpha2=0.06,
                               exact.pvals=FALSE, cont.correct=FALSE) {
  pms <- pm(object)
  mms <- mm(object)
  calls <- vector("character",ncol(pms))
  pvals <- vector("numeric",ncol(pms))
  for(i in 1:ncol(pms)){
    mat <- cbind(pms[,i],mms[,i])
    res <- mas5.detection(mat,tau=tau, alpha1=alpha1, alpha2=alpha2,
                           exact.pvals=exact.pvals, cont.correct=cont.correct)
    calls[i] <- res$call
    pvals[i] <- res$pval
  }
  return(list(call=calls,pval=pvals))
}

mas5calls.AffyBatch <- function(object, ids=NULL, verbose=TRUE,
                                tau=0.015, alpha1=0.04, alpha2=0.06,
                                exact.pvals=FALSE, cont.correct=FALSE) {
  
  n <- length(object)
  
  ## if NULL compute for all
  if (is.null(ids))
    ids <- geneNames(object)
  
  m <- length(ids)
  
  exp.mat <- matrix(NA, m, n)
  se.mat <- matrix(NA, m, n)

  if (verbose) {
    cat(m, "ids to be processed\n")
    countprogress <- 0
  }

  ## loop over the ids
  
  CDFINFO <- getCdfInfo(object) ##do it once!

  for (i in seq(along=ids)) {

    id <- ids[i]

    if (verbose) {
      if ( round(m/10) == countprogress) {
        cat(".")
        countprogress <- 0
      }
      else
        countprogress <- countprogress + 1
    }

    ## locations for an id
    loc <- get(id, envir=CDFINFO)
    l.pm <- loc[, 1]
    if (ncol(loc) == 2)
      l.mm <- loc[ ,2]
    else
      stop("No MMs available for id:",id,"\n")

    pms <- intensity(object)[l.pm, ,drop=FALSE]
    mms <- intensity(object)[l.mm, ,drop=FALSE]
    for(j in 1:n){
      mat <- cbind(pms[,j],mms[,j])

      res <- mas5.detection(mat, tau=tau, alpha1=alpha1, alpha2=alpha2,
                           exact.pvals=exact.pvals, cont.correct=cont.correct)
      exp.mat[i,j] <- res$call
      se.mat[i,j] <- res$pval
    }
  }
  if (verbose) cat("\n")
  
  ## instance exprSet
  dimnames(exp.mat) <- list(ids, sampleNames(object))
  dimnames(se.mat) <- list(ids, sampleNames(object))
  eset <- new("exprSet",
              exprs=exp.mat,
              se.exprs=se.mat,
              phenoData=phenoData(object),
              description=description(object),
              annotation=annotation(object),
              notes=c(notes(object)))
  return(eset)
}


mas5.detection <- function(mat, tau=0.015, alpha1=0.04, alpha2=0.06,
                           exact.pvals=FALSE, cont.correct=FALSE) { 

  mat.r <- (mat[,1]-mat[,2])/(mat[,1]+mat[,2])
  ## CONSTANTS
  saturation.point <- 46000			# not a user parameter
     
     ## SANITY CHECKING
    if ( !is.matrix(mat) || length(dim(mat))!=2 || dim(mat)[2]!=2 ||
         dim(mat)[1] < 1 || !is.numeric(mat) )
        stop("Invalid mat matrix.")
    if ( !is.numeric(tau) )
        stop("Invalid tau.")
    if ( !is.numeric(alpha1) || !is.numeric(alpha2) ||
          alpha1 <= 0 || alpha1 >= alpha2 || alpha2 >= 0.5 )
        stop("Invalid alpha1 or alpha2.")
    if ( !is.logical(exact.pvals) )
        stop("Invalid exact.pvals.")
    if ( !is.logical(cont.correct) )
        stop("Invalid cont.correct.")
        
     ## DEALING WITH SATURATION; COMPUTING THE P-VALUE
     ## According to the Bioinformatics paper:
     ## * If all MM's are saturated, then call present
     ## * Otherwise discard pairs with a saturated MM
     ## According to the Affymetrix whitepaper:
     ## * If all probe-pairs are saturated, then call present with pval=0
     ## * If an MM is saturated, then we discard the pair
     ## * If a PM and MM are within tau of each other, we discard the pair
     ## So we're going with:
     ## * If all MM's are saturated, set pval=0 and don't use Wilcoxon
     ## * Discard probe-pairs when MM is saturated or the PM,MM are within tau
     ##   of each other
     ## * Compute the p-value using Wilcoxon's signed rank test on the retained
     ##   probe-pairs
    is.mm.saturated <- function(probe.pair, saturation.point)
        probe.pair[2] >= saturation.point
    is.retained <- function(probe.pair, saturation.point, tau)
        !(is.mm.saturated(probe.pair,saturation.point) ||
          abs(diff(probe.pair)) <= tau)
    if ( all(apply(mat,1,is.mm.saturated,saturation.point)) )
        pval <- 0
    else {
      retained <- apply(mat, 1, is.retained, saturation.point, tau)
      pval <- wilcox.test(mat.r[retained],
                          alternative="greater", mu=tau, paired=FALSE,
                          exact=exact.pvals, correct=cont.correct,
                          conf.int=FALSE)$p.value
    }
  
     ## DETECTION CALL
    if ( pval < 0 || pval > 1 )
        warning("Computed an unusual p-value outside the range [0,1].")
    if ( pval < alpha1 )
        call <- "P"
    else if ( pval < alpha2 )
        call <- "M"
    else
        call <- "A"
    
     ## DONE
    return(list(pval=pval, call=call))
}

merge.AffyBatch <- function(x, y, annotation=paste(annotation(x), annotation(y)),
                            description=NULL,
                            notes=paste(x@notes, y@notes), ...) {

  adim <- dim(intensity(x))[1]

  if ((nrow(x) != nrow(y)) || (ncol(x) != ncol(y)))
    stop("cannot merge chips of different sizes !")

  if (x@cdfName != y@cdfName)
    warning("cdfName mismatch (using the cdfName of x)!")

  if (is.null(description)){
    description <- new("MIAME")
    description@title <- "Created from merging two AffyBatches. No description was supplied. The description of the two original AffyBatches was not kept."
  }                       

  lx <- length(x)
  ly <- length(y)

  phenodata <- phenoData(x)
  pData(phenodata) <- rbind(pData(x),pData(y))
  return(new("AffyBatch",
             exprs=cbind(intensity(x),intensity(y)),
             cdfName=x@cdfName,
             nrow=nrow(x),
             ncol=ncol(x),
             phenoData=phenodata,
             annotation=x@annotation,
             description=description, ##need to write a merge for MIAME
             notes=paste("Merge from two AffyBatches with notes: 1)",
               x@notes,", and 2)",y@notes))
         )
}



library(modreg)
mva.pairs <- function(x,labels=colnames(x),log.it=TRUE,span=2/3,family.loess="gaussian",digits=3,line.col=2,main="MVA plot",...){
  if(log.it) x <-log2(x)
  J <- dim(x)[2]
  frame()
  old.par <- par(no.readonly = TRUE)
  on.exit(par(old.par))
  par(mfrow=c(J,J),mgp=c(0,.2,0),mar=c(1,1,1,1),oma=c(1,1.4,2,1))
  for(j in 1:(J-1)){
    par(mfg=c(j,j));plot(1,1,type="n",xaxt="n",yaxt="n",xlab="",ylab="");text(1,1,labels[j],cex=2)
    for(k in (j+1):J){
      par(mfg=c(j,k))
      yy <- x[,j]-x[,k]
      xx <-(x[,j]+x[,k])/2
      aux <- loess(yy~xx,degree=1,span=span,family=family.loess)$fitted
      plot(xx,yy,pch=".",xlab="",ylab="",tck=0,...)
      o <- order(xx)
      xx <- xx[o]
      yy <- aux[o]
      o <-which(!duplicated(xx))
      lines(approx(xx[o],yy[o]),col=line.col)
      par(mfg=c(k,j))
      sigma <- IQR(yy)
      txt <- format(c(sigma,0.123456789),digits=digits)
      plot(c(0,1),c(0,1),type="n",ylab="",xlab="",xaxt="n",yaxt="n")
      text(0.5,0.5,txt,cex=2)
    }
  }
  par(mfg=c(J,J));plot(1,1,type="n",xaxt="n",yaxt="n",xlab="",ylab="");
  text(1,1,labels[J],cex=2)
  mtext("A",1,outer=TRUE,cex=1.5)
  mtext("M",2,outer=TRUE,cex=1.5,las=1)
  mtext(main,3,outer=TRUE,cex=1.5)
  invisible()
}
normalize.AffyBatch.constant <- function(abatch, refindex=1, FUN=mean, na.rm=TRUE) {
  
  n <- length( abatch )
  
  if (! (refindex %in% 1:n)) stop("invalid reference index for normalization")
  refconstant <- FUN(intensity(abatch)[,refindex], na.rm=na.rm)
  
  #set.na.spotsd(abatch)

  normhisto <- vector("list", length=n)
  for (i in (1:n)[-refindex]) {
    m <- normalize.constant(intensity(abatch)[,i], refconstant, FUN=FUN, na.rm=na.rm)
    myhistory <- list(name="normalized by constant",
                      constant=attr(m,"constant"))
    attr(m,"constant") <- NULL
    intensity(abatch)[, i] <- m
    normhisto[[i]] <- myhistory
  }
  attr(abatch, "normalization") <- normhisto
  return(abatch)
}       


normalize.constant <- function(x, refconstant, FUN=mean, na.rm=TRUE) {
  thisconstant <- FUN(x, na.rm=na.rm)
  r <- x / thisconstant * refconstant
  attr(r,"constant") <- thisconstant * refconstant
  return(r)
}




normalize.AffyBatch.contrasts <- function(abatch,span=2/3,choose.subset=TRUE,subset.size=5000,verbose=TRUE,family="symmetric",type=c("together","pmonly","mmonly","separate")) {

  type <- match.arg(type)
  
  if (type == "pmonly"){
    Index <- unlist(pmindex(abatch))
  } else if (type == "mmonly"){
    Index <- unlist(mmindex(abatch))
  } else if (type == "together"){
    Index <- unlist(indexProbes(abatch,"both"))
  } else if (type == "separate"){
    abatch <- normalize.AffyBatch.contrasts(abatch,span=span,choose.subset=choose.subset,subset.size=subset.size,verbose=verbose,family=family,type="pmonly")
    Index <- unlist(mmindex(abatch))
  }
  
  ##we need default argumetns becuase they are used in this transitional file
  alldata <- intensity(abatch)[Index,]
  
  if(choose.subset)
    subset1 <- maffy.subset(alldata,verbose=verbose,subset.size=subset.size)$subset
  else
    subset1 <- sample(1:dim(alldata)[1],subset.size)
  aux <-   maffy.normalize(alldata,subset=subset1,verbose=verbose,span=span,family=family)
  
  intensity(abatch)[Index,] <- aux

  ##attr(abatch, "normalization") <- normhisto
  return(abatch)
}









normalize.AffyBatch.invariantset <- function(abatch, prd.td=c(0.003,0.007), verbose=FALSE,baseline.type=c("mean","median","pseudo-mean","pseudo-median"),type=c("separate","pmonly","mmonly","together")) {

  do.normalize.Affybatch.invariantset <- function(abatch, pms, prd.td, baseline.type){


    nc  <-  length(abatch)                                 # number of CEL files

    if (baseline.type == "mean"){
                                        # take as a reference the array having the median overall intensity
      m <- vector("numeric", length=nc)
      for (i in 1:nc)
        m[i] <- mean(intensity(abatch)[pms, i])
      refindex <- trunc(median(rank(m)))
      rm(m)
      baseline.chip <-  c(intensity(abatch)[pms, refindex])
      if (verbose) cat("Data from", sampleNames(abatch)[refindex], "used as baseline.\n")
    }
    else if (baseline.type == "median"){
                                        # take as a reference the array having the median median intensity
      m <- vector("numeric", length=nc)
      for (i in 1:nc)
        m[i] <- median(intensity(abatch)[pms, i])
      refindex <- trunc(median(rank(m)))
      rm(m)
      baseline.chip <-  c(intensity(abatch)[pms, refindex])
      if (verbose) cat("Data from", sampleNames(abatch)[refindex], "used as baseline.\n")
    } else if (baseline.type == "pseudo-mean"){
                                        # construct a psuedo chip to serve as the baseline by taking probewise means
      refindex <- 0
      baseline.chip <- apply(intensity(abatch)[pms,],1,mean)    
    } else if (baseline.type == "pseudo-median"){
    # construct a pseudo chip to serve as the baseline by taking probewise medians
      refindex <- 0
      baseline.chip <- apply(intensity(abatch)[pms,],1,median)
    }
  
    
  ##set.na.spotsd(cel.container)
  
    normhisto <- vector("list", length=nc)
#  normhisto[[refindex]] <- list(name="reference for the invariant set")
  
  ## loop over the CEL files and normalize them
  
    for (i in (1:nc)) {
      if (i != refindex){
        if (verbose) cat("normalizing array", chipNames(abatch)[i], "...")
        
        ##temporary
        tmp <- normalize.invariantset(c(intensity(abatch)[pms, i]),
                                      c(baseline.chip),
                                      prd.td)
                                        #i.set <- which(i.pm)[tmp$i.set]
        tmp <- as.numeric(approx(tmp$n.curve$y, tmp$n.curve$x,
                               xout=intensity(abatch)[pms, i], rule=2)$y)
        attr(tmp,"invariant.set") <- NULL
        intensity(abatch)[pms, i] <- tmp
        
        ## storing information about what has been done
                                        #normhisto[[i]] <- list(name="normalized by invariant set",
                                        #                       invariantset=i.set)
      
        if (verbose) cat("done.\n")
        
      }
    } 
    attr(abatch, "normalization") <- normhisto
    return(abatch)
  }
  
  type <- match.arg(type)
  baseline.type <- match.arg(baseline.type) 
  require(modreg, quietly=TRUE)
  
  if (type == "pmonly"){
    pms <- unlist(pmindex(abatch))
    do.normalize.Affybatch.invariantset(abatch, pms, prd.td, baseline.type)
  } else if (type == "mmonly"){
    pms <- unlist(mmindex(abatch))
    do.normalize.Affybatch.invariantset(abatch, pms, prd.td, baseline.type)
  } else if (type == "together"){
    pms <- nunlist(indexProbes(abatch,"both"))
    do.normalize.Affybatch.invariantset(abatch, pms, prd.td, baseline.type)
  } else if (type == "separate"){
    pms <- unlist(pmindex(abatch))
    abatch <- do.normalize.Affybatch.invariantset(abatch, pms, prd.td, baseline.type)
    pms <- unlist(mmindex(abatch))
    do.normalize.Affybatch.invariantset(abatch, pms, prd.td, baseline.type)
  }

}



##  The 'common-to-all' part of the algorithm. Operates on two vectors of numeric data
##
normalize.invariantset <- function(data, ref, prd.td=c(0.003,0.007)) {

  np <- length(data)
  r.ref <- rank(ref)
  r.array <- rank(data)
  
  ## init
  prd.td.adj <- prd.td*10                           # adjusted threshold things
  i.set <- rep(TRUE, np)                            # index all the PM probes as being in the invariant set
  ns <- sum(i.set)                                  # number of probes in the invariant set
  ns.old <- ns+50+1                                 # number of probes previously in the invariant set
    
  ## iterate while the number of genes in the invariant set (ns) still varies...
  while ( (ns.old-ns) > 50 ) {
    air <- (r.ref[i.set] + r.array[i.set]) / (2*ns)  # average intensity rank for the probe intensities
    prd <- abs(r.ref[i.set] - r.array[i.set]) / ns
    threshold <- (prd.td.adj[2]-prd.td[1]) * air + prd.td.adj[1]
    i.set[i.set] <- (prd < threshold)
    
    ns.old <- ns
    ns <- sum(i.set)
    
    if (prd.td.adj[1] > prd.td[1])
      prd.td.adj <- prd.td.adj * 0.9  # update the adjusted threshold parameters
  }
  
  ## the index i.set corresponds to the 'invariant genes'
  n.curve <- smooth.spline(ref[i.set], data[i.set])
  ## n.curve$x contains smoothed reference intensities
  ## n.curve$y contains smoothed i-th array intensities
  
  ##data <- as.numeric(approx(n.curve$y, n.curve$x, xout=data)$y)
  ##attr(data,"invariant.set") <- i.set
  ##return(data)
  return(list(n.curve=n.curve, i.set=i.set))
}







normalize.AffyBatch.loess <- function(abatch,type=c("together","pmonly","mmonly","separate"),...) {

  type <- match.arg(type)
  
  if (type == "separate"){
    Index <- unlist(indexProbes(abatch,"pm"))
    intensity(abatch)[Index,] <- normalize.loess(intensity(abatch)[Index,], ...)
    Index <- unlist(indexProbes(abatch,"mm"))
    intensity(abatch)[Index,] <- normalize.loess(intensity(abatch)[Index,], ...)
  } else if (type=="together"){
    Index <- unlist(indexProbes(abatch,"both"))
    intensity(abatch)[Index,] <- normalize.loess(intensity(abatch)[Index,], ...)
  } else if (type=="pmonly"){
    Index <- unlist(indexProbes(abatch,"pm"))
    intensity(abatch)[Index,] <- normalize.loess(intensity(abatch)[Index,], ...)
  } else if (type=="mmonly"){
    Index <- unlist(indexProbes(abatch,"mm"))
    intensity(abatch)[Index,] <- normalize.loess(intensity(abatch)[Index,], ...)
  }
  ##set.na.spotsd(listcel) # set 'sd' to nothing (meaningless after normalization)
  ##cat(cols,rows)


  ##need to use MIAME
  ##for (i in 1:abatch@nexp) {
  ##  history(abatch)[[i]] <- list(name="normalized by loess")
  ##}

  return(abatch)
}



normalize.loess <- function(mat, subset=sample(1:(dim(mat)[1]), min(c(5000, nrow(mat)))),
                            epsilon=10^-2, maxit=1, log.it=TRUE, verbose=TRUE, span=2/3,
                            family.loess="symmetric"){
  
  J <- dim(mat)[2]
  II <- dim(mat)[1]
  newData <- mat
  if(log.it){
    mat <- log2(mat)
    newData <- log2(newData)
  }
  
  change <- epsilon +1
  fs <- matrix(0, II, J)##contains what we substract
  iter <- 0
  w <- c(0, rep(1,length(subset)), 0) ##this way we give 0 weight to the
                                      ##extremes added so that we can interpolate
  
  while(iter < maxit){
    iter <- iter + 1
    means <- matrix(0,II,J) ##contains temp of what we substract
    
    for (j in 1:(J-1)){
      for (k in (j+1):J){
        y <- newData[,j] - newData[,k]
        x <- (newData[,j] + newData[,k]) / 2
        index <- c(order(x)[1], subset, order(-x)[1])
        ##put endpoints in so we can interpolate
        xx <- x[index]
        yy <- y[index]
        aux <-loess(yy~xx, span=span, degree=1, weights=w, family=family.loess)
        aux <- predict(aux, data.frame(xx=x)) / J
        means[, j] <- means[, j] + aux 
        means[, k] <- means[, k] - aux
        if (verbose)
          cat("Done with",j,"vs",k," in iteration ",iter,"\n")
      }
    }
    fs <- fs + means
    newData <- mat - fs
    change <- max(apply((means[subset,])^2, 2, mean))
    
    if(verbose)
      cat(iter, change,"\n")
    
    oldfs <- fs
    
  }
  
  if ((change > epsilon) & (maxit > 1))
    warning(paste("No convergence after", maxit, "iterations.\n"))
  
  if(log.it) {
    return(2^newData)
  } else
    return(newData)
}
normalize.AffyBatch.qspline <- function(abatch, type=c("together","pmonly","mmonly","separate"),...) {

  type <- match.arg(type)
  
  if (type == "together"){
    Index <- unlist(indexProbes(abatch,"both"))
    intensity(abatch)[Index,] <- normalize.qspline(intensity(abatch)[Index,], ...)
  } else if (type == "pmonly"){
    Index <- unlist(indexProbes(abatch,"pm"))
    intensity(abatch)[Index,] <- normalize.qspline(intensity(abatch)[Index,], ...)
  } else if (type == "mmonly"){
    Index <- unlist(indexProbes(abatch,"mm"))
    intensity(abatch)[Index,] <- normalize.qspline(intensity(abatch)[Index,], ...)
  } else if (type == "separate"){
    Index <- unlist(indexProbes(abatch,"pm"))
    intensity(abatch)[Index,] <- normalize.qspline(intensity(abatch)[Index,], ...)
    Index <- unlist(indexProbes(abatch,"mm"))
    intensity(abatch)[Index,] <- normalize.qspline(intensity(abatch)[Index,], ...)
  }
  #set.na.spotsd(listcel)
  normhisto <- vector("list", length=ncol(intensity(abatch)))
  ##need to use MIAME for this
  for (i in 1:length(abatch)) {
    normhisto[[i]] <- list(name="normalized by qspline")
  }
  
  attr(abatch, "normalization") <- normhisto
  
  return(abatch)
}

normalize.qspline <- function(x,
                              target        = NULL,
                              samples       = NULL,
                              fit.iters     = 5, 
                              min.offset    = 5,
                              spline.method = "natural", # c("fmm", "natural", "periodic")
                              smooth        = TRUE,
                              spar          = 0,     # smoothing parameter 
                              p.min         = 0, 
                              p.max         = 1.0, 
                              incl.ends     = TRUE,
                              converge      = FALSE,
                              verbose       = TRUE,
                              na.rm         = FALSE
                              ){

  require(modreg, quietly=TRUE) || error("library modreg is required !")
  
  if (is.null(target))
    target <- exp(apply(log(x), 1, mean))
  
  x.n <- dim(x)[1]
  m   <- dim(x)[2]

  if (is.null(samples))
    samples <- max(round(x.n/1000), 100)
  else
    if (samples < 1)
      samples <- round(samples * x.n)
  
  p <- (1:samples) / samples
  p <- p[ which(p <= p.max) & which(p >= p.min) ]
  samples <- length(p)
  
  k <- fit.iters
  
  if (na.rm==TRUE)
    y.n <- sum(!is.na(target))
  else
    y.n <- length(target)
  
  py.inds  <- as.integer(p * y.n)
  y.offset <- round(py.inds[1]/fit.iters)
  
  if (y.offset <= min.offset) { 
    y.offset <- min.offset;
    k <- round(py.inds[1]/min.offset)
  }

  if (k <= 1) {
    warning("'k' found is non-sense. using default 'fit.iter'")
    k <- fit.iters
  }
  
  y.offset <- c(0, array(y.offset, (k-1)))
  y.order <- order(target)

  fx <- matrix(0, x.n,m)
  if(verbose==TRUE)
    print(paste("samples=",samples, "k=", k, "first=", py.inds[1]))
  
  for (i in 1:m) {
                                        # to handel NA values for each array
    if (na.rm==TRUE)
      x.valid <- which(!is.na(x[,i])) 
    else
      x.valid <- 1:x.n
    
    x.n <- length(x.valid)
    px.inds  <- as.integer(p * x.n)
      x.offset <- round(px.inds[1]/fit.iters)
    
    if (x.offset<=min.offset) { 
      x.offset <- min.offset; 
      k <- min(round(px.inds[1]/min.offset), k) 
    }
    
    x.offset <- c(0, array(x.offset, (k-1)))
    x.order  <- order(x[,i]) # NA's at the end (?)
    
    y.inds   <- py.inds ## must be reset each iteration
    x.inds   <- px.inds 

    for (j in 1:k) {
         y.inds <- y.inds - y.offset[j]
         x.inds <- x.inds - x.offset[j]
         ty.inds <- y.inds
         tx.inds <- x.inds
         if (verbose==TRUE)
           print(paste("sampling(array=", i, "iter=", j, "off=",
                       x.inds[1], -x.offset[j], y.inds[1], -y.offset[j], ")"))
         
         if (converge==TRUE) {
           ty.inds <- as.integer(c(1, y.inds))
           tx.inds <- as.integer(c(1, x.inds))
           
           if (j > 1) {
             ty.inds <- c(ty.inds, y.n)
             tx.inds <- c(tx.inds, x.n)
           }
         }
         qy <- target[y.order[ty.inds]]
         qx <-  x[x.order[tx.inds],i]
         
         if (smooth==TRUE) {
           sspl <- smooth.spline(qx, qy, spar=spar)
           qx <- sspl$x
           qy <- sspl$y
         }
         
         fcn <- splinefun(qx, qy, method=spline.method)
         fx[x.valid,i] <- fx[x.valid,i] + fcn(x[x.valid,i])/k
       }
    
    if (na.rm==TRUE) {
      invalid <- which(is.na(x[,i]))
      fx[invalid,i] <- NA
    }
  }
  return(fx)
}
##################################################################
##
## file: normalize.quantiles.R
##
## For a description of quantile normalization method see
##
##  Bolstad, B. M., Irizarry R. A., Astrand, M, and Speed, T. P. (2003)(2003) 
##  A Comparison of Normalization Methods for High 
##  Density Oligonucleotide Array Data Based on Bias and Variance.
##  Bioinformatics 19,2,pp 185-193
##
## History
## Pre Aug 23, 2003 Two years worth of stuff
## Aug 23, 2003 - Added use.log2 to "robust",
##                added ability to pass additional parameters
##                to normalize.AffyBatch.Quantiles.robust
##                changed pmonly parameters on functions
##                so that it is now a string argument "type"
##                the options are pmonly, mmonly, together, separate
##
##
##################################################################

normalize.AffyBatch.quantiles <- function(abatch,type=c("separate","pmonly","mmonly","together")) {


  type <- match.arg(type)

  if ((type == "pmonly")|(type == "separate")){
    pms <- unlist(pmindex(abatch))
    noNA <- apply(intensity(abatch)[pms,,drop=FALSE],1,function(x) all(!is.na(x)))
    pms <- pms[noNA]
    intensity(abatch)[pms,] <- normalize.quantiles(intensity(abatch)[pms,,drop=FALSE ],copy=FALSE)
  }
  if((type == "mmonly") | (type == "separate")){ 
    mms <- unlist(mmindex(abatch))
    noNA <- apply(intensity(abatch)[mms,,drop=FALSE],1,function(x) all(!is.na(x)))
    mms <- mms[noNA]

    intensity(abatch)[mms,] <- normalize.quantiles(intensity(abatch)[mms,,drop=FALSE ],copy=FALSE)
  }
  if (type == "together"){
    pms <- unlist(indexProbes(abatch,"both"))
    intensity(abatch)[pms,]  <- normalize.quantiles(intensity(abatch)[pms,,drop=FALSE ],copy=FALSE)
  }
    
  ##this is MIAME we need to decide how to do this properly.
  ##for (i in 1:length(abatch)) {
  ##  history(abatch)[[i]]$name <- "normalized by quantiles"
  ##}

  return(abatch)
}
  
normalize.quantiles <- function(x,copy=TRUE){

  rows <- dim(x)[1]
  cols <- dim(x)[2]

  if (!is.matrix(x)){
    stop("Matrix expected in normalize.quantiles")
  }
  
  #matrix(.C("qnorm_c", as.double(as.vector(x)), as.integer(rows), as.integer(cols))[[1]], rows, cols)

  .Call("R_qnorm_c",x,copy);
}


normalize.AffyBatch.quantiles.robust <- function(abatch, type=c("separate","pmonly","mmonly","together"),weights=NULL,remove.extreme=c("variance","mean","both","none"),n.remove=1,approx.meth = FALSE,use.median=FALSE,use.log2=FALSE) {

  type <- match.arg(type)

  if ((type == "pmonly")|(type == "separate")){
    pms <- unlist(pmindex(abatch))
    intensity(abatch)[pms, ] <- normalize.quantiles.robust(intensity(abatch)[pms, ], weights,remove.extreme,n.remove,approx.meth,use.median,use.log2)
  }
  if ((type == "mmonly")|(type == "separate")){
    mms <- unlist(mmindex(abatch))
    intensity(abatch)[mms, ] <- normalize.quantiles.robust(intensity(abatch)[mms, ],weights,remove.extreme,n.remove,approx.meth,use.median,use.log2)
  }

  if (type == "together"){
    intensity(abatch)  <- normalize.quantiles.robust(intensity(abatch),weights,remove.extreme,n.remove,approx.meth,use.median,use.log2)
  }

  
  
  ##this is MIAME we need to decide how to do this properly.
  ##for (i in 1:length(abatch)) {
  ##  history(abatch)[[i]]$name <- "normalized by quantiles"
  ##}

  return(abatch)
}

normalize.quantiles.robust <- function(x,weights=NULL,remove.extreme=c("variance","mean","both","none"),n.remove=1,approx.meth = FALSE,use.median=FALSE,use.log2=FALSE){
  
  calc.var.ratios <- function(x){
    cols <- dim(x)[2]
    vars <- apply(x,2,var)
    results <- matrix(0,cols,cols)
    for (i in 1:cols-1)
      for (j in (i+1):cols){
        results[i,j] <- vars[i]/vars[j]
        results[j,i] <- vars[j]/vars[i]
      }
    results
  }

  calc.mean.dists <- function(x){
    cols <- dim(x)[2]
    means <- apply(x,2,mean)
    results <- matrix(0,cols,cols)
    for (i in 1:cols-1)
      for (j in (i+1):cols){
        results[i,j] <- means[i] - means[j]
        results[j,i] <- means[j] - means[i]
      }
    results
  }


  remove.extreme <- match.arg(remove.extreme)
  
  rows <- dim(x)[1]
  cols <- dim(x)[2]
  
  if (is.null(weights)){
    weights <- rep(1,cols)
    if (remove.extreme == "variance"){
      var.ratios <- calc.var.ratios(x)
      vars.big <- apply(var.ratios,1,sum)
      vars.small <- apply(var.ratios,2,sum)
      var.adj <- vars.big + vars.small
      remove.order <- order(-var.adj)
      weights[remove.order[1:n.remove]] <- 0
    }
    if (remove.extreme == "mean"){
      means <- abs(apply(calc.mean.dists(x),2,sum))
      remove.order <- order(-means)
      weights[remove.order[1:n.remove]] <- 0
    }
    if (remove.extreme == "both"){
      var.ratios <- calc.var.ratios(x)
      vars.big <- apply(var.ratios,1,sum)
      vars.small <- apply(var.ratios,2,sum)
      var.adj <- vars.big + vars.small
      means <- abs(apply(calc.mean.dists(x),2,sum))
      # by convention we will remove first the most extreme variance, then the most extreme mean
      remove.order <- order(-var.adj)
      weights[remove.order[1]] <- 0
      remove.order <- order(-means)
      weights[remove.order[1]] <- 0
    }
  }
  if (length(weights) != cols){
    stop("Weights vector incorrect length\n")
  }
  if (sum(weights > 0) < 2){
    stop("Need at least two non negative weights\n")
  }
  cat("Chip weights are ",weights,"\n") 
  if (approx.meth == FALSE){
    matrix(.C("qnorm_robust_c",as.double(as.vector(x)),as.double(weights),as.integer(rows),as.integer(cols),as.integer(use.median),as.integer(use.log2))[[1]],rows,cols)
  } else {
    cat("Approximation currently not implemented \nFalling back to standard Quantile method\n")
    matrix(.C("qnorm_robust_c",as.double(as.vector(x)),as.double(weights),as.integer(rows),as.integer(cols),as.integer(use.median),as.integer(use.log2))[[1]],rows,cols)
  }
}
pairs.AffyBatch <- function(x, panel=points, ..., transfo=I, main=NULL, oma=NULL,
                            font.main = par("font.main"), cex.main = par("cex.main"),
                            cex.labels = NULL, 
                            lower.panel=panel, upper.panel=NULL,
                            diag.panel=NULL,
                                        #text.panel = textPanel,
                                        #label.pos = 0.5 + has.diag/3,                                
                            font.labels = 1, row1attop = TRUE, gap = 1) {

  #label1 <- chipNames(x)
  #label2 <- unlist(lapply(history(x), function(z) z$name))
  
  #textPanel <- function(x = 0.5, y = 0.5, txt, cex, font) {
  #  text(x, y, txt, cex = cex, font = font)
  #}
  
  ##labels <- paste(sampleNames(x), unlist(lapply(history(x), function(z) if (is.null(z$name)) "" else z$name)), sep="\n")
  labels <- sampleNames(x)
  ##y <- matrix(intensity(x)[, , seq(along=x)], ncol=length(x))
  y <- intensity(x)
  
  pairs(transfo(y), labels=labels,
        panel=panel, ..., main=main, oma=oma,
        font.main = font.main, cex.main = cex.main,
        lower.panel=lower.panel, upper.panel=upper.panel, diag.panel=diag.panel,
        #text.panel = text.panel,
        #label.pos = label.pos,
        cex.labels = cex.labels, 
        font.labels = font.labels, row1attop = row1attop, gap = gap
        )
   
}
# matdensity <- function(x,
#                        ylab="density", xlab="x", type="l", plot=TRUE,
#                        ...) {
  
#   x.density <- apply(mat, 2, density)

#   all.x <- do.call("cbind", lapply(x.density, function(x) x$x))
#   all.y <- do.call("cbind", lapply(x.density, function(x) x$y))
  
#   if (plot)
#     matplot(all.x, all.y, ylab=ylab, xlab=xlab, ...)
 
#   invisible(list(all.x=all.x, all.y=all.y))                     
                       
# }

plotDensity <- function(mat,
                        ylab="density", xlab="x", type="l", col=1:6,
                        ...) {
  
  x.density <- apply(mat, 2, density)

  all.x <- do.call("cbind", lapply(x.density, function(x) x$x))
  all.y <- do.call("cbind", lapply(x.density, function(x) x$y))
  
  matplot(all.x, all.y, ylab=ylab, xlab=xlab, type=type, col=col, ...)

  invisible(list(all.x=all.x, all.y=all.y))
}
 

plotDensity.AffyBatch <- function(x, col=1:6, log=TRUE,
                                  which=c("pm","mm","both"),
                                  ylab="density",
                                  xlab=NULL,
                                  ...){
  
  Index <- unlist(indexProbes(x, which=which))
  
  x <- intensity(x)[Index, ,drop=FALSE]
  
  if(log){
    x <- log2(x)
    if(is.null(xlab)) xlab <- "log intensity"
  }
  else  if(is.null(xlab)) xlab <- "intensity"
  
  rv <- plotDensity(x, ylab=ylab, xlab=xlab, col=col, ...)

  invisible(rv)
}
plotLocation <- function(x, col="green", pch=22, ...) {
  if (is.list(x)) {
    x <- cbind(unlist(lapply(x, function(x) x[,1])),
               unlist(lapply(x, function(x) x[,2])))
  }
  points(x[,1], x[,2]
         , pch=pch, col=col, ...)
}
plot.ProbeSet <- function(x, which=c("pm", "mm"), xlab="probes", type="l", ylim=NULL, ...) {

  which <- match.arg(which)
  if (which == "pm")
    f <- getMethod("pm", "ProbeSet")
  else
    g <- getMethod("mm", "ProbeSet")
  
  if (is.null(ylim))
    ylim = range(c(f(x)), na.rm=TRUE)
  
  if (is.na(xlab))
    xlab="probes"
  
  matplot(f(x), xlab=xlab, type=type, ylim=ylim, ...)  
}
############################
##MPM Changed delta, Affy SADD states delta as 2e-20
pmcorrect.mas <- function (object, contrast.tau = 0.03, scale.tau = 10, delta = 2.0e-20)
#function (object, contrast.tau = 0.03, scale.tau = 10, delta = 9.536743e-07) 
###########################
{
    all.pps.pm <- pm(object)
    all.pps.mm <- mm(object)
    diff <- log2(all.pps.pm) - log2(all.pps.mm)
    delta <- rep(delta, nrow(diff))
    for (i in 1:ncol(diff)) {
        sb <- tukey.biweight(diff[, i])
        pps.pm <- all.pps.pm[, i]
        pps.mm <- all.pps.mm[, i]
        pps.im <- pps.mm
        j <- (pps.mm >= pps.pm) & (sb > contrast.tau)
        pps.im[j] <- pps.pm[j]/2^sb
        j <- (pps.mm >= pps.pm) & (sb <= contrast.tau)
        pps.im[j] <- pps.pm[j]/2^(contrast.tau/(1 + (contrast.tau - 
            sb)/scale.tau))
	#########################
	#MPM SADD Need to substract the PM-IM, I think this is the culprit
	pm.corrected <- apply(cbind(pps.pm-pps.im, delta), 1, 
            max)
        #pm.corrected <- apply(cbind(pps.pm, pps.im, delta), 1, 
        #    max)
	##########################
        diff[, i] <- pm.corrected
    }
    return(diff)
}
if (debug.affy123) cat("-->initAffyBatch\n")

## Inherits from Affybatch
## The accessor 'intensity' gets what is in the slot 'exprs'
setClass("AffyBatch",
         representation(cdfName="character",
                        nrow="numeric",
                        ncol="numeric"),
         prototype=list(exprs=matrix(nr=0,nc=0),
         se.exprs = matrix(nr=0,nc=0),
         description=new("MIAME"),
         annotation="",
         notes="",
         cdfName="",
         nrow=0,
         ncol=0), contains="exprSet")

#######################################################
### accessors
#######################################################

if (debug.affy123) cat("--->accessors\n")

if (is.null(getGeneric("cdfName")))
    setGeneric("cdfName", function(object)
               standardGeneric("cdfName"))

setMethod("cdfName", "AffyBatch", function(object)
          object@cdfName)

##intensity
if ( !isGeneric("intensity") ) {
  setGeneric("intensity", function(object)
             standardGeneric("intensity"))
} else
cat("intensity is already generic, could be a problem.\n")


setMethod("intensity", signature(object="AffyBatch"),
          function(object) exprs(object))


if( !isGeneric("intensity<-") )
  setGeneric("intensity<-", function(object, value)
             standardGeneric("intensity<-"))

setReplaceMethod("intensity", signature(object="AffyBatch"),
                 function(object, value){
                   exprs(object) <- value
                   colnames(exprs(object)) <- sampleNames(object)
                   return(object)
                 })

##for now, there is no accessor for se.exprs. we could use this to store
##sd, but if no one uses it... why do it

setMethod("length",signature(x="AffyBatch"),
          function(x) ncol(exprs(x))) ##RI: assumes matrices

if(is.null(getGeneric("ncol")))
  setGeneric("ncol")

setMethod("ncol",signature(x="AffyBatch"),
          function(x) x@ncol) ##RI: assumes matrices

if( is.null(getGeneric("nrow")))
  setGeneric("nrow")

  setMethod("nrow",signature(x="AffyBatch"),
            function(x) x@nrow) ##RI: assumes matrices


#######################################################
### methods
#######################################################

##geneNames method
if (debug.affy123) cat("--->geneNames\n")

if( is.null(getGeneric("geneNames") ))
  setGeneric("geneNames", function(object)
             standardGeneric("geneNames"))

setMethod("geneNames",signature("AffyBatch"),
            function(object){
              cdf.envir <- getCdfInfo(object)
              return(ls(env=cdf.envir))
            })


##show method
if (debug.affy123) cat("--->show\n")

setMethod("show", "AffyBatch",
          function(object) {

            ## Location from cdf env
            cdf.env <- try( getCdfInfo(object) )
            if (! inherits(cdf.env, "try-error")) {
              num.ids <- length(ls(env=cdf.env))
            } else {
              warning("missing cdf environment !")
              num.ids <- "???"
            }

            cat("AffyBatch object\n")
            cat("size of arrays=", nrow(object), "x", ncol(object),
                " features (", object.size(object) %/% 1024, " kb)\n", sep="")
            cat("cdf=", object@cdfName,
                " (", num.ids, " affyids)\n",
                sep="")
            cat("number of samples=",length(object),"\n",sep="")
            cat("number of genes=", length(geneNames(object)), "\n",sep="")
            cat("annotation=",object@annotation,"\n",sep="")
            if(length(object@notes)>0)
              if(nchar(object@notes)>0)
                cat("notes=",object@notes,"\n",sep="")
          })


if (is.null(getGeneric("index2xy"))) {
  setGeneric("indexProbes", function(object, which, ...)
             standardGeneric("indexProbes"))
}


## indexProbes
if( is.null(getGeneric("indexProbes")))
  setGeneric("indexProbes", function(object, which, ...)
             standardGeneric("indexProbes"))

setMethod("indexProbes", signature("AffyBatch", which="character"),
          function(object, which=c("pm", "mm","both"),
                   genenames=NULL, xy=FALSE) {

            which <- match.arg(which)

            i.probes <- match(which, c("pm", "mm", "both"))
            ## i.probes will know if "[,1]" or "[,2]"
            ## if both then [,c(1,2)]
            if(i.probes==3) i.probes=c(1,2)

            envir <- getCdfInfo(object)

            if(is.null(genenames))
              genenames <- ls(envir )

            ## shorter code, using the features of multiget
            ## (eventually more readable too)
            ## note: genenames could be confusing (the same gene can be
            ## found in several affyid (ex: the 3' and 5' controls)

            ans <-  multiget(genenames, pos, envir, iffail=NA)

            ## this kind of thing could be included in 'multiget' as
            ## an extra feature. A function could be specified to
            ## process what is 'multi'-get on the fly
            for (i in seq(along=ans)) {

              if ( is.na(ans[[i]][1]) )
                next

              ##as.vector cause it might be a matrix if both
              tmp <- as.vector(ans[[i]][, i.probes])


              if (xy) {
                warning("flag 'xy' is deprecated (because confusing)")
                x <- tmp %% nrow(object)
                x[x == 0] <- nrow(object)
                y <- tmp %/% nrow(object) + 1
                tmp <- cbind(x, y)
              }

              ans[[i]] <- tmp
            }

            return(ans)
          })


  ##pmindex method
if( is.null(getGeneric("pmindex")))
  setGeneric("pmindex", function(object,...)
             standardGeneric("pmindex"))

##wrapper
setMethod("pmindex", "AffyBatch",
            function(object, genenames=NULL, xy=FALSE)
            indexProbes(object, "pm", genenames=genenames, xy=xy))

  ##mmindex method
if( is.null(getGeneric("mmindex")))
  setGeneric("mmindex", function(object,...)
             standardGeneric("mmindex"))

##wrapper
setMethod("mmindex", "AffyBatch",
          function(object,genenames=NULL, xy=FALSE)
          indexProbes(object, "mm", genenames=genenames, xy=xy))


##probeNames method
if( is.null(getGeneric("probeNames")))
  setGeneric("probeNames", function(object, ...)
             standardGeneric("probeNames"))

setMethod("probeNames","AffyBatch",
          function(object, genenames=NULL, mm=FALSE){
            if(mm) Index <- mmindex(object,genenames)
            else Index <- pmindex(object,genenames)
            reps <- unlist(lapply(Index,length))
            rep(names(Index),reps)
          })


if( is.null(getGeneric("probes")) )
  setGeneric("probes", function(object, ...)
             standardGeneric("probes"))

setMethod("probes", signature("AffyBatch"),
          function(object, which=c("pm", "mm"),
                   genenames=NULL, LISTRUE=FALSE, drop=FALSE){

            which <- match.arg(which)

            index <- indexProbes(object, which, genenames)

            if(LISTRUE)
              ans <- lapply(index, function(i) exprs(object)[i, ,drop=drop])
            else{
              index <- unlist(index)
              ans <- exprs(object)[index, ,drop=drop]
              colnames(ans) <- sampleNames(object)
              rownames(ans) <- names(index)
            }

            return(ans)
          })

##pm method
if( is.null(getGeneric("pm") ))
  setGeneric("pm", function(object, ...)
             standardGeneric("pm"))

setMethod("pm","AffyBatch",
          function(object, genenames=NULL, LISTRUE=FALSE)
          probes(object, "pm", genenames, LISTRUE=LISTRUE))

if( is.null(getGeneric("pm<-") ))
  setGeneric("pm<-", function(object, value)
             standardGeneric("pm<-"))
setReplaceMethod("pm", "AffyBatch", function(object, value){
  Dimnames <- dimnames(intensity(object))
  pmIndex <- unlist(pmindex(object))
  intensity(object)[pmIndex,] <- value
  dimnames(intensity(object)) <- Dimnames
  object
})



##mm method
if( is.null(getGeneric("mm") ))
  setGeneric("mm", function(object, ...)
             standardGeneric("mm"))

setMethod("mm",signature("AffyBatch"),
          function(object, genenames=NULL, LISTRUE=FALSE) probes(object, "mm", genenames, LISTRUE=LISTRUE))

if( is.null(getGeneric("mm<-") ))
  setGeneric("mm<-", function(object, value)
             standardGeneric("mm<-"))

setReplaceMethod("mm", "AffyBatch", function(object, value){
  Dimnames <- dimnames(intensity(object))
  mmIndex <- unlist(mmindex(object))
  intensity(object)[mmIndex,] <- value
  dimnames(intensity(object)) <- Dimnames
  object
})

###probeset
if( is.null(getGeneric("probeset") ))
  setGeneric("probeset", function(object, ...)
             standardGeneric("probeset"))

setMethod("probeset", "AffyBatch", function(object, genenames=NULL,
                                            locations=NULL){
  oldoptions <- getOption("BioC")

  if(is.null(locations)) ##use info in cdf
    envir <- getCdfInfo(object)
  else{
    ##if the user gives a list of locations let them use that as enviromnet
    envir <- new.env()
    multiassign(names(locations), locations, envir)
    object@cdfName <- "envir"
    newoptions <- oldoptions
    newoptions$affy$probesloc[[1]]$what <- "environment"
    newoptions$affy$probesloc[[1]]$where <- parent.env(envir)
    options("BioC"=newoptions)
  }
  if(is.null(genenames))
    genenames <- ls(envir)

  p.pps <- vector("list", length(genenames))
  names(p.pps) <- genenames

  for (i in seq(along=genenames)) {

    i.pm <- indexProbes(object, "pm", genenames[i])[[1]]
    if (is.na(i.pm)[1])
      intensity.pm <- matrix()
    else
      intensity.pm <- intensity(object)[i.pm, , drop=FALSE]

    i.mm <- indexProbes(object, "mm", genenames[i])[[1]]
    if (is.na(i.mm)[1])
      intensity.mm <- matrix()
    else
      intensity.mm <- intensity(object)[i.mm, , drop=FALSE]

    p.pps[[i]] <- new("ProbeSet", id = genenames[i], pm = intensity.pm, mm = intensity.mm)
  }

  options("BioC"=oldoptions)
  return(p.pps)
})

if (debug.affy123) cat("--->[[\n")


##[[: no more [[, because no more cel class
# setMethod("[[", "AffyBatch",
#           function(x, i, j, ...) { ##no need for j
#             return(new("Cel",
#                        intensity = matrix(intensity(x)[, i], ncol(x), nrow(x)),
#                        name = sampleNames(x)[i],
#                        cdfName = x@cdfName,
#                        history = description(x)@preprocessing))
#           })

##[[ we need replacement that takes an entry by the Cel in value

##[ subseting. can only happen by sample. for now not by gene
setMethod("[", "AffyBatch", function(x, i, j,..., drop=FALSE) {
  if( !missing(i) ) {
    phenoData(x) <- phenoData(x)[i, , ..., drop=FALSE]
    intensity(x) <- intensity(x)[ ,i, ..., drop=FALSE]
  }
  return(x)
})

setReplaceMethod("[", "AffyBatch", function(x, i, j,..., value) {
  phenoData(x)[i,, ...] <- phenoData(value)[i, , ..., drop=FALSE]
  intensity(x)[,i]      <- intensity(value)[ ,i,... , drop=FALSE]
  return(x)
})


## --- bg.correct

if (debug.affy123) cat("--->bg.correct\n")

if( is.null(getGeneric("bg.correct") ))
  setGeneric("bg.correct", function(object, method, ...)
             standardGeneric("bg.correct"))

setMethod("bg.correct", signature(object="AffyBatch", method="character"),
          function(object, method=getOption("BioC")$affy$bgcorrect.method, ...) {

            ## simple for system to let one add background correction methods
            ## relies on naming convention

            method <- match.arg(method, bgcorrect.methods)

            methodname <- paste("bg.correct.", method, sep="")

            if (! exists(methodname))
              stop(paste("Unknown method (cannot find function", methodname, ")"))

            r <- do.call(methodname, alist(object, ...))

            return(r)
          })


## --- normalize.methods
if( is.null(getGeneric("normalize.methods")))
  setGeneric("normalize.methods", function(object)
             standardGeneric("normalize.methods"))


setMethod("normalize.methods", signature(object="AffyBatch"),
          function(object) {
            normalize.AffyBatch.methods
          })

  ## ---normalize
if (is.null(getGeneric("normalize")))
  setGeneric("normalize", function(object, ...) standardGeneric("normalize"))

  setMethod("normalize", signature(object="AffyBatch"),
            function(object, method=getOption("BioC")$affy$normalize.method, ...) {
              method <- match.arg(method, normalize.AffyBatch.methods)
              if (is.na(method))
                stop("unknown method")
              method <- paste("normalize.AffyBatch", method, sep=".")
              object <- do.call(method, alist(object, ...))
              ## collect info in the attribute "normalization"
              preproc <- c(description(object)@preprocessing,
                           list(normalization = attr(object, "normalization")))
              attr(object, "normalization") <- NULL
              ## and store it in MIAME
              MIAME <- description(object)
              MIAME@preprocessing <- preproc
              description(object) <- MIAME
              ##
              return(object)
            })


## --- expression value computation
if (debug.affy123) cat("--->computeExprSet\n")
if( is.null(getGeneric("computeExprSet")))
  setGeneric("computeExprSet",
             function(x, pmcorrect.method, summary.method, ...)
             standardGeneric("computeExprSet"))

setMethod("computeExprSet", signature(x="AffyBatch", pmcorrect.method="character", summary.method="character"),
          function(x, pmcorrect.method, summary.method, ids=NULL,
                   verbose=TRUE, summary.param=list(),
                   pmcorrect.param=list())
          {

            pmcorrect.method<- match.arg(pmcorrect.method, pmcorrect.methods)
            summary.method <- match.arg(summary.method, express.summary.stat.methods)

            n <- length(x)

            ## if NULL compute for all
            if (is.null(ids))
              ids <- geneNames(x)

            m <- length(ids)
            pps.warnings <- vector("list", length=m)

            ## cheap trick to (try to) save time
            c.pps <- new("ProbeSet",
                         pm=matrix(),
                         mm=matrix())

            ## matrix to hold expression values
            exp.mat <- matrix(NA, m, n)
            se.mat <- matrix(NA, m, n)

            if (verbose) {
              cat(m, "ids to be processed\n")
              countprogress <- 0
            }

            ## loop over the ids
            mycall <- as.call(c(getMethod("express.summary.stat",
                                          signature=c("ProbeSet","character", "character")),
                                list(c.pps, pmcorrect=pmcorrect.method, summary=summary.method,
                                     summary.param=summary.param, pmcorrect.param=pmcorrect.param))
                              )
            ##only one character cause no more bg correct
            ##bg.correct=bg.method, param.bg.correct=bg.param,

            ##WHy not show error? took it out cause sometimes we
            ##get errors and couldnt see them.
            ##options(show.error.messages = FALSE)
            ##on.exit(options(show.error.messages = TRUE))

            CDFINFO <- getCdfInfo(x) ##do it once!

            for (i in seq(along=ids)) {

              id <- ids[i]

              if (verbose) {
                if ( round(m/10) == countprogress) {
                  cat(".")
                  countprogress <- 0
                }
                else
                  countprogress <- countprogress + 1
              }
              ## locations for an id
              loc <- get(id, envir=CDFINFO)
              l.pm <- loc[, 1]
              if (ncol(loc) == 2)
                l.mm <- loc[ ,2]
              else
                l.mm <- integer()

              np <- length(l.pm)

              ##names are skipped

              c.pps@pm <- intensity(x)[l.pm, , drop=FALSE]
              c.pps@mm <- intensity(x)[l.mm, , drop=FALSE]

              ## generate expression values
              ## (wrapped in a sort of try/catch)
              mycall[[2]] <- c.pps
              ev <- try(eval(mycall))

              if (! inherits(ev, "try-error")) {
                exp.mat[i, ] <- ev$exprs
                se.mat[i,] <- ev$se.exprs
                ##
              } else {
                pps.warnings[[i]] <- "Error"
                ##warning(paste("Error with affyid:", id))
              }

            }

            ##options(show.error.messages = TRUE)
            ## on.exit(NULL)

            if (verbose) cat("\n")

            ## instance exprSet
            ##if (verbose) cat("instancianting an exprSet.....")
            dimnames(exp.mat) <- list(ids, sampleNames(x))
            dimnames(se.mat) <- list(ids, sampleNames(x))
            eset <- new("exprSet",
                        exprs=exp.mat,
                        se.exprs=se.mat,
                        phenoData=phenoData(x),
                        description=description(x),
                        annotation=annotation(x),
                        notes=c(notes(x)))
            ##if (verbose) cat(".....done.\n")

            attr(eset, "pps.warnings") <- pps.warnings
            return(eset)
            ##return(list(exprSet=eset, pps.warnings=pps.warnings))
          })


##some methods i was asked to add

if( is.null(getGeneric("image")))
  setGeneric("image")

setMethod("image",signature(x="AffyBatch"),
          function(x, transfo=log, col=gray(c(0:64)/64),xlab="",ylab="", ...){
            scn <- prod(par("mfrow"))
            ask <- dev.interactive()
            which.plot <- 0

            NCOL <- ncol(x)
            NROW <- nrow(x)

            for(i in 1:length(sampleNames(x))){
              which.plot <- which.plot+1;
              if(trunc((which.plot-1)/scn)==(which.plot-1)/scn && which.plot>1 && ask)  par(ask=TRUE)
              m <- x@exprs[,i]
              if (is.function(transfo)) {
                m <- transfo(m)
              }

              image(1:NROW, 1:NCOL, matrix(m,nrow=NROW,ncol=NCOL),
                    col=col, main=sampleNames(x)[i],
                    xlab=xlab, ylab=ylab, ...)
              par(ask=FALSE)}
          })


###boxplot
if( is.null(getGeneric("boxplot")))
  setGeneric("boxplot")

setMethod("boxplot",signature(x="AffyBatch"),
          function(x,which="both",range=0,...){
            tmp <- description(x)
            if(class(tmp)=="MIAME") main <- tmp@title

            tmp <- unlist(indexProbes(x,which))
            tmp <- tmp[seq(1,length(tmp),len=5000)]

            boxplot(data.frame(log2(intensity(x)[tmp,])),main=main,range=range, ...)
          })

###hist
if (debug.affy123) cat("--->hist\n")

if( is.null(getGeneric("hist")) )
  setGeneric("hist")

setMethod("hist",signature(x="AffyBatch"),function(x,...) plotDensity.AffyBatch(x,...))


if( is.null(getGeneric("mas5calls")) )
  setGeneric("mas5calls", function(object,...) standardGeneric("mas5calls"))

setMethod("mas5calls",signature(object="AffyBatch"),
          function(object,...) mas5calls.AffyBatch(object,...))


##like for exprSet

"$.AffyBatch" <- function(affybatch, val)
    (pData(affybatch))[[as.character(val)]]

 ## A ProbeSet holds probe values for a probe pairs set(*) accross a batch of experiments.
  ## methods 'express.summary.stat' returns of expression value per experiement in the
  ## batch, and 'bg.correct' does background correction (in some sense... the MM probes
  ## were created to measure unspecific hybridization. People thought that doing
  ## PM - MM would remove background noise. The method 'bg.correct' accepts extra parameters
  ## through '...' (can be used to pass background correction parameters common to different
  ## ProbeSet)
  ##
  ## -
  ## (*) : a probe pair set is the set of probes pairs(**) related to an affyid. Generally a
  ##       a probe pair set has 20 elements.
  ## (**): a probe pair (or atom) is a pair of PM/MM values
  ##

if (debug.affy123) cat("-->initProbeSet\n")

setClass("ProbeSet",
         representation(id="character", pm="matrix", mm="matrix"),
         prototype=list(pm=matrix(), mm=matrix()))

setMethod("show", "ProbeSet",
          function(object) {
            cat("ProbeSet object:\n")
            cat("  id=", object@id, "\n", sep="")
            cat("  pm=", nrow(object@pm), "probes x ", ncol(object@pm), " chips\n")
          })

##DEBUG: what to do with that ?
## --> with what ?

if( is.null(getGeneric("colnames")))
                                        #setGeneric("colnames", function(x, do.NULL, prefix)
  setGeneric("colnames")

##for consistency also use sampleNames
if( is.null(getGeneric("sampleNames")))
  setGeneric("sampleNames", function(object)
             standardGeneric("sampleNames"))
setMethod("sampleNames", "ProbeSet",
          function(object) colnames(object))

setMethod("colnames", signature(x="ProbeSet"),
          function(x ,do.NULL=FALSE, prefix="row") {
            
            cnames<-colnames(pm(x))
            
            if (is.null(cnames)) {
              
              if (do.NULL) {
                warning("No column names for ProbeSet")
              }
              else {
                cnames <- paste(prefix, 1:ncols(x@pm))
              }
              
            }
            return(cnames)
          })

## pm
if( is.null(getGeneric("pm")))
  setGeneric("pm", function(object) standardGeneric("pm"))

setMethod("pm", "ProbeSet", function(object) object@pm)

if( is.null(getGeneric("pm<-")))
  setGeneric("pm<-", function(object, value) standardGeneric("pm<-"))

setReplaceMethod("pm", signature=c("ProbeSet", "matrix"),
                 function(object, value) {
                   if (! all(dim(value) == dim(object@mm)))
                     stop("dimension mismatch between 'pm' and 'mm'")
                   object@pm <- value
                 })

## mm
if( is.null(getGeneric("mm")))
  setGeneric("mm", function(object) standardGeneric("mm"))

setMethod("mm", "ProbeSet", function(object) object@mm)


if( is.null(getGeneric("mm<-")))
  setGeneric("mm<-", function(object, value) standardGeneric("mm<-"))

setReplaceMethod("mm", signature=c("ProbeSet", "matrix"),
                 function(object, value) {
                   if (sum(dim(value) == dim(object@mm)) != 2)
                     stop("dimension mismatch between 'pm' and 'mm'")
                   object@mm <- value
                 })

## method express.summary.stat
if( is.null(getGeneric("express.summary.stat")))
  setGeneric("express.summary.stat", function(x, pmcorrect, summary, ...)
             standardGeneric("express.summary.stat"))

setMethod("express.summary.stat",signature(x="ProbeSet",  pmcorrect="character", summary="character"),
          function(x, pmcorrect, summary, summary.param=list(), pmcorrect.param=list()) {
            
            pmcorrect <- match.arg(pmcorrect, pmcorrect.methods)
            summary  <- match.arg(summary, express.summary.stat.methods)
            
            ## simple for system to let one add background correction methods
            ## relies on naming convention
            pmcorrect.methodname <- paste("pmcorrect.", pmcorrect, sep="")
            summary.methodname <- paste("generateExprVal.method.", summary, sep="")
            
            if (! exists(summary.methodname))
              stop(paste("Unknown method (cannot find function", summary.methodname, ")"))
            if (! exists(pmcorrect.methodname))
              stop(paste("Unknown method (cannot find function", pmcorrect.methodname, ")"))
            
            ## NOTE: this could change...
                                        #m <- do.call(bg.correct, c(alist(x@pm, x@mm), param.bg.correct))
            pm.corrected <- do.call(pmcorrect.methodname, c(alist(x), pmcorrect.param))
            r <- do.call(summary.methodname, c(alist(pm.corrected), summary.param))
            
            ##DEBUG: name stuff to sort
                                        #names(r) <- names(allprobes)
            
            return(r)
          })

if( is.null(getGeneric("barplot")))
  setGeneric("barplot")

setMethod("barplot",signature(height="ProbeSet"),function(height,...) barplot.ProbeSet(height,...))

if( is.null(getGeneric("mas5calls")) )
  setGeneric("mas5calls", function(object,...) standardGeneric("mas5calls"))

setMethod("mas5calls",signature(object="ProbeSet"),
          function(object,...) mas5calls.ProbeSet(object,...))

#############################################################
##
## read.affybatch.R
##
## Adapted by B. M. Bolstad from read.affybatch in the affy
## package version 1.2.  The goal is a faster, less memory hungry
## ReadAffy. To do this we will shunt more work off to
## the c code.
##
## History
## Jun 13-15 Intial version
## Jun 16    Verbose flag passed to C routine
## Jun 17    New method for checking header of first cel
##           file.
## Jul 7     Added the function read.probematrix which
##           reads in PM, MM or both into matrices
## Sep 28    changed name from read.affybatch2 to read.affybatch
##           and cleaned up some old commented stuff
#############################################################


read.affybatch <- function(..., filenames=character(0),
                           ##sd=FALSE,
                           phenoData=new("phenoData"),
                           description=NULL,
                           notes="",
                           compress = getOption("BioC")$affy$compress.cel,
                           rm.mask = FALSE, rm.outliers=FALSE, rm.extra=FALSE,
                           verbose = FALSE) {
  
  auxnames <- as.list(substitute(list(...)))[-1]
  filenames <- .Primitive("c")(filenames, auxnames)
  
  n <- length(filenames)
  
  ## error if no file name !
  if (n == 0)
    stop("No file name given !")
  
  pdata <- pData(phenoData)
  ##try to read sample names form phenoData. if not there use CEL filenames
  if(dim(pdata)[1] != n) {
    ##if empty pdata filename are samplenames
    warning("Incompatible phenoData object. Created a new one.\n")
    
    samplenames <- sub("^/?([^/]*/)*", "", unlist(filenames), extended=TRUE)
    pdata <- data.frame(sample=1:n, row.names=samplenames)
    phenoData <- new("phenoData",pData=pdata,varLabels=list(sample="arbitrary numbering"))
  }
  else samplenames <- rownames(pdata)
  
  if (is.null(description))
    {
      description <- new("MIAME")
      description@preprocessing$filenames <- filenames
      description@preprocessing$affyversion <- library(help=affy)$info[[2]][[2]][2]
    }
  ## read the first file to see what we have
  if (verbose) cat(1, "reading",filenames[[1]],"...")

  headdetails <- .Call("ReadHeader",filenames[[1]],compress)

  #print(headdetails)
  
  
  
  ##now we use the length
  dim.intensity <- headdetails[[2]]   ##dim(intensity(cel))
  ##and the cdfname as ref
  ref.cdfName <- headdetails[[1]]   #cel@cdfName
  
  if (verbose)
    cat(paste("instanciating an AffyBatch (intensity a ", prod(dim.intensity), "x", length(filenames), " matrix)...", sep=""))
  

 
  if (verbose)
    cat("done.\n")

  #### this is where the code changes from the original read.affybatch.
  #### what we will do here is read in from the 1st to the nth CEL file
  
  return(new("AffyBatch",
               exprs  = .Call("read_abatch",filenames,compress, rm.mask,
                 rm.outliers, rm.extra, ref.cdfName, dim.intensity,verbose),       
               ##se.exprs = array(NaN, dim=dim.sd),
               cdfName    = ref.cdfName,   ##cel@cdfName,
               phenoData  = phenoData,
               nrow       = dim.intensity[1],
               ncol       = dim.intensity[2],
               annotation = cleancdfname(ref.cdfName, addcdf=FALSE),
               description= description,
               notes      = notes))
}





######################################################################################

read.probematrix <- function(..., filenames = character(0), phenoData = new("phenoData"),
    description = NULL, notes = "", compress = getOption("BioC")$affy$compress.cel,
    rm.mask = FALSE, rm.outliers = FALSE, rm.extra = FALSE, verbose = FALSE,which="pm"){

  auxnames <- as.list(substitute(list(...)))[-1]
  filenames <- .Primitive("c")(filenames, auxnames)

  match.arg(which,c("pm","mm","both"))
  
  if (verbose)
        cat(1, "reading", filenames[[1]], "to get header informatio")
    headdetails <- .Call("ReadHeader", filenames[[1]], compress)
    dim.intensity <- headdetails[[2]]
    ref.cdfName <- headdetails[[1]]
  
  Data <- new("AffyBatch", cdfName = ref.cdfName, annotation = cleancdfname(ref.cdfName,addcdf = FALSE))
  
  cdfInfo <- multiget(ls(getCdfInfo(Data)),-1,getCdfInfo(Data))
  .Call("read_probeintensities", filenames,
        compress, rm.mask, rm.outliers, rm.extra, ref.cdfName,
        dim.intensity, verbose, cdfInfo,which)
}


list.celfiles <-   function(...){
  files <- list.files(...)
  return(files[grep("\.[cC][eE][lL]\.gz$|\.[cC][eE][lL]$", files)])
}

###this is user friendly wrapper for read.affybatch
ReadAffy <- function(..., filenames=character(0),
                     widget=getOption("BioC")$affy$use.widgets,
                     compress=getOption("BioC")$affy$compress.cel,
                     celfile.path=getwd(),
                     sampleNames=NULL,
                     phenoData=NULL,
                     description=NULL,
                     notes="",
                     rm.mask=FALSE, rm.outliers=FALSE, rm.extra=FALSE,
                     verbose=FALSE) {
  
  ##first figure out filenames
  auxnames <- unlist(as.list(substitute(list(...)))[-1])

  if (widget){
    require(tkWidgets)
    widgetfiles <- fileBrowser(textToShow="Choose CEL files",
                               testFun=hasSuffix("[cC][eE][lL]"))
  }
  else
    widgetfiles <- character(0)
  
  filenames <- .Primitive("c")(filenames, auxnames, widgetfiles)
  
  if(length(filenames)==0) filenames <- list.celfiles(celfile.path,full.names=TRUE)
  
  if(length(filenames)==0) stop("No cel filennames specified and no cel files in specified directory:",celfile.path,"\n")
  
  
  ##now assign sampleNames if phenoData not given
  if(is.null(phenoData)){
    if(is.null(sampleNames)){
      if(widget){
        require(tkWidgets)
        tksn <- tkSampleNames(filenames=filenames)
        sampleNames <- tksn[,1]
        ##notice that a description of the files is ingored for now
        ##soon to go into MIAME
      }
      else{
        sampleNames <- sub("^/?([^/]*/)*", "", filenames, extended=TRUE)
      }
    }
    else{
      if(length(sampleNames)!=length(filenames)){
        warning("sampleNames not same length as filenames. Using filenames as sampleNames instead\n")
        sampleNames <- sub("^/?([^/]*/)*", "", filenames, extended=TRUE)
      }
    }
  }
  
  ##now get phenoData
  if(is.character(phenoData)) ##if character read file
    phenoData <- read.phenoData(filename=phenoData)
  else{
    if(class(phenoData)!="phenoData"){
      if(widget){
        require(tkWidgets)
        phenoData <- read.phenoData(sampleNames=sampleNames,widget=TRUE)
      }
      else
        phenoData <- read.phenoData(sampleNames=sampleNames,widget=FALSE)
    }
  }
  
  ##get MIAME information
  if(is.character(description)){
    description <- read.MIAME(filename=description,widget=FALSE)
  }
  else{
    if(class(description)!="MIAME"){
      if(widget){
        require(tkWidgets)
        description <- read.MIAME(widget=TRUE)
      }
      else
        description <- new("MIAME")
    }
  }
  
  ##MIAME stuff
  description@preprocessing$filenames <- filenames
  if(exists("tksn")) description@samples$description <- tksn[,2]
  description@preprocessing$affyversion <- library(help=affy)$info[[2]][[2]][2]

  ##and now we are ready to read cel files
  return(read.affybatch(filenames=filenames,
                        phenoData=phenoData,
                        description=description,
                        notes=notes,
                        compress=compress,
                        rm.mask=rm.mask,
                        rm.outliers=rm.outliers,
                        rm.extra=rm.extra,
                        verbose=verbose))
}








######################################################
#
# rma - RMA interface to c code
#
# the RMA method implemented in c code
#
# this code serves as interface to the c code.
# currently
# implemented (version 0.25) background correction
#
# Background correction code has been added.
#
# note this function does not leave the supplied
# AffyBatch unchanged if you select DESTRUCTIVE=TRUE. this is 
# for memory purposes but can be quite
# dangerous if you are not careful. Use destructive=FALSE if this is
# deemed likely to be a problem.
#
# UPDATE: note that the affybatch is now not affected if you use
# destructive=TRUE and you might actually save a little memory.
# the destructive refers only to Plobs, which would be destroyed.
#
########################################################

rma <- function(object,subset=NULL, verbose=TRUE, destructive = TRUE,normalize=TRUE,background=TRUE,bgversion=2,...){

  rows <- length(probeNames(object))
  cols <- length(object)
 
  ngenes <- length(geneNames(object))
  
  #background correction
  bg.dens <- function(x){density(x,kernel="epanechnikov",n=2^14)}

  if (destructive){
  	exprs <- .Call("rma_c_complete",pm(object),mm(object),probeNames(object),ngenes,body(bg.dens),new.env(),normalize,background,bgversion)
  } else {
	exprs <- .Call("rma_c_complete_copy",pm(object),mm(object),probeNames(object),ngenes,body(bg.dens),new.env(),normalize,background,bgversion)
  }
  colnames(exprs) <- sampleNames(object)
  se.exprs <- array(NA, dim(exprs)) # to be fixed later, besides which don't believe much in nominal se's with medianpolish
  
  phenodata <- phenoData(object)
  annotation <- annotation(object)
  description <- description(object) 
  notes <- notes(object)
  
  new("exprSet", exprs = exprs, se.exprs = se.exprs, phenoData = phenodata, 
       annotation = annotation, description = description, notes = notes)
}
split.AffyBatch <- function(x, f) {

  sp <- getMethod("split", c("exprSet", "vector"))(x, f)

  sp.abatch <- lapply(sp, function(eset) {
    new("AffyBatch", cdfName = x@cdfName,
        nrow = nrow(x), ncol= ncol(x))
  })

  for (i in seq(along = sp)) {
    ## function to cast classes included in next release
    exprs(sp.abatch[[i]]) <- exprs(sp[[i]])
    se.exprs(sp.abatch[[i]]) <- se.exprs(sp[[i]])
    annotation(sp.abatch[[i]]) <- annotation(sp[[i]])
    phenoData(sp.abatch[[i]]) <- phenoData(sp[[i]])
    description(sp.abatch[[i]]) <- description(sp[[i]])
    notes(sp.abatch[[i]]) <- notes(sp[[i]])
    ## try to lower memory usage
    sp[[i]] <- NA
    gc()
  }

  return(sp.abatch)
  
}


###these are summary functions they take matrices of probes x chips
###and return expression and se (when applicable)

##DEBUG: appending the se to the expression values in a same vector
##       is too much hackish (I think)... we need to think about something
##       better

avdiff <- function(x,constant=3){
  e <- apply(x,2,function(y){
    o <- order(y)
    yy <- y[-c(o[1],o[length(y)])] #take out biggest and smallest
    if(length(yy)<2)  # SK, some genes have only one probe
      mean(y)
    else
      mean(y[abs(y-mean(yy))<constant*sd(yy)])
  })
  list(exprs=e,se.exprs=apply(x,2,sd)/sqrt(nrow(x)))
}

li.wong <- function(data.matrix,remove.outliers=TRUE,
                    normal.array.quantile=0.5,
                    normal.resid.quantile=0.9,
                    large.threshold=3,
                    large.variation=0.8,
                    outlier.fraction=0.14,
                    delta = 1e-06,maxit=50,outer.maxit=50,verbose=FALSE){

  e <-  fit.li.wong(t(data.matrix),remove.outliers,normal.array.quantile,normal.resid.quantile,large.threshold,large.variation,outlier.fraction,delta,maxit,outer.maxit,verbose)
  list(exprs=e$theta,se.exprs=e$sigma.theta)
}


medianpolish <- function(x, ...){
  tmp <- medpolish(log2(x), trace.iter=FALSE, ...)
  ##rough estimate
  sigma <- 1.483*median(abs(as.vector(tmp$residuals)))/sqrt(nrow(x))
  list(exprs=tmp$overall + tmp$col,se.exprs=rep(sigma, ncol(x)))
}

tukeybiweight <- function(x, c=5, epsilon=0.0001){
  tmp <- function(x, c=5, epsilon=0.0001)
    {
      m <- median(x)
      s <- median(abs(x - m))
      u <- (x - m) / (c * s + epsilon)
      w <- rep(0, length(x))
      i <- abs(u) <= 1
      w[i] <- ((1 - u^2)^2)[i]
      t.bi <- sum(w * x) / sum(w)
      return(t.bi)
    }
  list(exprs=apply(log2(x),2,tmp),se.exprs=rep(NA,ncol(x)))
}


tukey.biweight <- function(x, c=5, epsilon=0.0001)
  {
    m <- median(x)
    s <- median(abs(x - m))
    u <- (x - m) / (c * s + epsilon)
    w <- rep(0, length(x))
    i <- abs(u) <= 1
    w[i] <- ((1 - u^2)^2)[i]
    t.bi <- sum(w * x) / sum(w)
    return(t.bi)
  }

tukeybiweight <-  function(x, c=5, epsilon=0.0001)
  list(exprs=apply(x,2,tukey.biweight,c=c,epsilon=epsilon),se.exprs=rep(NA,ncol(x)))


##this function changes the affymetrix cdf file name to the Bioconductor
##annotation name for that cdf file
## note: we had a hard time finding exact rules to match what is in the
## CEL file with what is in the CDF file
## ex: CEL says 'ecoli' while CDF says 'ecoligenome'
## or: CEL says '' while CDF says hu6800.1sq
cleancdfname <- function(cdfname, addcdf=TRUE) {
  if( !is.character(cdfname) )
                stop(paste("invalid CDF name:", cdfname))
  if ( nchar(cdfname)[1] == 0 )
               stop("supplied cdf name has zero length")
  i <- match(cdfname, mapCdfName$inCDF)
  if (is.na(i)) {
    tmp <- tolower(cdfname) #make lower case
    tmp <- gsub("_", "", tmp) #take out underscore
    tmp <- gsub("-", "", tmp) #take out underscore
    tmp <- gsub("\ ", "", tmp) ##take out spaces
    if(addcdf) tmp <- paste(tmp, "cdf", sep="")
  } else {
    tmp <- mapCdfName$inBioC[1]
  }
  return(tmp)
}

##this function gets the cdf from a celfile
whatcdf <- function(filename, compress=getOption("BioC")$affy$compress.cel)
  return(.Call("ReadHeader",filename,compress)[[1]])
 

  
xy2indices <- function(x, y, nr=NULL, cel=NULL, abatch=NULL) {
  if (any(x <= 0) || any(y <= 0))
    stop("Xs and Ys must start at 1 (please refer to the help file) !")
  ct <- sum(c(is.null(nr), is.null(cel), is.null(abatch)))
  if (ct != 2)
    stop("One and only one of 'nr', 'cel', 'abatch' should be specified.")
  if (! is.null(cel))
    stop("Cel class no longer supported") #nr <- nrow(intensity(cel))
  if (! is.null((abatch)))
    nr <- nrow(abatch)
  
  return(x + nr * (y - 1))
}

indices2xy <- function(i, nr=NULL, cel=NULL, abatch=NULL) {
  if (any(i)<= 0)
    stop("Indices must start at 1 (please refer to the help file) !")
  
  ct <- sum(c(is.null(nr), is.null(cel), is.null(abatch)))
  
  if (ct != 2)
    stop("One and only one of 'nr', 'cel', 'abatch' should be specified.")
  if (! is.null(cel))
    stop("Cel class no longer supported")#    nr <- nrow(intensity(cel))
  if (! is.null((abatch)))
    nr <- nrow(abatch)
  
  x <- i %% nr
  x[x == 0] <- nr
  y <- (i-1) %/% nr + 1
  xy <- cbind(x, y)
  colnames(xy) <- c("x", "y")
  return(xy)
}
.initNormalize <- function(where, all.affy) {
  if (debug.affy123) cat("-->detecting normalization methods from naming convention\n")

  ## this could move into the respective methods of AffyBatch later

  start <- nchar("normalize.AffyBatch.")
  assign("normalize.AffyBatch.methods",
         substr(all.affy[grep("normalize\.AffyBatch\.*", all.affy)], start+1, 100),
         envir=as.environment(where))
}

.initExpression <- function(where, all.affy) {
  if (debug.affy123) cat("-->detecting expression value methods from naming convention\n")

  ## the first one is deprecated (well... "should be"...)
  assign("generateExprSet.methods",
         substr(all.affy[grep("generateExprVal\.method\.*", all.affy)], 24,100),
         envir=as.environment(where))
  assign("express.summary.stat.methods",
         substr(all.affy[grep("generateExprVal\.method\.*", all.affy)], 24,100),
         envir=as.environment(where))
}

.initBackgroundCorrect <- function(where, all.affy) {
  if (debug.affy123) cat("-->detecting background correction methods from naming convention\n")
  ##assign("bg.correct.methods",
  ##       substr(ls(where)[grep("bg.correct\.*", ls(where))], 12,100),
  ##       envir=as.environment(where))
  start <- nchar("bg.correct.")
  assign("bgcorrect.methods",
         substr(all.affy[grep("bg\.correct\.", all.affy)], start+1, 100),
         envir=as.environment(where))
       }

.initPmCorrect <- function(where, all.affy) {
  if (debug.affy123) cat("-->detecting pm correction methods from naming convention\n")
  start <- nchar("pmcorrect.")
  assign("pmcorrect.methods",
         substr(all.affy[grep("pmcorrect\.*", all.affy)], start+1, 100),
         envir=as.environment(where))
}

.initMapCdfName <- function(where) {
  filepath <- file.path(.path.package("affy"), "data", "mapCdfName.tab")
  mapCdfName <- read.table(filepath, colClasses=rep("character", 3), quote="\"", sep="\t", comment="#", row.names=NULL, header=TRUE)
  assign("mapCdfName", mapCdfName, envir=as.environment(where))
}

.setAffyOptions <- function(affy.opt=NA) {

  if (! any(is.na(affy.opt))) {
    if (class(affy.opt) != "BioCPkg")
      stop("obviously invalid package options !")

    BioC <- getOption("BioC")
    BioC$affy <- affy.opt
    options("BioC"=BioC)
    return()
  }

  ## add affy specific options
  ## (not unlike what is done in 'Biobase')
  if (is.null(getOption("BioC"))) {
    BioC <- list()
    class(BioC) <- "BioCOptions"
    options("BioC"=BioC)
  }

  probesloc.first <- list(what="environment", where=.GlobalEnv)
  probesloc.second <- list(what="libPath", where=NULL)
  probesloc.third <- list(what="data", where="affy")
  probesloc.fourth <- list(what="bioC", where=.libPaths()[1])


  ## default for the methods
  bgcorrect.method <- "mas"
  normalize.method <- "quantiles"
  pmcorrect.method <- "pmonly"
  summary.method <- "liwong"

  affy.opt <- list(compress.cdf=FALSE, compress.cel=FALSE,
                   use.widgets=FALSE,
                   probesloc = list(probesloc.first, probesloc.second,
                   probesloc.third, probesloc.fourth),
                   bgcorrect.method = bgcorrect.method,
                   normalize.method = normalize.method,
                   pmcorrect.method = pmcorrect.method,
                   summary.method = summary.method)

  class(affy.opt) <- "BioCPkg"

  BioC <- getOption("BioC")
  BioC$affy <- affy.opt
  options("BioC"=BioC)
  ## ---
}

.First.lib <- function(libname, pkgname, where) {


  where <- match(paste("package:", pkgname, sep=""), search())
  all.affy <- ls(where)
 message <- FALSE

  if (message) {
    cat(rep("*",13),"\n",sep="")
    cat("affy: development version\n")
    cat(rep("*",13),"\n",sep="")
    cat(rep("*",13),"\n",sep="")
    cat("IMPORTANT: you need the latest versions of the required packages too.\n")
    cat(rep("*",13),"\n",sep="")
  }

  library.dynam("affy", pkgname, libname)

  .initNormalize(match(paste("package:", pkgname, sep=""), search()), all.affy)
  .initExpression(match(paste("package:", pkgname, sep=""), search()), all.affy)
  .initBackgroundCorrect(match(paste("package:", pkgname, sep=""), search()), all.affy)
  .initPmCorrect(match(paste("package:", pkgname, sep=""), search()), all.affy)
  .initMapCdfName(match(paste("package:", pkgname, sep=""), search()))

  .setAffyOptions()

  cacheMetaData(as.environment(where))


}





