.packageName <- "MAGEML"


.packageName<-"MAGEML"
##############################################################################
##                                                                          ##
## This is an R Package for handling MAGE-ML documents in Bioconductor.     ##
##                                                                          ## 
##@Authors Joke Allemeersch, Steffen Durinck                                ##
## ESAT-SCD KULeuven                                                        ##
##############################################################################

require(XML) || stop("can't load without package \"XML\"")
require(Biobase)
require(marrayInput) || stop("can't load without package \"marrayInput\"")
require(methods)

setClass("bioSequence",
         representation(bioSequenceID="list", 
			bioSequenceName="list", 
			dataBaseEntry="list",
                        databaseRef="list"))
setClass("arrayDesign",
         representation(feature="list",
                        zone="list",
                        location="list"))
setClass("bioAssayData",
         representation(featureRef="list",
                        array="character"))
setClass("quantitationType",
         representation(quantitationTypes="list",
                        dataExternal="list",
                        experiment="character",
                        dataID="list"))
setClass("designElement",
         representation(feature="list",
                        reporterBioSeq="list"))
setClass("bioSample",
         representation(measuredDataID="character",
                        compound="character",
                        bioSampleName="character"))


setMethod("show", "bioSequence", function(object)
          { 
            cat("Object of class BioSequence. \n\n")
            cat("Slot `bioSequenceID': \n")
            print(object@bioSequenceID[1:min(length(object@bioSequenceID),10)])
            if(length(object@bioSequenceID) > 10)  cat("... \n")
            cat("Slot `bioSequenceName': \n")
            print(object@bioSequenceName[1:min(length(object@bioSequenceName),10)])
            if(length(object@bioSequenceName) > 10)  cat("... \n")
            cat("Slot `dataBaseEntry': \n")
            print(object@dataBaseEntry[1:min(nrow(object@dataBaseEntry), 10)])
            if(length(object@dataBaseEntry) > 10)  cat("... \n")
            cat("Slot `databaseRef': \n")
            print(object@databaseRef[1:min(nrow(object@databaseRef), 10)])
            if(length(object@databaseRef) > 10)  cat("... \n")

          })

setMethod("show", "arrayDesign", function(object)
          { 
            cat("Object of class arrayDesign. \n\n")
            cat("Slot `feature': \n")
            print(object@feature[1:min(length(object@feature),10)])
            if(length(object@feature) > 10)  cat("... \n")
            cat("Slot `zone': \n")
            print(object@zone[1:min(length(object@zone),10)])
            if(length(object@zone) > 10)  cat("... \n")
            cat("Slot `location': \n")
            print(object@location[1:min(nrow(object@location), 10)])
            if(length(object@location) > 10)  cat("... \n")
          })

setMethod("show", "bioAssayData", function(object)
          { 
            cat("Object of class bioAssayData. \n\n")
            cat("Slot `feature': \n")
            print(object@featureRef[1:min(length(object@featureRef),10)])
            if(length(object@featureRef) > 10)  cat("... \n")
          })

setMethod("show", "designElement", function(object)
          { 
            cat("Object of class designElement. \n\n")
            cat("Slot `feature': \n")
            print(object@feature[1:min(length(object@feature),10)])
            if(length(object@feature) > 10)  cat("... \n")
            cat("Slot `reporterBioSeq': \n")
            print(object@reporterBioSeq[1:min(length(object@reporterBioSeq),10)])
            if(length(object@reporterBioSeq) > 10)  cat("... \n")
          })

setMethod("show", "quantitationType", function(object)
          { 
            cat("Object of class quantitationType. \n\n")
            cat("Slot `quantitationTypes': \n")
            print(object@quantitationTypes[1:min(length(object@quantitationTypes),10)])
            if(length(object@quantitationTypes) > 10)  cat("... \n")
            cat("Slot `dataExternal': \n")
            print(object@dataExternal[1:min(length(object@dataExternal),10)])
            if(length(object@dataExternal) > 10)  cat("... \n")
            cat("Slot `experiment': \n")
            print(object@experiment[1:min(length(object@experiment),10)])
            if(length(object@experiment) > 10)  cat("... \n")
            cat("Slot `dataID': \n")
            print(object@dataID[1:min(length(object@dataID),10)])
            if(length(object@dataID) > 10)  cat("... \n")
          })

setMethod("show", "bioSample", function(object)
          { 
            cat("Object of class bioSample. \n\n")
            cat("Slot `bioSample': \n")
            print(object@measuredDataID[1:min(length(object@measuredDataID),10)])
            if(length(object@measuredDataID) > 10)  cat("... \n")
            cat("Slot `dataExternal': \n")
            print(object@compound[1:min(length(object@compound),10)])
            if(length(object@compound) > 10)  cat("... \n")
            cat("Slot `dataID': \n")
            print(object@bioSampleName[1:min(length(object@bioSampleName),10)])
            if(length(object@bioSampleName) > 10)  cat("... \n")
          })


.First.lib <- function(libname, pkgname, where) {
  require(methods)
  require(Biobase) || stop("can't load without Biobase")
 
  require(marrayInput)
}
                      

                      
                      
readMageDesign <- function(file = NULL, datadir = ".", fArrayDesign = NULL, fBioAssayData = NULL, fDesignElement = NULL, fBioSequence = NULL, ngc=4, skip=0, sep="\t",quote="",...){
  reporter <- list()
  reporterDatabaseEntry <- list()
  zone <- list()
  row <- list()
  column <- list()
  warning <- ""
  p <- 0 

  
  
  # The following functions parse the MAGE-ML files.
  # The ArrayDesignHandler will extract information on the location of the features spotted on the microarray.
  # This information is stored in the ArrayDesign.xml file and cosist of the zone (= block), the row and column
  # within the zone.
  # The BioAssayDataHandler extracts the features that were assayed in the experiment this information is stored in the BioAssayData.xml file
  # The DesignElementHandler extracts the reporter information from the corresponding features, this information is stored
  # in the DesignElement.xml file   
  if ((is.null(fArrayDesign) || is.null(fBioAssayData) || is.null(fDesignElement)|| is.null(fBioSequence)) && is.null(file)){
    writeLines("Input file argument is missing")
    return() 
  }
  if (!is.null(fArrayDesign) && !is.null(fBioAssayData) && !is.null(fDesignElement) && !is.null(fBioSequence) && !is.null(file)){
    writeLines("To many input file arguments")
    return()
  } 
  if (is.null(fArrayDesign) || is.null(fBioAssayData) || is.null(fBioSequence) || is.null(fDesignElement) && !is.null(file)){
    fArrayDesign <- file
    fDesignElement <- file
    fBioAssayData <- file
    fBioSequence <- file
  }
  
  writeLines("Parsing XML documents ... (can take several minutes)")
  
  ArrayDesign <- getArrayDesign(file = fArrayDesign, datadir = datadir)           
  BioAssayData <- getBioAssayData(file= fBioAssayData, datadir = datadir)           
  DesignElement <- getDesignElement(file = fDesignElement, datadir = datadir)     
  BioSequence <- getBioSequence(file = fBioSequence,datadir = datadir)
  
# Next we want to combine all the extracted information (e.g map the gene names to the features that were used in the experiment,
  # map the location information to the corresponding gene names.  This will return a list contain all mapped information in which the order correspond
  # to the order of the actual measured data in the tab delimited text file. This list can then be use for making marrayLayout object
  s<-0
  writeLines("Making marrayLayout and marrayInfo objects ... (can take several minutes)")

  for (j in 1:length(BioAssayData@featureRef)){ 
    index <- match(BioAssayData@featureRef[[j]],ArrayDesign@feature)
    zone[[j]] <- ArrayDesign@zone[[index]]
    row[[j]] <- ArrayDesign@location[[index]][[1]]
    column[[j]] <-  ArrayDesign@location[[index]][[2]]
    if(!is.na(match(BioAssayData@featureRef[[j]], DesignElement@feature))){         #This check is only needed if XML files are corrupti
      if(!is.na(match(DesignElement@reporterBioSeq[[match(BioAssayData@featureRef[[j]],DesignElement@feature)]], BioSequence@bioSequenceID))){
        if(!is.null(BioSequence@bioSequenceName[[match(DesignElement@reporterBioSeq[[match(BioAssayData@featureRef[[j]],DesignElement@feature)]], BioSequence@bioSequenceID)]])&!is.null(BioSequence@dataBaseEntry[[match(DesignElement@reporterBioSeq[[match(BioAssayData@featureRef[[j]],DesignElement@feature)]], BioSequence@bioSequenceID)]])){
          reporter[[j]] <- BioSequence@bioSequenceName[[match(DesignElement@reporterBioSeq[[match(BioAssayData@featureRef[[j]],DesignElement@feature)]], BioSequence@bioSequenceID)]]
        }
        reporterDatabaseEntry[[j]] <- BioSequence@dataBaseEntry[[match(DesignElement@reporterBioSeq[[match(BioAssayData@featureRef[[j]],DesignElement@feature)]], BioSequence@bioSequenceID)]]
      }
      else{
        reporter[[j]]<- DesignElement@reporterBioSeq[[match(BioAssayData@featureRef[[j]],DesignElement@feature)]]
        reporterDatabaseEntry[[j]]<-NA 
      }
    }
    else{
      p<-p+1
      reporter[[j]] <- NA
      warning <- paste("Warning: ",p," reporter names were not available and have been replaced by NA")
    }
  }

  # Next we make a marrayLayout object
  
  Lblock <- as.integer(unlist(zone))
  Lrow <- as.integer(unlist(row))
  Lcolumn <- as.integer(unlist(column))
  ngr <- max(Lblock) / ngc
  ngc <- ngc
  nsr <- max(Lrow)
  nsc <- max(Lcolumn)
  nspots <- as.integer(ngr) * as.integer(ngc) * as.integer(nsr) * as.integer(nsc)
  temp <- rep(FALSE, nspots)
  ind <- (nsr * nsc) * (Lblock - 1) + (Lrow - 1) * nsc + Lcolumn

  temp[ind] <- TRUE
  mlayout <- new("marrayLayout", maNgr = as.integer(ngr),
                 maNgc = as.integer(ngc),
                 maNsr = as.integer(nsr),
                 maNsc = as.integer(nsc),
                 maNspots = nspots,
                 maNotes = paste("Array design is of type",BioAssayData@array),
                 maSub=temp)
  databases <- NULL
  for (a in 1: length(unique(unlist(BioSequence@databaseRef)))){
   databases <- paste(databases, unique(unlist(BioSequence@databaseRef))[a])
  }
  #Next we make a marrayInfo object
  mInfo <- new("marrayInfo", maNotes = paste("Identification corresponds to accession numbers from following database(s):\n",databases))
  maInfo <- as.data.frame(cbind(unlist(reporterDatabaseEntry),unlist(reporter)))
  names(maInfo) <- c("Identification","Gene Name")
  mInfo@maInfo <- maInfo
  
  writeLines(warning)
  return(list(layout = mlayout,info = mInfo))
}

#returns list of assayed features

getBioAssayData <- function(file = NULL, datadir = NULL){
 BioAssay <- xmlEventParse(file = file.path(datadir, file), BioAssayDataHandler())
 BioAssayData <- BioAssay$BioAssayData()[["BioAssayData"]]
 return(BioAssayData)
}

#returns list of feature location information 

getArrayDesign <- function(file = NULL, datadir = NULL){
 require(XML)
 ArrayDesign <- xmlEventParse(file = file.path(datadir, file),ArrayDesignHandler())
 ArrayDesign <- ArrayDesign$ArrayDesignData()[["ArrayDesign"]]
 return(ArrayDesign)
}

#returns list representing a feature-reporter mapping

getDesignElement <- function(file = NULL, datadir = "."){
 DesignElement <- xmlEventParse(file = file.path(datadir, file),DesignElementHandler())     
 DesignElement <- DesignElement$DesignElementData()[["DesignElement"]]
 return(DesignElement)
}

getBioSequence <- function(file=NULL, datadir="."){
 BioSequence <- xmlEventParse(file = file.path(datadir, file),BioSequenceHandler())
 BioSequence <- BioSequence$Biosequence()[["BioSequence"]]
 return(BioSequence)
}

magePackageDetector <- function(fnames = NULL, datadir="."){
  for (f in 1:length(fnames)){
    writeLines(paste("\n File ", fnames[f]," contains the following packages:\n"))
    xmlEventParse(file =file.path(datadir, fnames[f]),PackageDetectorHandler())
  }
}

#reads in raw cDNA datafile and returns raw object

readMageRaw <- function(fname,datadir=".",
                name.Gf = "",
                name.Gb = "",
                name.Rf = "",
                name.Rb = "",
                name.W=NULL,
                layout=NULL,
                gnames=NULL,
                notes=NULL,
                skip=0,
                sep="\t",
                quote="",
                ...)
{
  QuantType <- getQuantitationTypes(file=fname,datadir=datadir)
  fname <- file.path(datadir, fname)

  fnames <- QuantType@dataExternal
  experiment <- QuantType@experiment
  DataId <- QuantType@dataID
  
  ##If fnames is not specified

  if (is.null(fnames))
    writeLines("There are no files with raw data specified.")
  else
    fullnames<-file.path(datadir,fnames)
  
  ## Intensity data
  Gf<-Gb<-Rf<-Rb<-W<- NULL
  names(Gb)<-names(Rf)<-names(Rb)<-names(W)<-NULL
  if(is.null(name.Gb)) Gb <- matrix(0,0,0)
  if(is.null(name.Rb)) Rb <- matrix(0,0,0)
  fileLabels<-NULL
  i<-0
  NrTxtFiles<-length(fullnames)
  NrHyb<-rep(0,NrTxtFiles)
  for (f in  fullnames){
    i<-i+1
    writeLines(paste("Reading ",fnames[i]))
    dat<-read.table(file=f,header=FALSE,sep=sep) 
    ##Determine the number of hybridisation contained in one single txt file
    NrHyb[i]<-ncol(dat)/length(QuantType@quantitationTypes)
    if(round(NrHyb[i])!=NrHyb[i]){
      writeLines("The quantitation types can not correspond to the tab-delimited files, that contain the raw data.")
      break}
    writeLines(paste("This file describes ",NrHyb[i]," hybridisation(s)."))
    nam<-NULL
    for (j in 1:NrHyb[i]){
      nam<-c(nam,paste(QuantType@quantitationTypes,"_",j,sep=""))
    }
    names(dat)<-nam
    for (k in 1:NrHyb[i]){
      Gf<-cbind(Gf,as.numeric(dat[[paste(name.Gf,"_",k,sep="")]]))
      if(!is.null(name.Gb)) Gb<-cbind(Gb,as.numeric(dat[[paste(name.Gb,"_",k,sep="")]])) 
      Rf<-cbind(Rf,as.numeric(dat[[paste(name.Rf,"_",k,sep="")]]))
      if(!is.null(name.Rb)) Rb<-cbind(Rb,as.numeric(dat[[paste(name.Rb,"_",k,sep="")]]))
      if(!is.null(name.W)) W <-cbind(W,as.numeric(dat[[paste(name.W,"_",k,sep="")]]))
    }
  }
  colnam <- NULL 
  for (k in 1:length(fnames)){
    for (i in 1:NrHyb[k]){
      colnam<-c(colnam,paste(gsub("_measured","",gsub(".txt","",fnames[[k]][1])),"_Hyb_",i,sep=""))
    }
  }
  if(length(colnam)!=ncol(Gb)){
    writeLines("The name of the Green Background intensities is not correct.")
  }
  else{
    if(length(colnam)!=ncol(Gf)){
      writeLines("The name of the Green Foreground intensities is not correct.")
    }
    else{
      if(length(colnam)!=ncol(Rb)){
        writeLines("The name of the Red Background intensities is not correct.")
      }
      else{
        if(length(colnam)!=ncol(Rf)){
          writeLines("The name of the Red Foreground intensities is not correct.")
        }
        else{
          if(!is.null(name.W)) colnames(W) <- colnam
          if(!is.null(name.Gb)) colnames(Gb) <- colnam
          if(!is.null(name.Rb)) colnames(Rb) <- colnam
          colnames(Gf)<-colnames(Rf)<-colnam
          ## Add Notes
          if(is.null(notes)) notes <- ""
          mraw<-new("marrayRaw", maRf=Rf, maRb=Rb, maGf=Gf, maGb=Gb, maNotes =experiment)
          
          ## Making the Targets...##
          target <- xmlEventParse(fname, TargetHandler())
          BioSample <- target$TargetList()[["BioSample"]]
          n <- max(length(BioSample@bioSampleName),length(BioSample@measuredDataID),length(BioSample@compound))
          if(n!=sum(NrHyb)*2){
            writeLines("The targets described in the MAGE-ML file don`t correspond to the data, found in the tab-delimited file.  Probably your tab delimited file does not have the correct number of comlumns.")
          }
          LabelsUnique <- unique(BioSample@measuredDataID)
          NrHybTa<-rep(NA,length(LabelsUnique))
          for (l in 1:length(LabelsUnique)){
            count<-0
            for (u in 1:length(BioSample@measuredDataID)){
              if (BioSample@measuredDataID[u]==LabelsUnique[l]){
                count <- count+1
              }
            }
            NrHybTa[l] <- count/2
          }
          nrrow <- sum(NrHybTa)
          if(sum(sort(NrHyb)!=sort(NrHybTa))==length(NrHyb)){
            writeLines("Information obtained from the Target Handler does not correspond with the information obtained from the QuantitationType Handler.  The number of hybridisations per tab delimited file differs.")
          }
          else{
            TargetRG<-matrix(data = NA, nrow = nrrow, ncol = 2)
            colnames(TargetRG)<-c("Cy3","Cy5")
            TargetRG<-as.data.frame(TargetRG)
            rownames <- rep("U",nrrow)
            rownr<-1
            complete <- rep(0,nrrow)
            for (i in 1:length(BioSample@measuredDataID)){
              if (complete[rownr]==0){
                rownames[rownr] <- paste(BioSample@measuredDataID[i],".Hyb.",rownr,sep="")
              }
              if (BioSample@compound[i]=="COMP:cy3"){
                TargetRG[rownr,1] <- BioSample@bioSampleName[[i]]
                complete[rownr] <- complete[rownr]+1
                if(paste(BioSample@measuredDataID[i],".Hyb.",rownr,sep="")!=rownames[rownr]){
                  writeLines("The Cy3 sample is missing or the samples of the 2 samples are in a wrong order.")
                }
              }
              if (BioSample@compound[i]=="COMP:cy5"){
                TargetRG[rownr,2]<-BioSample@bioSampleName[[i]]
                complete[rownr] <- complete[rownr]+1
                if(paste(BioSample@measuredDataID[i],".Hyb.",rownr,sep="")!=rownames[rownr]){
                  writeLines("The Cy5 sample is missing or the samples of the 2 samples are in a wrong order.")
                }
              }
              if (complete[rownr]==2){
                rownr<-rownr+1
              }
            }
            
            HybLabel <- NULL
            for (i in 1:length(NrHybTa)){
              found<-FALSE
              k<-1
              while (found==FALSE & k<=length(DataId)){ 
                if (LabelsUnique[i]==DataId[k]){
                  for (p in 1:NrHybTa[i]){
                    HybLabel<-c(HybLabel,paste(gsub("_measured","",gsub(".txt","",fnames[[k]][1])),"_Hyb_",p,sep=""))
                  }
                  found<-TRUE
                }
                else {k<-k+1}
              }
            }
            mraw@maTargets<-new("marrayInfo",maLabels=HybLabel,maInfo=TargetRG,maNotes="Description of the targets")
          }
          
          
          if(!is.null(layout)) mraw@maLayout <- layout
          if(!is.null(gnames)) mraw@maGnames <- gnames
          if(!is.null(W)) mraw@maW <- W
          return(mraw)
        }
      }
    }
  }
}

getQuantitationTypes <- function(file = "", datadir="."){
  fullname<-file.path(datadir, file)
  Quant <- xmlEventParse(fullname,QuantitationTypeHandler())
  QuantType <- Quant$QuantType()[["QuantitationType"]]
  return (QuantType=QuantType)
}

showQuantitationTypes <- function(file = "", datadir="."){
  fullname<-file.path(datadir, file)
  QuantType <- getQuantitationTypes(file=fullname, datadir="")
  writeLines("\n These are the available quantitation types:\n")
  for(i in 1: length(QuantType@quantitationTypes)){
    writeLines(QuantType@quantitationTypes[[i]])
  }
}

##############################################################
##
## XML parser handlers
##
##############################################################

ArrayDesignHandler <- function(){
  
  Feature <- list()
  Zone <- list()
  zone <- NULL
  Location <- list()
  i<-0
  j<-0
  k<-0
  row <- 0
  col <- 0
  ArrayDesignFlag <- FALSE
  zoneLoc <-NULL
  zonestr <- NULL
  zoneCount <- NULL
   startElement <- function(x, atts, ...){
     
     if (x=="ArrayDesign_package"){
       ArrayDesignFlag <<- TRUE
     }
     if (x=="Feature" & ArrayDesignFlag){
       i <<- i+1
       Feature[[i]] <<- atts[["identifier"]]
     }
     if (x=="Zone_ref" & ArrayDesignFlag){
       j <<- j+1
       zone <<- atts
       if(is.null(zoneLoc)){zoneLoc<-length(strsplit(zone, ":")[[1]])}
       if(is.null(zonestr)){
         zonestr <<- strsplit(zone, ":")[[1]][[zoneLoc]]
         zonestr<<-length(strsplit(zonestr, ".")[[1]])
       }
       if(zonestr == 1){
         Zone[[j]] <<- strsplit(zone, ":")[[1]][[zoneLoc]]
       }
       else{
         zoneCount <<- strsplit(zone, ":")[[1]][[zoneLoc]]
         row <<- as.numeric(strsplit(zoneCount, "\\.")[[1]][[1]])
         col <<- as.numeric(strsplit(zoneCount, "\\.")[[1]][[2]])
         Zone[[j]] <<- ((row - 1)*4) + col
       }
     }
     if (x == "FeatureLocation" & ArrayDesignFlag){
       k <<- k+1
       Location[[k]] <<- atts
     }
   }
  endElement <- function(x){
    if(x == "ArrayDesign_package"){
      ArrayDesignFlag<<-FALSE
    }
  }
  ArrayDesignData <- function(){
    ArrayDesign <- new("arrayDesign")
    ArrayDesign@feature <-Feature
    ArrayDesign@zone <- Zone
    ArrayDesign@location <- Location
    list(ArrayDesign=ArrayDesign)
  }
  list(startElement = startElement,endElement = endElement, ArrayDesignData = ArrayDesignData)
}



BioAssayDataHandler <- function(){
  
   FeatureRef <- list()
   BioAssayDataFlag <- FALSE
   i <- 0
   ArrayFlag<-FALSE
   array<- NULL
   arrayRef <- NULL
   
   startElement <- function(x, atts, ...){
     
     if (x=="BioAssayData_package"){
       BioAssayDataFlag<<-TRUE
     } 
     if (x == "Feature_ref" & BioAssayDataFlag){
       i <<- i+1
       FeatureRef[[i]] <<- atts[["identifier"]]
     }
     if ( x == "PhysicalArrayDesign_ref" & ArrayFlag){
       array <<- atts[["identifier"]]
       if(is.null(arrayRef)){arrayRef<<-array}
       if(array != arrayRef){
       stop("This experiment covers multiple array types.  Handling experiments which include different array types will be implemented in the next version of the package, the current version of the MAGEML package can not handle this...sorry!") 
      }
     }
     if (x=="Array_package"){ArrayFlag<<-TRUE}
   }
   endElement <- function(x){
     if (x == "BioAssayData_package"){
       BioAssayDataFlag<<-FALSE
     }
     if (x == "Array_package"){ArrayFlag<<-FALSE}
   }
   BioAssayData <- function(){
     BioAssayData <- new("bioAssayData")
     BioAssayData@featureRef <- FeatureRef
     if(!is.null(array)){
       BioAssayData@array <- array
     }
     else {BioAssayData@array <- "NA"}
     list(BioAssayData = BioAssayData)
   }
   list(startElement = startElement, endElement = endElement, BioAssayData = BioAssayData)
 }

TargetHandler <- function(){
   
  BioAssayFlag <- FALSE
  BioMaterialFlag <- FALSE
  CompoundListFlag <- FALSE
  LabelFlag <- FALSE
  BioAssayDataFlag <- FALSE
  LabeledExtractID <- list()
  BioSampleRef <- list()
  Compound <- NULL
  PhysicalBioAssayFlag <- FALSE
  MeasuredBioAssayDataassnreflistFlag <- FALSE
  ref<- list()
  row<-list()
  LabeledExtractRef <- list(row=row, ref=ref)
  CompoundRef <- list()
  BioSampleID <- list()
  BioSampleName <- list()
  BioSample <- NULL
  DataRef <- list()
  labeledExtractFlag <- FALSE
  BioAssayDataassnlistFlag <- FALSE
  hasMeasuredDataRef <- FALSE
  MeasuredDataID <- NULL
  MeasuredFirstFlag <- FALSE
  PhysicalFirstFlag <- FALSE
  g <- 0
  i <- 0
  t <- 0
  w <- 0

  startElement <- function(x, atts, ...){
    if (x == "BioMaterial_package"){BioMaterialFlag <<- TRUE}   
    if (x == "Compound_assnlist" && BioMaterialFlag){CompoundListFlag <<- TRUE} 
    if (x == "Compound_ref" & labeledExtractFlag){
      CompoundRef[[t]] <<- atts[["identifier"]]
    }
    if (x == "BioSample" && BioMaterialFlag){
      w <<- w+1
      BioSampleID[[w]] <<- atts[["identifier"]]
      if(length(atts)>=2){
        BioSampleName[[w]] <<- atts[["name"]]
      }
      else{
        BioSampleName[[w]] <<- BioSampleID[[w]]
      }
    }
    if (x=="LabeledExtract" && BioMaterialFlag){
      t <<- t+1
      LabeledExtractID[[t]] <<- atts[["identifier"]][[1]]
      labeledExtractFlag <<- TRUE
    } 
    if (x == "PhysicalBioAssay" & BioAssayFlag){
      if(!MeasuredFirstFlag){
      g <<- g+1
      PhysicalFirstFlag <<- TRUE
      }
      PhysicalBioAssayFlag <<- TRUE
      hasMeasuredDataRef <<- FALSE
    }   
    if (x=="LabeledExtract_ref" & PhysicalBioAssayFlag){
      if(i==0){i <<- 1}
      else {
        if(i==1){i <<- 2}
        else {
          if(i==2){
            i <<- 1
          }
        }
      }
      if (i==1){
        LabeledExtractRef[[g]]<-list()
      }
      LabeledExtractRef[[g]][[i]] <<- atts[["identifier"]]
    }
    if( x == "MeasuredBioAssay" & BioAssayFlag){
       if(!PhysicalFirstFlag){
       g<<-g+1
       MeasuredFirstFlag <<- TRUE
       }
    }
    if (x == "MeasuredBioAssayData_ref" & BioAssayFlag){
      DataRef[[g]] <<- atts[["identifier"]]
      hasMeasuredDataRef <<- TRUE
    }
    if (x == "DerivedBioAssayData_ref" & BioAssayFlag & !hasMeasuredDataRef){
      DataRef[[g]] <<-atts[["identifier"]]
    }   
    if (x=="BioSample_ref" && labeledExtractFlag){
      BioSampleRef[[t]] <<- atts[["identifier"]]
    }
    if (x=="Labels_assnreflist" && BioMaterialFlag){
      LabelFlag <<-TRUE
    }
    if (x=="BioAssay_package"){
      BioAssayFlag<<-TRUE
    } 
    if (x == "BioAssayData_assnlist"){ 
      BioAssayDataassnlistFlag <<- TRUE
    }
    if (x == "BioAssayData_package"){BioAssayDataFlag <<- TRUE}
  }

  endElement <- function(x){
    if (x=="BioMaterial_package"){
      BioMaterialFlag<<-FALSE
      LabelFlag <<- FALSE
    }
    if (x == "BioAssayData_assnlist"){BioAssayDataassnlistFlag <<- FALSE}
    if (x == "Compound_assnlist"){CompoundListFlag <<-FALSE}
    if (x == "BioAssay_package"){BioAssayDataFlag<<-FALSE}
    if (x == "BioAssayData_package"){BioAssayDataFlag <<- FALSE}
    if (x == "MeasuredBioAssayData_assnreflist"){MeasuredBioAssayDataassnreflistFlag <<- FALSE}
    if (x == "BioAssayData_assnlist"){BioAssayDataassnlistFlag <<- FALSE}
    if (x == "PhysicalBioAssay"){PhysicalBioAssayFlag <<- FALSE}
    if (x == "LabeledExtract"){labeledExtractFlag <<- FALSE}
    if (x == "MAGE-ML"){ 
      for (u in 1:length(DataRef)){
        for (e in 1:2){
          MeasuredDataID[2*(u-1)+e] <<- DataRef[[u]] 
          Compound[2*(u-1)+e] <<- CompoundRef[[match(LabeledExtractRef[[u]][[e]],LabeledExtractID)]] 
          BioSample[2*(u-1)+e] <<- BioSampleName[[match(BioSampleRef[[match(LabeledExtractRef[[u]][[e]],LabeledExtractID)]], BioSampleID)]]
        }          
      }
    } 
  }


  TargetList <- function(){
    BioTarget <- new("bioSample")
    BioTarget@measuredDataID <-  MeasuredDataID
    BioTarget@compound <- Compound
    BioTarget@bioSampleName <- BioSample
    list(BioSample = BioTarget)
  }
  list(startElement = startElement,endElement = endElement, TargetList = TargetList)
}



BioSequenceHandler <- function(){
  
  BioSequenceFlag <- FALSE
  databaseEntry <- list()
  bioSequenceId <- list()
  bioSequenceName <- list()
  databaseRef <- list()
  hadNoDBEntry <- TRUE
  
  q<-0
  
  startElement <- function(x, atts, ...){
    if (x == "BioSequence_package"){BioSequenceFlag <<- TRUE}   
    if (x == "BioSequence" & BioSequenceFlag){
      q<<-q+1
      bioSequenceId[[q]]<<-atts["identifier"][[1]]
      bioSequenceName[[q]] <<- atts["name"][[1]]
    }   
    if (x == "DatabaseEntry" & BioSequenceFlag){
      databaseEntry[[q]] <<- atts["accession"][[1]]
      hadNoDBEntry<<-FALSE
    }
    if (x == "Database_ref" & BioSequenceFlag){
      databaseRef[[q]] <<- atts["identifier"][[1]] 
    }
  }
  endElement <- function(x){
    if (x=="BioSequence_package"){BioSequenceFlag <<- FALSE}
    if (x=="BioSequence" & BioSequenceFlag){
      if(hadNoDBEntry){
        databaseEntry[[q]]<<- NA
        databaseRef[[q]]<<-NULL
      }   #Checks if Biosequence had a databaseEntry
      hadNoDBEntry<<-TRUE			   #	
    }
  }

  Biosequence <- function(){
     BioSequence <- new("bioSequence")
     BioSequence@bioSequenceID <- bioSequenceId
     BioSequence@bioSequenceName <- bioSequenceName
     BioSequence@dataBaseEntry <- databaseEntry
     BioSequence@databaseRef <- databaseRef
     list(BioSequence = BioSequence)
  }
  list(startElement = startElement,endElement = endElement, Biosequence = Biosequence)
}

QuantitationTypeHandler <-function(){

  QuantitationTypes <- list()
  DataExternal <- list()
  DerivedBioAssayData <- list()
  DataID <- list()
  ExperimentFlag <- FALSE
  MeasuredDataFlag<- FALSE
  experiment <- NULL
  QuantitationTypeDimension <- list()
  QTdim <- list()
  QTypes <- list()
  currentDimension <- NULL
  z <- 0
  q <- 0
  n <- 0
  v <- 0
  BioAssayDataFlag <- FALSE
  QTFlag <-FALSE
  MeasuredPresentFlag <- FALSE
  DerivedDataFlag <- FALSE
  hasDerivedFlag <- FALSE

  startElement <- function(x, atts, ...){
    if (x=="BioAssayData_package"){BioAssayDataFlag <<-TRUE}
    if (x=="MeasuredBioAssayData" & BioAssayDataFlag){
      MeasuredDataFlag <<-TRUE
      MeasuredPresentFlag <<-TRUE
      q <<-q+1
      DataID[[q]] <<- atts[["identifier"]]
    }
    if (x =="QuantitationTypeDimension_ref" & MeasuredDataFlag){
      QTdim[[q]] <<- atts[["identifier"]]  
    }  
    if (x =="QuantitationTypeDimension_ref" & DerivedDataFlag){
      if(!MeasuredPresentFlag){
        QTdim[[q]] <<- atts[["identifier"]]  
      }
    }  
    if (x=="DerivedBioAssayData" & BioAssayDataFlag){
      if(!MeasuredPresentFlag){
        DerivedDataFlag <<-TRUE
        hasDerivedFlag <<- TRUE
        q <<- q+1
        DataID[[q]]<<- atts[["identifier"]]
      }
    }
    if (x == "QuantitationTypeDimension" & BioAssayDataFlag){
      QTFlag<<-TRUE
      currentDimension <<- atts[["identifier"]]
    }
       
    if (x == "MeasuredSignal_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }
    
    if (x == "Error_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }
    if (x == "PValue_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }
    
    if (x == "Ratio_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }
    
    if (x == "DerivedSignal_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }

    if (x == "SpecializedQuantitationType_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }

    if (x == "PresentAbsent_ref" & BioAssayDataFlag & QTFlag){
      z <<- z+1
      QTypes[[z]] <<- gsub("QT:","",atts[["identifier"]])
      QuantitationTypeDimension[[z]] <<- currentDimension
    }

    if (x == "DataExternal" && BioAssayDataFlag && MeasuredDataFlag){
      DataExternal[[q]] <<- atts[["filenameURI"]]
    }
       
    if (x == "DataExternal" && BioAssayDataFlag && DerivedDataFlag && !MeasuredPresentFlag){
      DataExternal[[q]] <<- atts[["filenameURI"]]
    }

    if (x == "Experiment_package"){ExperimentFlag <<-TRUE}  

    if (x == "Experiment" & ExperimentFlag){
      experiment <<- atts[["identifier"]]
    } 
  }

  endElement <- function(x){
    if(x == "BioAssayData_package"){
      BioAssayDataFlag<<-FALSE
      for( a in 1: length(QuantitationTypeDimension)){
        if(QuantitationTypeDimension[[a]]==QTdim[[1]]){
          v<<-v+1 
          QuantitationTypes[[v]]<<-QTypes[[a]]
        }
      }
    }
    if (x == "QuantitationTypes_assnreflist"){QTFlag<<-FALSE}
    if (x == "Experiment_package"){ExperimentFlag <<-FALSE}
    if (x == "MeasuredBioAssayData"){MeasuredDataFlag <<-FALSE}
    if (x == "DerivedBioAssayData"){DerivedDataFlag <<-FALSE}
  } 
  
  QuantType <- function(){
    QuantitationType <- new("quantitationType")
    QuantitationType@quantitationTypes <- QuantitationTypes
    QuantitationType@dataExternal <- DataExternal
    if(!is.null(experiment)){
    QuantitationType@experiment <- experiment
    }
    else{QuantitationType@experiment <- "NA"}
    QuantitationType@dataID <- DataID
    list(QuantitationType=QuantitationType)
  }
  
  list(startElement = startElement,endElement = endElement,QuantType = QuantType)
}


DesignElementHandler <- function(){

  Feature <- list()
  Reporter <- NULL
  CurrentReporter <- NULL
  i <- 0
  Current <- NULL
  ReporterId <- NULL 
  ReporterName <- NULL
  DesignElementFlag <- FALSE
  ReporterBioSeq <- list()
  ReporterBioSeqRef <- NULL
  ReporterassnlistFlag <- FALSE
  FeatureReporterMapFlag <- FALSE
  hasBioSeq <- FALSE
  warning <- NULL
  q <- 0
  j <- 0
  h <- 0
  f <- 0

  startElement <- function(x, atts, ...){
    if (x=="DesignElement_package"){DesignElementFlag <<- TRUE}  
    if (x=="Reporter_assnlist"){ReporterassnlistFlag <<- TRUE} 
    if (x=="ImmobilizedCharacteristics_assnreflist" & ReporterassnlistFlag){hasBioSeq<<-FALSE}
    if (x=="Reporter" & DesignElementFlag & ReporterassnlistFlag){
      h<<-h+1
      hasBioSeq <<- FALSE
      ReporterId[[h]] <<- atts["identifier"][[1]]
    }
    
    if (x=="BioSequence_ref" & DesignElementFlag & ReporterassnlistFlag & !hasBioSeq){
      ReporterBioSeqRef[[h]] <<- atts["identifier"][[1]]
      hasBioSeq<<-TRUE
    }
    if (x == "FeatureReporterMap_assnlist" & DesignElementFlag){FeatureReporterMapFlag <<- TRUE}
    if (x == "Reporter_ref" & DesignElementFlag & FeatureReporterMapFlag){
      Current <<- atts[["identifier"]][[1]]
    }
        
    if (x == "Feature_ref" & DesignElementFlag & FeatureReporterMapFlag){
      f <<- f+1
      Feature[[f]] <<-  atts[["identifier"]][[1]]
      ReporterBioSeq[[f]] <<- ReporterBioSeqRef[[match(Current, ReporterId)]]
    }   
  }

  endElement <- function(x){
    if(x == "DesignElement_package"){DesignElementFlag <<- FALSE
               if(!is.null(warning)){   
               print(warning)}}
    if(x == "FeatureReporterMap_assnlist"){FeatureReporterMapFlag <<- FALSE}
    if(x == "Reporter_assnlist"){ReporterassnlistFlag <<- FALSE}

    if(x == "Reporter" & DesignElementFlag & ReporterassnlistFlag & !hasBioSeq){
           warning<-"Some reporters have no BioSequence, of these no database entry or name of the sequence will be available"  
           ReporterBioSeqRef[[h]] <<- NA
    }
  } 

  DesignElementData <- function(){
    DesignElementData <- new("designElement")
    DesignElementData@feature <- Feature
    DesignElementData@reporterBioSeq <- ReporterBioSeq
    list(DesignElementData=DesignElementData)
  }
  list(startElement = startElement, endElement = endElement, DesignElementData = DesignElementData)
}

PackageDetectorHandler <- function(){
  startElement <- function(x, atts, ...){
    if (x == "DesignElement_package"){writeLines("\t DesignElement_package")}
    if (x == "BioAssayData_package"){writeLines("\t BioAssayData_package")}
    if (x == "BioAssay_package"){writeLines("\t BioAssay_package")}
    if (x == "ArrayDesign_package"){writeLines("\t ArrayDesign_package")}
    if (x == "BioMaterial_package"){writeLines("\t BioMaterial_package")}
    if (x == "BioSequence_package"){writeLines("\t BioSequence_package")}
    if (x == "QuantitationType_package"){writeLines("\t QuantitationType_package")}
  }
  
  endElement <- function(x){
  }
  list(startElement = startElement, endElement = endElement)
}


######################################################################
##
## Widgets
##
######################################################################


widget.readMageRaw <- function(
                               skip=0,
                               sep="\t",
                               quote="",
                               ...)

{
  require(tcltk)
  require(tkWidgets)
  args <- list(...)

  ## Calling read.marrayLayout
  inputRaw <- function(fname,
                       skip=skip,
                       sep=sep,
                       quote=quote,
                       ...)
    {
      dir<-tkfile.dir(fname)
      fname<-gsub(paste(tkfile.dir(fname),"/",sep=""),"",fname)

      if(tclvalue("rawName")=="")
        newname <- "Raw"
      else
        newname <- tclvalue("rawName")
      if(tclvalue("notes") =="")
        notes <- ""
      else
        notes <- tclvalue("notes")
      ## Deal with Names
      if(tclvalue("nameW") == "") name.W <- NULL else name.W <- tclvalue("nameW")
      if(tclvalue("nameRf") == "")
        {name.Rf = "F635 Mean"} else name.Rf <- tclvalue("nameRf")
      if(tclvalue("nameGf") == "")
        {name.Gf = "F532 Mean"} else name.Gf <- tclvalue("nameGf")
      if(tclvalue("nameRb") == "") {name.Rb = "B635 Median"} else name.Rb <- tclvalue("nameRb")
      if(tclvalue("nameGb") == "") {name.Gb = "B532 Median"} else name.Gb <- tclvalue("nameGb")

      ## Call readMageRaw
      newRaw <- readMageRaw(fname=fname,datadir=dir,
                            name.Gf = name.Gf,
                            name.Gb = name.Gb,
                            name.Rf = name.Rf,
                            name.Rb = name.Rb,
                            name.W= name.W,
                            layout=eval(parse(text=tclvalue("layout"))),
                            gnames=eval(parse(text=tclvalue("genenames"))),
                            targets=eval(parse(text=tclvalue("targets"))),
                            ...)

      if (!is.null(newRaw)){
        cat(paste("\n Finish creating a new marrayInfo: ", newname, "\n", sep=""))
        if (tclvalue("targetName")!=""){
          cat(paste("and the object containing the Targets, ", tclvalue("targetName"), "\n", sep=""))
          assign(tclvalue("targetName"),newRaw@maTargets, env= .GlobalEnv)
        }
        assign(newname, newRaw, env = .GlobalEnv)
      }
    }
  layBrowser <- function(){
    browserToEntry(layoutEntry, txt="Select the Layout object")
  }
  genBrowser <- function(){
    browserToEntry(geneEntry,txt="Select the Gnames object")
  }

  browserToEntry <- function(anEntry,txt){
    first = TRUE
    temp <- objectBrowser(textToShow = txt,nSelect = 1)
    tkdelete(anEntry, 0, "end")
    for(i in 1:length(temp)){
      if(first){
        tkinsert(anEntry, "end", temp[[i]]$name)
        first <- FALSE
      }
      else{
        toPut <- paste(",", temp[[i]]$name, sep = "")
        tkinsert(anEntry, "end", toPut)
      }
    }
  }

  filesClick2 <- function(){
    if(!is.null(args$path))
      fnames <<- fileBrowser(args$path,textToShow="Select a BioAssay Data file")
    else
    fnames <<- fileBrowser(textToShow="Select a BioAssay Data File")
    tkdelete(mageEntry, 0, "end")
    tkinsert(mageEntry,0,fnames)
  }

  filesClick <- function(){
    fnames <<- tclvalue(tkcmd("tk_getOpenFile"))
    tkdelete(mageEntry, 0, "end")
    tkinsert(mageEntry,0,fnames)
  }

  showPack <- function(){
    fnames <<- tclvalue("fname")
    magePackageDetector(fnames,datadir="")
  }


  newName <- NULL

  base <- tktoplevel()
  tkwm.title(base, "marrayRaw Builder")
  mainfrm <- tkframe(base, borderwidth=2)

  # Input Name of Raw Object
  notesvarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)
  tkpack(notesvarfr, tklabel(notesvarfr, text="Name of the marrayRaw object"),
         side='top')
  rawEntry<-tkentry(notesvarfr, width=78, textvariable="rawName")
  tkpack(notesvarfr,rawEntry , side='left')
  tkpack(mainfrm, notesvarfr, side="top")

  # Input Name of Target Object
  notesvarfr2 <- tkframe(mainfrm, relief="groove", borderwidth=2)
  tkpack(notesvarfr2, tklabel(notesvarfr2, text="Name of the Target object"),
         side='top')
  targetEntry<-tkentry(notesvarfr2, width=78, textvariable="targetName")
  tkpack(notesvarfr2,targetEntry , side='left')
  tkpack(mainfrm, notesvarfr2, side="top")

  ## MAGEML file
  notesvarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)
  mageFrame <- tkframe(notesvarfr)
  mageLbl <- tklabel(mageFrame, text="Enter the MAGE-ML file describing the experiment (containing the \n BioAssay, BioAssayData, and BioMaterial package):")
  mageEntry <- tkentry(mageFrame, width=49, textvariable="fname")
  filesClick <- tkbutton(mageFrame, text = "Browse",
                         command = filesClick)
  showPack <- tkbutton(mageFrame, text = "Show Packages",
                       command = showPack)
  tkgrid(mageLbl, columnspan = 2)
  tkgrid(mageEntry,filesClick, showPack)
  tkpack(mageFrame, side = "top")
  tkpack(mainfrm, notesvarfr, side="top")
  # Select fg and bg
  coordvarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)
  heading <- tklabel(coordvarfr, text="Foreground and Background Intensities")

  gf.label <- tklabel(coordvarfr, text="Green Foreground")
  gb.label <- tklabel(coordvarfr, text="Green Background")
  rf.label <- tklabel(coordvarfr, text="Red Foreground")
  rb.label <- tklabel(coordvarfr, text="Red Background")
  W.label <- tklabel(coordvarfr, text="Weights")

  gf.entry <- tkentry(coordvarfr, width=22, textvariable = "nameGf" )
  gb.entry <- tkentry(coordvarfr, width=22, textvariable = "nameGb" )
  rf.entry <- tkentry(coordvarfr, width=22, textvariable = "nameRf" )
  rb.entry <- tkentry(coordvarfr, width=22, textvariable = "nameRb" )
  W.entry <-  tkentry(coordvarfr, width=22, textvariable = "nameW" )


  tkgrid(heading, columnspan = 4)
  tkgrid(gf.label, gf.entry, gb.label, gb.entry)
  tkgrid(rf.label, rf.entry, rb.label, rb.entry)
  tkgrid(W.label, W.entry)
  tkgrid.configure(gf.label, gf.entry, gb.label, gb.entry)
  tkgrid.configure(rf.label, rf.entry, rb.label, rb.entry)
  tkgrid.configure(W.label, W.entry)

  tkpack(mainfrm, coordvarfr, side="top")

  # Input Targets, Layout, GeneNames info
  notesvarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)

  ## Layout
  layoutFrame <- tkframe(notesvarfr)
  layoutLbl <- tklabel(layoutFrame, text="Layout")
  layoutEntry <- tkentry(layoutFrame, width=66, textvariable="layout")
  layoutBrowser <- tkbutton(layoutFrame, text = "Browse",
                            command = layBrowser)
  tkgrid(layoutLbl, columnspan = 2)
  tkgrid(layoutEntry, layoutBrowser)
  tkpack(layoutFrame, side = "top")
  tkpack(mainfrm, notesvarfr, side="top")


  ## Gene / Probe
  notesvarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)
  geneFrame <- tkframe(notesvarfr)
  geneLbl <- tklabel(geneFrame, text="Gene Information")
  geneEntry <- tkentry(geneFrame, width=66, textvariable="genenames")
  geneBrowser <- tkbutton(geneFrame, text = "Browse",
                          command = genBrowser)
  tkgrid(geneLbl, columnspan = 2)
  tkgrid(geneEntry, geneBrowser)
  tkpack(geneFrame, side = "top")
  tkpack(mainfrm, notesvarfr, side="top")

  ## Notes
  notesvarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)
  tkpack(notesvarfr, tklabel(notesvarfr, text="Notes"), side='top')
  noteEntry<-tkentry(notesvarfr, width=78, textvariable="notes")
  tkpack(notesvarfr,noteEntry, side='left')
  tkpack(mainfrm, notesvarfr, side="top")

  ## Button

  butfrm <- tkframe(mainfrm, borderwidth= 1, relief="groove")
  a.but <- tkbutton(butfrm, command=function(){
    inputRaw(tclvalue("fname"),
             sep=sep,
             skip=skip,
             quote=quote,
             ...)
             }, text="Build")
  b.but<-tkbutton(butfrm,command=function(){
    tkdelete(geneEntry, 0, "end")
    tkdelete(noteEntry, 0, "end")
    tkdelete(layoutEntry , 0, "end")
    tkdelete(gf.entry , 0, "end")
    tkdelete(rf.entry , 0, "end")
    tkdelete(gb.entry , 0, "end")
    tkdelete(rb.entry , 0, "end")
    tkdelete(mageEntry , 0, "end")
    tkdelete(targetEntry,0,"end")
    tkdelete(rawEntry , 0, "end")
    tkdelete( W.entry, 0, "end")
    },text="Clear")
  c.but <- tkbutton(butfrm, command=function()tkdestroy(base), text="Quit")
  d.but <- tkbutton(butfrm, command=function(){
                    showQuantitationTypes(tclvalue("fname"), datadir="")}
                    , text="Show Quantitation Type")
  tkpack(butfrm, a.but, b.but, c.but,  d.but, side="right")
  tkpack(mainfrm, butfrm,anchor='w')

  return(invisible())
}



###################################################################################
widget.readMageDesign<-function(
                                skip=0,
                                sep="\t",
                                quote="",
                                ...)
{
  require(tcltk)
  require(tkWidgets)

  create<-function(){
    inputLayoutGnames(skip=skip,quote=quote,sep=sep,...)
  }
  inputLayoutGnames<-function(
                              skip=skip,
                              sep="\t",
                              quote="",
                              ...)
    {
      if (tclvalue("Layout2")==""){
        writeLines("The marrayLayout object that will be created, will have the name 'Layout'")
        newLay<-"Layout"
      }
      else newLay<-tclvalue("Layout2")
      if (tclvalue("Gnames2")==""){
        writeLines("The marrayInfo object that will contain the spotted probe sequences, will have the name 'Gnames'")
        newGnames<-"Gnames"
      }
      else newGnames<-tclvalue("Gnames2")
      if (tclvalue("OneFile")==""){
        fdesignelement<-tclvalue("DesignElement")
        farraydesign<-tclvalue("ArrayDesign")
        fbioassaydata<-tclvalue("BioAssay")
        fbiosequence <- tclvalue("BioSequence")
        fName<-NULL
      }
      else{
        fbiosequence <- NULL
        fdesignelement<-NULL
        farraydesign<-NULL
        fbioassaydata<-NULL
        fName<-tclvalue("OneFile")
      }
      if (is.null(fName)){
        if( (file.access(fdesignelement, mode = 0)+file.access(farraydesign,mode=0)+file.access(fbioassaydata,mode=0))==0){
          result<-readMageDesign(datadir="",
                                 fDesignElement=fdesignelement,
                                 fArrayDesign=farraydesign,
                                 fBioAssayData=fbioassaydata,
                                 fBioSequence=fbiosequence)

        }
        else {
          if (file.access(fdesignelement, mode = 0)<0) cat(paste("\n The DesignElement filename ", fdesignelement, " is invalid.", "\n",sep=""))
          if (file.access(farraydesign, mode = 0)<0) cat(paste("\n The ArrayDesign filename  ", farraydesign, " is invalid.", "\n",sep=""))
          if (file.access(fbioassaydata, mode = 0)<0) cat(paste("\n The BioAssay filename ", fbioassaydata, " is invalid.", "\n",sep=""))
          if (file.access(fbiosequence, mode = 0)<0) cat(paste("\n The BioSequence filename ", fbiosequence, " is invalid.", "\n",sep=""))
        }
      }
      else {
        if(file.access(fName, mode=0)==0){
          result<-readMageDesign(datadir="",
                                 file=fName)

        }
        else cat(paste("\n The filename ", fName, " is invalid.", "\n",sep=""))
      }
      TempLay<-result[[1]]
      TempInfo<-result[[2]]
      assign(newLay,TempLay,env=.GlobalEnv)
  assign(newGnames,TempInfo,env=.GlobalEnv)
      cat(paste("\n finished creating a new marrayLayout object ", newLay, " and a new marrayInfo object ", newGnames, "\n", sep=""))
    }

  filesClick1 <- function(){
    fnames <<- tclvalue(tkcmd("tk_getOpenFile"))
    tkdelete(DesignEntry, 0, "end")
    tkinsert(DesignEntry,0,fnames)
  }
  filesClick2 <- function(){
    fnames <<- tclvalue(tkcmd("tk_getOpenFile"))
    tkdelete(ArrayDesignEntry, 0, "end")
    tkinsert(ArrayDesignEntry,0,fnames)
  }
  filesClick3 <- function(){
    fnames <<- tclvalue(tkcmd("tk_getOpenFile"))
    tkdelete(BioAssayEntry, 0, "end")
    tkinsert(BioAssayEntry,0,fnames)
  }
  filesClick4 <- function(){
    fnames <<- tclvalue(tkcmd("tk_getOpenFile"))
    tkdelete(BioSequenceEntry, 0, "end")
    tkinsert(BioSequenceEntry,0,fnames)
  }
  filesClick <- function(){
    fnames <<- tclvalue(tkcmd("tk_getOpenFile"))
    tkdelete(FileEntry, 0, "end")
    tkinsert(FileEntry,0,fnames)
  }

  showPack1 <- function(){
    fnames <<- tclvalue("DesignElement")
    magePackageDetector(fnames,datadir="")
  }

  showPack2 <- function(){
    fnames <<- tclvalue("ArrayDesign")
    magePackageDetector(fnames,datadir="")
  }

  showPack4 <- function(){
    fnames <<- tclvalue("BioSequence")
    magePackageDetector(fnames,datadir="")
  }
  showPack <- function(){
    fnames <<- tclvalue("OneFile")
    magePackageDetector(fnames,datadir="")
  }

  showPack3 <- function(){
    fnames <<- tclvalue("BioAssay")
    magePackageDetector(fnames,datadir="")
  }
  base <- tktoplevel()
  tkwm.title(base, "Array Design builder")
  mainfrm <- tkframe(base, borderwidth=2)
  Magevarfr <- tkframe(mainfrm, relief="groove", borderwidth=2)
  LayFrame<-tkframe(Magevarfr)
  heading4 <- tklabel(LayFrame, text="Enter Layout and Gene information from MAGE-ML files")
  tkgrid(heading4, columnspan = 4)
  layout.label<-tklabel(LayFrame, text="Name of Layout Object (default:'Layout')")
  info.label <- tklabel(LayFrame, text="  Name of Gnames Object (default:'Gnames')")
  layout.entry<-tkentry(LayFrame, width=20, textvariable = "Layout2" )
  info.entry <- tkentry(LayFrame, width=20, textvariable = "Gnames2" )
  tkgrid(layout.label,layout.entry)
  tkgrid(info.label,info.entry)

  LayLbl <- tklabel(LayFrame, text="Enter  a file containing the DesignElement_package")
  DesignEntry <- tkentry(LayFrame, width=43, textvariable="DesignElement")
  filesClick1<- tkbutton(LayFrame, text = "Browse",
                         command = filesClick1)
  showPack1<- tkbutton(LayFrame, text = "Show Packages",
                       command =  showPack1)
  tkgrid(LayLbl, columnspan =2)
  tkgrid(DesignEntry,  filesClick1, showPack1)

  LayLbl2 <- tklabel(LayFrame, text="Enter a file containing the ArrayDesign_package")
  ArrayDesignEntry <- tkentry(LayFrame, width=43, textvariable="ArrayDesign")
  filesClick2<- tkbutton(LayFrame, text = "Browse",
                         command =  filesClick2)
  showPack2<- tkbutton(LayFrame, text = "Show Packages",
                         command =  showPack2)
  tkgrid(LayLbl2, columnspan =2)
  tkgrid(ArrayDesignEntry,  filesClick2, showPack2)

  LayLbl5 <- tklabel(LayFrame, text="Enter a file containing the BioSequence_package")
  BioSequenceEntry <- tkentry(LayFrame, width=43, textvariable="BioSequence")
  filesClick4<- tkbutton(LayFrame, text = "Browse",
                         command =  filesClick4)
  showPack4<- tkbutton(LayFrame, text = "Show Packages",
                       command =  showPack4)
  tkgrid(LayLbl5, columnspan =2)
  tkgrid(BioSequenceEntry,  filesClick4, showPack4)

  LayLbl3 <- tklabel(LayFrame, text="Enter a file containing the BioAssayData_package")
  BioAssayEntry <- tkentry(LayFrame, width=43, textvariable="BioAssay")
  filesClick3<- tkbutton(LayFrame, text = "Browse",
                         command =  filesClick3)
  showPack3<- tkbutton(LayFrame, text = "Show Packages",
                       command =  showPack3)
  tkgrid(LayLbl3, columnspan =2)
  tkgrid(BioAssayEntry,filesClick3, showPack3)

  LayLbl4 <- tklabel(LayFrame, text="Or enter one single file containing all information")
  FileEntry <- tkentry(LayFrame, width=43, textvariable="OneFile")
  filesClick<- tkbutton(LayFrame, text = "Browse",
                        command = filesClick)
  showPack<- tkbutton(LayFrame, text = "Show Packages",
                      command =  showPack)
  tkgrid(LayLbl4, columnspan =2)
  tkgrid(FileEntry,  filesClick, showPack)



  a.but <- tkbutton(Magevarfr, command=function()
                    inputLayoutGnames(skip=skip,quote=quote,sep=sep,...)
                    , text="Run")
  c.but<-tkbutton(Magevarfr,command=function(){
    tkdelete(DesignEntry, 0, "end")
    tkdelete(ArrayDesignEntry, 0, "end")
    tkdelete(BioAssayEntry, 0, "end")
    tkdelete(BioSequenceEntry, 0, "end")
    tkdelete(layout.entry , 0, "end")
    tkdelete(info.entry , 0, "end")
    tkdelete(FileEntry , 0, "end")
    },text="Clear")
  b.but<-tkbutton(Magevarfr, command=function() tkdestroy(base),text="Quit")
  e.but<-tkbutton(Magevarfr, command=function() finish(),text="finish")
  d.but<-tkbutton(Magevarfr, command=function() create(),text="Run")
  tkpack(LayFrame, side = "top")
  tkpack(mainfrm, Magevarfr, side="top")
  tkpack(Magevarfr, d.but,c.but,b.but, side="right")
  tkpack(mainfrm, Magevarfr,anchor='w')
}


