params <-
list(seed = 2924)

## ---- eval=FALSE-----------------------------------------------------------
#  if (!require("BiocManager"))
#    install.packages("BiocManager")
#  BiocManager::install("glmSparseNet")

## ----packages, message=FALSE, warning=FALSE--------------------------------
library(dplyr)
library(ggplot2)
library(survival)
library(loose.rock)
library(futile.logger)
library(curatedTCGAData)
library(TCGAutils)
#
library(glmSparseNet)
#
# Some general options for futile.logger the debugging package
.Last.value <- flog.layout(layout.format('[~l] ~m'))
.Last.value <- loose.rock::show.message(FALSE)
# Setting ggplot2 default theme as minimal
theme_set(ggplot2::theme_minimal())

## ----curated_data, include=FALSE-------------------------------------------
# chunk not included as it produces to many unnecessary messages
prad <- curatedTCGAData(diseaseCode = "PRAD", assays = "RNASeq2GeneNorm", FALSE)

## ----curated_data_non_eval, eval=FALSE-------------------------------------
#  prad <- curatedTCGAData(diseaseCode = "PRAD", assays = "RNASeq2GeneNorm", FALSE)

## ----data.show, warning=FALSE, error=FALSE---------------------------------
# keep only solid tumour (code: 01)
prad.primary.solid.tumor <- TCGAutils::splitAssays(prad, '01')
xdata.raw <- t(assay(prad.primary.solid.tumor[[1]]))

# Get survival information
ydata.raw <- colData(prad.primary.solid.tumor) %>% as.data.frame %>% 
  # Find max time between all days (ignoring missings)
  rowwise %>%
  mutate(time = max(days_to_last_followup, days_to_death, na.rm = TRUE)) %>%
  # Keep only survival variables and codes
  select(patientID, status = vital_status, time) %>% 
  # Discard individuals with survival time less or equal to 0
  filter(!is.na(time) & time > 0) %>% as.data.frame

# Set index as the patientID
rownames(ydata.raw) <- ydata.raw$patientID

# keep only features that have standard deviation > 0
xdata.raw  <- xdata.raw[TCGAbarcode(rownames(xdata.raw)) %in% 
                          rownames(ydata.raw),]
xdata.raw  <- xdata.raw %>% 
  { (apply(., 2, sd) != 0) } %>% 
  { xdata.raw[, .] } %>% 
  scale

# Order ydata the same as assay
ydata.raw    <- ydata.raw[TCGAbarcode(rownames(xdata.raw)), ]

set.seed(params$seed)
small.subset <- c(geneNames(c('ENSG00000103091', 'ENSG00000064787', 
                              'ENSG00000119915', 'ENSG00000120158', 
                              'ENSG00000114491', 'ENSG00000204176', 
                              'ENSG00000138399'))$external_gene_name, 
                  sample(colnames(xdata.raw), 100)) %>% unique %>% sort

xdata <- xdata.raw[, small.subset[small.subset %in% colnames(xdata.raw)]]
ydata <- ydata.raw %>% select(time, status)

## ----fit-------------------------------------------------------------------
set.seed(params$seed)
fitted <- cv.glmHub(xdata, Surv(ydata$time, ydata$status),
                    family  = 'cox',
                    nlambda = 1000,
                    network = 'correlation', 
                    network.options = networkOptions(cutoff = .6, 
                                                     min.degree = .2))

## ----results---------------------------------------------------------------
plot(fitted)

## ----show_coefs------------------------------------------------------------
coefs.v <- coef(fitted, s = 'lambda.min')[,1] %>% { .[. != 0]}
coefs.v %>% { 
  data.frame(ensembl.id  = names(.), 
             gene.name   = geneNames(names(.))$external_gene_name, 
             coefficient = .,
             stringsAsFactors = FALSE)
  } %>%
  arrange(gene.name) %>%
  knitr::kable()

## ----hallmarks-------------------------------------------------------------
geneNames(names(coefs.v)) %>% { hallmarks(.$external_gene_name)$heatmap }

## --------------------------------------------------------------------------
separate2GroupsCox(as.vector(coefs.v), 
                   xdata[, names(coefs.v)], 
                   ydata, 
                   plot.title = 'Full dataset', legend.outside = FALSE)

