## ----style, echo = FALSE, results = 'asis'---------------------------------
BiocStyle::markdown(css.files = c('custom.css'))

## ---- echo=FALSE-----------------------------------------------------------
suppressPackageStartupMessages(
  library(tRNA)
)
data("gr", package = "tRNA", envir = environment())

## ---- eval=FALSE-----------------------------------------------------------
#  library(tRNA)
#  data("gr", package = "tRNA", envir = environment())

## --------------------------------------------------------------------------
# just get the coordinates of the anticodonloop
gettRNAstructureGRanges(gr,
                        structure = "anticodonLoop")
gettRNAstructureSeqs(gr,
                     joinFeatures = TRUE,
                     structure = "anticodonLoop")

## --------------------------------------------------------------------------
seqs <- gettRNAstructureSeqs(gr[1:10],
                             joinCompletely = TRUE)
seqs
# getting the tRNA structure boundaries
metadata(seqs)[["tRNA_structures"]]

## ----echo=TRUE, results="hide"---------------------------------------------
gr[hasAcceptorStem(gr, unpaired = TRUE)]
# mismatches and bulged are subsets of unpaired
gr[hasAcceptorStem(gr, mismatches = TRUE)]
gr[hasAcceptorStem(gr, bulged = TRUE)]
# combination of different structure parameters
gr[hasAcceptorStem(gr, mismatches = TRUE) & 
     hasDloop(gr, length = 8)]

## --------------------------------------------------------------------------
# load tRNA data for E. coli and H. sapiens
data("gr_eco", package = "tRNA", envir = environment())
data("gr_human", package = "tRNA", envir = environment())

# get summary plots
grl <- GRangesList(Sce = gr,
                   Hsa = gr_human,
                   Eco = gr_eco)
plots <- gettRNAFeaturePlots(grl)

## ----plot1, fig.cap = "tRNA length."---------------------------------------
plots$length

## ----plot2, fig.cap = "tRNAscan-SE scores."--------------------------------
plots$tRNAscan_score

## ----plot3, fig.cap = "tRNA GC content."-----------------------------------
plots$gc

## ----plot4, fig.cap = "tRNAs with introns."--------------------------------
plots$tRNAscan_intron

## ----plot5, fig.cap = "Length of the variable loop."-----------------------
plots$variableLoop_length

## ---- eval=FALSE-----------------------------------------------------------
#  # score column will be used
#  plots <- gettRNAFeaturePlots(grl,
#                               plotScores = TRUE)

## --------------------------------------------------------------------------
plots <- gettRNAFeaturePlots(grl,
                             scores = list(runif(length(grl[[1]]),0,100),
                                           runif(length(grl[[2]]),0,100),
                                           runif(length(grl[[3]]),0,100)))

## ----plot6, fig.cap = "tRNA length and score correlation."-----------------
plots$length

## ----plot7, fig.cap = "variable loop length and score correlation."--------
plots$variableLoop_length

## ----plot8, fig.cap = "Customized plot switching out the point and violin plot into a boxplot."----
plots$length$layers <- plots$length$layers[c(-1,-2)]
plots$length + ggplot2::geom_boxplot()

## --------------------------------------------------------------------------
options("tRNA_colour_palette")
options("tRNA_colour_yes")
options("tRNA_colour_no")

## --------------------------------------------------------------------------
head(gettRNABasePairing(gr)[[1]])
head(getBasePairing(gr[1]$tRNA_str)[[1]])

## --------------------------------------------------------------------------
gettRNALoopIDs(gr)[[1]]
getLoopIDs(gr[1]$tRNA_str)

## --------------------------------------------------------------------------
sessionInfo()

