params <-
list(seed = 29221)

## ---- eval=FALSE-----------------------------------------------------------
#  if (!require('BiocManager'))
#    install.packages('BiocManager')
#  BiocManager::install('glmSparseNet')

## ----packages, message=FALSE, warning=FALSE--------------------------------
library(dplyr)
library(ggplot2)
library(survival)
library(loose.rock)
library(futile.logger)
library(curatedTCGAData)
library(TCGAutils)
#
library(glmSparseNet)
#
# Some general options for futile.logger the debugging package
.Last.value <- flog.layout(layout.format('[~l] ~m'))
.Last.value <- loose.rock::show.message(FALSE)
# Setting ggplot2 default theme as minimal
theme_set(ggplot2::theme_minimal())

## ----curated_data, include=FALSE-------------------------------------------
# chunk not included as it produces to many unnecessary messages
skcm <- curatedTCGAData(diseaseCode = 'SKCM', assays = 'RNASeq2GeneNorm', FALSE)

## ----curated_data_non_eval, eval=FALSE-------------------------------------
#  skcm <- curatedTCGAData(diseaseCode = 'SKCM', assays = 'RNASeq2GeneNorm', FALSE)

## ----data.show, warning=FALSE, error=FALSE---------------------------------
skcm.metastatic <- TCGAutils::splitAssays(skcm, '06')
xdata.raw <- t(assay(skcm.metastatic[[1]]))

# Get survival information
ydata.raw <- colData(skcm.metastatic) %>% as.data.frame %>% 
  # Find max time between all days (ignoring missings)
  rowwise %>%
  mutate(time = max(days_to_last_followup, days_to_death, na.rm = TRUE)) %>%
  # Keep only survival variables and codes
  select(patientID, status = vital_status, time) %>% 
  # Discard individuals with survival time less or equal to 0
  filter(!is.na(time) & time > 0) %>% as.data.frame

# Get survival information
ydata.raw <- colData(skcm) %>% as.data.frame %>% 
  # Find max time between all days (ignoring missings)
  rowwise %>%
  mutate(time = max(days_to_last_followup, days_to_death, na.rm = TRUE)) %>%
  # Keep only survival variables and codes
  select(patientID, status = vital_status, time) %>% 
  # Discard individuals with survival time less or equal to 0
  filter(!is.na(time) & time > 0) %>% as.data.frame

# Set index as the patientID
rownames(ydata.raw) <- ydata.raw$patientID

# keep only features that have standard deviation > 0
xdata.raw      <- xdata.raw[TCGAbarcode(rownames(xdata.raw)) %in% 
                              rownames(ydata.raw),]
xdata.raw      <- xdata.raw %>% 
  { (apply(., 2, sd) != 0) } %>% 
  { xdata.raw[, .] } %>% 
  scale

# Order ydata the same as assay
ydata.raw    <- ydata.raw[TCGAbarcode(rownames(xdata.raw)), ]

set.seed(params$seed)
small.subset <- c('FOXL2', 'KLHL5', 'PCYT2', 'SLC6A10P', 'STRAP', 'TMEM33',
                  'WT1-AS', sample(colnames(xdata.raw), 100))

xdata <- xdata.raw[, small.subset[small.subset %in% colnames(xdata.raw)]]
ydata <- ydata.raw %>% select(time, status)

## ----fit-------------------------------------------------------------------
fitted <- cv.glmHub(xdata, 
                    Surv(ydata$time, ydata$status), 
                    family  = 'cox', 
                    foldid  = balanced.cv.folds(!!ydata$status)$output,
                    network = 'correlation', 
                    network.options = networkOptions(min.degree = .2, 
                                                     cutoff = .6))

## ----results---------------------------------------------------------------
plot(fitted)

## ----show_coefs------------------------------------------------------------
coefs.v <- coef(fitted, s = 'lambda.min')[,1] %>% { .[. != 0]}
coefs.v %>% { 
  data.frame(ensembl.id  = names(.), 
             gene.name   = geneNames(names(.))$external_gene_name, 
             coefficient = .,
             stringsAsFactors = FALSE)
  } %>%
  arrange(gene.name) %>%
  knitr::kable()

## ----hallmarks-------------------------------------------------------------
geneNames(names(coefs.v)) %>% { hallmarks(.$external_gene_name)$heatmap }

## --------------------------------------------------------------------------
separate2GroupsCox(as.vector(coefs.v), 
                   xdata[, names(coefs.v)], 
                   ydata, 
                   plot.title = 'Full dataset', legend.outside = FALSE)

