## ----setup, include=FALSE--------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)

## ---- eval=FALSE-----------------------------------------------------------
#  # new S4 function (note the capital W)
#  ?Waterfall
#  
#  # older established function
#  ?waterfall

## ---- message=FALSE, tidy=TRUE---------------------------------------------
# set a seed
set.seed(426)

# load GenVisR into R
library(GenVisR)

## ---- message=FALSE, tidy=TRUE---------------------------------------------
# get the disk location for maf test file
testFileDir <- system.file("extdata", package="GenVisR")
testFile <- Sys.glob(paste0(testFileDir, "/brca.maf"))

# define the objects for testing
mafObject <- MutationAnnotationFormat(testFile)

## ---- message=FALSE, tidy=TRUE---------------------------------------------
# get the disk location for test files
testFileDir <- system.file("extdata", package="GenVisR")
testFile <- Sys.glob(paste0(testFileDir, "/*.vep"))

# define the object for testing
vepObject <- VEP(testFile)

## ---- tidy=TRUE------------------------------------------------------------
# get the disk location for test files
testFileDir <- system.file("extdata", package="GenVisR")
testFile <- Sys.glob(paste0(testFileDir, "/FL.gms"))

# define the objects for testing
gmsObject <- GMS(testFile)

## ---- tidy=TRUE, eval=FALSE------------------------------------------------
#  # view the samples from the VEP file
#  getSample(vepObject)

## ---- tidy=TRUE, warning=FALSE---------------------------------------------
waterfallPlot <- Waterfall(vepObject, recurrence=.40)

## ----eval=FALSE------------------------------------------------------------
#  # extract data by the slot name
#  getData(waterfallPlot, name="primaryData")
#  
#  # extract data by the slot index (same as above)
#  getData(waterfallPlot, index=1)

## ----eval=FALSE------------------------------------------------------------
#  # draw the plot
#  drawPlot(waterfallPlot)
#  
#  # draw the plot and save it to a pdf
#  pdf(file="waterfall.pdf", height=10, width=15)
#  drawPlot(waterfallPlot)
#  dev.off()

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE----
# draw a waterfall plot for the maf object
drawPlot(Waterfall(vepObject, recurrence=.20))

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE----
# show those genes which recur in 50% of the cohort for these 4 samples
drawPlot(Waterfall(vepObject, recurrence=.50, samples=c("FLX0040Naive", "FLX0070Naive", "FLX0050Naive", "FLX0030Naive")))

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE----
# define a coverage for each sample
sampCov <- c("FLX0040Naive"=1.45e7, "FLX0070Naive"=1.39e7, "FLX0050Naive"=1.21e7, "FLX0030Naive"=1.3e7, "FLX0010Naive"=1.1e7)

drawPlot(Waterfall(vepObject, recurrence=.50, coverage=sampCov, plotA="burden", plotATally="complex", drop=FALSE))

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE----
# find which mutations are in the input data
mutations <- getMutation(vepObject)$Conse

# define a new color and hierarchy for mutations
library(data.table)
newHierarchy <- data.table("mutation"=c("splice_region_variant", "splice_acceptor_variant",  "splice_donor_variant", "missense_variant", "stop_gained"), "color"=c("tomato1", "tomato2", "tomato3", "purple", "cyan"))

# draw the plot
drawPlot(Waterfall(vepObject, recurrence=.50, mutationHierarchy = newHierarchy, plotATally="complex", drop=FALSE))

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE, message=FALSE----
# determine the appropriate BSgenome object to use from the vep header
assembly <- getHeader(vepObject)
assembly <- assembly[grepl("assembly", assembly$Info),]

# load in the correct BSgenome object
library(BSgenome.Hsapiens.UCSC.hg19)

# create a MutSpectra plot
drawPlot(MutSpectra(vepObject, BSgenome=BSgenome.Hsapiens.UCSC.hg19))

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE, message=FALSE----
# determine the appropriate BSgenome object to use from the vep header
assembly <- getHeader(vepObject)
assembly <- assembly[grepl("assembly", assembly$Info),]

# load in the correct BSgenome object
library(BSgenome.Hsapiens.UCSC.hg19)

# create a Rainfall plot
drawPlot(Rainfall(vepObject, BSgenome=BSgenome.Hsapiens.UCSC.hg19))

## ---- fig.keep='last', fig.width=10, fig.height=6.5, tidy=TRUE, warning=FALSE, message=FALSE----
# create a Rainfall plot limiting the chromosomes and samples plotted
drawPlot(Rainfall(vepObject, BSgenome=BSgenome.Hsapiens.UCSC.hg19, sample=c("FLX0010Naive"), chromosomes=c("chr1", "chr2", "chr3")))

## ---- message=FALSE, tidy=TRUE---------------------------------------------
sessionInfo()

