## ---- echo=FALSE---------------------------------------------------------

    library(knitr)

    opts_chunk$set(tidy = TRUE, message = FALSE, warning = FALSE)


## ----getPackage, eval=FALSE----------------------------------------------
#  
#      if (!requireNamespace("BiocManager", quietly = TRUE))
#  
#          install.packages("BiocManager")
#  
#          BiocManager::install("EnhancedVolcano")
#  

## ----getPackageDevel, eval=FALSE-----------------------------------------
#  
#      devtools::install_github("kevinblighe/EnhancedVolcano")
#  

## ----Load, message=FALSE-------------------------------------------------

    library(EnhancedVolcano)


## ------------------------------------------------------------------------

    library(airway)

    library(magrittr)

    data("airway")

    airway$dex %<>% relevel("untrt")


## ------------------------------------------------------------------------

    library("DESeq2")

    dds <- DESeqDataSet(airway, design = ~ cell + dex)
    dds <- DESeq(dds, betaPrior=FALSE)
    res1 <- results(dds,
        contrast = c("dex","trt","untrt"))
    res1 <- lfcShrink(dds,
        contrast = c("dex","trt","untrt"), res=res1)
    res2 <- results(dds,
        contrast = c("cell", "N061011", "N61311"))
    res2 <- lfcShrink(dds,
        contrast = c("cell", "N061011", "N61311"), res=res2)


## ----ex1, fig.height = 8, fig.width = 6, fig.cap = "Plot the most basic volcano plot."----

    EnhancedVolcano(res1,

        lab = rownames(res1),

        x = "log2FoldChange",

        y = "pvalue")


## ----ex2, fig.height = 8, fig.width = 6, fig.cap = "Modify cut-offs for log2FC and P value; add title; adjust point and label size."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "pvalue",

        pCutoff = 10e-12,

        FCcutoff = 1.5,

        transcriptPointSize = 1.5,

        transcriptLabSize = 3.0,

        title = "N061011 versus N61311")


## ----ex3, fig.height = 8, fig.width = 6, fig.cap = "Adjust colour and alpha for point shading."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "pvalue",

        pCutoff = 10e-12,

        FCcutoff = 1.5,

        transcriptPointSize = 1.5,

        transcriptLabSize = 3.0,

        title = "N061011 versus N61311",

        col=c("black", "black", "black", "red3"),

        colAlpha = 1)


## ----ex4, fig.height = 8, fig.width = 9, fig.cap = "Adjust axis limits."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "pvalue",

        pCutoff = 10e-12,

        FCcutoff = 1.5,

        transcriptPointSize = 1.5,

        transcriptLabSize = 3.0,

        title = "N061011 versus N61311",

        colAlpha = 1,

        xlim = c(-8, 8),

        ylim = c(0, -log10(10e-32)))


## ----ex5, fig.height = 8, fig.width = 9, fig.cap = "Adjust cut-off lines."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "pvalue",

        pCutoff = 10e-12,

        FCcutoff = 1.5,

        transcriptPointSize = 1.5,

        transcriptLabSize = 3.0,

        title = "N061011 versus N61311",

        colAlpha = 1,

        xlim = c(-8, 8),

        ylim = c(0, -log10(10e-32)),

        cutoffLineType = "twodash",

        cutoffLineCol = "red3",

        cutoffLineWidth = 1.5)


## ----ex6, fig.height = 8, fig.width = 12, fig.cap = "Adjust legend position, size, and text."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "pvalue",

        pCutoff = 10e-12,

        FCcutoff = 1.5,

        transcriptPointSize = 1.5,

        transcriptLabSize = 3.0,

        colAlpha = 1,

        cutoffLineType = "twodash",

        cutoffLineCol = "red4",

        cutoffLineWidth = 1.0,

        legend=c("NS","Log (base 2) fold-change","P value",
            "P value & Log (base 2) fold-change"),

        legendPosition = "right",

        legendLabSize = 14,

        legendIconSize = 5.0)


## ----eval=FALSE----------------------------------------------------------
#  
#  legend=c("","","",""), legendLabSize=-1, legendIconSize=-1
#  

## ----ex7, fig.height = 8, fig.width = 6, fig.cap = "Plot adjusted p-values."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.0001,

        FCcutoff = 1.0,

        xlim=c(-6,6),

        transcriptLabSize = 3.0,

        colAlpha = 1,

        legend=c("NS","Log2 FC","Adjusted p-value",
            "Adjusted p-value & Log2 FC"),

        legendPosition = "bottom",

        legendLabSize = 10,

        legendIconSize = 3.0)


## ----ex8, fig.height = 8, fig.width = 6.5, fig.cap = "Fit more labels by adding connectors."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.0001,

        FCcutoff = 2.0,

        xlim = c(-6,6),

        transcriptLabSize = 3.0,

        colAlpha = 1,

        legend=c("NS","Log2 FC","Adjusted p-value",
            "Adjusted p-value & Log2 FC"),

        legendPosition = "bottom",

        legendLabSize = 10,

        legendIconSize = 3.0,

        DrawConnectors = TRUE,

        widthConnectors = 0.2,

        colConnectors = "grey30")


## ----ex9, fig.height = 8, fig.width = 6, fig.cap = "Only label key transcripts."----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        selectLab = c("ENSG00000106565","ENSG00000187758"),

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.0001,

        FCcutoff = 2.0,

        xlim = c(-6,6),

        transcriptPointSize = 1.8,

        transcriptLabSize = 5.0,

        colAlpha = 1,

        legend=c("NS","Log2 FC","Adjusted p-value",
            "Adjusted p-value & Log2 FC"),

        legendPosition = "bottom",

        legendLabSize = 10,

        legendIconSize = 3.0)


## ----ex10, fig.height = 8, fig.width = 6, fig.cap = "Modify border and remove gridlines"----

    EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        selectLab = c("ENSG00000106565","ENSG00000187758"),

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.00000000000000001,

        FCcutoff = 2.0,

        xlim = c(-6,6),

        transcriptLabSize = 3.0,

        colAlpha = 1,

        legend=c("NS","Log2 FC","Adjusted p-value",
            "Adjusted p-value & Log2 FC"),

        legendPosition = "bottom",

        legendLabSize = 10,

        legendIconSize = 3.0,

        DrawConnectors = FALSE,

        border = "full",

        borderWidth = 1.5,

        borderColour = "black",

        gridlines.major = FALSE,

        gridlines.minor = FALSE)


## ----ex11, fig.height = 8, fig.width = 11, fig.cap = "Plot multiple volcanos on the same page."----

    p1 <- EnhancedVolcano(res1,

        lab = rownames(res1),

        x = "log2FoldChange",

        y = "pvalue",

        pCutoff = 10e-24,

        FCcutoff = 2.0,

        transcriptLabSize = 2.5,

        colAlpha = 1,

        legendPosition = "bottom",

        legendLabSize = 10,

        legendIconSize = 3.0)


    p2 <- EnhancedVolcano(res2,

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        selectLab = c("ENSG00000106565","ENSG00000187758"),

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.0001,

        FCcutoff = 2.0,

        xlim = c(-6,6),

        transcriptLabSize = 5.0,

        colAlpha = 1,

        legend=c("NS","Log2 FC","Adjusted p-value",
            "Adjusted p-value & Log2 FC"),

        legendPosition = "bottom",

        legendLabSize = 10,

        legendIconSize = 3.0)


    library(gridExtra)

    library(grid)

    grid.arrange(p1, p2, ncol=2, top="EnhancedVolcano")

    grid.rect(gp=gpar(fill=NA))


## ----ex12, fig.height = 8, fig.width = 11, fig.cap = "Override colour"----

    # create custom key-value pairs for high, low, mid expression
    keyvals <- rep("black", nrow(res2))
    names(keyvals) <- rep("Mid / NA", nrow(res2))

    keyvals[which(res2$log2FoldChange > 2)] <- "gold"
    names(keyvals)[which(res2$log2FoldChange > 2)] <- "High"

    keyvals[which(res2$log2FoldChange < -2)] <- "royalblue"
    names(keyvals)[which(res2$log2FoldChange < -2)] <- "Low"

    p1 <- EnhancedVolcano(res2,

        title = "Over-ride",

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        selectLab = rownames(res2)[which(names(keyvals) %in% c("High", "Low"))],

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.00000000000000001,

        FCcutoff = 1.0,

        xlim = c(-6,6),

        transcriptLabSize = 2.0,

        colAlpha = 1,

        legendPosition = "top",

        transcriptPointSize = 1.2,

        legendLabSize = 15,

        legendIconSize = 5.0,

        DrawConnectors = TRUE,

        widthConnectors = 0.3,

        colConnectors = "grey50",

        border = "partial",

        borderWidth = 1.5,

        borderColour = "black",

        gridlines.major = TRUE,

        gridlines.minor = FALSE,

        colOverride = keyvals)


    p2 <- EnhancedVolcano(res2,

        title = "No over-ride",

        lab = rownames(res2),

        x = "log2FoldChange",

        y = "padj",

        selectLab = rownames(res2)[which(names(keyvals) %in% c("High", "Low"))],

        xlab = bquote(~Log[2]~ "fold change"),

        ylab = bquote(~-Log[10]~adjusted~italic(P)),

        pCutoff = 0.00000000000000001,

        FCcutoff = 1.0,

        xlim = c(-6,6),

        transcriptLabSize = 2.0,

        colAlpha = 1,

        legendPosition = "top",

        transcriptPointSize = 1.2,

        legendLabSize = 15,

        legendIconSize = 5.0,

        DrawConnectors = FALSE,

        widthConnectors = 0.3,

        colConnectors = "grey50",

        border = "full",

        borderWidth = 1.0,

        borderColour = "black",

        gridlines.major = TRUE,

        gridlines.minor = FALSE,

        colOverride = NULL)


    library(gridExtra)

    library(grid)

    grid.arrange(p1, p2, ncol=2, top="EnhancedVolcano")

    grid.rect(gp=gpar(fill=NA))


## ------------------------------------------------------------------------

sessionInfo()


