## ---- fig.show='hold', message=FALSE, warning=FALSE, eval=FALSE------------
#  ## try http:// if https:// URLs are not supported
#  source("https://bioconductor.org/biocLite.R")
#  biocLite("hipathia")

## ---- fig.show='hold', message=FALSE, warning=FALSE------------------------
library(hipathia)
data("brca")
brca

## ---- fig.show='hold', message=FALSE, warning=FALSE------------------------
hhead(assay(brca), 4)

## ---- fig.show='hold', message=FALSE, warning=FALSE------------------------
colData(brca)

## ---- fig.show='hold', message=FALSE, eval=FALSE---------------------------
#  brca <- SummarizedExperiment(assays=SimpleList(raw=brca_data),
#                               colData=brca_design)

## ---- fig.show='hold'------------------------------------------------------
data(brca_data)
trans_data <- translate_data(brca_data, "hsa")

## ---- fig.show='hold'------------------------------------------------------
exp_data <- normalize_data(trans_data)

## ---- fig.show='hold', fig.cap="BRCA data before scaling"------------------
boxplot(trans_data)

## ---- fig.show='hold', fig.cap="BRCA data after scaling"-------------------
boxplot(exp_data)

## ---- fig.show='hold', fig.cap="BRCA data after a Quantiles normalization"----
exp_data <- normalize_data(trans_data, by_quantiles = TRUE)
boxplot(exp_data)

## ---- fig.show='hold', fig.cap="BRCA data after normalizing by percentil"----
exp_data <- normalize_data(trans_data, percentil = TRUE)
boxplot(exp_data)

## ---- fig.show='hold', fig.cap="BRCA data after truncating by percentil 0.95"----
exp_data <- normalize_data(trans_data, truncation_percentil = 0.95)
boxplot(exp_data)

## ---- fig.show='hold'------------------------------------------------------
pathways <- load_pathways(species = "hsa")

## ---- fig.show='hold'------------------------------------------------------
pathways_only2 <- load_pathways(species = "hsa", pathways_list = c("hsa03320",
                                                                   "hsa04014"))

## ---- fig.show='hold'------------------------------------------------------
length(get_pathways_list(pathways))
get_pathways_list(pathways)[1:10]

## ---- fig.show='hold'------------------------------------------------------
length(get_pathways_list(pathways_only2))
get_pathways_list(pathways_only2)

## ---- fig.show='hold'------------------------------------------------------
results <- hipathia(exp_data, pathways, decompose = FALSE, verbose=FALSE)

## ---- fig.show='hold'------------------------------------------------------
results

## ---- fig.show='hold'------------------------------------------------------
path_vals <- get_paths_data(results, matrix = TRUE)
path_vals <- get_paths_data(results)
hhead(path_vals, 4)

## ---- echo=FALSE, results='asis'-------------------------------------------
tab <- t(sapply(c("hsa", "mmu", "rno"), function(species){
    p <- suppressMessages(load_pathways(species))
    effs <- sum(sapply(p$pathigraphs, function(pathi) length(
        pathi$effector.subgraphs)))
    decs <- sum(sapply(p$pathigraphs, function(pathi) length(pathi$subgraphs)))
    n <- length(p$pathigraphs)
    c(n, effs, decs)
}))
colnames(tab) <- c("Pathways", "Effector subpathways", "Decomposed subpathways")
knitr::kable(tab)

## ---- fig.show='hold'------------------------------------------------------
uniprot_vals <- quantify_terms(results, pathways, dbannot = "uniprot")
go_vals <- quantify_terms(results, pathways, dbannot = "GO")

## ---- fig.show='hold'------------------------------------------------------
sample_group <- brca_design[colnames(path_vals),"group"]
comp <- do_wilcoxon(path_vals, sample_group, g1 = "Tumor", g2 = "Normal")
hhead(comp)

## ---- fig.show='hold'------------------------------------------------------
pathways_summary <- get_pathways_summary(comp, pathways)
head(pathways_summary, 4)

## ---- fig.show='hold'------------------------------------------------------
ranked_path_vals <- path_vals[order(comp$p.value, decreasing = FALSE),]
pca_model <- do_pca(ranked_path_vals[1:ncol(ranked_path_vals),])

## ---- fig.show='hold', fig.cap="Heatmap plot", fig.small=TRUE--------------
heatmap_plot(path_vals, group = sample_group)

## ---- fig.show='hold', fig.cap="Heatmap plots with variable clustering", fig.small=TRUE----
heatmap_plot(uniprot_vals, group = sample_group, colors="hipathia", 
          variable_clust = TRUE)

## ---- fig.show='hold', fig.cap="Different colors of heatmaps: `redgreen`", fig.small=TRUE----
heatmap_plot(go_vals, group = sample_group, colors="redgreen", 
          variable_clust = TRUE)

## ---- fig.show='hold', fig.cap="PCA plot"----------------------------------
pca_plot(pca_model, sample_group, legend = TRUE)

## ---- fig.show='hold', fig.cap="PCA plot with 5 random colors"-------------
pca_plot(pca_model, group = rep(1:5, 8), main = "Random types", 
      legend = TRUE)

## ---- fig.show='hold', fig.cap="Multiple PCA plot with acumulated explained variance"----
multiple_pca_plot(pca_model, sample_group, cex=3, plot_variance = TRUE)

## ---- fig.show='hold', fig.cap= "Pathway comparison plot without node colors"----
pathway_comparison_plot(comp, metaginfo = pathways, pathway = "hsa03320")

## ---- fig.show='hold', fig.cap="Pathway comparison plot with node colors: `classic`"----
colors_de <- node_color_per_de(results, pathways, sample_group, "Tumor", 
                            "Normal")
pathway_comparison_plot(comp, metaginfo = pathways, pathway = "hsa03320", 
                     node_colors = colors_de)

## ---- fig.show='hold', fig.cap="Pathway comparison plot with node colors: `hipathia`"----
colors_de_hipathia <- node_color_per_de(results, pathways, sample_group, 
                                     "Tumor", "Normal", colors = "hipathia")
pathway_comparison_plot(comp, metaginfo = pathways, pathway = "hsa03320", 
                     node_colors = colors_de_hipathia, colors = "hipathia")

## ---- fig.show='hold'------------------------------------------------------
report <- create_report(comp, pathways, "save_noColors")
report_colors <- create_report(comp, pathways, "save_colors", 
                               node_colors = colors_de)

## ---- fig.show='hold'------------------------------------------------------
visualize_report(report_colors)

## ---- fig.show='hold'------------------------------------------------------
visualize_report(report, port = 4001)

## ---- fig.show='hold'------------------------------------------------------
servr::daemon_stop()

## ---- fig.show='hold', eval=FALSE------------------------------------------
#  colors_de_uni <- node_color_per_de(results, pathways, sample_group, "Tumor",
#                                  "Normal", group_by = "uniprot")
#  create_report(comp, pathways, "save_colors_uniprot",
#             node_colors = colors_de_uni, group_by = "uniprot")
#  visualize_report("save_colors_uniprot", port = 4002)

## ---- fig.show='hold'------------------------------------------------------
servr::daemon_stop()

## ---- fig.show='hold', message=FALSE, warning=FALSE------------------------
class(brca)
hhead(brca, 4)

## ---- fig.show='hold', message=FALSE, warning=FALSE------------------------
class(assay(brca))
hhead(assay(brca), 4)

## ---- fig.show='hold'------------------------------------------------------
get_path_names(pathways, c("P-hsa03320-37", "P-hsa04010-15"))

