## ----GlobalOptions, results="hide", include=FALSE, cache=FALSE-------------
knitr::opts_chunk$set(fig.align="center", cache=FALSE, cache.path = "clusterExperimentTutorial_cache/",  fig.path="clusterExperimentTutorial_figure/",error=FALSE, #make it stop on error
fig.width=6,fig.height=6,autodep=TRUE,out.width="600px",out.height="600px",
message=FALSE)
#knitr::opts_knit$set(stop_on_error = 2L) #really make it stop
#knitr::dep_auto()

## ----cache=FALSE-----------------------------------------------------------
set.seed(14456) ## for reproducibility, just in case
library(scRNAseq)
data("fluidigm")

## --------------------------------------------------------------------------
assay(fluidigm)[1:5,1:10]
colData(fluidigm)[,1:5]
NCOL(fluidigm) #number of samples

## ----filter_high-----------------------------------------------------------
se <- fluidigm[,colData(fluidigm)[,"Coverage_Type"]=="High"]

## ----filter----------------------------------------------------------------
wh_zero <- which(rowSums(assay(se))==0)
pass_filter <- apply(assay(se), 1, function(x) length(x[x >= 10]) >= 10)
se <- se[pass_filter,]
dim(se)

## ----normalization---------------------------------------------------------
fq <- round(limma::normalizeQuantiles(assay(se)))
assays(se) <- list(normalized_counts=fq)

## ----colnames--------------------------------------------------------------
wh<-which(colnames(colData(se)) %in% c("Cluster1","Cluster2"))
colnames(colData(se))[wh]<-c("Published1","Published2")

## ----RSEC, eval=FALSE------------------------------------------------------
#  library(clusterExperiment)
#  system.time(rsecFluidigm<-RSEC(se, isCount = TRUE,
#  	reduceMethod="PCA", nReducedDims=10,combineMinSize=3,
#  	ncores=1, random.seed=176201))

## ----RSECLoad--------------------------------------------------------------
#don't call this routine if you ran the above code.
#it will overwrite the rsecFluidigm you made
library(clusterExperiment)
data("rsecFluidigm")

## ----RSECprint-------------------------------------------------------------
rsecFluidigm

## ----primaryCluster--------------------------------------------------------
head(primaryCluster(rsecFluidigm),20)

## ----clusterMatrix---------------------------------------------------------
head(clusterMatrix(rsecFluidigm)[,1:4])

## ----clusterLabels---------------------------------------------------------
head(clusterLabels(rsecFluidigm))

## ----clusterParams---------------------------------------------------------
head(getClusterManyParams(rsecFluidigm))

## ----plotClusterRSEC-------------------------------------------------------
defaultMar<-par("mar")
plotCMar<-c(1.1,8.1,4.1,1.1)
par(mar=plotCMar)
plotClusters(rsecFluidigm,main="Clusters from RSEC", whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), axisLine=-1)

## ----plotClustersWorkflow--------------------------------------------------
par(mar=plotCMar)
plotClustersWorkflow(rsecFluidigm)

## ----plotBarplotRSEC-------------------------------------------------------
plotBarplot(rsecFluidigm,main="Final Clustering")

## ----plotBarplotRSEC.2-----------------------------------------------------
plotBarplot(rsecFluidigm,whichClusters=c("combineMany" ))

## ----plotBarplot2----------------------------------------------------------
plotBarplot(rsecFluidigm,whichClusters=c("mergeClusters" ,"combineMany"))

## ----plotCoClustering_rsec-------------------------------------------------
plotCoClustering(rsecFluidigm,whichClusters=c("mergeClusters","combineMany"))

## ----plotDendroRSEC--------------------------------------------------------
plotDendrogram(rsecFluidigm,whichClusters=c("combineMany","mergeClusters"),plotType="colorblock",leafType="sample")

## ----plotReducedDims-------------------------------------------------------
plotReducedDims(rsecFluidigm)

## ----plotReducedDimsMany---------------------------------------------------
plotReducedDims(rsecFluidigm,whichDims=c(1:4))

## ----recallRSEC------------------------------------------------------------
rsecFluidigm<-RSEC(rsecFluidigm,isCount=TRUE,combineProportion=0.6,mergeMethod="JC",mergeCutoff=0.05)

## ----plotClusterRSECRecall-------------------------------------------------
defaultMar<-par("mar")
plotCMar<-c(1.1,8.1,4.1,1.1)
par(mar=plotCMar)
plotClusters(rsecFluidigm,main="Clusters from RSEC", whichClusters=c("mergeClusters.1","combineMany.1","mergeClusters","combineMany"), sampleData=c("Biological_Condition","Published2"), axisLine=-1)

## ----plotClusterRSECReset--------------------------------------------------
rsecFluidigm<-RSEC(rsecFluidigm,combineMinSize=3)

## ----getBestFeatures-------------------------------------------------------
pairsAllRSEC<-getBestFeatures(rsecFluidigm,contrastType="Pairs",p.value=0.05,
                          number=nrow(rsecFluidigm))
head(pairsAllRSEC)

## ----getBestFeatures_size--------------------------------------------------
length(pairsAllRSEC$Feature)==length(unique(pairsAllRSEC$Feature))

## ----getBestFeatures_heatmap-----------------------------------------------
plotHeatmap(rsecFluidigm, whichClusters=c("combineMany","mergeClusters"),clusterSamplesData="dendrogramValue",
            clusterFeaturesData=unique(pairsAllRSEC[,"IndexInOriginal"]),
            main="Heatmap of features w/ significant pairwise differences",
            breaks=.99)

## ----plotContrastHeatmap---------------------------------------------------
plotContrastHeatmap(rsecFluidigm, signif=pairsAllRSEC,nBlankLines=40,whichCluster="primary")

## ----clusterMany-----------------------------------------------------------
ce<-clusterMany(se, clusterFunction="pam",ks=5:10, minSizes=5,
      isCount=TRUE,reduceMethod=c("PCA","var"),nFilterDims=c(100,500,1000),
      nReducedDims=c(5,15,50),run=TRUE)

## ----plotClusterEx1--------------------------------------------------------
defaultMar<-par("mar")
par(mar=plotCMar)
plotClusters(ce,main="Clusters from clusterMany", whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), axisLine=-1)

ce<-clusterMany(se, clusterFunction="pam",ks=5:10, minSizes=5,
      isCount=TRUE,reduceMethod=c("PCA","var"),nFilterDims=c(100,500,1000),
      nReducedDims=c(5,15,50),run=TRUE)
test<-getClusterManyParams(ce)
ord<-order(test[,"k"],test[,"nFilterDims"],test[,"nReducedDims"])
plotClusters(ce,main="Clusters from clusterMany", whichClusters=test$clusteringIndex[ord], sampleData=c("Biological_Condition","Published2"), axisLine=-1)

## ----plotCluster_params----------------------------------------------------
cmParams<-getClusterManyParams(ce)
head(cmParams)

## ----plotCluster_newOrder--------------------------------------------------
ord<-order(cmParams[,"reduceMethod"],cmParams[,"nReducedDims"])
ind<-cmParams[ord,"clusteringIndex"]
par(mar=plotCMar)
plotClusters(ce,main="Clusters from clusterMany", whichClusters=ind, sampleData=c("Biological_Condition","Published2"), axisLine=-1)

## ----plotCluster_newLabels-------------------------------------------------
clOrig<-clusterLabels(ce)
clOrig<-gsub("Features","",clOrig)
plotClusters(ce,main="Clusters from clusterMany", whichClusters=ind, clusterLabels=clOrig[ind], sampleData=c("Biological_Condition","Published2"), axisLine=-1)

## --------------------------------------------------------------------------
ce<-combineMany(ce,proportion=1)

## ----lookAtCombineMany-----------------------------------------------------
head(clusterMatrix(ce)[,1:3])
par(mar=plotCMar)
plotClustersWorkflow(ce)


## ----combineMany_changeLabel-----------------------------------------------
wh<-which(clusterLabels(ce)=="combineMany")
if(length(wh)!=1) stop() else clusterLabels(ce)[wh]<-"combineMany,1"

## ----combineMany_newCombineMany--------------------------------------------
ce<-combineMany(ce,proportion=0.7,clusterLabel="combineMany,0.7")
par(mar=plotCMar)
plotClustersWorkflow(ce)


## ----combineMany_changeMinSize---------------------------------------------
ce<-combineMany(ce,proportion=0.7,minSize=3,clusterLabel="combineMany,final")
par(mar=plotCMar)
plotClustersWorkflow(ce,whichClusters=c("combineMany,final","combineMany,0.7","combineMany,1"),main="Min. Size=3")

## ----plotCoClustering_quickstart-------------------------------------------
plotCoClustering(ce)

## ----makeDendrogram--------------------------------------------------------
ce<-makeDendrogram(ce,reduceMethod="var",nDims=500)
plotDendrogram(ce)

## ----makeDendrogram2-------------------------------------------------------
plotDendrogram(ce,plotType="colorblock",leafType="sample")

## ----makeDendrogram_show---------------------------------------------------
ce

## ----mergeClustersPlot-----------------------------------------------------
mergeClusters(ce,mergeMethod="adjP",plotInfo="mergeMethod")

## ----mergeClusters---------------------------------------------------------
ce<-mergeClusters(ce,mergeMethod="adjP",cutoff=0.01)
par(mar=plotCMar)

## ----mergeClusters_coClustering--------------------------------------------
plotClustersWorkflow(ce,whichClusters="workflow") #, sampleData=c("Biological_Condition","Published2")
plotCoClustering(ce,whichClusters=c("mergeClusters","combineMany"),
                 sampleData=c("Biological_Condition","Published2"),annLegend=FALSE)

## ----plotHeatmap-----------------------------------------------------------
plotHeatmap(ce,clusterSamplesData="dendrogramValue",breaks=.99,
            sampleData=c("Biological_Condition", "Published1", "Published2"))

## ----quasiRsecCode, eval=FALSE---------------------------------------------
#  rsecOut<-RSEC(se, reduceMethod="PCA", nReducedDims=c(50,10), k0s=4:15,
#          alphas=c(0.1,0.2,0.3),betas=c(0.8,0.9), minSizes=c(1,5), clusterFunction="hierarchical01",
#          combineProportion=0.7, combineMinSize=5,
#          dendroReduce="mad",dendroNDims=500,
#          mergeMethod="adjP",mergeCutoff=0.05,
#  		)

## ----stepCode,eval=FALSE---------------------------------------------------
#  ce<-clusterMany(se,ks=4:15,alphas=c(0.1,0.2,0.3),betas=c(0.8,0.9),minSizes=c(1,5),
#  	clusterFunction="hierarchical01", sequential=TRUE,subsample=TRUE,
#           reduceMethod="PCA",nFilterDims=c(50,10))
#  ce<-combineMany(ce, proportion=0.7, minSize=5)
#  ce<-makeDendrogram(ce,reduceMethod="mad",nDims=500)
#  ce<-mergeClusters(ce,mergeMethod="adjP",cutoff=0.05,plot=FALSE)

## ----show------------------------------------------------------------------
ce

## ----CEHelperCommands1-----------------------------------------------------
head(clusterMatrix(ce))[,1:5]
primaryCluster(ce)

## ----CEHelperCommands2-----------------------------------------------------
head(clusterLabels(ce),10)

## ----CEHelperCommands3-----------------------------------------------------
head(clusterTypes(ce),10)

## ----SECommandsOnCE--------------------------------------------------------
colData(ce)[,1:5]

## ----CEClusterLengend------------------------------------------------------
length(clusterLegend(ce))
clusterLegend(ce)[1:2]

## ----CEClusterLengendAssign------------------------------------------------
existingColors<-clusterLegend(ce)[[1]]
existingColors[,"name"]<-gsub("m","M",existingColors[,"name"])
clusterLegend(ce)[[1]]<-existingColors
print(ce)

## ----plotClusterEx1_redo---------------------------------------------------
par(mar=plotCMar)
plotClusters(ce,main="Clusters from clusterMany", whichClusters="workflow", 
             axisLine=-1)

## ----plotClusterEx1_newOrder-----------------------------------------------
par(mar=plotCMar)
plotClusters(ce,whichClusters="clusterMany",
               main="Only Clusters from clusterMany",axisLine=-1)

## ----plotClusterEx1_addData------------------------------------------------
par(mar=plotCMar)
plotClusters(ce,whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), 
               main="Workflow clusters plus other data",axisLine=-1)

## ----plotClusterEx1_origColors---------------------------------------------
clusterLegend(ce)[c("mergeClusters","combineMany,final")]

## ----plotClusterLegend_before,out.width="300px",out.height="300px"---------
plotClusterLegend(ce,whichCluster="combineMany,final")

## ----plotClusterEx1_setColors----------------------------------------------
ce_temp<-plotClusters(ce,whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), 
               main="Cluster Alignment of Workflow Clusters",clusterLabels=FALSE, axisLine=-1, resetNames=TRUE,resetColors=TRUE,resetOrderSamples=TRUE)

## ----plotClusterEx1_newColors----------------------------------------------
clusterLegend(ce_temp)[c("mergeClusters","combineMany,final")]

## ----plotClusterEx1_clusterLegend------------------------------------------
ce_temp<-ce
clusterLegend(ce_temp)[["mergeClusters"]]

## ----plotClusterEx1_assignColors-------------------------------------------
clusterLegend(ce_temp)[["mergeClusters"]][,"color"]<-c("white","blue","green","cyan","purple")
clusterLegend(ce_temp)[["mergeClusters"]][,"name"]<-c("Not assigned","Cluster1","Cluster2","Cluster3","Cluster4")

## ----plotClusterLegend,out.width="300px",out.height="300px"----------------
plotClusterLegend(ce_temp,whichCluster="mergeClusters")

## ----plotClusterEx1_firstOnlyNoSave----------------------------------------
plotClusters(ce_temp,whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), clusterLabels=FALSE,
               main="Clusters from clusterMany, different order",axisLine=-1,existingColors="firstOnly")

## ----plotClusterEx1_firstOnly----------------------------------------------
ce_temp<-plotClusters(ce_temp,whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), resetColors=TRUE,
               main="Clusters from clusterMany, different order",axisLine=-1, clusterLabels=FALSE, existingColors="firstOnly",plot=FALSE)

## ----plotClusterEx1_forceColors,fig.width=18,fig.height=9------------------
par(mfrow=c(1,2))
plotClusters(ce_temp, sampleData=c("Biological_Condition","Published2"),
             whichClusters="workflow", existingColors="all", clusterLabels=FALSE,
             main="All Workflow Clusters, use existing colors",axisLine=-1)

plotClusters(ce_temp, sampleData=c("Biological_Condition","Published2"), 
             existingColors="all", whichClusters="clusterMany", clusterLabels=FALSE,
             main="clusterMany Clusters, use existing colors",
             axisLine=-1)

## ----plotClusterEx1_compareForceColors,fig.width=18,fig.height=18----------
par(mfrow=c(2,2))
plotClusters(ce_temp, sampleData=c("Biological_Condition","Published2"), 
             existingColors="all", whichClusters="clusterMany", clusterLabels=FALSE,
             main="clusterMany Clusters, use existing colors",
             axisLine=-1)
plotClusters(ce_temp, sampleData=c("Biological_Condition","Published2"), 
               existingColors="firstOnly", whichClusters="clusterMany", clusterLabels=FALSE,
               main="clusterMany Clusters, use existing of first row only",
               axisLine=-1)
plotClusters(ce_temp, sampleData=c("Biological_Condition","Published2"), 
            existingColors="ignore", whichClusters="clusterMany", clusterLabels=FALSE,
            main="clusterMany Clusters, default\n(ignoring assigned colors)",
            axisLine=-1)

## ----showPalette-----------------------------------------------------------
showPalette()

## ----showPaletteOptions----------------------------------------------------
showPalette(which=1:10)
showPalette(palette())

## ----showPaletteMassive----------------------------------------------------
showPalette(massivePalette,cex=0.5)

## ----removeBlack-----------------------------------------------------------
plotClusters(ce,whichClusters="workflow", sampleData=c("Biological_Condition","Published2"), unassignedColor="black",colPalette=bigPalette[-grep("black",bigPalette)],
               main="Setting unassigned color",axisLine=-1)


## ----plotWorkflow----------------------------------------------------------
plotClustersWorkflow(ce, axisLine= -1)

## ----plotWorkflow_resort---------------------------------------------------
par(mar=plotCMar)
plotClustersWorkflow(ce,whichClusters=c("combineMany,final","combineMany,0.7","combineMany,1"),main="Different choices of proportion in combineMany",sortBy="clusterMany",highlightOnTop=FALSE, axisLine= -1)

## ----plotHeatmap_Ex1-------------------------------------------------------
par(mfrow=c(1,1))
par(mar=defaultMar)
plotHeatmap(ce,main="Heatmap with clusterMany")

## ----plotHeatmap_Ex1.1-----------------------------------------------------
whClusterPlot<-1:2
plotHeatmap(ce,whichClusters=whClusterPlot, annLegend=FALSE)

## ----plotHeatmap_primaryCluster--------------------------------------------
plotHeatmap(ce,clusterSamplesData="primaryCluster",
            whichClusters="primaryCluster",
            main="Heatmap with clusterMany",annLegend=FALSE)

## ----showCE_dendrogram-----------------------------------------------------
show(ce)

## ----plotHeatmap_dendro----------------------------------------------------
plotHeatmap(ce,clusterSamplesData="dendrogramValue",
            whichClusters=c("mergeClusters","combineMany"),
            main="Heatmap with clusterMany",
            sampleData=c("Biological_Condition","Published2"),annLegend=FALSE)

## ----plotHeatmap_break99---------------------------------------------------
plotHeatmap(ce,clusterSamplesData="primaryCluster",
            whichClusters="primaryCluster", breaks=0.99,
            main="Heatmap with clusterMany, breaks=0.99",annLegend=FALSE)

## ----plotHeatmap_break95---------------------------------------------------
plotHeatmap(ce,clusterSamplesData="primaryCluster",
            whichClusters="primaryCluster", breaks=0.95,
            main="Heatmap with clusterMany, breaks=0.95",annLegend=FALSE)

## ----builtInFunctions------------------------------------------------------
listBuiltInFunctions()

## ----getInputType----------------------------------------------------------
inputType(c("kmeans","pam","hierarchicalK"))

## ----getAlgorithmType------------------------------------------------------
algorithmType(c("kmeans","hierarchicalK","hierarchical01"))

## ----builtInKFunctions-----------------------------------------------------
listBuiltInTypeK()

## ----builtIn01Functions----------------------------------------------------
listBuiltInType01()

## ----requiredArgs----------------------------------------------------------
requiredArgs(c("hierarchical01","hierarchicalK"))

## ----mainClusterArgsSyntax,eval=FALSE--------------------------------------
#  clusterMany(x,clusterFunction="hierarchicalK", ... , mainClusterArgs=list(clusterArgs=list(method="single") ))

## ----subsampleArgsSyntax,eval=FALSE----------------------------------------
#  clusterMany(x,..., subsampleArgs=list(resamp.num=100,samp.p=0.5,clusterFunction="hiearchicalK", clusterArgs=list(method="single") ))

## ----seqArgsSyntax,eval=FALSE----------------------------------------------
#  clusterMany(x,..., seqArgs=list( remain.n=10))

## ----tableArguments, echo=FALSE, message=FALSE, warnings=FALSE, results='asis'----
# simple table creation here
tabl <- "  
Argument| Dependencies |  Passed to | Argument passed to
---------------|-----------------|:-------------:|------:|
ks             | sequential=TRUE |  seqCluster   |  k0 
-    | sequential=FALSE, findBestK=FALSE, clusterFunction of type 'K' | mainClustering | k
-              | sequential=FALSE, findBestK=FALSE, subsample=TRUE | subsampleClustering | k
-               | sequential=FALSE, findBestK=TRUE, clusterFunction of type 'K' | mainClustering | kRange
reduceMethod      | none            | transform     | reduceMethod
nFilterDims       | reduceMethod in 'mad','cv','var' | transform | nFilterDims
nReducedDims       | reduceMethod='PCA' | transform     | nReducedDims
clusterFunction| none            | mainClustering      | clusterFunction    
minSizes       | none            | mainClustering      | minSize
distFunction   | subsample=FALSE | mainClustering      | distFunction
alphas         | clusterFunction of type '01'| mainClustering | alpha
findBestK      | clusterFunction of type 'K' | mainClustering | findBestK
removeSil      | clusterFunction of type 'K' | mainClustering | removeSil
silCutoff      | clusterFunction of type 'K' | mainClustering | silCutoff
betas          | sequential=TRUE | seqCluster    | beta
"
cat(tabl) # output the table in a format good for HTML/PDF/docx conversion

## ----defineDist------------------------------------------------------------
corDist<-function(x){(1-cor(t(x),method="pearson"))/2}
spearDist<-function(x){(1-cor(t(x),method="spearman"))/2}

## ----visualizeSubsamplingD-------------------------------------------------
ceSub<-clusterSingle(ce,reduceMethod="mad",nDims=1000,subsample=TRUE,subsampleArgs=list(clusterFunction="pam",clusterArgs=list(k=8)),clusterLabel="subsamplingCluster",mainClusterArgs=list(clusterFunction="hierarchical01",clusterArgs=list(alpha=0.1),minSize=5), replaceCoClustering=TRUE)
plotCoClustering(ceSub,colorScale=rev(seqPal5))

## ----visualizeSpearmanDist-------------------------------------------------
dSp<-spearDist(t(transformData(ce,reduceMethod="mad",nFilterDims=1000)))
plotHeatmap(dSp,isSymmetric=TRUE,colorScale=rev(seqPal5))

## ----clusterManyDiffDist_calculateMad,fig.width=15,fig.height=6------------
ceDist<-makeFilterStats(ce,filterStats="mad")
ceDist

## ----clusterManyDiffDist,fig.width=15,fig.height=6-------------------------
ceDist<-clusterMany(ceDist, k=7:9, alpha=c(0.35,0.4,0.45), 
	 clusterFunction=c("tight","hierarchical01","pam","hierarchicalK"),
	 findBestK=FALSE,removeSil=c(FALSE),dist=c("corDist","spearDist"),
          reduceMethod=c("mad"),nFilterDims=1000,run=TRUE)
clusterLabels(ceDist)<-gsub("clusterFunction","alg",clusterLabels(ceDist))
clusterLabels(ceDist)<-gsub("Dist","",clusterLabels(ceDist))
clusterLabels(ceDist)<-gsub("distFunction","dist",clusterLabels(ceDist))
clusterLabels(ceDist)<-gsub("hierarchical","hier",clusterLabels(ceDist))
par(mar=c(1.1,15.1,1.1,1.1))
plotClusters(ceDist,axisLine=-2,sampleData=c("Biological_Condition"))

## ----clusterManyCheckParam-------------------------------------------------
checkParam<-clusterMany(se, clusterFunction="pam", ks=2:10,
                        removeSil=c(TRUE,FALSE), isCount=TRUE,
                        reduceMethod=c("PCA","var"),
                        nFilterDims=c(100,500,1000),nReducedDims=c(5,15,50),run=FALSE)
dim(checkParam$paramMatrix) #number of rows is the number of clusterings

## ----clusterManyCheckParam2,eval=FALSE-------------------------------------
#  # ce<-clusterMany(se,  paramMatrix=checkParam$paramMatrix, mainClusterArgs=checkParam$mainClusterArgs, seqArgs=checkParam$seqArgs,subsampleArgs=checkParam$subsampleArgs)
#  ce<-clusterMany(ce, clusterFunction="pam",ks=2:10,findBestK=TRUE,removeSil=c(TRUE), isCount=TRUE,reduceMethod=c("PCA","var"),nFilterDims=c(100,500,1000),nReducedDims=c(5,15,50),run=TRUE)

## ----combineMany_detailed--------------------------------------------------
plotCoClustering(ce)

## ----combineMany_chooseClusters--------------------------------------------
wh<-getClusterManyParams(ce)$clusteringIndex[getClusterManyParams(ce)$reduceMethod=="var"]
ce<-combineMany(ce,whichCluster=wh,proportion=0.7,minSize=3,
                clusterLabel="combineMany,nVAR")
plotCoClustering(ce)

## ----combineMany_showDifferent---------------------------------------------
wh<-grep("combineMany",clusterTypes(ce))
par(mar=plotCMar)
plotClusters(ce,whichClusters=rev(wh),axisLine=-1)

## ----makeDendrogram_reducedFeatures----------------------------------------
ce<-makeDendrogram(ce,reduceMethod="var",nDims=500)
plotDendrogram(ce)

## ----showCe----------------------------------------------------------------
show(ce)

## ----remakeMakeDendrogram--------------------------------------------------
ce<-makeDendrogram(ce,reduceMethod="var",nDims=500,
                   whichCluster="combineMany,final")

## ----plotRemadeDendrogram--------------------------------------------------
plotDendrogram(ce,leafType="sample",plotType="colorblock")

## ----plotRemadeDendrogram_compare------------------------------------------
whCM<-grep("combineMany",clusterTypes(ce))
plotDendrogram(ce,whichClusters=whCM,leafType="sample",plotType="colorblock")

## ----clusterTypeOfCombineMany----------------------------------------------
clusterTypes(ce)[which(clusterLabels(ce)=="combineMany,final")]

## ----getBackCombineMany----------------------------------------------------
ce<-setToCurrent(ce,whichCluster="combineMany,final")
show(ce)

## ----checkWhatDendro-------------------------------------------------------
ce

## ----runMergeDetail--------------------------------------------------------
ce<-mergeClusters(ce,mergeMethod="adjP",plotInfo=c("adjP"),calculateAll=FALSE)

## ----getMergeInfo----------------------------------------------------------
mergeMethod(ce)
mergeCutoff(ce)
nodeMergeInfo(ce)

## ----mergeClusters_plot,fig.width=12---------------------------------------
ce<-mergeClusters(ce,mergeMethod="none",plotInfo="all")

## ----showMergeAll----------------------------------------------------------
nodeMergeInfo(ce)

## ----mergeClusters_ex------------------------------------------------------
ce<-mergeClusters(ce,cutoff=0.05,mergeMethod="adjP",clusterLabel="mergeClusters,v2",plot=FALSE)
ce

## ----mergeClusters_reexamineNode-------------------------------------------
nodeMergeInfo(ce)

## ----mergeClusters_redo----------------------------------------------------
ce<-mergeClusters(ce,cutoff=0.15,mergeMethod="MB",
                  clusterLabel="mergeClusters,v3",plot=FALSE)
ce

## ----mergeClusters_compareMerges-------------------------------------------
par(mar=c(1.1,1.1,6.1,2.1))
plotDendrogram(ce,whichClusters=c("mergeClusters,v3","mergeClusters,v2"),mergeInfo="mergeMethod")

## ----workflowTable---------------------------------------------------------
workflowClusterTable(ce)

## ----workflowDetails-------------------------------------------------------
head(workflowClusterDetails(ce),8)

## ----markFinal-------------------------------------------------------------
ce<-setToFinal(ce,whichCluster="mergeClusters,v2",
               clusterLabel="Final Clustering")
par(mar=plotCMar)
plotClusters(ce,whichClusters="workflow")

## ----rsecTable, echo=FALSE, message=FALSE, warnings=FALSE, results='asis'----
# simple table creation here
tabl <- "  
| | Arguments in original function internally fixed | Arguments in RSEC for the user |   |  |
|:-----------------|:-----------------|:-------------|:------|:------|
     | | *Name of Argument in original function (if different)* | *Notes*|
*clusterMany*| sequential=TRUE | k0s              | ks | RSEC only sets 'k0', no other k 
- | distFunction=NA | clusterFunction | |
- | removeSil=FALSE | reduceMethod | | 
- | subsample=TRUE | nFilterDims | | 
- |  silCutoff=0 | nReducedDims | | 
- |  | alphas | |
- |  | betas | |
- |  | minSizes | |
-  | | mainClusterArgs | |
-  | | subsampleArgs | |
- |  | seqArgs | |
- |  | run | |
- |  | ncores | |
- |  | random.seed | |
- |  | isCount | |
- |  | transFun | |
*combineMany*  | propUnassigned = *(default)* | combineProportion | proportion 
-  | combineMinSize    | minSize | |
*makeDendrogram* | ignoreUnassignedVar=TRUE | dendroReduce      | reduceMethod | 
-  | unassignedSamples= *(default)* | dendroNDims       | nDims |  
*mergeClusters*   | plot=FALSE | mergeMethod     | | 
-  | | mergeCutoff | cutoff | 
-  |  | isCount | | argument used for both mergeMethod and clusterMany 
"

cat(tabl) # output the table in a format good for HTML/PDF/docx conversion


## ----getBestFeatures_onlyTopPairs------------------------------------------
pairsAllTop<-getBestFeatures(rsecFluidigm,contrastType="Pairs",p.value=0.05)
dim(pairsAllTop)
head(pairsAllTop)

## ----getBestFeatures_oneAgainstAll-----------------------------------------
best1vsAll<-getBestFeatures(rsecFluidigm,contrastType="OneAgainstAll",p.value=0.05,number=NROW(rsecFluidigm))
head(best1vsAll)

## ----getBestFeatures_oneHeatmap--------------------------------------------
plotContrastHeatmap(rsecFluidigm,signifTable=best1vsAll,nBlankLines=10, whichCluster="primary")

## ----getBestFeatures_dendro------------------------------------------------
bestDendro<-getBestFeatures(rsecFluidigm,contrastType="Dendro",p.value=0.05,number=NROW(rsecFluidigm))
head(bestDendro)

## ----dendroContrastLevels--------------------------------------------------
levels((bestDendro)$Contrast)

## ----getBestFeatures_dendroHeatmap-----------------------------------------
plotContrastHeatmap(rsecFluidigm,signifTable=bestDendro,nBlankLines=10)

## ----dendroWithNodeNames---------------------------------------------------
plotDendrogram(rsecFluidigm,show.node.label=TRUE,whichClusters=c("combineMany","mergeClusters"),leaf="samples",plotType="colorblock")

## ----sessionInfo-----------------------------------------------------------
sessionInfo()

