## ----'installDer', eval = FALSE------------------------------------------
#  ## try http:// if https:// URLs are not supported
#  source("https://bioconductor.org/biocLite.R")
#  biocLite("derfinderPlot")
#  
#  ## Check that you have a valid Bioconductor installation
#  biocValid()

## ----'citation'----------------------------------------------------------
## Citation info
citation('derfinderPlot')

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE-----------
## Track time spent on making the vignette
startTime <- Sys.time()

## Bib setup
library('knitcitations')

## Load knitcitations with a clean bibliography
cleanbib()
cite_options(hyperlink = 'to.doc', citation_format = 'text', style = 'html')
# Note links won't show for now due to the following issue
# https://github.com/cboettig/knitcitations/issues/63

## Write bibliography information
bibs <- c(knitcitations = citation('knitcitations'),
    derfinderPlot = citation('derfinderPlot')[1], 
    #BiocStyle = citation('BiocStyle'),
    BiocStyle = RefManageR::BibEntry(bibtype = 'unpublished', key = 'BiocStyle', title = 'BiocStyle: Standard styles for vignettes and other Bioconductor documents', author = 'Martin Morgan and Andrzej Oles and Wolfgang Huber', note = 'R package version 1.8.0', year = '2015'), 
    knitr = citation('knitr')[3],
    rmarkdown = citation('rmarkdown'),
    derfinder = citation('derfinder')[1],
    ggbio = citation('ggbio'),
    brainspan = RefManageR::BibEntry(bibtype = 'Unpublished', key = 'brainspan', title = 'Atlas of the Developing Human Brain [Internet]. Funded by ARRA Awards 1RC2MH089921-01, 1RC2MH090047-01, and 1RC2MH089929-01.', author = 'BrainSpan', year = 2011, url = 'http://www.brainspan.org/'),
    R = citation(),
    IRanges = citation('IRanges'),
    devtools = citation('devtools'),
    testthat = citation('testthat'),
    #GenomeInfoDb = citation('GenomeInfoDb'),
    GenomeInfoDb = RefManageR::BibEntry(bibtype = 'unpublished', key = 'GenomeInfoDb', title = "GenomeInfoDb: Utilities for manipulating chromosome and other 'seqname' identifiers", author = 'Sonali Arora and Martin Morgan and Marc Carlson and H. Pages', note = 'R package version 1.7.3', year = '2015'),
    GenomicRanges = citation('GenomicRanges'),
    ggplot2 = citation('ggplot2'),
    plyr = citation('plyr'),
    RColorBrewer = citation('RColorBrewer'),
    reshape2 = citation('reshape2'),
    scales = citation('scales'),
    #biovizBase = citation('biovizBase'),
    biovizBase = RefManageR::BibEntry(bibtype = 'unpublished', key = 'biovizBase', title = 'biovizBase: Basic graphic utilities for visualization of genomic data.', author = 'Tengfei Yin and Michael Lawrence and Dianne Cook', note = 'R package version 1.19.0', year = '2015'),
    bumphunter = citation('bumphunter')[1],
    #TxDb.Hsapiens.UCSC.hg19.knownGene = citation('TxDb.Hsapiens.UCSC.hg19.knownGene'),
    TxDb.Hsapiens.UCSC.hg19.knownGene = RefManageR::BibEntry(bibtype = 'unpublished', key = 'TxDb.Hsapiens.UCSC.hg19.knownGene', title = 'TxDb.Hsapiens.UCSC.hg19.knownGene: Annotation package for TxDb object(s)', author = 'Marc Carlson and Bioconductor Package Maintainer', note = 'R package version 3.2.2', year = '2015'),
    bumphunterPaper = RefManageR::BibEntry(bibtype = 'article', key = 'bumphunterPaper', title = 'Bump hunting to identify differentially methylated regions in epigenetic epidemiology studies', author = 'Jaffe, Andrew E and Murakami, Peter and Lee, Hwajin and Leek, Jeffrey T and Fallin, M Daniele and Feinberg, Andrew P and Irizarry, Rafael A', year = 2012, journal = 'International Journal of Epidemiology'),
    derfinderData = citation('derfinderData')
)
write.bibtex(bibs,
    file = 'derfinderPlotRef.bib')
bib <- read.bibtex('derfinderPlotRef.bib')

## Assign short names
names(bib) <- names(bibs)

## Working on Windows?
windowsFlag <- .Platform$OS.type == 'windows'

## ----'start'-------------------------------------------------------------
## Load libraries
suppressPackageStartupMessages(library('derfinder'))
library('derfinderData')
library('derfinderPlot')

## For tMatrix() colors
library('grDevices')
library('RColorBrewer')

## ----'phenoData', results = 'asis'---------------------------------------
library('knitr')
## Get pheno table
pheno <- subset(brainspanPheno, structure_acronym == 'A1C')

## Display the main information
p <- pheno[, -which(colnames(pheno) %in% c('structure_acronym', 'structure_name', 'file'))]
rownames(p) <- NULL
kable(p, format = 'html', row.names = TRUE)

## ----'getData', eval = !windowsFlag--------------------------------------
#  ## Determine the files to use and fix the names
#  files <- rawFiles(system.file('extdata', 'A1C', package = 'derfinderData'), samplepatt = 'bw', fileterm = NULL)
#  names(files) <- gsub('.bw', '', names(files))
#  
#  ## Load the data from disk
#  system.time(fullCov <- fullCoverage(files = files, chrs = 'chr21'))

## ----'getDataWindows', eval = windowsFlag, echo = FALSE------------------
## Load data in Windows case
foo <- function() { 
    load(system.file('extdata', 'fullCov', 'fullCovA1C.RData', package = 'derfinderData'))
    return(fullCovA1C) 
}
fullCov <- foo()

## ----'webData', eval = FALSE---------------------------------------------
#  ## Determine the files to use and fix the names
#  files <- pheno$file
#  names(files) <- gsub('.A1C', '', pheno$lab)
#  
#  ## Load the data from the web
#  system.time(fullCov <- fullCoverage(files = files, chrs = 'chr21'))

## ----'models'------------------------------------------------------------
## Get some idea of the library sizes
sampleDepths <- sampleDepth(collapseFullCoverage(fullCov), 1)

## Define models
models <- makeModels(sampleDepths, testvars = pheno$group, adjustvars = pheno[, c('gender')])

## ----'analyze'-----------------------------------------------------------
## Filter coverage
filteredCov <- lapply(fullCov, filterData, cutoff = 3)

## Perform differential expression analysis
suppressPackageStartupMessages(library('bumphunter'))
system.time(results <- analyzeChr(chr = 'chr21', filteredCov$chr21, models, groupInfo = pheno$group, writeOutput = FALSE, cutoffFstat = 5e-02, nPermute = 20, seeds = 20140923 + seq_len(20)))

## Quick access to the results
regions <- results$regions$regions

## Annotation database to use
suppressPackageStartupMessages(library('TxDb.Hsapiens.UCSC.hg19.knownGene'))
txdb <- TxDb.Hsapiens.UCSC.hg19.knownGene

## ----'plotOverview', bootstrap.show.warning=FALSE------------------------
## Q-values overview
plotOverview(regions = regions, annotation = results$annotation, type = 'qval')

## ----'plotOverview2', bootstrap.show.warning=FALSE-----------------------
## Annotation overview
plotOverview(regions = regions, annotation = results$annotation, type = 'annotation')

## ----'regionData'--------------------------------------------------------
## Get required information for the plots
annoRegs <- annotateRegions(regions, genomicState$fullGenome)
regionCov <- getRegionCoverage(fullCov, regions)

## ----'plotRegCov'--------------------------------------------------------
## Plot top 10 regions
plotRegionCoverage(regions = regions, regionCoverage = regionCov, 
    groupInfo = pheno$group, nearestAnnotation = results$annotation, 
    annotatedRegions = annoRegs, whichRegions=1:10, txdb = txdb, scalefac = 1, ask = FALSE)

## ----'plotCluster', warning=FALSE----------------------------------------
## First cluster
plotCluster(idx = 1, regions = regions, annotation = results$annotation, coverageInfo = fullCov$chr21, txdb = txdb, groupInfo = pheno$group, titleUse = 'pval')

## ----'plotCluster2', bootstrap.show.warning=FALSE------------------------
## Second cluster
plotCluster(idx = 2, regions = regions, annotation = results$annotation, coverageInfo = fullCov$chr21, txdb = txdb, groupInfo = pheno$group, titleUse = 'pval')

## ----'vennRegions'-------------------------------------------------------
## Make venn diagram
venn <- vennRegions(annoRegs)

## It returns the actual venn counts information
venn

## ----createVignette, eval=FALSE------------------------------------------
#  ## Create the vignette
#  library('rmarkdown')
#  system.time(render('derfinderPlot.Rmd', 'BiocStyle::html_document'))
#  
#  ## Extract the R code
#  library('knitr')
#  knit('derfinderPlot.Rmd', tangle = TRUE)

## ----createVignette2-----------------------------------------------------
## Clean up
unlink('chr21', recursive = TRUE)
file.remove('derfinderPlotRef.bib')

## ----reproducibility1, echo=FALSE----------------------------------------
## Date the vignette was generated
Sys.time()

## ----reproducibility2, echo=FALSE----------------------------------------
## Processing time in seconds
totalTime <- diff(c(startTime, Sys.time()))
round(totalTime, digits=3)

## ----reproducibility3, echo=FALSE-------------------------------------------------------------------------------------
## Session info
library('devtools')
options(width = 120)
session_info()

## ----vignetteBiblio, results = 'asis', echo = FALSE, warning = FALSE--------------------------------------------------
## Print bibliography
bibliography()

