# Test suite for the hypothesis ranking and ordering function (which also acts as the top-level interface to the whole calculation)

# Copyright: Copyright 2012 - 2013 GSK. All rights reserved

# *** SVN *** LastChanged: $Date: 2016-02-03 10:42:06 +0000 (Wed, 03 Feb 2016) $ Changed By: $Author: pipm $ Version: $Revision: 588 $ Source:
# $HeadURL: https://stevenagefs:7777/7412/trunk/RankingHypotheses/test_RankTheHypotheses.R $

# A helper function that will create the CG for the purpose of testing in RankTheHypotheses
CreateCGForTest_RankTheHypotheses <- function() {
    
    # Create test_network
    test_network <- igraph::graph_from_literal("node0" - +"node1", "node0" - +"node2")
    test_network <- igraph::set_edge_attr(test_network, "Interaction", 1, "Activates")
    test_network <- igraph::set_edge_attr(test_network, "Interaction", 2, "Inhibits")
    test_network <- igraph::set_edge_attr(test_network, "Weight", 1, 1)
    test_network <- igraph::set_edge_attr(test_network, "Weight", 2, -1)
    test_network <- igraph::set_vertex_attr(test_network, "ID", 1:3, 1:3)
    test_network$isCCG <- FALSE
    
    return(test_network)
}

test_RankTheHypothesesProducesTheCorrectOutputWithCG <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesProducesTheCorrectOutputWithCCG <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesProducesTheCorrectOutputWithCGUsingQuarticAlgorithm <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, useCubicAlgorithm = FALSE)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesProducesTheCorrectOutputWithCCGUsingQuarticAlgorithm <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, useCubicAlgorithm = FALSE)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_OrderHypothesesProducesTheCorrectResult <- function() {
    matrix1 <- matrix(c("GeneA", "GeneB", "GeneC", 1, -1, 1, 1, 3, 2), nrow = 3)
    orderedTable <- CausalR:::OrderHypotheses(matrix1)
    checkEquals(orderedTable[1, 1], "GeneB")
    checkEquals(orderedTable[1, 2], "-1")
    checkEquals(orderedTable[1, 3], "3")
    checkEquals(orderedTable[2, 1], "GeneC")
    checkEquals(orderedTable[2, 2], "1")
    checkEquals(orderedTable[2, 3], "2")
    checkEquals(orderedTable[3, 1], "GeneA")
    checkEquals(orderedTable[3, 2], "1")
    checkEquals(orderedTable[3, 3], "1")
}

test_AnalyseExperimentalData <- function() {
    expData <- matrix(c(1, 2, 3, 4, 5, 6, 7, -1, 1, 1, 0, -1, 1, +1), ncol = 2)
    expDataStats <- CausalR:::AnalyseExperimentalData(expData)
    checkEquals(expDataStats[1], 4)
    checkEquals(expDataStats[2], 2)
    checkEquals(expDataStats[3], 1)
}

test_RankTheHypothesesWillWorkWithANonSymmetricCCG <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1+"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2-"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2+"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1-"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -2)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 2)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesWorksWithACCGWhenGivenASingleNodeIDInTheSpecifiedListOfNodes <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, listOfNodes = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 1)
    checkEquals(resultsTable[1, 4], 1)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (2/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -1)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 1)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACCGWhenGivenASingleNameInTheSpecifiedListOfNodes <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, listOfNodes = "node0")
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -3)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 3)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACCGWhenGivenMoreThanOneNodeInTheSpecifiedListOfNodes <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, listOfNodes = c("node0", "node1"))
    checkEquals(dim(resultsTable)[1], 4)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], -1)
    checkEquals(resultsTable[3, 4], 0)
    checkEquals(resultsTable[3, 5], 1)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[4, 2], -1)
    checkEquals(resultsTable[4, 3], -3)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 3)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
}

test_RankTheHypothesesWorksWithACGWhenGivenASingleNodeIDInTheSpecifiedListOfNodes <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, listOfNodes = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 1)
    checkEquals(resultsTable[1, 4], 1)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (2/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -1)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 1)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACGWhenGivenASingleNameInTheSpecifiedListOfNodes <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, listOfNodes = "node0")
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -3)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 3)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACGWhenGivenMoreThanOneNodeInTheSpecifiedListOfNodes <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, listOfNodes = c("node0", "node1"))
    checkEquals(dim(resultsTable)[1], 4)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], -1)
    checkEquals(resultsTable[3, 4], 0)
    checkEquals(resultsTable[3, 5], 1)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[4, 2], -1)
    checkEquals(resultsTable[4, 3], -3)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 3)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
}

test_RankTheHypothesesIsAbleToFilterByTheNumberOfCorrectPredictionsForACCG <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, correctPredictionsThreshold = 0)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkTrue(is.na(resultsTable[4, 7]))
    checkTrue(is.na(resultsTable[4, 8]))
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkTrue(is.na(resultsTable[5, 7]))
    checkTrue(is.na(resultsTable[5, 8]))
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkTrue(is.na(resultsTable[6, 7]))
    checkTrue(is.na(resultsTable[6, 8]))
}

test_RankTheHypothesesWorksWithNonSymetricCCGWithSingleNodeIDInTheSpecifiedNodesList <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, listOfNodes = 1)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -2)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 2)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithNonSymetricCCGWithSingleNameInTheSpecifiedNodesList <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, listOfNodes = "node0")
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -2)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 2)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithNonSymetricCCGWithMoreThanOneNodeInTheSpecifiedNodesList <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, listOfNodes = c(1, 2))
    checkEquals(dim(resultsTable)[1], 4)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1+"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node1-"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], -1)
    checkEquals(resultsTable[3, 4], 0)
    checkEquals(resultsTable[3, 5], 1)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[4, 2], -1)
    checkEquals(resultsTable[4, 3], -2)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 2)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
}

test_RankTheHypothesesHandlesNoOverlap <- function() {
    # Test that RankTheHypotheses works correctly if there is no overlap between the predictions and the nodes in the experimental data.
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # call RankTheHypotheses looking at node2, but only providing experimental data on node0 and node1
    output <- CausalR::RankTheHypotheses(test_CCG, experimentalData[1:2, ], 2, listOfNodes = "node2")
    # the calculated part of output should be all zeros, with 1 for the p values
    checkTrue(all(output[, 3:6] == 0))
    checkTrue(all(output[, 7:8] == 1))
} 


###########################################################
## Same again in parallel
## Note: CRAN checks only allow tests and examples to use 2 cores
###########################################################

test_RankTheHypothesesProducesTheCorrectOutputWithCGInParallelMode <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesProducesTheCorrectOutputWithCCGInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesProducesTheCorrectOutputWithCGUsingQuarticAlgorithmInParallelMode <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, useCubicAlgorithm = FALSE, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_TheHypothesesProducesTheCorrectOutputWithCCGUsingQuarticAlgorithmInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, useCubicAlgorithm = FALSE, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesWillWorkWithANonSymmetricCCGInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1+"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2-"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2+"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1-"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkEquals(resultsTable[5, 7], 1)
    checkEquals(resultsTable[5, 8], 1)
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -2)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 2)
    checkEquals(resultsTable[6, 6], 0)
    checkEquals(resultsTable[6, 7], 1)
    checkEquals(resultsTable[6, 8], 1)
}

test_RankTheHypothesesWorksWithACCGWhenGivenASingleNodeIDInTheSpecifiedListOfNodesInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, listOfNodes = 2, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 1)
    checkEquals(resultsTable[1, 4], 1)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (2/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -1)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 1)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACCGWhenGivenASingleNameInTheSpecifiedListOfNodesInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, listOfNodes = "node0", doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -3)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 3)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACCGWhenGivenMoreThanOneNodeInTheSpecifiedListOfNodesInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, listOfNodes = c("node0", "node1"), doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 4)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], -1)
    checkEquals(resultsTable[3, 4], 0)
    checkEquals(resultsTable[3, 5], 1)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[4, 2], -1)
    checkEquals(resultsTable[4, 3], -3)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 3)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
}

test_RankTheHypothesesWorksWithACGWhenGivenASingleNodeIDInTheSpecifiedListOfNodesInParallelMode <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, listOfNodes = 2, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 1)
    checkEquals(resultsTable[1, 4], 1)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (2/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -1)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 1)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACGWhenGivenASingleNameInTheSpecifiedListOfNodesInParallelMode <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, listOfNodes = "node0", doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -3)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 3)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithACGWhenGivenMoreThanOneNodeInTheSpecifiedListOfNodesInParallelMode <- function() {
    test_CG <- CreateCGForTest_RankTheHypotheses()
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CG)
    resultsTable <- CausalR::RankTheHypotheses(test_CG, experimentalData, 2, listOfNodes = c("node0", "node1"), doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 4)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CG, "node1"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], -1)
    checkEquals(resultsTable[3, 4], 0)
    checkEquals(resultsTable[3, 5], 1)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CG, "node0"))
    checkEquals(resultsTable[4, 2], -1)
    checkEquals(resultsTable[4, 3], -3)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 3)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
}

test_RankTheHypothesesIsAbleToFilterByTheNumberOfCorrectPredictionsForACCGInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, correctPredictionsThreshold = 0, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 6)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], 1)
    checkEquals(resultsTable[3, 4], 1)
    checkEquals(resultsTable[3, 5], 0)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1/3)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node2"))
    checkEquals(resultsTable[4, 2], 1)
    checkEquals(resultsTable[4, 3], -1)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 1)
    checkEquals(resultsTable[4, 6], 0)
    checkTrue(is.na(resultsTable[4, 7]))
    checkTrue(is.na(resultsTable[4, 8]))
    checkEquals(resultsTable[5, 1], CausalR:::GetNodeID(test_CCG, "node1"))
    checkEquals(resultsTable[5, 2], -1)
    checkEquals(resultsTable[5, 3], -1)
    checkEquals(resultsTable[5, 4], 0)
    checkEquals(resultsTable[5, 5], 1)
    checkEquals(resultsTable[5, 6], 0)
    checkTrue(is.na(resultsTable[5, 7]))
    checkTrue(is.na(resultsTable[5, 8]))
    checkEquals(resultsTable[6, 1], CausalR:::GetNodeID(test_CCG, "node0"))
    checkEquals(resultsTable[6, 2], -1)
    checkEquals(resultsTable[6, 3], -3)
    checkEquals(resultsTable[6, 4], 0)
    checkEquals(resultsTable[6, 5], 3)
    checkEquals(resultsTable[6, 6], 0)
    checkTrue(is.na(resultsTable[6, 7]))
    checkTrue(is.na(resultsTable[6, 8]))
}

test_RankTheHypothesesWorksWithNonSymetricCCGWithSingleNodeIDInTheSpecifiedNodesListInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, listOfNodes = 1, doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -2)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 2)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithNonSymetricCCGWithSingleNameInTheSpecifiedNodesListInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, listOfNodes = "node0", doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 2)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[2, 2], -1)
    checkEquals(resultsTable[2, 3], -2)
    checkEquals(resultsTable[2, 4], 0)
    checkEquals(resultsTable[2, 5], 2)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 1)
    checkEquals(resultsTable[2, 8], 1)
}

test_RankTheHypothesesWorksWithNonSymetricCCGWithMoreThanOneNodeInTheSpecifiedNodesListInParallelMode <- function() {
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # Delete an edge to make the graph non-symmetric
    test_CCG <- igraph::delete_edges(test_CCG, 4)
    resultsTable <- CausalR::RankTheHypotheses(test_CCG, experimentalData, 2, symmetricCCG = FALSE, listOfNodes = c(1, 2), doParallel = TRUE, numCores = 2)
    checkEquals(dim(resultsTable)[1], 4)
    checkEquals(dim(resultsTable)[2], 8)
    checkEquals(resultsTable[1, 1], CausalR:::GetNodeID(test_CCG, "node0+"))
    checkEquals(resultsTable[1, 2], 1)
    checkEquals(resultsTable[1, 3], 3)
    checkEquals(resultsTable[1, 4], 3)
    checkEquals(resultsTable[1, 5], 0)
    checkEquals(resultsTable[1, 6], 0)
    checkEquals(resultsTable[1, 7], (1/3))
    checkEquals(resultsTable[1, 8], 1)
    checkEquals(resultsTable[2, 1], CausalR:::GetNodeID(test_CCG, "node1+"))
    checkEquals(resultsTable[2, 2], 1)
    checkEquals(resultsTable[2, 3], 1)
    checkEquals(resultsTable[2, 4], 1)
    checkEquals(resultsTable[2, 5], 0)
    checkEquals(resultsTable[2, 6], 0)
    checkEquals(resultsTable[2, 7], 2/3)
    checkEquals(resultsTable[2, 8], 1)
    checkEquals(resultsTable[3, 1], CausalR:::GetNodeID(test_CCG, "node1-"))
    checkEquals(resultsTable[3, 2], -1)
    checkEquals(resultsTable[3, 3], -1)
    checkEquals(resultsTable[3, 4], 0)
    checkEquals(resultsTable[3, 5], 1)
    checkEquals(resultsTable[3, 6], 0)
    checkEquals(resultsTable[3, 7], 1)
    checkEquals(resultsTable[3, 8], 1)
    checkEquals(resultsTable[4, 1], CausalR:::GetNodeID(test_CCG, "node0-"))
    checkEquals(resultsTable[4, 2], -1)
    checkEquals(resultsTable[4, 3], -2)
    checkEquals(resultsTable[4, 4], 0)
    checkEquals(resultsTable[4, 5], 2)
    checkEquals(resultsTable[4, 6], 0)
    checkEquals(resultsTable[4, 7], 1)
    checkEquals(resultsTable[4, 8], 1)
}

test_RankTheHypothesesHandlesNoOverlapInParallelMode <- function() {
    # Test that RankTheHypotheses works correctly if there is no overlap between the predictions and the nodes in the experimental data.
    test_CCG <- CausalR::CreateCCG(system.file("testData", "test_network.sif", package = "CausalR"))
    experimentalData <- CausalR::ReadExperimentalData(system.file("testData", "test_ExperimentalData.txt", package = "CausalR"), test_CCG)
    # call RankTheHypotheses looking at node2, but only providing experimental data on node0 and node1
    output <- CausalR::RankTheHypotheses(test_CCG, experimentalData[1:2, ], 2, listOfNodes = "node2", doParallel = TRUE, numCores = 2)
    # the calculated part of output should be all zeros, with 1 for the p values
    checkTrue(all(output[, 3:6] == 0))
    checkTrue(all(output[, 7:8] == 1))
} 