## ---- echo=FALSE, results="hide", message=FALSE-------------------------------
knitr::opts_chunk$set(error=FALSE, message=FALSE, warning=FALSE)
library(BiocStyle)

## -----------------------------------------------------------------------------
suppressPackageStartupMessages(library(escape))
suppressPackageStartupMessages(library(dittoSeq))
suppressPackageStartupMessages(library(SingleCellExperiment))
suppressPackageStartupMessages(library(Seurat))
suppressPackageStartupMessages(library(SeuratObject))
pbmc_small <- get("pbmc_small")
pbmc_small <- DietSeurat(suppressMessages(UpdateSeuratObject(pbmc_small)))
sce <- as.SingleCellExperiment(pbmc_small, assay = "RNA")

## -----------------------------------------------------------------------------
GS.hallmark <- getGeneSets(library = "H")

## ---- eval = FALSE------------------------------------------------------------
#  data("escape.gene.sets", package="escape")
#  gene.sets <- escape.gene.sets

## ---- eval=FALSE--------------------------------------------------------------
#  gene.sets <- list(Tcell_signature = c("CD2","CD3E","CD3D"),
#  			Myeloid_signature = c("SPI1","FCER1G","CSF1R"))

## -----------------------------------------------------------------------------
ES.seurat <- enrichIt(obj = pbmc_small, 
                      gene.sets = GS.hallmark, 
                      groups = 1000, cores = 2, 
                      min.size = 5)

ES.sce <- enrichIt(obj = sce, 
                   gene.sets = GS.hallmark, 
                   method = "UCell",
                   groups = 1000, cores = 2, 
                   min.size = 5)

## -----------------------------------------------------------------------------
## if working with a Seurat object
pbmc_small <- Seurat::AddMetaData(pbmc_small, ES.seurat)

## if working with an SCE object
met.data <- merge(colData(sce), ES.sce, by = "row.names", all=TRUE)
row.names(met.data) <- met.data$Row.names
met.data$Row.names <- NULL
colData(sce) <- met.data

## -----------------------------------------------------------------------------
colors <- colorRampPalette(c("#0D0887FF","#7E03A8FF","#CC4678FF","#F89441FF","#F0F921FF"))

## -----------------------------------------------------------------------------
dittoHeatmap(pbmc_small, genes = NULL, metas = names(ES.seurat), 
             annot.by = "groups", 
             fontsize = 7, 
             cluster_cols = TRUE,
             heatmap.colors = colors(50))

## -----------------------------------------------------------------------------
dittoHeatmap(sce, genes = NULL, 
             metas = c("HALLMARK_IL2_STAT5_SIGNALING", 
                       "HALLMARK_IL6_JAK_STAT3_SIGNALING", 
                       "HALLMARK_INFLAMMATORY_RESPONSE"), 
             annot.by = "groups", 
             fontsize = 7,
             heatmap.colors = colors(50))

## -----------------------------------------------------------------------------
multi_dittoPlot(sce, vars = c("HALLMARK_IL2_STAT5_SIGNALING", 
                       "HALLMARK_IL6_JAK_STAT3_SIGNALING", 
                       "HALLMARK_INFLAMMATORY_RESPONSE"), 
                group.by = "groups", plots = c("jitter", "vlnplot", "boxplot"), 
                ylab = "Enrichment Scores", 
                theme = theme_classic() + theme(plot.title = element_text(size = 10)))

## -----------------------------------------------------------------------------
dittoScatterHex(sce, x.var = "HALLMARK_IL2_STAT5_SIGNALING", 
                    y.var = "HALLMARK_IL6_JAK_STAT3_SIGNALING", 
                    do.contour = TRUE) + 
        scale_fill_gradientn(colors = colors(11)) 

## -----------------------------------------------------------------------------
dittoScatterHex(sce, x.var = "HALLMARK_IL2_STAT5_SIGNALING", 
                    y.var = "HALLMARK_IL6_JAK_STAT3_SIGNALING", 
                do.contour = TRUE,
                split.by = "groups")  + 
        scale_fill_gradientn(colors = colors(11)) 

## -----------------------------------------------------------------------------
## Seurat object example
ES2 <- data.frame(pbmc_small[[]], Idents(pbmc_small))
colnames(ES2)[ncol(ES2)] <- "cluster"

## plot
ridgeEnrichment(ES2, gene.set = "HALLMARK_IL2_STAT5_SIGNALING", group = "cluster", add.rug = TRUE)

## -----------------------------------------------------------------------------
ridgeEnrichment(ES2, gene.set = "HALLMARK_IL2_STAT5_SIGNALING", group = "cluster", 
                facet = "letter.idents", add.rug = TRUE)

## -----------------------------------------------------------------------------
splitEnrichment(ES2, split = "groups", gene.set = "HALLMARK_IL2_STAT5_SIGNALING")
splitEnrichment(ES2, x.axis = "cluster", split = "groups", gene.set = "HALLMARK_IL2_STAT5_SIGNALING")

## -----------------------------------------------------------------------------
enrichmentPlot(pbmc_small, 
               gene.set = "HALLMARK_IL2_STAT5_SIGNALING",
               gene.sets = GS.hallmark,
               group = "groups")

## -----------------------------------------------------------------------------
PCA <- performPCA(enriched = ES2, gene.sets = names(GS.hallmark), groups = c("groups", "cluster"))

## -----------------------------------------------------------------------------
pcaEnrichment(PCA, PCx = "PC1", PCy = "PC2", contours = TRUE)

## -----------------------------------------------------------------------------
pcaEnrichment(PCA, PCx = "PC1", PCy = "PC2", contours = FALSE, facet = "cluster") 

## -----------------------------------------------------------------------------
masterPCAPlot(ES2, gene.sets = names(GS.hallmark), PCx = "PC1", PCy = "PC2", top.contribution = 10)

## -----------------------------------------------------------------------------
output <- getSignificance(ES2, 
                          group = "groups", 
                          gene.sets = names(ES.seurat),
                          fit = "T.test")

## -----------------------------------------------------------------------------
sessionInfo()

