## ----eval=FALSE---------------------------------------------------------------
#  if (!requireNamespace("BiocManager", quietly=TRUE))
#      install.packages("BiocManager")
#  BiocManager::install("UCell")

## -----------------------------------------------------------------------------
library(UCell)

data(sample.matrix)
gene.sets <- list(Tcell_signature = c("CD2","CD3E","CD3D"),
                  Myeloid_signature = c("SPI1","FCER1G","CSF1R"))

scores <- ScoreSignatures_UCell(sample.matrix, features=gene.sets)
head(scores)

## ----message=F, warning=F, results=F------------------------------------------
library(scRNAseq)

#For this demo, limit the analysis to 5000 immune cells
lung <- ZilionisLungData()

immune <- lung$Used & lung$used_in_NSCLC_immune
lung <- lung[,immune]
lung <- lung[,1:5000]

exp.mat <- Matrix::Matrix(counts(lung),sparse = TRUE)

## -----------------------------------------------------------------------------
signatures <- list(
    Immune = c("PTPRC"),
    Tcell = c("CD3D","CD3E","CD3G","CD2","TRAC"),
    Bcell = c("MS4A1","CD79A","CD79B","CD19","BANK1"),
    Myeloid = c("CD14","LYZ","CSF1R","FCER1G","SPI1","LCK-"),
    NK = c("KLRD1","NCAM1","NKG7","CD3D-","CD3E-"),
    Plasma_cell = c("IGKC","IGHG3","IGHG1","IGHA1","CD19-")
)

## -----------------------------------------------------------------------------
u.scores <- ScoreSignatures_UCell(exp.mat,features=signatures)
u.scores[1:10,1:3]

## -----------------------------------------------------------------------------
library(reshape2)
library(ggplot2)
melted <- reshape2::melt(u.scores)
colnames(melted) <- c("Cell","Signature","UCell_score")
p <- ggplot(melted, aes(x=Signature, y=UCell_score)) + 
    geom_violin(aes(fill=Signature), scale = "width") +
    geom_boxplot(width=0.1, outlier.size=0) +
    theme_bw() + theme(axis.text.x=element_blank())
p

## -----------------------------------------------------------------------------
set.seed(123)
ranks <- StoreRankings_UCell(exp.mat)
ranks[1:5,1:5]

## -----------------------------------------------------------------------------
set.seed(123)
u.scores.2 <- ScoreSignatures_UCell(features=signatures,
                                    precalc.ranks = ranks)

melted <- reshape2::melt(u.scores.2)
colnames(melted) <- c("Cell","Signature","UCell_score")
p <- ggplot(melted, aes(x=Signature, y=UCell_score)) + 
    geom_violin(aes(fill=Signature), scale = "width") +
    geom_boxplot(width=0.1, outlier.size = 0) + 
    theme_bw() + theme(axis.text.x=element_blank())
p

## ----fig.small=TRUE-----------------------------------------------------------
new.signatures <- list(Mast.cell = c("TPSAB1","TPSB2","CPA3","MS4A2"),
                       Lymphoid = c("LCK"))

u.scores.3 <- ScoreSignatures_UCell(features=new.signatures,
                                    precalc.ranks = ranks)
melted <- reshape2::melt(u.scores.3)
colnames(melted) <- c("Cell","Signature","UCell_score")
p <- ggplot(melted, aes(x=Signature, y=UCell_score)) + 
    geom_violin(aes(fill=Signature), scale = "width") +
    geom_boxplot(width=0.1, outlier.size=0) + 
    theme_bw() + theme(axis.text.x=element_blank())
p

## -----------------------------------------------------------------------------
BPPARAM <- BiocParallel::MulticoreParam(workers=1)
u.scores <- ScoreSignatures_UCell(exp.mat,features=signatures,
                                  BPPARAM=BPPARAM)

## ----message=F, warning=F-----------------------------------------------------
library(SingleCellExperiment)

sce <- SingleCellExperiment(list(counts=exp.mat))
sce <- ScoreSignatures_UCell(sce, features=signatures, 
                                 assay = 'counts', name=NULL)
altExp(sce, 'UCell')

## ----message=F, warning=F-----------------------------------------------------
library(scater)
library(patchwork)
#PCA
sce <- logNormCounts(sce)
sce <- runPCA(sce, scale=TRUE, ncomponents=20)

#UMAP
set.seed(1234)
sce <- runUMAP(sce, dimred="PCA")

## ----fig.wide=TRUE------------------------------------------------------------
pll <- lapply(names(signatures), function(x) {
    plotUMAP(sce, colour_by = x, by_exprs_values = "UCell",
             point_size=0.2)
})
wrap_plots(pll)

## ----message=F, warning=F-----------------------------------------------------
library(Seurat)
seurat.object <- CreateSeuratObject(counts = exp.mat, 
                                    project = "Zilionis_immune")
seurat.object <- AddModuleScore_UCell(seurat.object, 
                                      features=signatures, name=NULL)
head(seurat.object[[]])

## ----message=F, warning=F-----------------------------------------------------
seurat.object <- NormalizeData(seurat.object)
seurat.object <- FindVariableFeatures(seurat.object, 
                     selection.method = "vst", nfeatures = 500)
  
seurat.object <- ScaleData(seurat.object)
seurat.object <- RunPCA(seurat.object, npcs = 20, 
                        features=VariableFeatures(seurat.object)) 
seurat.object <- RunUMAP(seurat.object, reduction = "pca", 
                         dims = 1:20, seed.use=123)

## ----fig.wide=TRUE------------------------------------------------------------
FeaturePlot(seurat.object, reduction = "umap", 
            features = names(signatures), ncol=3, order=TRUE)

## -----------------------------------------------------------------------------
sessionInfo()

