## ----setup, include=FALSE-----------------------------------------------------
library(knitr)
opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL,
    fig.width = 6,
    dpi = 72
)

## ----"libraries", message=FALSE, warning=FALSE--------------------------------
library(TargetDecoy)
library(ggplot2)

## ----"install-devel", eval=FALSE----------------------------------------------
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  
#  BiocManager::install("statOmics/TargetDecoy")

## ----"citation"---------------------------------------------------------------
## Citation info
citation("TargetDecoy")

## ---- echo=FALSE, fig.cap="Illustration of the target and decoy distributions. In grey the histogram of the target PSM scores."----
include_graphics("img/Grijs_Target.png")

## ---- echo=FALSE, fig.cap="The blue bars are the histogram of the decoy PSM scores. We see that the decoy PSMs match well with the incorrect target hits that are more likely to occur at low scores."----
include_graphics("img/Kleur.png")

## ---- fig.cap="Histogram of the scores of targets and decoys."----------------
data("ModSwiss")
evalTargetDecoysHist(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE, nBins = 50
)

## ---- fig.cap="PP plot"-------------------------------------------------------
evalTargetDecoysPPPlot(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE
)

## ----example1.1-data----------------------------------------------------------
## Load the example SwissProt dataset
data("ModSwiss")

## ----evalTargetDecoys-plot, fig.cap="Histogram and PP plot, as well as a zoom for both plots."----
evalTargetDecoys(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE, nBins = 50
)

## -----------------------------------------------------------------------------
msgfSwiss_ppPlot <- evalTargetDecoysPPPlot(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE
)
msgfSwiss_ppPlot_zoomed <- evalTargetDecoysPPPlot(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE, zoom = TRUE
)
msgfSwiss_hist <- evalTargetDecoysHist(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE, nBins = 50
)
msgfSwiss_hist_zoomed <- evalTargetDecoysHist(ModSwiss,
    decoy = "isdecoy", score = "ms-gf:specevalue",
    log10 = TRUE, nBins = 50, zoom = TRUE
)

## ---- fig.cap="Zoom of the first part of the PP plot"-------------------------
msgfSwiss_ppPlot +
    coord_cartesian(xlim = c(NA, 0.25), ylim = c(NA, 0.05))

## -----------------------------------------------------------------------------
## Load X!Tandem example data
data("ModSwissXT")

## ---- fig.cap="Histogram and PP plot, each with a zoom level."----------------
evalTargetDecoys(ModSwissXT,
    decoy = "isdecoy", score = "x!tandem:expect",
    log10 = TRUE, nBins = 50
)

## ---- fig.asp=1, fig.cap="PP plot and standardized PP plot"-------------------
## Run createPPlotScores with necessary arguments
## Omitting 'decoy', 'scores' or 'log10' will launch the Shiny app
createPPlotScores(ModSwissXT,
    decoy = "isdecoy",
    scores = c("omssa:evalue", "ms-gf:specevalue",
               "x!tandem:expect", "peptideshaker psm score"),
    ## We can choose to log-transform some scores but not others
    log10 = c(TRUE, TRUE, TRUE, FALSE)
)

## ---- fig.asp=1, fig.cap="PP plot and standardized PP plot for two searches (same search engine)"----
## Create list of input objects, list names will be reused in the plots
mzObjects <- list(
    ModSwiss_data = ModSwiss,
    pyroSwissprot_data = ModSwissXT
)

createPPlotObjects(
    mzObjects, decoy = "isdecoy",
    score = "ms-gf:specevalue"
)

## ---- fig.asp=1, fig.cap="PP plot and standardized PP plot for two searches (different search engine)"----
## Create list of input objects, list names will be reused in the plots
mzObjects <- list(
    ModSwiss_data = ModSwiss,
    ModSwissXT_data = ModSwissXT
)

plotDiffEngine <- createPPlotObjects(mzObjects,
    decoy = "isdecoy",
    score = c("ms-gf:specevalue", "x!tandem:expect")
)
plotDiffEngine

## -----------------------------------------------------------------------------
# Zoom in on the relevant or desired part of the plot
plotZoom <- plotDiffEngine +
    coord_cartesian(xlim = c(NA, 0.25), ylim = c(NA, 0.05))
plotZoom 

## ---- fig.cap="Histogram and PP plot, each with a zoom level."----------------
evalTargetDecoys(mzObjects$ModSwissXT_data,
    decoy = "isdecoy", score = "x!tandem:expect",
    log10 = TRUE, nBins = 50
)

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
options(width = 120)
sessioninfo::session_info()

