## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE, dev = "png")

## ----load, eval=TRUE, message=FALSE-------------------------------------------
library(celda)

## ----load_10X, eval=TRUE, message=FALSE---------------------------------------
# Install TENxPBMCData if is it not already
if (!requireNamespace("TENxPBMCData", quietly = TRUE)) {
  if (!requireNamespace("BiocManager", quietly = TRUE)) {
    install.packages("BiocManager")
  }
  BiocManager::install("TENxPBMCData")
}

# Load PBMC data
library(TENxPBMCData)
sce <- TENxPBMCData("pbmc4k")
colnames(sce) <- paste(sce$Sample, sce$Barcode, sep = "_")
rownames(sce) <- rowData(sce)$Symbol_TENx

## ----decontX_background, eval=FALSE, message=FALSE----------------------------
#  sce <- decontX(sce, background = raw)

## ----decontX, eval=TRUE, message=FALSE----------------------------------------
sce <- decontX(sce)

## ----UMAP_Clusters------------------------------------------------------------
umap <- reducedDim(sce, "decontX_UMAP")
plotDimReduceCluster(x = sce$decontX_clusters,
    dim1 = umap[, 1], dim2 = umap[, 2])

## ----plot_decon---------------------------------------------------------------
plotDecontXContamination(sce)

## ----plot_feature, message=FALSE----------------------------------------------
library(scater)
sce <- logNormCounts(sce)
plotDimReduceFeature(as.matrix(logcounts(sce)),
    dim1 = umap[, 1],
    dim2 = umap[, 2],
    features = c("CD3D", "CD3E", "GNLY",
        "LYZ", "S100A8", "S100A9",
        "CD79A", "CD79B", "MS4A1"),
    exactMatch = TRUE)

## ----barplotCounts------------------------------------------------------------
markers <- list(Tcell_Markers = c("CD3E", "CD3D"),
    Bcell_Markers = c("CD79A", "CD79B", "MS4A1"),
    Monocyte_Markers = c("S100A8", "S100A9", "LYZ"),
    NKcell_Markers = "GNLY")
cellTypeMappings <- list(Tcells = 2, Bcells = 5, Monocytes = 1, NKcells = 6)
plotDecontXMarkerPercentage(sce,
    markers = markers,
    groupClusters = cellTypeMappings,
    assayName = "counts")

## ----barplotDecontCounts------------------------------------------------------
plotDecontXMarkerPercentage(sce,
    markers = markers,
    groupClusters = cellTypeMappings,
    assayName = "decontXcounts")

## ----barplotBoth--------------------------------------------------------------
plotDecontXMarkerPercentage(sce,
    markers = markers,
    groupClusters = cellTypeMappings,
    assayName = c("counts", "decontXcounts"))

## ----plotDecontXMarkerExpression----------------------------------------------
plotDecontXMarkerExpression(sce,
    markers = markers[["Monocyte_Markers"]],
    groupClusters = cellTypeMappings,
    ncol = 3)

## ----plot_norm_counts, eval = FALSE-------------------------------------------
#  sce <- scater::logNormCounts(sce,
#      exprs_values = "decontXcounts",
#      name = "dlogcounts")
#  
#  plotDecontXMarkerExpression(sce,
#      markers = markers[["Monocyte_Markers"]],
#      groupClusters = cellTypeMappings,
#      ncol = 3,
#      assayName = c("logcounts", "dlogcounts"))

## ----findDelta----------------------------------------------------------------
metadata(sce)$decontX$estimates$all_cells$delta

## ----newDecontX, eval=TRUE, message=FALSE-------------------------------------
sce.delta <- decontX(sce, delta = c(9, 20), estimateDelta = FALSE)

plot(sce$decontX_contamination, sce.delta$decontX_contamination,
     xlab = "DecontX estimated priors",
     ylab = "Setting priors to estimate higher contamination")
abline(0, 1, col = "red", lwd = 2)

## -----------------------------------------------------------------------------
sessionInfo()

