## ----setup, echo=FALSE, results="hide", warning=FALSE-------------------------
suppressPackageStartupMessages({
    library("srnadiff")
    library("BiocManager")
    library("BiocStyle")
    library("knitr")
    library("rmarkdown")
    library("grid")
})

knitr::opts_chunk$set(
    error=FALSE,
    fig.height=5,
    fig.width=8,
    message=FALSE,
    warning=FALSE,
    tidy=FALSE
)

## ----echo=FALSE---------------------------------------------------------------
basedir    <- system.file("extdata", package="srnadiff", mustWork=TRUE)
sampleInfo <- read.csv(file.path(basedir, "dataInfo.csv"))
bamFiles   <- paste(file.path(basedir, sampleInfo$FileName), "bam", sep=".")
gtfFile    <- file.path(basedir, "Homo_sapiens.GRCh38.76.gtf.gz")
annotReg   <- readAnnotation(gtfFile, feature="gene", source="miRNA")

## ----eval=TRUE, echo=FALSE----------------------------------------------------
x <- citation("srnadiff")

## ----eval=FALSE, message=FALSE, warning=FALSE---------------------------------
#  help("srnadiff")

## ----eval=FALSE, message=FALSE, warning=FALSE---------------------------------
#  ?srnadiff

## ----message=FALSE, warning=FALSE, include=FALSE, results="hide"--------------
#-- Data preparation
srnaExp <- srnadiffExp(bamFiles, sampleInfo)

#-- Performing srnadiff
srnaExp <- srnadiff(srnaExp)

#-- Visualization of the results
plotRegions(srnaExp, regions(srnaExp)[1])

## ---- bioconductor, eval=FALSE------------------------------------------------
#  if (!requireNamespace("BiocManager", quietly=TRUE))
#      install.packages("BiocManager")

## ---- install, eval=FALSE-----------------------------------------------------
#  BiocManager::install("srnadiff", version="3.8")

## ---- loadLibrary, eval=FALSE-------------------------------------------------
#  library(srnadiff)

## ---- helpSearch, eval=FALSE--------------------------------------------------
#  help.search("srnadiff")

## ----message=FALSE, warning=FALSE---------------------------------------------
## Determine the path to data files
basedir <- system.file("extdata", package="srnadiff", mustWork=TRUE)

## Vector with the full paths to the BAM files to use
bamFiles <- paste(file.path(basedir, sampleInfo$FileName), "bam", sep=".")

## Reads sample information file and creates a data frame from it
sampleInfo <- read.csv(file.path(basedir, "dataInfo.csv"))

## Vector with the full paths to the BAM files to use
bamFiles <- paste(file.path(basedir, sampleInfo$FileName), "bam", sep = ".")

## Creates an srnadiffExp object
srnaExp <- srnadiffExp(bamFiles, sampleInfo)

## ---- eval=FALSE--------------------------------------------------------------
#  srnaExp <- srnadiffExp(bamFiles, sampleInfo, annotReg)

## ----message=FALSE, warning=FALSE---------------------------------------------
srnaExp <- srnadiffExp(bamFiles, sampleInfo)
annotReg(srnaExp) <- annotReg

## -----------------------------------------------------------------------------
srnaExp

## -----------------------------------------------------------------------------
srnaExp <- srnadiffExample()

## ----message=FALSE, warning=FALSE---------------------------------------------
basedir    <- system.file("extdata", package="srnadiff", mustWork=TRUE)
sampleInfo <- read.csv(file.path(basedir, "dataInfo.csv"))
gtfFile    <- file.path(basedir, "Homo_sapiens.GRCh38.76.gtf.gz")
annotReg   <- readAnnotation(gtfFile, feature="gene", source="miRNA")
bamFiles   <- paste(file.path(basedir, sampleInfo$FileName), "bam", sep=".")
srnaExp    <- srnadiffExp(bamFiles, sampleInfo, annotReg)

## ----message=FALSE, warning=FALSE---------------------------------------------
gtfFile  <- file.path(basedir, "Homo_sapiens.GRCh38.76.gtf.gz")
annotReg <- readAnnotation(gtfFile, feature="gene", source="miRNA")

## ----message=FALSE, warning=FALSE---------------------------------------------
gffFile  <- file.path(basedir, "mirbase21_GRCh38.gff3")
annotReg <- readAnnotation(gffFile, feature="miRNA_primary_transcript")

## ----message=FALSE, warning=FALSE---------------------------------------------
gffFile  <- file.path(basedir, "mirbase21_GRCh38.gff3")
annotReg <- readAnnotation(gffFile, feature="miRNA")

## ----message=FALSE, warning=FALSE---------------------------------------------
annotation <- readAnnotation(gtfFile, source="miRNA", feature="gene")

## -----------------------------------------------------------------------------
srnaExp <- srnadiff(srnaExp)

## -----------------------------------------------------------------------------
sampleInfo(srnaExp)

## -----------------------------------------------------------------------------
chromosomeSizes(srnaExp)

## -----------------------------------------------------------------------------
parameters(srnaExp)

## ----message=FALSE, warning=FALSE---------------------------------------------
regions <- regions(srnaExp, pvalue=0.5)

## ---- fig.height=3.5, fig.width=4, fig.align='center', out.width='450pt'------
plotRegions(srnaExp, regions(srnaExp)[1])

## ---- echo=FALSE, fig.height=9------------------------------------------------
types <- list("p", "l", "b", "a", c("a", "confint"), c("a", "p", "confint"))
ncols <- 2
nrows <- 3
grid.newpage()
pushViewport(viewport(layout=grid.layout(nrows, ncols)))
for(i in 1:length(types)){
    pushViewport(viewport(layout.pos.col=((i - 1) %% ncols) + 1,
                          layout.pos.row=(((i) - 1) %/% ncols) + 1))
    trN <- paste(types[[i]], collapse=", ")
    plotRegions(srnaExp, regions(srnaExp)[1], type=types[[i]],
                trNames=c("DER", trN), chrTitle=FALSE, add=TRUE)
    popViewport(1)
}

## ---- eval=FALSE--------------------------------------------------------------
#  parameters(srnaExp) <- list(noDiffToDiff=0.01, emissionThreshold=0.2)

## ---- eval=FALSE--------------------------------------------------------------
#  parameters(srnaExp) <- list(minLogFC=1)

## ---- eval=FALSE--------------------------------------------------------------
#  parameters(srnaExp) <- list(cutoff=1.5)

## ---- general_parameter-------------------------------------------------------
parameters(srnaExp) <- list(minDepth=1)
parameters(srnaExp) <- list(minSize=15, maxSize=1000)

## ---- strategies--------------------------------------------------------------
srnaExp <- srnadiffExample()
srnaExp <- srnadiff(srnaExp, segMethod=c("hmm", "IR"))

## ---- minOverlap--------------------------------------------------------------
parameters(srnaExp) <- list(minOverlap=1000)

## ---- threads, eval=FALSE-----------------------------------------------------
#  exp <- setNThreads(exp, nThreads=4)

## ---- session_info------------------------------------------------------------
devtools::session_info()

