## ----knitr-options, include = FALSE-------------------------------------------
knitr::opts_chunk$set(collapse = TRUE, comment = "#>")

## ----setup--------------------------------------------------------------------
suppressPackageStartupMessages({
  library("splatter")
  library("scater")
  library("VariantAnnotation")
  library("ggplot2")
})
set.seed(42)

## ----quick-start--------------------------------------------------------------
vcf <- mockVCF()
gff <- mockGFF()

sim <- splatPopSimulate(vcf = vcf, gff = gff, sparsify = FALSE)

sim <- logNormCounts(sim)
sim <- runPCA(sim, ncomponents = 5)
plotPCA(sim, colour_by = "Sample")

## ----eqtlEstimate-------------------------------------------------------------
bulk.means <- mockBulkMatrix(n.genes=100, n.samples=100)
bulk.eqtl <- mockBulkeQTL(n.genes=100)
counts <- mockSCE()

params.est <- splatPopEstimate(means = bulk.means,
                               eqtl = bulk.eqtl,
                               counts = counts)
params.est

## ----splatPopSimulateMeans----------------------------------------------------
params <- newSplatPopParams()

sim.means <- splatPopSimulateMeans(vcf = vcf, gff = gff,
                                   params = params)

round(sim.means$means[1:5, 1:3], digits = 2)

print(sim.means$key[1:5, ], digits = 2)

## ----splatPopSimulateMeans-from-key-------------------------------------------
sim.means.rep2 <- splatPopSimulateMeans(vcf = vcf, key=sim.means$key,
                                        params = newSplatPopParams())

round(sim.means.rep2$means[1:5, 1:5], digits = 2)

## ----quant-normalize-population-data------------------------------------------
bulk.qnorm <- splatPopQuantNorm(newSplatPopParams(), bulk.means)
round(bulk.qnorm[1:5, 1:5], 3)

## ----eqtl-splatPopSimulateSC-simple-object------------------------------------
sim.sc <- splatPopSimulateSC(params=params, 
                             key = sim.means$key,
                             sim.means=sim.means$means, 
                             batchCells=50, 
                             sparsify = FALSE)
sim.sc

## ----eqtl-splatPopSimulateSC-simple-plots-------------------------------------
sim.sc <- logNormCounts(sim.sc)
sim.sc <- runPCA(sim.sc, ncomponents = 10)
plotPCA(sim.sc, colour_by = "Sample")

## ----splatPop-simulation-nCells-sampled---------------------------------------
params.nCells <- newSplatPopParams(nCells.sample = TRUE,
                                   nCells.shape = 1.5,
                                   nCells.rate = 0.01)

sim.sc.nCells <- splatPopSimulate(vcf = vcf, gff = gff, params = params.nCells, 
                               sparsify = FALSE)

sim.sc.nCells <- logNormCounts(sim.sc.nCells)
sim.sc.nCells <- runPCA(sim.sc.nCells, ncomponents = 2)
plotPCA(sim.sc.nCells, colour_by = "Sample")

## ----group-specific-eQTL-simulations------------------------------------------
params.group <- newSplatPopParams(batchCells = 50,
                                  group.prob = c(0.5, 0.5))

sim.sc.gr2 <- splatPopSimulate(vcf = vcf, gff = gff, params = params.group, 
                               sparsify = FALSE)

sim.sc.gr2 <- logNormCounts(sim.sc.gr2)
sim.sc.gr2 <- runPCA(sim.sc.gr2, ncomponents = 10)
plotPCA(sim.sc.gr2, colour_by = "Group", shape_by = "Sample")

## ----group-specific-eQTL-simulations-bigger-----------------------------------
params.group <- newSplatPopParams(batchCells = 50,
                                  similarity.scale = 8,
                                  eqtl.group.specific = 0.6,
                                  de.prob = 0.5,
                                  de.facLoc = 0.5, 
                                  de.facScale = 0.5,
                                  group.prob = c(0.5, 0.5))

sim.sc.gr2 <- splatPopSimulate(vcf = vcf, gff = gff, params = params.group, 
                               sparsify = FALSE)

sim.sc.gr2 <- logNormCounts(sim.sc.gr2)
sim.sc.gr2 <- runPCA(sim.sc.gr2, ncomponents = 10)
plotPCA(sim.sc.gr2, colour_by = "Group", shape_by = "Sample")

## ----simulate-population-with-condition-effects-------------------------------
params.cond <- newSplatPopParams(eqtl.n = 0.5, 
                                 batchCells = 50,
                                 similarity.scale = 5,
                                 condition.prob = c(0.5, 0.5),
                                 eqtl.condition.specific = 0.5,
                                 cde.facLoc = 0.5, 
                                 cde.facScale = 0.5)

sim.pop.cond <- splatPopSimulate(vcf = vcf, gff = gff, params = params.cond, 
                                 sparsify = FALSE)

sim.pop.cond <- logNormCounts(sim.pop.cond)
sim.pop.cond <- runPCA(sim.pop.cond, ncomponents = 10)
plotPCA(sim.pop.cond, colour_by = "Condition", shape_by = "Sample", point_size=3)

## ----simulate-population-with-batch-effects-----------------------------------
params.batches <- newSplatPopParams(similarity.scale = 4,
                                    batchCells = c(20, 20),
                                    batch.size = 3,
                                    batch.facLoc = 0.1,
                                    batch.facScale = 0.2)

sim.pop.batches <- splatPopSimulate(vcf = vcf, gff = gff, 
                                    params = params.batches, sparsify = FALSE)
sim.pop.batches <- logNormCounts(sim.pop.batches)
sim.pop.batches <- runPCA(sim.pop.batches, ncomponents = 10)
plotPCA(sim.pop.batches, colour_by = "Sample", shape_by = "Batch", point_size=3)

## ----simulate-population-with-different-sized-batch-effects-------------------
params.batches <- newSplatPopParams(similarity.scale = 8,
                                    batchCells = c(5, 5, 5),
                                    batch.size = 3,
                                    batch.facLoc = c(0.5, 0.1,  0.1),
                                    batch.facScale = c(0.6, 0.15, 0.15))

sim.pop.batches <- splatPopSimulate(vcf = vcf, gff = gff, 
                                    params = params.batches, sparsify = FALSE)
sim.pop.batches <- logNormCounts(sim.pop.batches)
sim.pop.batches <- runPCA(sim.pop.batches, ncomponents = 10)
plotPCA(sim.pop.batches, colour_by = "Sample", shape_by = "Batch", point_size=3)

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

