## ---- include = FALSE---------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## -----------------------------------------------------------------------------
if (!requireNamespace("BiocManager", quietly = TRUE))
    install.packages("BiocManager")

BiocManager::install("miaViz")

## ----setup, message=FALSE-----------------------------------------------------
library(miaViz)
data(GlobalPatterns, package = "mia")

## -----------------------------------------------------------------------------
plotAbundance(GlobalPatterns, rank = NULL, 
              features = "549322", abund_values = "counts")

## -----------------------------------------------------------------------------
GlobalPatterns <- transformCounts(GlobalPatterns, method = "relabundance")

## -----------------------------------------------------------------------------
plotAbundance(GlobalPatterns, rank = "Kingdom", abund_values = "relabundance")

## -----------------------------------------------------------------------------
prev_phylum <- getPrevalentTaxa(GlobalPatterns, rank = "Phylum",
                                detection = 0.01)

## -----------------------------------------------------------------------------
plotAbundance(GlobalPatterns[rowData(GlobalPatterns)$Phylum %in% prev_phylum],
              rank = "Phylum",
              abund_values = "relabundance")

## -----------------------------------------------------------------------------
library(patchwork)
plots <- plotAbundance(GlobalPatterns[rowData(GlobalPatterns)$Phylum %in% prev_phylum],
                       features = "SampleType",
                       rank = "Phylum",
                       abund_values = "relabundance")
plots$abundance / plots$SampleType +
     plot_layout(heights = c(9, 1))

## -----------------------------------------------------------------------------
plotTaxaPrevalence(GlobalPatterns, rank = "Phylum",
                   detections = c(0, 0.001, 0.01, 0.1, 0.2))

## -----------------------------------------------------------------------------
plotPrevalentAbundance(GlobalPatterns, rank = "Family",
                       colour_by = "Phylum") +
    scale_x_log10()

## -----------------------------------------------------------------------------
plotPrevalence(GlobalPatterns,
               rank = "Phylum",
               detections = c(0.01, 0.1, 1, 2, 5, 10, 20)/100,
               prevalences = seq(0.1, 1, 0.1))

## ---- message=FALSE-----------------------------------------------------------
library(scater)
library(mia)

## -----------------------------------------------------------------------------
altExp(GlobalPatterns,"Genus") <- agglomerateByRank(GlobalPatterns,"Genus")
altExp(GlobalPatterns,"Genus") <- addPerFeatureQC(altExp(GlobalPatterns,"Genus"))
rowData(altExp(GlobalPatterns,"Genus"))$log_mean <-
    log(rowData(altExp(GlobalPatterns,"Genus"))$mean)
rowData(altExp(GlobalPatterns,"Genus"))$detected <- 
    rowData(altExp(GlobalPatterns,"Genus"))$detected / 100
top_taxa <- getTopTaxa(altExp(GlobalPatterns,"Genus"),
                       method="mean",
                       top=100L,
                       abund_values="counts")

## ----plot1, fig.cap="Tree plot using ggtree with tip labels decorated by mean abundance (colour) and prevalence (size)"----
plotRowTree(altExp(GlobalPatterns,"Genus")[top_taxa,],
            tip_colour_by = "log_mean",
            tip_size_by = "detected")

## ----plot2, fig.cap="Tree plot using ggtree with tip labels decorated by mean abundance (colour) and prevalence (size). Tip labels of the tree are shown as well."----
plotRowTree(altExp(GlobalPatterns,"Genus")[top_taxa,],
            tip_colour_by = "log_mean",
            tip_size_by = "detected",
            show_label = TRUE)

## ----plot3, fig.cap="Tree plot using ggtree with tip labels decorated by mean abundance (colour) and prevalence (size). Selected node and tip labels are shown."----
labels <- c("Genus:Providencia", "Genus:Morganella", "0.961.60")
plotRowTree(altExp(GlobalPatterns,"Genus")[top_taxa,],
            tip_colour_by = "log_mean",
            tip_size_by = "detected",
            show_label = labels,
            layout="rectangular")

## ----plot4, fig.cap="Tree plot using ggtree with tip labels decorated by mean abundance (colour) and edges labeled Kingdom (colour) and prevalence (size)"----
plotRowTree(altExp(GlobalPatterns,"Genus")[top_taxa,],
            edge_colour_by = "Phylum",
            tip_colour_by = "log_mean")

## -----------------------------------------------------------------------------
data(col_graph)

## -----------------------------------------------------------------------------
plotColGraph(col_graph,
             altExp(GlobalPatterns,"Genus"),
             colour_by = "SampleType",
             edge_colour_by = "weight",
             edge_width_by = "weight",
             show_label = TRUE)

## -----------------------------------------------------------------------------
metadata(altExp(GlobalPatterns,"Genus"))$graph <- col_graph

## ----include=FALSE------------------------------------------------------------
plotColGraph(altExp(GlobalPatterns,"Genus"),
             name = "graph",
             colour_by = "SampleType",
             edge_colour_by = "weight",
             edge_width_by = "weight",
             show_label = TRUE)

## -----------------------------------------------------------------------------
library(microbiomeDataSets)
silverman <- SilvermanAGutData()
silverman <- transformCounts(silverman, method = "relabundance")
taxa <- getTopTaxa(silverman, 2)

## -----------------------------------------------------------------------------
plotSeries(silverman,
           x = "DAY_ORDER",
           y = taxa,
           colour_by = "Family")

## -----------------------------------------------------------------------------
plotSeries(silverman[taxa,],
           x = "DAY_ORDER",
           colour_by = "Family",
           linetype_by = "Phylum",
           abund_values = "relabundance")

## -----------------------------------------------------------------------------
plotSeries(silverman,
           x = "DAY_ORDER",
           y = getTopTaxa(silverman, 5),
           colour_by = "Family",
           linetype_by = "Phylum",
           abund_values = "counts")

## -----------------------------------------------------------------------------
sessionInfo()

