## ----chunk_setup, include = FALSE---------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----vignette_setup, echo=FALSE, message=FALSE, warning = FALSE---------------
# Track time spent on making the vignette.
start_time <- Sys.time()

# Bib setup.
library(RefManageR)

# Write bibliography information
bib <- c(
    R = citation(),
    BiocStyle = citation("BiocStyle")[1],
    knitr = citation("knitr")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    RefManageR = citation("RefManageR")[1],
    decoupleR = citation("decoupleR")[1],
    GSVA = citation("GSVA")[1],
    viper = citation("viper")[1]
)

## ----bioconductor_install, eval=FALSE-----------------------------------------
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#      install.packages("BiocManager")
#  }
#  
#  BiocManager::install("decoupleR")
#  
#  # Check that you have a valid Bioconductor installation
#  BiocManager::valid()

## ----github_install, eval=FALSE-----------------------------------------------
#  # install.packages("devtools")
#  devtools::install_github("saezlab/decoupleR")

## ----decoupleR citation-------------------------------------------------------
citation("decoupleR")

## ----load_library, message=FALSE----------------------------------------------
library(decoupleR)
library(dplyr)

## ----read_example_data--------------------------------------------------------
inputs_dir <- system.file("testdata", "inputs", package = "decoupleR")

mat <- file.path(inputs_dir, "input-expr_matrix.rds") %>%
    readRDS() %>%
    glimpse()

network <- file.path(inputs_dir, "input-dorothea_genesets.rds") %>%
    readRDS() %>%
    glimpse()

## ----usage-decouple_function, message=TRUE------------------------------------
decouple(
    mat = mat,
    network = network,
    .source = "tf",
    .target = "target",
    statistics = c("gsva", "mean", "pscira", "scira", "viper", "ora"),
    args = list(
        gsva = list(verbose = FALSE),
        mean = list(.mor = "mor", .likelihood = "likelihood"),
        pscira = list(.mor = "mor"),
        scira = list(.mor = "mor"),
        viper = list(
            .mor = "mor",
            .likelihood = "likelihood",
            verbose = FALSE
        ),
        ora = list()
    )
) %>%
    glimpse()

## ----see gsvas individual calls-----------------------------------------------
gsvas_res <- decouple(
    mat = head(mat, 5000),
    network = network,
    .source = "tf",
    .target = "target",
    statistics = c("gsva"),
    args = list(
        gsva_default = list(verbose = FALSE),
        gsva_minsize = list(verbose = FALSE, ssgsea.norm = FALSE)
    ),
    show_toy_call = TRUE
)

## ----run_individual_gsvas-----------------------------------------------------
gsva_1 <- run_gsva(
    mat = head(mat, 5000),
    network = network,
    .source = "tf",
    .target = "target",
    verbose = FALSE
)

gsva_2 <- run_gsva(
    mat = head(mat, 5000),
    network = network,
    .source = "tf",
    .target = "target",
    verbose = FALSE,
    ssgsea.norm = FALSE
)

gsvas_res_2 <- bind_rows(gsva_1, gsva_2, .id = "run_id")

## ----see_not_differences------------------------------------------------------
all.equal(gsvas_res, gsvas_res_2)

## ----show_columns-------------------------------------------------------------
network %>%
    colnames()

## -----------------------------------------------------------------------------
this_column <- "target"
viper_res <- decouple(
    mat = mat,
    network = network,
    .source = tf,
    .target = !!this_column,
    statistics = c("viper"),
    args = list(
        viper = list(
            .mor = 4,
            .likelihood = "likelihood",
            verbose = FALSE
        )
    ),
    show_toy_call = TRUE
)

## ----create_vignette, eval=FALSE----------------------------------------------
#  # Create the vignette
#  library(rmarkdown)
#  system.time(render("decoupleR.Rmd", "BiocStyle::html_document"))
#  
#  # Extract the R code
#  library(knitr)
#  knit("decoupleR.Rmd", tangle = TRUE)

## ----reproduce_time, echo=FALSE-----------------------------------------------
# Processing time in seconds
total_time <- diff(c(start_time, Sys.time()))
round(total_time, digits = 3)

## ----session_info, echo=FALSE-----------------------------------------------------------------------------------------
options(width = 120)
sessioninfo::session_info()

## ----vignetteBiblio, results = "asis", echo = FALSE, warning = FALSE, message = FALSE---------------------------------
## Print bibliography
PrintBibliography(bib, .opts = list(hyperlink = "to.doc", style = "html"))

