## ---- code=readLines(system.file('vignettes_inc.R', package='biodb')), echo=FALSE----
# Disable automatic loading of extra biodb packages
Sys.setenv(BIODB_AUTOLOAD_EXTRA_PKGS="FALSE")

biodbVignettes <- data.frame()
files <- Sys.glob('*.Rmd')
for (f in files) {
	name <- sub('^(.*)\\.Rmd', '\\1', f, perl=TRUE)
	firstLines <- readLines(f, n=20)
	title <- grep("^title:", firstLines, value=TRUE)
	title <- sub('^title: *"(.*)\\.?"$', '\\1', title, perl=TRUE)
	desc <- grep("%\\\\VignetteIndexEntry", firstLines, value=TRUE)
	desc <- sub('^.*VignetteIndexEntry{(.*)}.*$', '\\1', desc, perl=TRUE)
    html <- paste0(name, '.html')
    link <- paste0('[', title, '](', html, ')')
    biodbVignettes <- rbind(biodbVignettes, data.frame(name=name, title=title,
                                                       desc=desc, html=html,
                                                       link=link))
}

make_vignette_ref <- function(name) {
	cat(biodbVignettes[biodbVignettes$name == name, 'link', drop=TRUE])
}

insert_features_table <- function() {
    featuresFile <- system.file("features.tsv",
                                package='biodb')
    featuresDf <- read.table(featuresFile, sep="\t", header=TRUE, quote="",
                             stringsAsFactors=FALSE)
    knitr::kable(featuresDf, "pipe", label="features",
                 caption="*biodb* main features. These are generic features (i.e.: present at top-level of architecture or present in at least a group of connectors), unless specified otherwise.")
}

## ---- echo=FALSE, results='asis'----------------------------------------------
make_vignette_ref('entries')

## -----------------------------------------------------------------------------
mybiodb <- biodb::newInst()

## -----------------------------------------------------------------------------
fileUrl <- system.file("extdata", "massbank_extract_lcms_2.tsv", package='biodb')

## -----------------------------------------------------------------------------
conn <- mybiodb$getFactory()$createConn('mass.csv.file', url=fileUrl)

## -----------------------------------------------------------------------------
conn$addField('ms.level', 1)
conn$addField('chrom.rt.unit', 's')

## -----------------------------------------------------------------------------
conn$getMatchingMzField()

## -----------------------------------------------------------------------------
conn$setMatchingMzField('peak.mztheo')

## -----------------------------------------------------------------------------
conn$searchForMassSpectra(mz.min=145.96, mz.max=147.13)

## -----------------------------------------------------------------------------
conn$searchForMassSpectra(mz=c(86.000002, 146.999997), mz.tol=10, mz.tol.unit='ppm')

## -----------------------------------------------------------------------------
conn$searchForMassSpectra(mz.min=72.78, mz.max=73.12, rt=45.5, rt.tol=1, rt.unit='s')

## -----------------------------------------------------------------------------
input <- data.frame(mz=c(73.01, 116.04, 174.2),
                    rt=c(79, 173, 79))

## -----------------------------------------------------------------------------
conn$searchMsPeaks(input, mz.tol=0.1, rt.unit='s', rt.tol=10, match.rt=TRUE, prefix='match.')

## ----Set a list of M/Z values-------------------------------------------------
ms.tsv <- system.file("extdata", "ms.tsv", package='biodb')
mzdf <- read.table(ms.tsv, header=TRUE, sep="\t")

## ----Instantiate an in-house LCMS database------------------------------------
lcmsdb <- system.file("extdata", "massbank_extract_lcms_1.tsv", package="biodb")
massbank <- mybiodb$getFactory()$createConn('mass.csv.file', url=lcmsdb)

## ----Set missing fields on database-------------------------------------------
massbank$addField('ms.level', 1)
massbank$addField('chrom.rt.unit', 's')

## ----Annotate M/Z values with an in-house LCMS database-----------------------
massbank$searchMsPeaks(mzdf, mz.tol=1e-3, fields=c('accession', 'name', 'formula', 'chebi.id'), prefix='mydb.')

## ----Set chromatographic columns and output fields----------------------------
chromColIds <- c('TOSOH TSKgel ODS-100V  5um Part no. 21456')
fields <- c('accession', 'name', 'formula', 'chebi.id', 'chrom.rt', 'chrom.col.id')

## ----Annotate using RT values-------------------------------------------------
massbank$searchMsPeaks(mzdf, mz.tol=1e-3, fields=fields, prefix='mydb.', chrom.col.ids=chromColIds, rt.unit='s', rt.tol=10, match.rt=TRUE)

## -----------------------------------------------------------------------------
db.tsv <- system.file("extdata", "massbank_extract_msms.tsv", package='biodb')
conn <- mybiodb$getFactory()$createConn('mass.csv.file', url=db.tsv)

## -----------------------------------------------------------------------------
conn$searchForMassSpectra(mz.min=115, mz.max=115.1, max.results=5)

## -----------------------------------------------------------------------------
conn$searchForMassSpectra(mz=115, mz.tol=0.1, mz.tol.unit='plain', max.results=5)

## -----------------------------------------------------------------------------
spectrum <- data.frame(mz=c(286.1456, 287.1488, 288.1514), rel.int=c(100, 45, 18))

## -----------------------------------------------------------------------------
conn$msmsSearch(spectrum, precursor.mz=286.1438, mz.tol=0.1, mz.tol.unit='plain', ms.mode='pos')

## -----------------------------------------------------------------------------
sqliteFile <- system.file("extdata", "generated", "massbank_extract_full.sqlite", package='biodb')

## -----------------------------------------------------------------------------
sqliteConn <-  mybiodb$getFactory()$createConn('mass.sqlite', url=sqliteFile)

## -----------------------------------------------------------------------------
sqliteConn$searchMsPeaks(mzdf, mz.tol=1e-3, fields=c('accession', 'name', 'formula', 'chebi.id'), prefix='mydb.')

## -----------------------------------------------------------------------------
mybiodb$terminate()

