## ---- code=readLines(system.file('vignettes_inc.R', package='biodb')), echo=FALSE----
# Disable automatic loading of extra biodb packages
Sys.setenv(BIODB_AUTOLOAD_EXTRA_PKGS="FALSE")

biodbVignettes <- data.frame()
files <- Sys.glob('*.Rmd')
for (f in files) {
	name <- sub('^(.*)\\.Rmd', '\\1', f, perl=TRUE)
	firstLines <- readLines(f, n=20)
	title <- grep("^title:", firstLines, value=TRUE)
	title <- sub('^title: *"(.*)\\.?"$', '\\1', title, perl=TRUE)
	desc <- grep("%\\\\VignetteIndexEntry", firstLines, value=TRUE)
	desc <- sub('^.*VignetteIndexEntry{(.*)}.*$', '\\1', desc, perl=TRUE)
    html <- paste0(name, '.html')
    link <- paste0('[', title, '](', html, ')')
    biodbVignettes <- rbind(biodbVignettes, data.frame(name=name, title=title,
                                                       desc=desc, html=html,
                                                       link=link))
}

make_vignette_ref <- function(name) {
	cat(biodbVignettes[biodbVignettes$name == name, 'link', drop=TRUE])
}

insert_features_table <- function() {
    featuresFile <- system.file("features.tsv",
                                package='biodb')
    featuresDf <- read.table(featuresFile, sep="\t", header=TRUE, quote="",
                             stringsAsFactors=FALSE)
    knitr::kable(featuresDf, "pipe", label="features",
                 caption="*biodb* main features. These are generic features (i.e.: present at top-level of architecture or present in at least a group of connectors), unless specified otherwise.")
}

## ---- echo=FALSE, results='asis'----------------------------------------------
make_vignette_ref('entries')

## -----------------------------------------------------------------------------
mybiodb <- biodb::newInst()

## -----------------------------------------------------------------------------
csvUrl <- system.file("extdata", "chebi_extract.tsv", package='biodb')

## ----csvTable, echo=FALSE, results='asis'-------------------------------------
csvDf <- read.table(csvUrl, sep="\t", header=TRUE, quote="")
## Prevent RMarkdown from interpreting @ character as a reference:
csvDf$smiles <- vapply(csvDf$smiles, function(s) paste0('`', s, '`'), FUN.VALUE='')
knitr::kable(head(csvDf), "pipe", caption="Excerpt from compound database TSV file.")

## -----------------------------------------------------------------------------
conn <- mybiodb$getFactory()$createConn('comp.csv.file', url=csvUrl)

## -----------------------------------------------------------------------------
csvUrl2 <- system.file("extdata", "chebi_extract_custom.csv", package='biodb')

## -----------------------------------------------------------------------------
conn2 <- mybiodb$getFactory()$createConn('comp.csv.file', url=csvUrl2)

## -----------------------------------------------------------------------------
conn2$setCsvSep(';')

## -----------------------------------------------------------------------------
conn2$setCsvQuote('')

## -----------------------------------------------------------------------------
conn2$setField('accession',         'ID')
conn2$setField('kegg.compound.id',  'kegg')
conn2$setField('monoisotopic.mass', 'mass')
conn2$setField('molecular.mass',    'molmass')

## -----------------------------------------------------------------------------
entries <- conn$getEntry(c('1018', '1456', '16750', '64679'))
entries

## -----------------------------------------------------------------------------
entriesDf <- mybiodb$entriesToDataframe(entries)

## ----entriesTable, echo=FALSE, results='asis'---------------------------------
## Prevent RMarkdown from interpreting @ character as a reference:
entriesDf$smiles <- vapply(entriesDf$smiles, function(s) paste0('`', s, '`'), FUN.VALUE='')
knitr::kable(entriesDf, "pipe", caption="Some entries from the compound database.")

## ---- echo=FALSE, results='asis'----------------------------------------------
make_vignette_ref('entries')

## -----------------------------------------------------------------------------
conn$searchForEntries(list(monoisotopic.mass=list(value=283.0917, delta=0.1)))

## -----------------------------------------------------------------------------
conn$searchForEntries(list(monoisotopic.mass=list(value=283.0917, ppm=10)))

## -----------------------------------------------------------------------------
conn$searchForEntries(list(monoisotopic.mass=list(min=283.091, max=283.093)))

## -----------------------------------------------------------------------------
conn$searchForEntries(list(monoisotopic.mass=list(value=283.0917, delta=0.1)), max.results=2)

## -----------------------------------------------------------------------------
mybiodb$getEntryFields()$getFieldNames(type='mass')

## -----------------------------------------------------------------------------
mybiodb$getEntryFields()$get(c('monoisotopic.mass', 'nominal.mass'))

## -----------------------------------------------------------------------------
conn$isSearchableByField('monoisotopic.mass')

## -----------------------------------------------------------------------------
conn$getSearchableFields()

## -----------------------------------------------------------------------------
conn$searchForEntries(list(name='deoxyguanosine'))

## -----------------------------------------------------------------------------
conn$searchForEntries(list(name='guanosine', monoisotopic.mass=list(value=283.0917, delta=0.1)))

## -----------------------------------------------------------------------------
msTsv <- system.file("extdata", "ms.tsv", package='biodb')
mzDf <- read.table(msTsv, header=TRUE, sep="\t")

## ----mzTable, echo=FALSE, results='asis'--------------------------------------
knitr::kable(mzDf, "pipe", caption="Input M/Z values.")

## -----------------------------------------------------------------------------
annotDf <- conn$annotateMzValues(mzDf, mz.tol=1e-3, ms.mode='neg', mz.tol.unit='plain',
                                 fields=c('accession', 'name', 'formula',
                                          'molecular.mass', 'monoisotopic.mass'),
                                 prefix='mydb.', fieldsLimit=1)

## ----annotTable, echo=FALSE, results='asis'-----------------------------------
knitr::kable(annotDf, "pipe", caption="The annotated mass spectrum. Columns prefixed with \"mydb.\" come from the compound database.")

## ---- echo=FALSE, results='asis'----------------------------------------------
make_vignette_ref('in_house_mass_db')

## -----------------------------------------------------------------------------
sqliteFile <- system.file("extdata", "generated", "chebi_extract.sqlite", package='biodb')

## -----------------------------------------------------------------------------
sqliteConn <-  mybiodb$getFactory()$createConn('comp.sqlite', url=sqliteFile)

## -----------------------------------------------------------------------------
sqliteConn$searchForEntries(list(monoisotopic.mass=list(value=283.0917, delta=0.1)))

## -----------------------------------------------------------------------------
mybiodb$terminate()

