## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 5,
  fig.height = 4,
  dpi=200
)

## ---- message=FALSE, warning=FALSE--------------------------------------------
library(NanoStringNCTools)

## ---- message=FALSE, warning=FALSE--------------------------------------------
library(ggthemes)
library(ggiraph)

## -----------------------------------------------------------------------------
datadir <- system.file("extdata", "3D_Bio_Example_Data",
                       package = "NanoStringNCTools")
rcc_files <- dir(datadir, pattern = "SKMEL.*\\.RCC$", full.names = TRUE)
rlf_file <- file.path(datadir, "3D_SolidTumor_Sig.rlf")
sample_annotation <- file.path(datadir, "3D_SolidTumor_PhenoData.csv")
demoData <- readNanoStringRccSet(rcc_files, rlfFile = rlf_file, 
                                 phenoDataFile = sample_annotation)
class( demoData )
isS4( demoData )
is( demoData, "ExpressionSet" )
demoData

## -----------------------------------------------------------------------------
head( pData( demoData ), 2 )
protocolData( demoData )
svarLabels( demoData )
head( sData(demoData), 2 )

## -----------------------------------------------------------------------------
design( demoData ) <- ~ `Treatment`
design( demoData )

dimLabels( demoData )
protocolData(demoData)[["Sample ID"]] <- sampleNames(demoData)
dimLabels( demoData )[2] <- "Sample ID"
dimLabels( demoData )

## -----------------------------------------------------------------------------
head( summary( demoData , MARGIN = 1 ), 2 )
head( summary( demoData , MARGIN = 2 ), 2 )
unique( sData( demoData )$"Treatment" )
head( summary( demoData , MARGIN = 2, GROUP = "Treatment" )$VEM, 2 )
head( summary( demoData , MARGIN = 2, GROUP = "Treatment" )$"DMSO", 2 )
head( summary( demoData , MARGIN = 2, GROUP = "Treatment", log2 = FALSE )$"DMSO", 2 )

## -----------------------------------------------------------------------------
length( sampleNames( demoData ) )
length( sampleNames( subset( demoData , 
                             select = phenoData( demoData )[["Treatment"]] == "VEM" ) ) )

dim( exprs( demoData ) )
dim( exprs( endogenousSubset( demoData, 
                              select = phenoData( demoData )[["Treatment"]] == "VEM" ) ) )

with( housekeepingSubset( demoData ) , table( CodeClass ) )
with( negativeControlSubset( demoData ) , table( CodeClass ) )
with( positiveControlSubset( demoData ) , table( CodeClass ) )
with( controlSubset( demoData ) , table( CodeClass ) )
with( nonControlSubset( demoData ) , table( CodeClass ) )

## -----------------------------------------------------------------------------
assayDataElement( demoData, "demoElem" ) <- 
  assayDataApply( demoData, MARGIN=2, FUN=log, base=10, elt="exprs" )
assayDataElement( demoData, "demoElem" )[1:3, 1:2]
assayDataApply( demoData, MARGIN=1, FUN=mean, elt="demoElem")[1:5]

head( esBy( demoData, 
            GROUP = "Treatment", 
            FUN = function( x ) { 
              assayDataApply( x, MARGIN = 1, FUN=mean, elt="demoElem" ) 
            } ) )

## -----------------------------------------------------------------------------
demoData <- normalize( demoData , type="nSolver", fromELT = "exprs" , toELT = "exprs_norm" )
assayDataElement( demoData , elt = "exprs_norm" )[1:3, 1:2]

## -----------------------------------------------------------------------------
neg_set <- negativeControlSubset( demoData )
class( neg_set )
neg_ctrls <- munge( neg_set )
head( neg_ctrls, 2 )
class( neg_ctrls )
head( munge( demoData ), 2 )
munge( demoData, mapping = ~`BRAFGenotype` + GeneMatrix )

exprs_df <- transform( assayData( demoData )[["exprs_norm"]] )
class( exprs_df )
exprs_df[1:3, 1:2]

## ---- fig.cap="Housekeeping Genes QC Plot"------------------------------------
demoData <- setQCFlags( demoData )
tail( svarLabels( demoData ) )
head( protocolData( demoData )[["QCFlags"]], 2 )
head( protocolData( demoData )[["QCBorderlineFlags"]], 2 )

## -----------------------------------------------------------------------------
theme_set( theme_gray( base_family = "Arial" ) )
girafe( ggobj = autoplot( demoData , "bindingDensity-mean" ) )
girafe( ggobj = autoplot( demoData , "bindingDensity-sd" ) )

## -----------------------------------------------------------------------------
girafe( ggobj = autoplot( demoData , "lane-bindingDensity" ) )
girafe( ggobj = autoplot( demoData , "lane-fov" ) )

## -----------------------------------------------------------------------------
subData <- subset( demoData, select = phenoData( demoData )[["Treatment"]] == "DMSO" )
girafe( ggobj = autoplot( subData, "housekeep-geom" ) )

## -----------------------------------------------------------------------------
girafe( ggobj = autoplot( demoData , "ercc-linearity" ) )
girafe( ggobj = autoplot( subData , "ercc-lod" ) )

## -----------------------------------------------------------------------------
#girafe( ggobj = autoplot( demoData , "boxplot-feature" , index = featureNames(demoData)[3] , elt = "exprs" ) )
#autoplot( demoData , "heatmap-genes" , elt = "exprs_norm" )

## -----------------------------------------------------------------------------
sessionInfo()

