## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)
library("SharedObject")

## -----------------------------------------------------------------------------
library(parallel)
## Initiate the cluster
cl <- makeCluster(1)
## create data
n <- 3
A <- matrix(runif(n^2), n, n)
## create a shared object
shared_A <- share(A)
## export the shared object
clusterExport(cl,"shared_A")

stopCluster(cl)

## -----------------------------------------------------------------------------
## check the data
A
shared_A
## check the class
class(A)
class(shared_A)
## idential
identical(A, shared_A)

## -----------------------------------------------------------------------------
## `shared_A` is a shared object
is.shared(shared_A)

## The subset of `shared_A` is not
is.shared(shared_A[1:2])

## -----------------------------------------------------------------------------
## the element `a` is sharable and b is not
mydata <- list(a = 1:3, b = letters[1:3])

## Will get an error if we directly share the object
## share(mydata)

## Use the `mustWork` argument to suppress the error message
sharedList1 <- share(mydata, mustWork = FALSE)

## Use the function `tryShare`
## this is equivalent to `share(mydata, mustWork = FALSE)`
sharedList2 <- tryShare(mydata)

## Only the element `a` is a shared object
is.shared(sharedList1)

## -----------------------------------------------------------------------------
## Check if an object is a shared object
## This works for both vector and data.frame
is.shared(A)
is.shared(shared_A)

## -----------------------------------------------------------------------------
## get a summary report
getSharedObjectProperty(shared_A)

## Internal function to check the properties
## get the individual properties
getCopyOnWrite(shared_A)
getSharedSubset(shared_A)
getSharedCopy(shared_A)

## -----------------------------------------------------------------------------
getSharedObjectOptions()

## -----------------------------------------------------------------------------
## change the default setting
setSharedObjectOptions(mustWork = FALSE)
## Check if the change is made
getSharedObjectOptions("mustWork")

## -----------------------------------------------------------------------------
shared_A2 <- shared_A
shared_A[1,1] <- 10

shared_A
shared_A2

## shared_A became a regular R object
is.shared(shared_A)

## -----------------------------------------------------------------------------
shared_A <- share(A, copyOnWrite=FALSE)
shared_A2 <- shared_A
shared_A[1,1] <- 10

shared_A
shared_A2

## -----------------------------------------------------------------------------
shared_A <- share(A, copyOnWrite = FALSE)
shared_A
-shared_A
shared_A

## -----------------------------------------------------------------------------
shared_A <- share(A, copyOnWrite = FALSE)
shared_A2 <- shared_A
## change the value of shared_A
shared_A[1,1] <- 10
## Both shared_A and shared_A2 are affected
shared_A
shared_A2

## Enable copy-on-write
setCopyOnWrite(shared_A, TRUE)
## shared_A is now independent with shared_A2
shared_A[1,2] <- 10
shared_A
shared_A2

## -----------------------------------------------------------------------------
shared_A <- share(A, sharedSubset = TRUE, sharedCopy = TRUE)
getSharedObjectProperty(shared_A, property = c("sharedSubset", "sharedCopy"))

## Changing the value of `shared_A` will not 
## result in a regular R object
shared_A2 <- shared_A
shared_A[1,1] <- 10
is.shared(shared_A)

## -----------------------------------------------------------------------------
sessionInfo()

