## ---- echo = FALSE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
library(knitr)

## ----install, eval = FALSE-----------------------------------------------
#  BiocManager::install(c("SBGNview"))

## ---- echo = TRUE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
# load demo dataset and pathway information of built-in collection of SBGN-ML files. 
# We use a cancer microarray dataset 'gse16837.d' from package 'pathview'
library(SBGNview)
data("gse16873.d","pathways.info","sbgn.xmls")
# search for pathways with user defined keywords
input.pathways <- findPathways("Adrenaline and noradrenaline biosynthesis")
# render SBGN pathway graph and output image files
SBGNview.obj <- SBGNview(
          gene.data = gse16873.d[,1:3], 
          gene.id.type = "entrez",
          input.sbgn = input.pathways$pathway.id,
          output.file = "quick.start", 
          output.formats = c("png")
          ) 
print(SBGNview.obj)

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
list.files( pattern = "quick.start", full.names = TRUE) 

## ----quickStartFig, echo = FALSE,fig.cap="\\label{fig:quickStartFig}Quick start example: Adrenaline and noradrenaline biosynthesis pathway. "----
include_graphics("quick.start_P00001.svg")

## ---- echo = TRUE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
outputFile(SBGNview.obj) <- "quick.start.highlight.elements"
SBGNview.obj + 
        highlightArcs(class = "production",color = "red") +
        highlightArcs(class = "consumption",color = "blue") +
        highlightNodes(node.set = c("tyrosine", "(+-)-epinephrine"),
                       stroke.width = 4, stroke.color = "green") + 
        highlightPath(from.node = "tyrosine", to.node = "dopamine",
                      from.node.color = "green",
                      to.node.color = "blue",
                      shortest.paths.cols = "purple",
                      input.node.stroke.width = 6,
                      path.node.stroke.width = 5,
                      path.node.color = "purple",
                      path.stroke.width = 5,
                      tip.size = 10 )

## ----quickStartFigModified, echo = FALSE,fig.cap="\\label{fig:quickStartFigModified}Quick start example: Adrenaline and noradrenaline biosynthesis pathway. Highlight nodes and edges."----
include_graphics("quick.start.highlight.elements_P00001.svg")

## ---- echo = TRUE,   message = FALSE, warning = FALSE--------------------
data("pathways.stat")
gse16873 <- gse16873.d[,1:3]
input.pathway.ids = input.pathways$pathway.id
head(pathways.info)
pathways.stat

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
pathways <- findPathways(c("bile acid","bile salt"))
head(pathways)
pathways.local.file <- downloadSbgnFile(pathways$pathway.id[1:3])
pathways.local.file

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
pathways <- findPathways(c("tp53","Trp53"),keyword.type = "SYMBOL")
head(pathways)
pathways <- findPathways(c("K04451","K10136"),keyword.type = "KO")
head(pathways)

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
cpd.data <- sim.mol.data(mol.type = "cpd", id.type = "KEGG COMPOUND accession", nmol = 50000, nexp = 2)
head(cpd.data)

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
gene.data <- gse16873
head(gene.data[,1:2])

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
gene.data <- changeDataId(data.input.id = gene.data,
                           input.type = "entrez",
                           output.type = "pathwayCommons",
                           cpd.or.gene = "gene",
                           sum.method = "sum"
                           )

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
head(gene.data[,1:2])

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
SBGNview.obj <- SBGNview(
              gene.data = gene.data,
              input.sbgn = "P00001",
              output.file = "test_output",
              gene.id.type = "pathwayCommons",
              output.formats =  c("svg")
    )
SBGNview.obj

## ----figGeneData, echo = FALSE,fig.cap="\\label{fig:figGeneData}Visualization of gene expression data."----
include_graphics("test_output_P00001.svg")

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
cpd.data <- changeDataId(data.input.id = cpd.data,
                           input.type = "kegg.ligand",
                           output.type = "pathwayCommons",
                           cpd.or.gene = "compound",
                           sum.method = "sum"
                           )
head(cpd.data)

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
SBGNview.obj <- SBGNview(
                gene.data = gse16873,
                cpd.data = cpd.data,
                input.sbgn = "P00001",
                output.file = "test_output.gene.compound",
                gene.id.type = "entrez",
                cpd.id.type = "pathwayCommons",
                output.formats =  c("svg")
                )
SBGNview.obj

## ----figGeneAndCpdData, echo = FALSE,fig.cap="\\label{fig:figGeneAndCpdData}Visualization of both gene expression and compound abundance data."----
include_graphics("test_output.gene.compound_P00001.svg")

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
result.one.sbgn <- SBGNview.obj$data[[1]]
names(result.one.sbgn)
glyphs <- result.one.sbgn$glyphs.list
arcs <- result.one.sbgn$arcs.list
str(glyphs[[1]])
str(arcs[[1]])

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
outputFile(SBGNview.obj)
outputFile(SBGNview.obj) <- "test.change.output.file"
outputFile(SBGNview.obj)
SBGNview.obj
outputFile(SBGNview.obj) <- "test.print"
outputFile(SBGNview.obj)
print(SBGNview.obj)

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
download.file("https://raw.githubusercontent.com/datapplab/SBGNhub/master/data/SBGN.with.stamp/pathwayCommons/http___identifiers.org_panther.pathway_P00001.1.sbgn",destfile = "P00001.new.layout.sbgn")
SBGNview(
          gene.data = gse16873, 
          gene.id.type = "entrez",
          input.sbgn = "P00001.new.layout.sbgn",
          sbgn.gene.id.type = "pathwayCommons",
          output.file = "test.different.layout", 
          output.formats =  c("svg")
          )

## ----differentLayout, echo = FALSE,fig.cap="\\label{fig:differentLayoutFig}Graph with different layout."----
include_graphics("test.different.layout_P00001.new.layout.sbgn.svg")

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
highlight.all.nodes.sbgn.obj <-  SBGNview.obj +
        highlightNodes( 
# Here we set argument "node.set" to select all nodes
            node.set = "all",
            stroke.width = 4, stroke.color = "green")
outputFile(highlight.all.nodes.sbgn.obj) = "highlight.all.nodes"
print(highlight.all.nodes.sbgn.obj)

## ----highlightAllNodes, echo = FALSE,fig.cap="\\label{fig:highlightAllNodes}Highlight all nodes."----
include_graphics("highlight.all.nodes_P00001.svg")

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
highlight.macromolecule.sbgn.obj <-  SBGNview.obj +
        highlightNodes(
# Here we set argument "select.glyph.class" to select macromolecule nodes
            select.glyph.class = "macromolecule",
            stroke.width = 4, stroke.color = "green")
outputFile(highlight.macromolecule.sbgn.obj) = "highlight.macromolecule"
print(highlight.macromolecule.sbgn.obj)

## ----highlightMacromolecule, echo = FALSE,fig.cap="\\label{fig:highlightMacromolecule}Highlight macromolecule nodes."----
include_graphics("highlight.macromolecule_P00001.svg")

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
highlight.all.nodes.sbgn.obj <-  SBGNview.obj +
        highlightNodes( node.set = "(+-)-epinephrine", stroke.width = 4, stroke.color = "green",
# Here we set argument "show.glyph.id" to display node ID instead of the original label.
                         show.glyph.id = TRUE,
                        label.font.size = 10)
outputFile(highlight.all.nodes.sbgn.obj) = "highlight.all.id.nodes"
print(highlight.all.nodes.sbgn.obj)

## ----highlightNodes, echo = FALSE,fig.cap="\\label{fig:highlightNodes}Highlight nodes using node IDs."----
include_graphics("highlight.all.id.nodes_P00001.svg")

## ---- echo = TRUE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
my.labels <- c("Tyr","epinephrine")
names(my.labels) <- c("tyrosine", "(+-)-epinephrine")
SBGNview.obj.adjust.label <-  SBGNview.obj +
        highlightNodes( node.set = c("tyrosine", "(+-)-epinephrine"), stroke.width = 4, stroke.color = "green",
                         label.x.shift = 0,
# Labels are moved up a little bit                        
                         label.y.shift = -20,
                         label.color = "red",
                         label.font.size = 30,
                         label.spliting.string = "", 
# node labels can be customized by a named vector. The names of the vector is the IDs assigned to argument "node.set". Values of the vector are the new labels for display.                
                         labels = my.labels)
outputFile(SBGNview.obj.adjust.label) <- "adjust.label"
print(SBGNview.obj.adjust.label)

## ----changeLabel, echo = FALSE,fig.cap="\\label{fig:changeLabel}Modify node labels."----
include_graphics("adjust.label_P00001.svg")

## ---- echo = TRUE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
SBGNview.obj.change.label.wrapping <-  SBGNview.obj +
        highlightNodes( node.set = c("tyrosine", "(+-)-epinephrine"), stroke.width = 4, stroke.color = "green",
                         show.glyph.id = TRUE,
                         label.x.shift = 10,label.y.shift = 20,label.color = "red",
                         label.font.size = 10,label.spliting.string = "any")
outputFile(SBGNview.obj.change.label.wrapping) = "change.label.wrapping"
print(SBGNview.obj.change.label.wrapping)

## ----changeWrapping, echo = FALSE,fig.cap="\\label{fig:changeWrapping}Change how labels are wrapped."----
include_graphics("change.label.wrapping_P00001.svg")

## ---- echo = TRUE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
test.show.glyph.id <- SBGNview.obj+
    highlightNodes( node.set = c("tyrosine", "(+-)-epinephrine"), stroke.width = 4,
                     stroke.color = "green", show.glyph.id = TRUE,
                     label.x.shift = 10,label.y.shift = 20,label.color = "red",
                     label.font.size = 10,
# When "label.spliting.string" is set to a string that is not in the label (including an empty string ""), the label will not be wrapped into multiple lines.                     
                     label.spliting.string = "")
outputFile(test.show.glyph.id) <- "test.show.glyph.id"
print(test.show.glyph.id)

## ----displayNodeIds, echo = FALSE,fig.cap="\\label{fig:displayNodeIds}Show node IDs of mapped nodes."----
include_graphics("test.show.glyph.id_P00001.svg")

## ---- echo = TRUE, results = 'hide', eval = TRUE, message = FALSE, warning = FALSE----
mapping <- changeIds(input.ids = c("tyrosine", "(+-)-epinephrine"),
           input.type = "CompoundName",
           output.type = "pathwayCommons",
           cpd.or.gene = "compound",
           limit.to.pathways = input.pathway.ids[1] )

## ---- echo = TRUE, eval = TRUE, message = FALSE, warning = FALSE---------
mapping

## ---- echo =TRUE---------------------------------------------------------
outputFile(SBGNview.obj) <- "highlight.by.node.id"

## ---- echo = TRUE, eval = TRUE, results = 'hide', message = FALSE, warning = FALSE----
SBGNview.obj+  highlightNodes(node.set = c("tyrosine", "(+-)-epinephrine"),
                       stroke.width = 4, stroke.color = "red") + 
    highlightPath(from.node =  "SmallMolecule_96737c854fd379b17cb3b7715570b733",
                   to.node =   "SmallMolecule_7753c3822ee83d806156d21648c931e6",
                   node.set.id.type = "pathwayCommons",
                      from.node.color = "green",
                      to.node.color = "blue",
                      shortest.paths.cols = c("purple"),
                      input.node.stroke.width = 6,
                      path.node.stroke.width = 3,
                      path.node.color = "purple",
                      path.stroke.width = 5,
                      tip.size = 10)

## ----highlightNodesById, echo = FALSE,fig.cap="\\label{fig:highlightNodesById}Highlight nodes and shortest path using node IDs."----
include_graphics("highlight.by.node.id_P00001.svg")

## ---- echo = TRUE, results = 'hide', message = FALSE, warning = FALSE----
result.one.sbgn <- SBGNview.obj$data[[1]]
names(result.one.sbgn)
glyphs <- result.one.sbgn$glyphs.list
arcs <- result.one.sbgn$arcs.list
str(glyphs[[1]])
str(arcs[[1]])

## ---- echo = TRUE, message = FALSE,results='hide', warning=FALSE---------
node.info <- sbgnNodes(input.sbgn = c("P00001","P00002"),
                       output.gene.id.type = "SYMBOL",
                       output.cpd.id.type = "chebi",
                       species = "hsa"
                       )

## ---- echo = TRUE--------------------------------------------------------
head(node.info[[1]])

## ---- echo = TRUE--------------------------------------------------------

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
data("mapped.ids")

## ---- echo = TRUE, results = 'hide',message = FALSE, warning = FALSE-----
mapping <- changeIds(
  input.ids = c("tyrosine", "(+-)-epinephrine"),
  input.type = "CompoundName",
  output.type = "pathwayCommons",
  cpd.or.gene = "compound",
  limit.to.pathways = "P00001"
)

## ---- echo = TRUE,message = FALSE, warning = FALSE-----------------------
head(mapping)

## ---- echo = TRUE, results = 'hide',message = FALSE, warning = FALSE-----
mapping <- changeIds(
  input.ids = c("tyrosine", "(+-)-epinephrine"),
  input.type = "CompoundName",
  output.type = "chebi",
  cpd.or.gene = "compound"
)

## ---- echo = TRUE,message = FALSE, warning = FALSE-----------------------
head(mapping)

## ---- echo = TRUE, results = 'hide',message = FALSE, warning = FALSE-----
mapping <- changeIds(
  input.ids = c("tyrosine"),
  input.type = "CompoundName",
  output.type = "chebi",
  cpd.or.gene = "compound",
  SBGNview.data.folder = "./SBGNview.tmp.data"
)

## ---- echo = TRUE,message = FALSE, warning = FALSE-----------------------
head(mapping)

## ---- echo = TRUE, results='hide', message = FALSE, warning = FALSE------
mol.list <- getMolList(
                        database = "metacrop"
                        ,mol.list.ID.type = "ENZYME"
                        ,org = "ath"
                        ,output.pathway.name = FALSE
                        ,truncate.name.length = 50
)

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
mol.list[[1]]

## ---- echo = TRUE, results='hide', message = FALSE, warning = FALSE------
mol.list <- getMolList(
                 database = "pathwayCommons",
                 mol.list.ID.type = "ENTREZID",
                 org = "hsa"
)

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
mol.list[[1]]

## ---- echo = TRUE, results='hide', message = FALSE, warning = FALSE------
mol.list <- getMolList(
                 database = "pathwayCommons",
                 mol.list.ID.type = "ENTREZID",
                 org = "mmu"
)

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
mol.list[[2]]

## ---- echo = TRUE, results='hide', message = FALSE, warning = FALSE------
mol.list <- getMolList(
                 database = "MetaCyc",
                 mol.list.ID.type = "KO",
                 org = "eco"
)

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
mol.list[[2]]

## ---- echo = TRUE, results='hide', message = FALSE, warning = FALSE------
mol.list <- getMolList(
                 database = "pathwayCommons",
                 mol.list.ID.type = "chebi",
                 cpd.or.gene = "compound"
)

## ---- echo = TRUE, message = FALSE, warning = FALSE----------------------
mol.list[[2]]

## ---- echo = TRUE,results = 'hide', message = FALSE, warning = FALSE-----
is.reactome <- pathways.info[,"sub.database"]== "reactome"
reactome.ids <- pathways.info[is.reactome ,"pathway.id"]
SBGNview.obj <- SBGNview(
              gene.data = gse16873, 
              gene.id.type = "entrez",
              input.sbgn =  reactome.ids[1:2],
              output.file = "demo.reactome", 
              output.formats =  c("svg")
            )
SBGNview.obj

## ---- echo = TRUE,results = 'hide', message = FALSE, warning = FALSE-----
is.metacrop <- pathways.info[,"sub.database"]== "MetaCrop"
metacrop.ids <- pathways.info[is.metacrop ,"pathway.id"]
SBGNview.obj <- SBGNview(
              gene.data = c(), 
              input.sbgn =  metacrop.ids[1:2],
              output.file = "demo.metacrop", 
              output.formats =  c("svg")
            )
SBGNview.obj

## ---- echo = TRUE,results = 'hide', message = FALSE, warning = FALSE-----
downloadSbgnFile(c("AF_Reference_Card.sbgn"
                 ,"PD_Reference_Card.sbgn"
                 ,"ER_Reference_Card.sbgn"
                 ))

SBGNview.obj <- SBGNview(
             gene.data = c()
             ,input.sbgn = c("AF_Reference_Card.sbgn"
                             ,"PD_Reference_Card.sbgn"
                             ,"ER_Reference_Card.sbgn"
                             )
             ,sbgn.gene.id.type ="glyph"
             
             ,output.file = "./test.refcards" 
             ,output.formats = c("pdf")
             ,font.size = 1
             ,logic.node.font.scale = 10
             ,status.node.font.scale = 10
           )
SBGNview.obj



## ---- eval = FALSE, echo = TRUE,results = 'hide', message = FALSE, warning = FALSE----
#  # Not run!
#  SBGNview(
#   key.pos = "none"
#  )

## ------------------------------------------------------------------------
sessionInfo()

