## ----, echo = FALSE, results = "asis"-------------------------------------------------------------------------------------------
options(width = 130)
BiocStyle::markdown()

## -------------------------------------------------------------------------------------------------------------------------------
library(ClassifyR)
library(curatedOvarianData)
data(GSE26712_eset)
GSE26712_eset <- GSE26712_eset[1:1000, ]

## ----, results = "hold", tidy = FALSE-------------------------------------------------------------------------------------------
curatedClinical <- pData(GSE26712_eset)
ovarPoor <- curatedClinical[, "vital_status"] == "deceased" & curatedClinical[, "days_to_death"] < 365 * 1
ovarGood <- curatedClinical[, "vital_status"] == "living" & curatedClinical[, "days_to_death"] > 365 * 5
sum(ovarPoor, na.rm = TRUE)
sum(ovarGood, na.rm = TRUE)

## -------------------------------------------------------------------------------------------------------------------------------
ovarExpression <- exprs(GSE26712_eset)[, c(which(ovarPoor), which(ovarGood))]
ovarGroups <- factor(rep(c("Poor", "Good"), c(length(which(ovarPoor)), length(which(ovarGood)))),
                     levels = c("Poor", "Good"))

## ----, fig.width = 18, fig.height = 10, tidy = FALSE----------------------------------------------------------------------------
library(ggplot2)
plotData <- data.frame(expression = as.numeric(ovarExpression),
                       sample = factor(rep(1:ncol(ovarExpression), each = nrow(ovarExpression))))

ggplot(plotData, aes(x = sample, y = expression)) + geom_boxplot() +
       scale_y_continuous(limits = c(0, 15)) + xlab("Sample") + ylab("Expression Value") +
       ggtitle("Expression for All Arrays") 

## ----, tidy = FALSE-------------------------------------------------------------------------------------------------------------
groupsTable <- data.frame(class = ovarGroups)
rownames(groupsTable) <- colnames(ovarExpression)
ovarSet <- ExpressionSet(ovarExpression, AnnotatedDataFrame(groupsTable))
featureNames(ovarSet) <- rownames(ovarExpression)
dim(ovarSet)

## ----, tidy = FALSE-------------------------------------------------------------------------------------------------------------
library(sparsediscrim)
DEresults <- runTests(ovarSet, "Ovarian Cancer", "Differential Expression", validation = "bootstrap", resamples = 5, folds = 3,
                      params = list(SelectionParams(limmaSelection, resubstituteParams = ResubstituteParams(nFeatures = c(25, 50, 75, seq(100, 1000, 100)), performanceType = "balanced", better = "lower")),
                                    TrainParams(dlda, TRUE, doesTests = FALSE),
                                    PredictParams(predict, TRUE, FALSE, getClasses = function(result) result[["class"]])),
                      parallelParams = bpparam(), verbose = 1)
DEresults

## ----, fig.height = 12, fig.width = 12, results = "hold", message = FALSE-------------------------------------------------------
plotFeatureClasses(ovarSet, features(DEresults)[[1]][[2]][1:5])

## -------------------------------------------------------------------------------------------------------------------------------
DEresults <- calcPerformance(DEresults, "balanced")
DEresults
performance(DEresults)

## -------------------------------------------------------------------------------------------------------------------------------
DEresults <- calcPerformance(DEresults, "mat")
DEresults
performance(DEresults)

## ----, tidy = FALSE-------------------------------------------------------------------------------------------------------------
DVresults <- runTests(ovarSet, "Ovarian Cancer", "Differential Deviation",
                      validation = "bootstrap", resamples = 2, folds = 4,
                      params = list(SelectionParams(leveneSelection, resubstituteParams = ResubstituteParams(nFeatures = c(25, 50, 75, seq(100, 1000, 100)), performanceType = "balanced", better = "lower")),
                                    TransformParams(subtractFromLocation, location = "median"),
                                    TrainParams(fisherDiscriminant, FALSE, doesTests = TRUE),
                                    PredictParams(predictor = function(){}, FALSE, FALSE, getClasses = function(result) result)),
                      verbose = 1)
DVresults

## ----, fig.height = 12, fig.width = 12, results = "hold", message = FALSE-------------------------------------------------------
plotFeatureClasses(ovarSet, features(DVresults)[[1]][[2]][1:5])

## -------------------------------------------------------------------------------------------------------------------------------
DVresults <- calcPerformance(DVresults, "balanced")
DVresults
performance(DVresults)

## ----, tidy = FALSE-------------------------------------------------------------------------------------------------------------
DDresults <- runTests(ovarSet, "Ovarian Cancer", "Differential Distribution",
                      validation = "bootstrap", resamples = 2, folds = 2,
                      params = list(SelectionParams(KullbackLeiblerSelection, resubstituteParams = ResubstituteParams(nFeatures = c(25, 50, 75, seq(100, 1000, 100)), performanceType = "balanced", better = "lower")),
                                    TrainParams(naiveBayesKernel, FALSE, doesTests = TRUE),
                                    PredictParams(predictor = function(){}, FALSE, FALSE, getClasses = function(result) result, weighted = "weighted")),
                                verbose = 1)
DDresults

## ----, fig.height = 12, fig.width = 12, results = "hold", message = FALSE-------------------------------------------------------
plotFeatureClasses(ovarSet, features(DDresults)[[1]][[1]][1:5])

## -------------------------------------------------------------------------------------------------------------------------------
DDresults <- calcPerformance(DDresults, "balanced")
DDresults
performance(DDresults)

## ----, fig.width = 10, fig.height = 7-------------------------------------------------------------------------------------------
library(grid)
resultsList <- list(Expression = DEresults, Variability = DVresults)
errorPlot <- errorMap(resultsList)

