## ----include=FALSE-------------------------------------------------------
library(TxDb.Hsapiens.UCSC.hg19.knownGene)
library(clusterProfiler)
library(ChIPseeker)
library(org.Hs.eg.db)
library(GenomicFeatures)
library(GenomicRanges)
library(knitr)
opts_chunk$set(tidy_source=TRUE, 
               tidy.opts=list(blank=FALSE, width.cutoff=50),
               out.truncate=80,
               out.lines=6,
               cache=TRUE,
               dev='pdf',
               include=TRUE,
               fig.width=6,
               fig.height=6,
               resolution=100)

## ----options,results='hide',echo=FALSE----------------------------------------
options(digits=3, width=80, prompt=" ", continue=" ")

## ----loadPkgs-----------------------------------------------------------------
## loading packages
require(ChIPseeker)
require(TxDb.Hsapiens.UCSC.hg19.knownGene)
txdb <- TxDb.Hsapiens.UCSC.hg19.knownGene
require(clusterProfiler)

## ----samplefiles--------------------------------------------------------------
files <- getSampleFiles()
print(files)
peak <- readPeakFile(files[[4]])
peak

## ----chromosomeCoverage, fig.height=8, fig.width=7, out.width="0.8\\textwidth"----
covplot(peak, weightCol="V5")

## ----covplot, fig.height=3, fig.width=7, out.width="0.6\\textwidth"-----------
covplot(peak, weightCol="V5", chrs=c("chr17", "chr18"), xlim=c(4.5e7, 5e7))

## ----promoter-----------------------------------------------------------------
## promoter <- getPromoters(TxDb=txdb, upstream=3000, downstream=3000)
## tagMatrix <- getTagMatrix(peak, windows=promoter)
##
## to speed up the compilation of this vignettes, we use a precalculated tagMatrix
data("tagMatrixList")
tagMatrix <- tagMatrixList[[4]]

## ----heatmap, eval=TRUE, fig.cap="Heatmap of ChIP peaks binding to TSS regions", fig.align="center", fig.height=12, fig.width=4, out.width="0.2\\textwidth", fig.pos="h"----
tagHeatmap(tagMatrix, xlim=c(-3000, 3000), color="red")

## ----heatmap2, eval=FALSE-----------------------------------------------------
#  peakHeatmap(files[[4]], TxDb=txdb, upstream=3000, downstream=3000, color="red")

## ----avgprof, eval=TRUE, fig.cap="Average Profile of ChIP peaks binding to TSS region", fig.align="center", fig.height=7, fig.width=7, out.width="0.8\\textwidth", fig.pos="h"----
plotAvgProf(tagMatrix, xlim=c(-3000, 3000), xlab="Genomic Region (5'->3')", ylab = "Read Count Frequency")

## ----avgprof2, eval=FALSE-----------------------------------------------------
#  plotAvgProf2(files[[4]], TxDb=txdb, upstream=3000, downstream=3000, xlab="Genomic Region (5'->3')", ylab = "Read Count Frequency")

## ----peak annotation----------------------------------------------------------
peakAnno <- annotatePeak(files[[4]], tssRegion=c(-3000, 3000), 
                         TxDb=txdb, annoDb="org.Hs.eg.db")
peakAnno

## ----pieplot, fig.cap="Genomic Annotation by pieplot", fig.align="center", fig.height=6, fig.width=8, out.width="0.8\\textwidth", fig.pos="h"----
plotAnnoPie(peakAnno)

## ----barplot_anno, fig.cap="Genomic Annotation by barplot", fig.align="center", fig.height=4, fig.width=10, out.width="0.8\\textwidth", fig.pos="h"----
plotAnnoBar(peakAnno)

## ----vennpie, fig.cap="Genomic Annotation by vennpie", fig.align="center", fig.height=8, fig.width=10, out.width="0.8\\textwidth", fig.pos="h"----
vennpie(peakAnno)

## ----barplot_tss, fig.cap="Distribution of Binding Sites", fig.align="center", fig.height=2, fig.width=6, out.width="0.8\\textwidth", fig.pos="h"----
plotDistToTSS(peakAnno, title="Distribution of transcription factor-binding loci \n relative to TSS")

## ----enrichment---------------------------------------------------------------
require(clusterProfiler)
bp <- enrichGO(as.data.frame(peakAnno)$geneId, ont="BP", readable=TRUE)
head(summary(bp))

## ----avgprof3, eval=TRUE, fig.cap="Average Profiles of ChIP peaks among different experiments", fig.align="center", fig.height=7, fig.width=7, out.width="0.8\\textwidth", fig.pos="h"----
## promoter <- getPromoters(TxDb=txdb, upstream=3000, downstream=3000)
## tagMatrixList <- lapply(files, getTagMatrix, windows=promoter)
## 
## to speed up the compilation of this vigenettes, we load a precaculated tagMatrixList
data("tagMatrixList")
plotAvgProf(tagMatrixList, xlim=c(-3000, 3000))

## ----heatmap3, eval=TRUE, fig.cap="Heatmap of ChIP peaks among different experiments", fig.align="center", fig.height=9, fig.width=7, out.width="0.8\\textwidth", fig.pos="h"----
tagHeatmap(tagMatrixList, xlim=c(-3000, 3000), color=NULL)

## ----List_of_peakAnno---------------------------------------------------------
peakAnnoList <- lapply(files, annotatePeak, TxDb=txdb, tssRegion=c(-3000, 3000), verbose=FALSE)

## ----barplot_anno_list, fig.cap="Genomic Annotation among different ChIPseq data", fig.align="center", fig.height=4, fig.width=6, out.width="0.6\\textwidth", fig.pos="h"----
plotAnnoBar(peakAnnoList)

## ----barplot_tss_list, fig.cap="Distribution of Binding Sites among different ChIPseq data", fig.align="center", fig.height=5, fig.width=8, out.width="0.6\\textwidth", fig.pos="h"----
plotDistToTSS(peakAnnoList)

## ----compareCluster, eval=FALSE-----------------------------------------------
#  genes = lapply(peakAnnoList, function(i) as.data.frame(i)$geneId)
#  names(genes) = sub("_", "\n", names(genes))
#  compGO <- compareCluster(geneCluster=genes, fun="enrichGO", ont="MF", organism="human", pvalueCutoff=0.05, pAdjustMethod="BH")
#  plot(compGO, showCategory=20, title="Molecular Function Enrichment")

## ----vennplot, fig.cap="Overlap of annotated genes", fig.align="center", fig.height=7, fig.width=7, out.width="0.7\\textwidth", fig.pos="h"----
genes= lapply(peakAnnoList, function(i) as.data.frame(i)$geneId)
vennplot(genes)

## ----shuffle------------------------------------------------------------------
p <- GRanges(seqnames=c("chr1", "chr3"), ranges=IRanges(start=c(1, 100), end=c(50, 130)))
shuffle(p, TxDb=txdb)

## ----peakEnrichment-----------------------------------------------------------
enrichPeakOverlap(queryPeak=files[[5]], targetPeak=unlist(files[1:4]), TxDb=txdb, pAdjustMethod="BH", nShuffle=50, chainFile=NULL)

## ----getGEOspecies------------------------------------------------------------
getGEOspecies()

## ----getGEOgenomeVersion------------------------------------------------------
getGEOgenomeVersion()

## ----getGEOInfo---------------------------------------------------------------
hg19 <- getGEOInfo(genome="hg19", simplify=TRUE)
head(hg19)

## ----downloadGEO, eval=FALSE--------------------------------------------------
#  downloadGEObedFiles(genome="hg19", destDir="hg19")

## ----downloadGSM, eval=FALSE--------------------------------------------------
#  gsm <- hg19$gsm[sample(nrow(hg19), 10)]
#  downloadGSMbedFiles(gsm, destDir="hg19")

## ----sessInfo, results='asis', echo=FALSE-------------------------------------
toLatex(sessionInfo())

