
## ----style, eval=TRUE, echo=FALSE, results="asis"------------------------
BiocStyle::latex()


## ----setup, include=FALSE, cache=FALSE-----------------------------------
library("knitr")
opts_chunk$set(tidy.opts=list(width.cutoff=45, tidy=FALSE), fig.align='center', comment=NA, prompt=TRUE)


## ----libraries, echo=FALSE-----------------------------------------------
suppressPackageStartupMessages(library("UniProt.ws"))
suppressPackageStartupMessages(library("BRAIN"))


## ------------------------------------------------------------------------
library("cleaver")


## ----eval=FALSE----------------------------------------------------------
## help("cleave")


## ------------------------------------------------------------------------
cleave("LAAGKVEDSD", enzym="trypsin")


## ------------------------------------------------------------------------
## miss one cleavage position
cleave("LAAGKVEDSD", enzym="trypsin", missedCleavages=1)

## miss zero or one cleavage positions
cleave("LAAGKVEDSD", enzym="trypsin", missedCleavages=0:1)


## ------------------------------------------------------------------------
## create AAStringSet object
gaju <- AAStringSet("LAAGKVEDSD")

## cleave it
cleave(gaju, enzym="trypsin")


## ------------------------------------------------------------------------
## load UniProt.ws library
library("UniProt.ws")

## download sequences of Insulin/Somatostatin
s <- select(UniProt.ws, keys=c("P01308", "P61278"), columns=c("SEQUENCE"))

## fetch only sequences
sequences <- setNames(s$SEQUENCE, s$UNIPROTKB)

## remove whitespaces
sequences <- gsub(pattern="[[:space:]]", replacement="", x=sequences)


## ------------------------------------------------------------------------
cleave(sequences, enzym="pepsin")


## ------------------------------------------------------------------------
## load BRAIN library
library("BRAIN")

## cleave insulin
cleavedInsulin <- cleave(sequences[1], enzym="trypsin")[[1]]

## create empty plot area
plot(NA, xlim=c(150, 4300), ylim=c(0, 1),
     xlab="mass", ylab="relative intensity",
     main="tryptic digested insulin - isotopic distribution")

## loop through peptides
for (i in seq(along=cleavedInsulin)) {
  ## count C, H, N, O, S atoms in current peptide
  atoms <- BRAIN::getAtomsFromSeq(cleavedInsulin[[i]])
  ## calculate isotopic distribution
  d <- useBRAIN(atoms)
  ## draw peaks
  lines(d$masses, d$isoDistr, type="h", col=2)
}


## ----sessioninfo, echo=FALSE, results="asis"-----------------------------
toLatex(sessionInfo())


