# gaggleTest.R:  unit tests for functions in gaggle.R

# all of these tests require a 'reflector goose' -- one which sends the
# received broadcast straight back to this R goose -- to be running
# you can start one up here:
# 	http://gaggle.systemsbiology.net/2007-04/echo.jnlp

#---------------------------------------------------------------------------------
.First.lib <- function (libname, pkgname)
{
  cat ('running gaggleTest package 0.99-22\n')
  require (gaggle)
  require (RUnit)
  
} # first lib
#-------------------------------------------------------------------------------------
test.gaggle <- function ()
{
  test.roundTripBroadcastNameList ()
  test.roundTripBroadcastMatrix ()

  test.graphNELtoGaggleNetworkMinimalGraph ()
  test.graphNELtoGaggleNetwork_noOrphans ()
  test.graphNELtoGaggleNetwork_withOrphans ()
  test.gaggleNetworkToGraphNEL_noOrphans ()
  test.gaggleNetworkToGraphNEL_withOrphans ()
  test.graphToGaggleAndBack ()

  test.roundTripBroadcastMinimalGraph ()
  test.roundTripBroadcastSmallRandomEGraph ()
  test.roundTripBroadcastLargeRandomEGraph ()
  test.roundTripBroadcastNetwork ()
  test.roundTripBroadcastEnvironment ()
  test.roundTripBroadcastCluster ()

} # test.gaggle
#-------------------------------------------------------------------------------------
# ensure that if we broadcast a matrix to a 'reflector goose' -- one which sends the
# received matrix straight back to this R goose -- that we receive the identical matrix
#
test.roundTripBroadcastNameList <- function ()
{
  print ('test.roundTripBroadcastNameList')
  setTargetGoose ('all')
  names = c ('larry', 'moe', 'curly', 'harpo', 'zeppo', 'groucho')
  broadcast (names)
  reflectedNames = getNameList ()
  #print (paste ('reflected names:', reflectedNames))
  checkEquals (names, reflectedNames)
}
#-------------------------------------------------------------------------------------
# ensure that if we broadcast a matrix to a 'reflector goose' -- one which sends the
# received matrix straight back to this R goose -- that we receive the identical matrix
#
test.roundTripBroadcastCluster <- function ()
{
  print ('test.roundTripBroadcastCluster')
  setTargetGoose ('all')
  cluster = list ()
  cluster$name = "test cluster"
  cluster$rowNames = c ("YFL036W","YLR212C","YML085C","YML123C")
  cluster$columnNames = c ("YFL036W","YLR212C","YML085C","YML123C")
  broadcast (cluster, cluster$name)
  reflectedCluster = getCluster ()
  checkEquals (cluster, reflectedCluster)

}
#-------------------------------------------------------------------------------------
# ensure that if we broadcast a hashmap (an R environment) to a 'reflector goose' --
# one which sends the received matrix straight back to this R goose -- that we receive the 
# environment 
#
test.roundTripBroadcastEnvironment <- function ()
{
  print ('test.roundTripBroadcastEnvironment')
  hash = new.env ()
  hash$title = 'test hashmap'
  keys = c ("YFL036W", "YFL037W", "YLR212C", "YLR213C", "YML085C", "YML086C", "YML123C", "YML124C")
  values = c (0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8)
  for (i in seq (length (keys)))
    assign (keys [i], values [i], envir=hash)

  setTargetGoose ('all')
  broadcast (hash)
  reflectedHash = getTuple ()
  checkEquals (ls (hash), ls (reflectedHash))
  for (key in keys)
    checkEquals (get (key, hash), get (key, reflectedHash))

}
#-------------------------------------------------------------------------------------
# ensure that if we broadcast a matrix to an 'echo goose' -- one which broadcasts the
# received list straight back to all geese, and therefore this R goose -- that we
# receive the identical list
#
test.roundTripBroadcastMatrix <- function ()
{
  print ('test.roundTripBroadcastMatrix')
  setTargetGoose ('all')
  m = .createMatrix ()
  broadcast (m)
  #Sys.sleep (1)
  reflectedMatrix = getMatrix ()
  checkEquals (rownames (m), rownames (reflectedMatrix))
  checkEquals (colnames (m), colnames (reflectedMatrix))
  checkEquals (as.numeric (m), as.numeric (reflectedMatrix))
}
#-------------------------------------------------------------------------------------
.createNetworkData <- function (variety)
# this function is called with any of four arguments, which respectively 
# contribute to the three parts of a gaggle network: edges, edgesWithTwoOrphans,
# node attributes, edge attributes.  'edgesWithTwoOrphans' seems contradictory, but
# will make sense if you think of orphans nodes as degenerate edges.
{
  if (variety == "edges")
    return (c ("VNG0720G::VNG0723G::GeneCluster",
               "VNG0723G::VNG1038C::GeneNeighbor"))

  else if (variety == "edgesWithTwoOrphans")
    return (c ("VNG0720G::VNG0723G::GeneCluster",
               "VNG0723G::VNG1038C::GeneNeighbor",
               "VNG0212H", "VNG0214C"))

  else if (variety == "node attributes")
    return (c ("VNG1038C::lambda::3.812",
               "VNG0720G::lambda::15.406",
               "VNG0723G::lambda::4.395",
               "VNG1038C::commonName::VNG1038C",
               "VNG0720G::commonName::dip2",
               "VNG0723G::commonName::pepq1",
               "VNG1038C::log10 ratio::0.086",
               "VNG0720G::log10 ratio::0.295",
               "VNG0723G::log10 ratio::0.171",
               "VNG1038C::canonicalName::VNG1038C",
               "VNG0720G::canonicalName::VNG0720G",
               "VNG0723G::canonicalName::VNG0723G",
               "VNG1038C::species::Halobacterium sp.",
               "VNG0720G::species::Halobacterium sp.",
               "VNG0723G::species::Halobacterium sp."))

  else if (variety == "edge attributes")
    return (c ("VNG0720G::VNG0723G::GeneCluster::confidence::0.655",
               "VNG0720G::VNG0723G::GeneCluster::interaction::GeneCluster",
               "VNG0720G::VNG0723G::GeneCluster::canonicalName::VNG0720G (GeneCluster) VNG0723G",
               "VNG0720G::VNG0723G::GeneCluster::pValue::0.34",
               "VNG0723G::VNG1038C::GeneNeighbor::confidence::0.497",
               "VNG0723G::VNG1038C::GeneNeighbor::interaction::GeneNeighbor",
               "VNG0723G::VNG1038C::GeneNeighbor::canonicalName::VNG0723G (GeneNeighbor) VNG1038C",
               "VNG0723G::VNG1038C::GeneNeighbor::pValue::3.4e-17"))
}
#-------------------------------------------------------------------------------------
test.gaggleNetworkToGraphNEL_withOrphans <- function ()
# this is a simple 5-node, 2-edge network, with lots of node & edge attributes, originally
# obtained by selecting and broadcasting from the cytoscape network found at
# http://gaggle.systemsbiology.net/projects/rValidation/2006-03/networks/prolinks-halo/cy.jnlp
# this is a 'gaggle network' in the sense that, when the RShellGoose receives a nomal
# gaggle (Java) network object, it translates it into three lists of strings, each with their
# own simple format.  that format is replicated here, in the three calls to .createNetworkData ():
#
# note:  this test is written so that, in addition to being a test, it also returns the
# graphNEL object whose conversion is tested.  this is a tad duplicative, but works out
# nicely, providing a well-known graphNEL object to other tests which need one.
{
  print ('test.gaggleNetworkToGraphNEL_withOrphans')

  edges = .createNetworkData ("edgesWithTwoOrphans")
  nodeAttributes = .createNetworkData ("node attributes")
  edgeAttributes = .createNetworkData ("edge attributes")

  g  = gaggle:::.gaggleNetworkToGraphNEL (edges, nodeAttributes, edgeAttributes)
   
    #--------------------------------------------------------------
    # check the basic structure of the graph: 3 nodes, 2 edges
    # note the redundant storage
    #--------------------------------------------------------------

  checkEquals (nodes (g), c ("VNG0720G", "VNG0723G", "VNG1038C", "VNG0212H", "VNG0214C"))
  checkEquals (edges (g)[["VNG0720G"]], c ("VNG0723G"))
  checkEquals (edges (g)[["VNG0723G"]], c ("VNG0720G", "VNG1038C"))
  checkEquals (edges (g)[["VNG1038C"]], c ("VNG0723G"))


    #--------------------------------------------------------------
    # do some spot checks on node attributes
    #--------------------------------------------------------------

  #print (paste ('175, noa names: ', names (nodeDataDefaults (g))))
  checkEquals (names (nodeDataDefaults (g)), 
                   c ("lambda", "commonName", "log10 ratio", "canonicalName", "species"))

  checkEquals (nodeData (g, "VNG1038C", "log10 ratio")[[1]], 0.086)
  checkEquals (nodeData (g, "VNG1038C", "lambda")[[1]], 3.812)
  checkEquals (nodeData (g, "VNG0720G", "lambda")[[1]], 15.406)
  checkEquals (nodeData (g, "VNG0720G", "log10 ratio")[[1]], 0.295)

  checkEquals (nodeData (g, "VNG1038C", "species")[[1]], "Halobacterium sp.")
  checkEquals (nodeData (g, "VNG1038C", "species")[[1]], "Halobacterium sp.")
  checkEquals (nodeData (g, "VNG0720G", "species")[[1]], "Halobacterium sp.")
  checkEquals (nodeData (g, "VNG0720G", "species")[[1]], "Halobacterium sp.")

    # now check the two orphan nodes.  since -- in this test -- no node
    # attributes are assigned to them, they ought to have the default node
    # attribute values

  checkEquals (nodeData (g, "VNG0212H", "species")[[1]], "")
  checkEquals (nodeData (g, "VNG0214C", "species")[[1]], "")

  checkEquals (nodeData (g, "VNG0212H", "log10 ratio")[[1]], "")
  checkEquals (nodeData (g, "VNG0214C", "log10 ratio")[[1]], "")

  checkEquals (nodeData (g, "VNG0212H", "lambda")[[1]], "")
  checkEquals (nodeData (g, "VNG0214C", "lambda")[[1]], "")

    #--------------------------------------------------------------
    # do some spot checks on attributes of the two edges
    #--------------------------------------------------------------

  checkEquals (names (edgeDataDefaults (g)), 
                   c ("edgeType", "confidence", "interaction", "canonicalName", "pValue"))

  checkEquals (edgeData (g, "VNG0720G", "VNG0723G", "confidence")[[1]],  0.655)
  checkEquals (edgeData (g, "VNG0720G", "VNG0723G", "pValue")[[1]],      0.34)
  checkEquals (edgeData (g, "VNG0720G", "VNG0723G", "interaction")[[1]], "GeneCluster")

  checkEquals (edgeData (g, "VNG0723G", "VNG1038C", "confidence")[[1]],  0.497)
  checkEquals (edgeData (g, "VNG0723G", "VNG1038C", "pValue")[[1]],      3.4e-17)
  print (checkEquals (edgeData (g, "VNG0723G", "VNG1038C", "interaction")[[1]], "GeneNeighbor"))

  invisible (g)

}
#-------------------------------------------------------------------------------------
test.gaggleNetworkToGraphNEL_noOrphans <- function ()
# this is a simple 3-node, 2-edge network, with lots of node & edge attributes, originally
# obtained by selecting and broadcasting from the cytoscape network found at
# http://gaggle.systemsbiology.net/projects/rValidation/2006-03/networks/prolinks-halo/cy.jnlp
# this is a 'gaggle network' in the sense that, when the RShellGoose receives a normal
# gaggle (Java) network object, it translates it into three lists of strings, each with their
# own simple format.  that format is replicated here, in the three calls to .createNetworkData ():
#
# note:  this test is written so that, in addition to being a test, it also returns the
# graphNEL object whose conversion is tested.  this is a tad duplicative, but works out
# nicely, providing a well-known graphNEL object to other tests which need one.
{
  print ('test.gaggleNetworkToGraphNEL_noOrphans')

  edges = .createNetworkData ("edges")
  nodeAttributes = .createNetworkData ("node attributes")
  edgeAttributes = .createNetworkData ("edge attributes")

  g  = gaggle:::.gaggleNetworkToGraphNEL (edges, nodeAttributes, edgeAttributes)
   
    #--------------------------------------------------------------
    # check the basic structure of the graph: 3 nodes, 2 edges
    # note the redundant storage
    #--------------------------------------------------------------

  checkEquals (nodes (g), c ("VNG0720G", "VNG0723G", "VNG1038C"))

  checkEquals (edges (g)[["VNG0720G"]], c ("VNG0723G"))
  checkEquals (edges (g)[["VNG0723G"]], c ("VNG0720G", "VNG1038C"))
  checkEquals (edges (g)[["VNG1038C"]], c ("VNG0723G"))


    #--------------------------------------------------------------
    # do some spot checks on node attributes
    #--------------------------------------------------------------

  #print (paste ('256, noa names: ', names (nodeDataDefaults (g))))
  checkEquals (names (nodeDataDefaults (g)), 
                   c ("lambda", "commonName", "log10 ratio", "canonicalName", "species"))

  checkEquals (nodeData (g, "VNG1038C", "log10 ratio")[[1]], 0.086)
  checkEquals (nodeData (g, "VNG1038C", "lambda")[[1]], 3.812)
  checkEquals (nodeData (g, "VNG0720G", "lambda")[[1]], 15.406)
  checkEquals (nodeData (g, "VNG0720G", "log10 ratio")[[1]], 0.295)
  for (node in nodes (g)) 
    checkEquals (nodeData (g, node, "species")[[1]], "Halobacterium sp.")

    #--------------------------------------------------------------
    # do some spot checks on attributes of the two edges
    #--------------------------------------------------------------

  checkEquals (names (edgeDataDefaults (g)), 
                   c ("edgeType", "confidence", "interaction", "canonicalName", "pValue"))

  checkEquals (edgeData (g, "VNG0720G", "VNG0723G", "confidence")[[1]],  0.655)
  checkEquals (edgeData (g, "VNG0720G", "VNG0723G", "pValue")[[1]],      0.34)
  checkEquals (edgeData (g, "VNG0720G", "VNG0723G", "interaction")[[1]], "GeneCluster")

  checkEquals (edgeData (g, "VNG0723G", "VNG1038C", "confidence")[[1]],  0.497)
  checkEquals (edgeData (g, "VNG0723G", "VNG1038C", "pValue")[[1]],      3.4e-17)
  print (checkEquals (edgeData (g, "VNG0723G", "VNG1038C", "interaction")[[1]], "GeneNeighbor"))

  invisible (g)

} # test.gaggleNetworkToGraphNEL_withOrphans
#-------------------------------------------------------------------------------------
test.graphNELtoGaggleNetwork_noOrphans <- function ()
# get a simple 3-node, 2-edge, fully-connected  network, with lots of node & edge attributes, 
# and convert it to a 'gaggleNetwork' -- more properly, the list of strings representation
# used by RShellGoose in communication with gaggle.R; RShellGoose translates these
# string representations back and forth to real Gaggle Networks.
{
  print ('test.graphNELtoGaggleNetwork_noOrphans')

  g = test.gaggleNetworkToGraphNEL_noOrphans   ()  # a convenient place to get a simple graph, which
                                                   # mimics the gaggle network, deconstructed, which
                                                   # the RShellGoose provides to gaggle.R
  result = gaggle:::.graphNELtoGaggleNetwork (g)

  checkTrue (gaggle:::.listMember ("edges", names (result)))
  edgeStringList = result$edges
  expectedEdges = .createNetworkData ("edges")
  for (expected in expectedEdges)
    checkTrue (gaggle:::.listMember (expected, edgeStringList))

  checkTrue (gaggle:::.listMember ("noa", names (result)))
  noaStringList = result$noa
  expectedNodeAttributes = .createNetworkData ("node attributes")
  for (expected in expectedNodeAttributes) 
    checkTrue (gaggle:::.listMember (expected, noaStringList))

  checkTrue (gaggle:::.listMember ("eda", names (result)))
  edaStringList = result$eda
  expectedEdgeAttributes = .createNetworkData ("edge attributes")
  for (expected in expectedEdgeAttributes) {
    checkTrue (gaggle:::.listMember (expected, edaStringList))
    }

  return (TRUE)

} # test.graphNELtoGaggleNetwork_noOrphans
#-------------------------------------------------------------------------------------
test.graphNELtoGaggleNetwork_withOrphans <- function ()
# create a randomEGraph, with many more nodes than edges, guaranteeing some orphan
# (that is, unconnected) nodes.
# and convert it to a 'gaggleNetwork' -- more properly, the list of strings representation
# used by RShellGoose in communication with gaggle.R; RShellGoose translates these
# string representations back and forth to real Gaggle Networks.
{
  print ('test.graphNELtoGaggleNetwork_withOrphans')
  set.seed (123)  # guarantees we get the same 'random' graph each time
  g = randomEGraph (LETTERS [1:5], edges=2)
  result = gaggle:::.graphNELtoGaggleNetwork (g)

  checkTrue (gaggle:::.listMember ("edges", names (result)))
  edgeStringList = result$edges
  #print (edgeStringList)
  expectedEdges = c ("A", "B::E::edge", "B::C::edge", "D")
  for (expected in expectedEdges)
    checkTrue (gaggle:::.listMember (expected, edgeStringList))

     # node attributes do exist in a randomEGraph, though with value NA
  checkTrue (gaggle:::.listMember ("noa", names (result)))

  checkTrue (gaggle:::.listMember ("eda", names (result)))
  edaStringList = result$eda
    # two edges, each with a weight & edgeType attibute (for which the value 'edge' is the default
  expectedEdgeAttributes = c ("B::E::edge::weight::1", "B::C::edge::weight::1",
                              "B::E::edge::edgeType::edge", "B::C::edge::edgeType::edge")
  for (expected in expectedEdgeAttributes)
    checkTrue (gaggle:::.listMember (expected, edaStringList))

  return (TRUE)

} # test.graphNELtoGaggleNetwork_noOrphans
#-------------------------------------------------------------------------------------
test.randomEGraphToGaggleNetwork <- function ()
{
  print ('test.randomEGraphToGaggleNetwork')
  set.seed (123)                                   # make this deterministic
  g = randomEGraph (LETTERS [1:4], edges=3)

    # should have these edges:  A-C, A-D, C-A, C-D, D-A, D-C
    # since this is an undirected graph, this boils down to 3: A-C, A-D, C-D
  
  checkEquals (nodes (g),  c ("A", "B", "C", "D"))
  checkEquals (names (edges (g)), c ("A", "B", "C", "D"))
  checkEquals (edgeL (g)$A[[1]],  c (3, 4))
  checkEquals (edgeL (g)$B[[1]],  numeric (0))
  checkEquals (edgeL (g)$C[[1]], c (1, 4))
  checkEquals (edgeL (g)$D[[1]], c (1,3))
  
  result = gaggle:::.graphNELtoGaggleNetwork (g)
  checkEquals (sort (result$edges), c ("A::C::edge", "A::D::edge", "B", "C::D::edge"))
  checkEquals (sort (result$eda),
               c ("A::C::edge::edgeType::edge", "A::C::edge::weight::1",
                  "A::D::edge::edgeType::edge", "A::D::edge::weight::1",
                  "C::D::edge::edgeType::edge", "C::D::edge::weight::1")) 
               
   ## todo: keep on checking, then run all tests
  print (result)
  return (TRUE)

} # test.graphEGraphToGaggleNetwork
#-------------------------------------------------------------------------------------
test.graphToGaggleAndBack <- function ()
{
  print ('test.graphToGaggleAndBack')
  set.seed (123)                                   # make this deterministic

  g = randomEGraph (LETTERS [1:8], edges=6)
  nodeDataDefaults (g, attr="species") = "Homo sapiens"
  nodeData (g, attr="species", n="C") = "Cebus capucinus"
  edgeDataDefaults (g, attr="confidence") = 15.0
  edgeData (g, attr="confidence", from="A", to="F") = 99.99

  gg = gaggle:::.graphNELtoGaggleNetwork (g)
  ggg = gaggle:::.gaggleNetworkToGraphNEL (gg$edges, gg$noa, gg$eda)

  .checkGraphsEqual (g, ggg)

} # test.graphToGaggleAndBack
#-------------------------------------------------------------------------------------
test.roundTripBroadcastNetwork <- function ()
{
  print ('test.roundTripBroadcastNetwork')
  g = test.gaggleNetworkToGraphNEL_noOrphans ()  # a convenient place to get a simple graph, which
                                                 # mimics the gaggle network, deconstructed, which
                                                 # the RShellGoose provides to gaggle.R
  broadcast (g)
  reflectedGraph = getNetwork ()
  .checkGraphsEqual (g, reflectedGraph)

} # test.roundTripBroadcastNetwork
#-------------------------------------------------------------------------------------
test.roundTripBroadcastSmallRandomEGraph <- function ()
{
  print ('test.roundTripBroadcastSmallRandomEGraph')
  reg = randomEGraph (LETTERS [1:5], edges=1)

  broadcast (reg)
  reg2 = getNetwork ()
  regNodes = sort (nodes (reg))
  reg2Nodes = sort (nodes (reg2))

  checkEquals (regNodes, reg2Nodes)
  
  for (node in regNodes) {
    targetNodes1 = sort (edges (reg)[[node]])
    targetNodes2 = sort (edges (reg2)[[node]])
    checkEquals (targetNodes1, targetNodes2)
    }

     # should be no node attributes
  checkEquals (names (nodeDataDefaults (reg)), names (nodeDataDefaults (reg2)))
  

} # test.roundTripBroadcastSmallRandomEGraph
#-------------------------------------------------------------------------------------
test.roundTripBroadcastLargeRandomEGraph <- function ()
{
  print ('test.roundTripBroadcastLargeRandomEGraph')
  set.seed (123)
  reg = randomEGraph (LETTERS [1:20], edges=28)

  broadcast (reg)
  reg2 = getNetwork ()
  regNodes = sort (nodes (reg))
  reg2Nodes = sort (nodes (reg2))

  checkEquals (regNodes, reg2Nodes)
  
  for (node in regNodes) {
    targetNodes1 = sort (edges (reg)[[node]])
    targetNodes2 = sort (edges (reg2)[[node]])
    checkEquals (targetNodes1, targetNodes2)
    }

     # should be no node attributes
  checkEquals (names (nodeDataDefaults (reg)), names (nodeDataDefaults (reg2)))
  

} # test.roundTripBroadcastLargeRandomEGraph
#-------------------------------------------------------------------------------------
.createMatrix <- function ()
{
  data = c (0.00, 0.38, 0.76, 1.14, 1.52, 1.90, 2.28, 2.66, 0.09, 0.47, 0.85, 1.23, 1.61, 
            1.99, 2.37, 2.75, 0.18, 0.56, 0.94, 1.32, 1.70, 2.08, 2.46, 2.84, 0.27, 0.65, 
            1.03, 1.41, 1.79, 2.17,  2.55, 2.93)
  m = matrix (data)
  dim (m) = c (8,4)
  rownames (m) = c ("YFL036W","YFL037W","YLR212C","YLR213C","YML085C","YML086C",
                    "YML123C", "YML124C")
  colnames (m) = c ("T000", "T060", "T120", "T240")
  return (m)

}
#---------------------------------------------------------------------------------
.checkEdgeAttributesEqual <- function (g1, g2)
{
  g1EdgeAttributeNames = names (edgeDataDefaults (g1))
  g2EdgeAttributeNames = names (edgeDataDefaults (g2))

  g1EdgeNames = sort (names (edgeData (g1)))
  g2EdgeNames = sort (names (edgeData (g2)))

   # loop over the edge attributes, and inside that loop, loop over the edge names,
   # pulling out edge attribute values one at a time, and testing them
   # for equality. 
   # since .graphNELtoGaggleNetwork always adds an 'edgeType' attribute, but the
   # various R graph constructors do not, do not insist on its presence

  for (attributeName in g1EdgeAttributeNames) {
    if (attributeName == 'edgeType')
      next
    checkTrue (gaggle:::.listMember (attributeName, g2EdgeAttributeNames))
    for (edge in g1EdgeNames) {
      tokens = unlist (strsplit (edge, "\\|"))
      source = tokens [1]
      target = tokens [2]
      g1Value = edgeData (g1, attr=attributeName, from=source, to=target)
      g2Value = edgeData (g2, attr=attributeName, from=source, to=target)
      #print (paste (edge, ":", g1Value, g2Value))
      checkEquals (g1Value, g2Value)
      } # for edge
   } # for attributeName
   

  return (TRUE)

} # .checkEdgeAttributesEquals
#---------------------------------------------------------------------------------
.checkNodeAttributesEqual <- function (g1, g2)
{
  g1NodeAttributeNames = names (nodeDataDefaults (g1))
  g2NodeAttributeNames = names (nodeDataDefaults (g2))
  
  g1SortedAttributeNames = sort(g1NodeAttributeNames)
  g2SortedAttributeNames = sort(g2NodeAttributeNames)


  checkEquals (g1SortedAttributeNames, g2SortedAttributeNames)


  g1NodeNames = sort (names (nodeData (g1)))
  g2NodeNames = sort (names (nodeData (g2)))




   # loop over the node attributes, and inside that loop, loop over the node names,
   # pulling out node attribute values one at a time, and testing them
   # for equality. 

  for (attributeName in g1NodeAttributeNames) {
    checkTrue (gaggle:::.listMember (attributeName, g2NodeAttributeNames))
    for (node in g1NodeNames) {
      g1Value = nodeData (g1, attr=attributeName, n=node)
      g2Value = nodeData (g2, attr=attributeName, n=node)
      #print (paste (node, ":", g1Value, g2Value))
      checkEquals (g1Value, g2Value)
      } # for node
   } # for attributeName
   

  return (TRUE)

} # .checkNodeAttributesEqual
#---------------------------------------------------------------------------------
.checkGraphsEqual <- function (g1, g2)
{
  checkEquals (sort (nodes (g1)), sort (nodes (g2)))
  checkEquals (sort (names (edges (g1))), sort (names (edges (g2))))

  edgeSourceNodes = sort (names (edges (g1)))
  for (node in edgeSourceNodes) {
    g1Targets = sort (edges (g1)[[node]])
    g2Targets = sort (edges (g2)[[node]])
    checkEquals (g1Targets, g2Targets)
    }

  .checkEdgeAttributesEqual (g1, g2)
  .checkNodeAttributesEqual (g1, g2)

} # .checkGraphsEqual
#---------------------------------------------------------------------------------
.createMinimalGraph <- function ()
{
  g = new ("graphNEL")
  g = addNode ('A', g)
  g = addNode ('B', g)
  g = addEdge ('A', 'B', g)

} # createMinimalGraph
#---------------------------------------------------------------------------------
test.graphNELtoGaggleNetworkMinimalGraph <- function ()
# the minimal graph has no two nodes, one edge, no explicit edge attributes,
# no node attributes.  note, however, that '.graphNELtoGaggleNetwork'
# always creates a default edge attribute called 'edgeType' for which the
# default value is simply (and uninterstingly) 'edge' 
{
  print ('test.graphNELtoGaggleNetworkMinimalGraph')
  g = .createMinimalGraph ()
  gg = gaggle:::.graphNELtoGaggleNetwork (g)

  checkEquals (gg$edges,  "A::B::edge")
  checkTrue (is.na (gg$noa))
  checkEquals (gg$eda,  "A::B::edge::edgeType::edge")

  return (gg)
}
#---------------------------------------------------------------------------------
test.roundTripBroadcastMinimalGraph <- function ()
{
  print ('test.roundTripBroadcastMinimalGraph')
  g = .createMinimalGraph ()
  broadcast (g)
  reflectedGraph = getNetwork ()
  .checkGraphsEqual (g, reflectedGraph)

} # test.roundTripBroadcastGraphNoAttributes
#---------------------------------------------------------------------------------
test.map1 <- function (attributeName="testA", nodes="VNG1607G", values="hoopla", title="from R")
{
  .jcall (goose, "V", "createAndBroadcastHashMap", title, attributeName, nodes, values)
  invisible ()
}
#---------------------------------------------------------------------------------
