## ----warning=FALSE---------------------------------------------------------
# load required packages
suppressPackageStartupMessages({
    library(CATALYST)
    library(SummarizedExperiment)
})

## --------------------------------------------------------------------------
# load example data
data(PBMC_fs, PBMC_panel, PBMC_md)
PBMC_fs
head(PBMC_panel)
head(PBMC_md)

## ----eval=FALSE------------------------------------------------------------
#  # download exemplary set of FCS files
#  url <- "http://imlspenticton.uzh.ch/robinson_lab/cytofWorkflow"
#  fcs_zip <- "PBMC8_fcs_files.zip"
#  download.file(paste0(url, "/", fcs_zip), destfile = fcs_zip, mode = "wb")
#  unzip(fcs_zip)
#  
#  # read in FCS files as flowSet
#  library(flowCore)
#  fcs_files <- list.files(pattern = ".fcs$")
#  fs <- read.flowSet(fcs_files, transformation = FALSE, truncate_max_range = FALSE)

## --------------------------------------------------------------------------
# construct daFrame
(daF <- daFrame(PBMC_fs, PBMC_panel, PBMC_md))

## ----eval=FALSE------------------------------------------------------------
#  # alter panel column names
#  panel2 <- PBMC_panel
#  colnames(panel2) <- c("channel_name", "marker")
#  
#  # alter metadata column names & add 2nd condition
#  md2 <- PBMC_md
#  colnames(md2) <- c("file", "sampleID", "cond1", "patientID")
#  md2$cond2 <- rep(c("A", "B"), 4)
#  
#  # construct daFrame
#  daFrame(PBMC_fs, panel2, md2,
#      panel_cols = list(channel = "channel_name", antigen = "marker"),
#      md_cols = list(file = "file", id = "sampleID",
#          factors = c("cond1", "cond2", "patientID")))

## ----fig.width=6, fig.height=3.5-------------------------------------------
n_cells(daF)
plotCounts(daF, color_by = "condition")

## ----fig.width=5, fig.height=4.5-------------------------------------------
plotMDS(daF, color_by = "condition")

## ----fig.width=10, fig.height=6--------------------------------------------
plotExprHeatmap(daF, color_by = "condition")

## --------------------------------------------------------------------------
# specify markers to use for clustering
lineage_markers <- c("CD3", "CD45", "CD4", "CD20", 
    "CD33", "CD123", "CD14", "IgM", "HLA_DR", "CD7")
daF <- cluster(daF, cols_to_use = lineage_markers, 
    xdim = 10, ydim = 10, maxK = 20, verbose = FALSE, seed = 1)       

## ----fig.width=5, fig.height=3---------------------------------------------
# access & render delta area plot
metadata(daF)$delta_area

## --------------------------------------------------------------------------
data(merging_table)
head(merging_table)
daF <- mergeClusters(daF, table = merging_table, id = "merging1")
head(cluster_codes(daF))

## ----fig.width=8, fig.height=6---------------------------------------------
# median pS6 expression by sample as 2nd heatmap
plotClusterHeatmap(daF, hm2 = "pS6", k=12, m = 6)

## ----fig.width=10, fig.height=6--------------------------------------------
# population frequencies by sample as 2nd heatmap
plotClusterHeatmap(daF, hm2 = "abundances", 
    draw_freqs = TRUE, cluster_anno = FALSE)

## ----fig.width=6, fig.height=4---------------------------------------------
plotAbundances(daF, k = 12, by = "sample_id", group = "condition")
plotAbundances(daF, k = "merging1", by = "cluster_id", group = "condition")

## --------------------------------------------------------------------------
daF <- tSNE(daF, cols_to_use = lineage_markers, 
    n = 200, verbose = FALSE, seed = 42)

## ----fig.width=7, fig.height=3---------------------------------------------
# color by pS6 expression and split by condition
plotSNE(daF, color_by = "pS6", facet = "condition")

## ----fig.width=8, fig.height=4---------------------------------------------
# color by 8 metaclusters & split by sample ID
plotSNE(daF, color_by = 8, facet = "sample_id")

## ----message=FALSE, warning=FALSE, fig.show='hide'-------------------------
library(diffcyt)
# create design & constrast matrix
design <- createDesignMatrix(PBMC_md, cols_design=3)
contrast <- createContrast(c(0, 1))
# test for
# - differential abundance (DA) of clusters
# - differential states (DS) within clusters
res_DA <- diffcyt(daF, design = design, contrast = contrast, 
    analysis_type = "DA", method_DA = "diffcyt-DA-edgeR")
res_DS <- diffcyt(daF, design = design, contrast = contrast, 
    analysis_type = "DS", method_DS = "diffcyt-DS-limma", plot = FALSE)

## ----fig.width=10, fig.height=6--------------------------------------------
plotDiffHeatmap(daF, res_DA, th = 1e-3, top_n = 25)
plotDiffHeatmap(daF, res_DS, top_n = 10)

## --------------------------------------------------------------------------
sessionInfo()

