## ----use-libs, include = FALSE---------------------------------------------
library(mfa)
library(ggplot2)
library(dplyr)

knitr::opts_chunk$set(echo = TRUE, cache = TRUE, fig.width = 6, fig.height = 4,
                      warning = FALSE, message = FALSE)

## ----install-bioconductor, eval = FALSE------------------------------------
#  source("https://bioconductor.org/biocLite.R")
#  biocLite("mfa")
#  library(mfa)

## ----install-github, eval = FALSE------------------------------------------
#  install.packages("devtools") # If not already installed
#  devtools::install_github("kieranrcampbell/mfa")
#  library(mfa)

## ----synthetic-------------------------------------------------------------
synth <- create_synthetic(C = 100, G = 40)
print(str(synth))

## ----to-tidy---------------------------------------------------------------
df_synth <- as_data_frame(prcomp(synth$X)$x[,1:2]) %>% 
  mutate(pseudotime = synth$pst,
        branch = factor(synth$branch))

## ----pca-rep---------------------------------------------------------------
ggplot(df_synth, aes(x = PC1, y = PC2, color = pseudotime)) + geom_point()
ggplot(df_synth, aes(x = PC1, y = PC2, color = branch)) + geom_point()

## ----run-mfa---------------------------------------------------------------
m <- mfa(synth$X)
print(m)

## ----diagnostics-----------------------------------------------------------
plot_mfa_trace(m)
plot_mfa_autocorr(m)

## ----summary---------------------------------------------------------------
ms <- summary(m)
print(head(ms))

## ----compare-pst-----------------------------------------------------------
qplot(synth$pst, ms$pseudotime, color = factor(synth$branch)) +
  xlab('True pseudotime') + ylab('Inferred pseudotime') +
  scale_color_discrete(name = 'True\nbranch')

## ----pca-rep-with-branch---------------------------------------------------
mutate(df_synth, inferred_branch = ms[['branch']]) %>% 
  ggplot(aes(x = PC1, y = PC2, color = inferred_branch)) +
  geom_point() +
  scale_color_discrete(name = 'Inferred\nbranch')

## ----plot-chi--------------------------------------------------------------
plot_chi(m)

## ----posterior-mean-chi----------------------------------------------------
posterior_chi_df <- calculate_chi(m)
head(posterior_chi_df)

## ----str-mfa---------------------------------------------------------------
str(m, max.level = 1)

## ----str-traces------------------------------------------------------------
print(names(m$traces))

## ----tau-------------------------------------------------------------------
str(m$traces$tau_trace)

## ----print-k---------------------------------------------------------------
str(m$traces$k_trace)

## ----posterior-mean-of-k---------------------------------------------------
pmean_k <- apply(m$traces$k_trace, 3, colMeans)
str(pmean_k)

## ----sess-info-------------------------------------------------------------
sessionInfo()

