## ---- eval=FALSE---------------------------------------------------------
#  ## try http if https is not available
#  source("https://bioconductor.org/biocLite.R")
#  biocLite("EGAD")

## ---- eval=FALSE---------------------------------------------------------
#  browseVignettes("EGAD")

## ----library, results="hide"---------------------------------------------
library(EGAD)

## ----eval=FALSE----------------------------------------------------------
#  data <- get_phenocarta(species="human", type="all")

## ----eval=FALSE----------------------------------------------------------
#  data <- get_biogrid(species="9606")

## ----eval=FALSE----------------------------------------------------------
#  data <- get_biogrid(species="9606", version="3.4.131")

## ----eval=FALSE----------------------------------------------------------
#  exprs <- get_expression_matrix_from_GEO(gseid="GSE34308")

## ----eval=FALSE----------------------------------------------------------
#  exprs <- get_expression_data_gemma(gseid="GSE34308", filtered = "true")

## ----eval=TRUE-----------------------------------------------------------
data("example_annotations")
annotations <- make_annotations(example_annotations$gene2Annot, 
                                example_annotations$genes, 
                                example_annotations$annotationlist)
head(colnames(annotations))

## ----eval=TRUE-----------------------------------------------------------
data("example_binary_network")
binary_net <- build_binary_network(example_binary_network$genepairs, example_binary_network$genelist)
hist <- plot_distribution(node_degree(binary_net), b=10, 
                          xlim=c(0,14), ylim=c(0,2), xlab="Node degree")

## ----eval=TRUE-----------------------------------------------------------
extended_net <- extend_network(binary_net)
hist <- plot_distribution(node_degree(extended_net), b=10, xlab="Node degree", )

## ----eval=TRUE-----------------------------------------------------------
semantic_net <- build_semantic_similarity_network(annotations, example_binary_network$genelist)
hist <- plot_distribution(node_degree(semantic_net), b=20, xlab="Node degree", bars=TRUE)

## ----eval=FALSE----------------------------------------------------------
#  data("example_coexpression")
#  coexp_net <- build_coexp_expressionSet(exprsSet, genelist, method="spearman", flag="rank")
#  # or
#  coexp_net <- build_coexp_network(exprs, genelist, method="spearman", flag="rank")

## ----eval=TRUE-----------------------------------------------------------
data("example_neighbor_voting")
network <- example_neighbor_voting$network
gba_auc_nv <- neighbor_voting(annotations, network, nFold=3, output="AUROC")
# or 
gba_pr_nv <- neighbor_voting(annotations, network, nFold=3, output="PR")
head(gba_auc_nv)
head(gba_pr_nv)

## ----eval=TRUE-----------------------------------------------------------
multifunc_assessment <- calculate_multifunc(annotations)
auc_mf <- auc_multifunc(annotations, multifunc_assessment[,4])
hist <- plot_distribution(auc_mf, xlab="AUROC", med=FALSE, avg=FALSE)

## ----eval=TRUE-----------------------------------------------------------
X <- runif(1000)
Y <- runif(1000) + X*0.3
window <- 50
smoothedxy <- conv_smoother(X,Y, window, xlab="X",ylab="Y")

## ----eval=TRUE-----------------------------------------------------------
filt <- !is.na(gba_auc_nv[,1])
aucA <- gba_auc_nv[filt,1]
aucB <- gba_auc_nv[filt,3]
hist <- plot_distribution(aucA, xlab="AUROCs")
avgs <- plot_density_compare(aucA, aucB, xlab="AUROCs")
plot_value_compare(aucA, aucB)

## ----eval=TRUE, include=FALSE--------------------------------------------
scores <- 100:1 
labels <- sample(c(0,1), 100, replace=TRUE)
labels[1:13]<-1
labels[18:30]<-1

## ----eval=TRUE-----------------------------------------------------------
roc <- plot_roc(scores, labels)

## ----eval=FALSE----------------------------------------------------------
#  scores <- predictions(labels, network)
#  plot_roc_overlay(scores, labels)

## ----eval=TRUE-----------------------------------------------------------
roc <- get_roc(scores, labels)

## ----eval=TRUE-----------------------------------------------------------
auroc <- get_auc(roc[,1], roc[,2])
print(auroc)

## ----eval=TRUE-----------------------------------------------------------
auroc <- auroc_analytic(scores, labels)
print(auroc)

## ----eval=TRUE-----------------------------------------------------------
prc <- get_prc(scores, labels)
prc <- plot_prc(scores, labels)

## ----eval=TRUE-----------------------------------------------------------
auprc <- auprc(scores, labels)
print(auprc)

## ----eval=TRUE-----------------------------------------------------------
fbeta <- fmeasure(prc[,1], prc[,2], beta=1)

## ----eval=TRUE-----------------------------------------------------------
nd <- node_degree(network)
hist <- plot_distribution(nd, xlab="Node degree", med = FALSE)

## ----eval=TRUE-----------------------------------------------------------
assort <- assortativity(network)
print(assort)

## ----eval=FALSE----------------------------------------------------------
#  data("biogrid")
#  data("GO.human")

## ----eval=FALSE----------------------------------------------------------
#  genelist <- make_genelist(biogrid)
#  gene_network <- make_gene_network(biogrid,genelist)

## ----eval=FALSE----------------------------------------------------------
#  annotations <- make_annotations(GO.human[,c(2,3)],genelist,goterms)

## ----eval=FALSE----------------------------------------------------------
#  GO_groups_voted <- run_GBA(gene_network, annotations)

## ----eval=FALSE----------------------------------------------------------
#  GO_multifunc_assessment <- calculate_multifunc(annotations)

## ----eval=FALSE----------------------------------------------------------
#  # For genes
#  ord <- order(as.numeric(GO_multifunc_assessment[,2]),decreasing=TRUE)
#  GO_multifunc_assessment_s <- GO_multifunc_assessment[ord,c(1,2)]
#  optimallist_genes <- as.vector(unlist(GO_multifunc_assessment_s[,1]))
#  
#  # and for GO groups
#  ord <- order(as.numeric(GO_groups_voted[[1]][,2]),decreasing=TRUE)
#  GO_groups_voted_s <- GO_groups_voted[[1]][ord,]
#  optimallist_GO <- cbind(GO.term=rownames(GO_groups_voted_s),
#                      node.degree=GO_groups_voted_s[,2], roc=GO_groups_voted_s[,1])

## ----eval=FALSE----------------------------------------------------------
#  auc_GO_mf <- auc_multifunc(annotations, optimallist_genes)

## ----eval=FALSE----------------------------------------------------------
#  auc_GO_nv <- GO_groups_voted[,1]
#  plot_distribution(auc_GO_nv, xlab="Neighbor voting AUROC ",
#                    ylab="Number of functional terms",
#                    b=30, xlim=c(0.4,1), ylim=c(0, 440), col="gray64",
#                    density=FALSE, avg=FALSE, bars=TRUE)

## ----eval=FALSE----------------------------------------------------------
#  plot_distribution(auc_GO_mf, xlab="Optimal GO Ranking AUROC",
#                    ylab="Number of functional terms",
#                    b=20, xlim=c(0.2,1), ylim=c(0,4400), col="gray64",
#                    density=FALSE, avg=FALSE, bars=TRUE)

## ----eval=FALSE----------------------------------------------------------
#  data("attr.human")
#  genelist <- unique(attr.human$entrezID[attr.human$type == "protein_coding"
#                  & !is.na(attr.human$entrezID)] )
#  exprs <- get_expression_matrix_from_GEO("GSE34308")

## ----eval=FALSE----------------------------------------------------------
#  network <- build_coexp_network(exprs, gene.list)
#  network <- build_coexp_expressionSet(exprsSet, genelist)

## ----eval=FALSE----------------------------------------------------------
#  assort <- assortativity(network)
#  nd <- node_degrees(network)
#  plot_density(nd)

## ----eval=FALSE----------------------------------------------------------
#  data("GO.human")
#  annotations <- make_annotations(GO.human[,c(2,3)],genelist,goterms)
#  annotations_sub <- filter_network_cols(annotations, min=20, max=300)
#  GO_groups_voted <- run_GBA(network, annotations_sub)

## ----eval=FALSE----------------------------------------------------------
#  data("biogrid")
#  data("GO.human")
#  data("ortho")
#  
#  # Prepare gene network and gene labels/annotations
#  genelist <- make_genelist(BIOGRID )
#  gene_network <- make_gene_network(BIOGRID, genelist)
#  annotations <- make_annotations(GO.human[,c(2,3)],genelist,goterms)
#  # neighbor voting
#  nv_results <- run_GBA(gene_network, annotations)
#  mf_optimal <- calculate_multifunc(annotations)
#  optimal_list<- as.numeric(mf_optimal[,4])
#  mf_results <- auc_multifunc(annotations, optimal_list)
#  # neighbor voting, ortholog constraints (e.g. yeast)
#  annotations_yeast <- filter_orthologs(annotations, genelist, ortho_yeast)
#  
#  nv_results_yeast <- run_GBA(gene_network, annotations_yeast)
#  mf_optimal_yeast <- calculate_multifunc(annotations_yeast)
#  optimal_list_yeast<- as.numeric(mf_optimal_yeast[,4])
#  mf_results_yeast <- auc_multifunc(annotations_yeast, optimal_list_yeast)

## ----eval=FALSE----------------------------------------------------------
#  # Load data
#  load("net.RData")
#  load("ortho.RData")
#  
#  # Extend network
#  ext_gene_network <- extend_network(gene_network, max=6)
#  # neighbor voting
#  nv_results <- run_GBA(ext_gene_network, annotations)
#  # neighbor voting, ortholog constraints (e.g. yeast)
#  annotations_yeast <- filter_orthologs(annotations, genelist, ortho_yeast)
#  nv_results_yeast <- run_GBA(ext_gene_network, annotations_yeast)

