## ----style, echo = FALSE, results = 'asis'-------------------------------
BiocStyle::markdown()

## ----global_options, include=FALSE--------------------------------------------------------------------------
knitr::opts_chunk$set(fig.width=10, fig.height=7, warning=FALSE, message=FALSE)
options(width=110)
set.seed(123)

## -----------------------------------------------------------------------------------------------------------
## load ABAEnrichment package
require(ABAEnrichment)
## create input vector with candidate genes 
genes = rep(1,13)
names(genes) = c('NCAPG', 'APOL4', 'NGFR', 'NXPH4', 'C21orf59', 'CACNG2', 'AGTR1', 'ANO1', 'BTBD3', 'MTUS1', 'CALB1', 'GYG1', 'PAX2')
genes

## ----results='hide'-----------------------------------------------------------------------------------------
## run enrichment analysis
res = aba_enrich(genes,dataset='5_stages', cutoff_quantiles=c(0.5,0.7,0.9), n_randsets=100)

## -----------------------------------------------------------------------------------------------------------
## extract first element from the output list, which contains the statistics
fwers = res[[1]]
## see results for the brain regions with highest enrichment for children (3-11 yrs, age_category 3)
head(fwers[fwers[,1]==3,])

## -----------------------------------------------------------------------------------------------------------
res[2:3]

## ----eval=FALSE---------------------------------------------------------------------------------------------
#  ## run enrichment analysis, with randomsets dependent on gene length
#  aba_enrich(genes, gene_len = TRUE)

## -----------------------------------------------------------------------------------------------------------
## create input vector with a candidate region on chromosome 8
## and background regions on chromosome 7, 8 and 9
genes = c(1, rep(0,6))
names(genes) = c('8:82000000-83000000', '7:1300000-56800000', '7:74900000-148700000','8:7400000-44300000', '8:47600000-146300000', '9:0-39200000', '9:69700000-140200000')
genes

## ----results='hide'-----------------------------------------------------------------------------------------
## run enrichment analysis for the adult human brain
res = aba_enrich(genes, dataset='adult', cutoff_quantiles=c(0.5,0.7,0.9), n_randsets=100)

## -----------------------------------------------------------------------------------------------------------
## look at the results from the enrichment analysis
fwers = res[[1]]
head(fwers)
## see which genes are located in the candidate regions
input_genes = res[[2]]
candidate_genes = input_genes[input_genes==1]
candidate_genes

## -----------------------------------------------------------------------------------------------------------
## create input vector with a candidate region on chromosome 8
## and background regions on chromosome 8
genes = c(1,0,0)
names(genes) = c('8:82000000-82700000', '8:7400000-44300000', '8:47600000-146300000')
genes

## ----results='hide'-----------------------------------------------------------------------------------------
## run enrichment analysis for the 5 developmental stages using the "circ_chrom" option
res = aba_enrich(genes, dataset='5_stages', cutoff_quantiles=c(0.5,0.7,0.9), n_randsets=100, circ_chrom=TRUE)

## -----------------------------------------------------------------------------------------------------------
## look at the results from the enrichment analysis for children (age_category 3)
fwers = res[[1]]
head(fwers[fwers$age_category==3,])

## -----------------------------------------------------------------------------------------------------------
## get expression data (list) for the first 5 brain structures of the enrichment analysis output
expr = get_expression(fwers[1:5,"structure_id"], background=FALSE)
## look at the structure of the expression data
str(expr)
## see expression for children (age_category 3 -> 3rd element of expression list)
head(expr[[3]])

## -----------------------------------------------------------------------------------------------------------
## plot expression in 5 structures with dendrogram for the prenatal stage (age_category 1)
plot_expression(fwers[1:5,"structure_id"], age_category=1)

## -----------------------------------------------------------------------------------------------------------
## plot expression for the 10 highest scoring brain structures in age category 3 (children, 3-11 yrs) without dendrogram
plot_expression(fwers[fwers$age_category==3, 'structure_id'][1:10], dendro=FALSE, age_category=3)

## -----------------------------------------------------------------------------------------------------------
## get expression data for six genes in the brain structure 'Allen:4010' 
get_expression(structure_ids=c('Allen:4010'), gene_ids=c(324,8312,673,1029,64764,1499), dataset='adult')
## plot this expression 
plot_expression(structure_ids=c('Allen:4010'), gene_ids=c(324,8312,673,1029,64764,1499), dataset='adult')

## -----------------------------------------------------------------------------------------------------------
## get ids of the substructres of 'Allen:4010' that have expression data annotated
subs = get_sampled_substructures('Allen:4010')
## get the full name of those substructures
get_name(subs)

## -----------------------------------------------------------------------------------------------------------
## get ids of the superstructures of 'Allen:4010'
supers = get_superstructures('Allen:4010')
## get the full name of those superstructures
get_name(supers)

## -----------------------------------------------------------------------------------------------------------
## create input vector with random scores associated with the candidate genes
genes = sample(1:50, 12)
names(genes) = c('ENSG00000168036', 'ENSG00000157764', 'ENSG00000163041', 'ENSG00000182158', 'ENSG00000147889', 'ENSG00000103126', 'ENSG00000182054', 'ENSG00000184634', 'ENSG00000134982', 'ENSG00000138413', 'ENSG00000133703', 'ENSG00000132475')
genes

## ----results='hide'-----------------------------------------------------------------------------------------
## run enrichment analysis
res = aba_enrich(genes, dataset='adult', test='wilcoxon', cutoff_quantiles=c(0.2,0.5,0.8), n_randsets=100)

## -----------------------------------------------------------------------------------------------------------
## see the 5 brain regions with the lowest FWERs
res[[1]][1:5,]

## -----------------------------------------------------------------------------------------------------------
## plot expression of the 5 brain strcutures with the lowest FWERs, with genes ordered by user defined score from Wilcoxon rank test
plot_expression(res[[1]][1:5,"structure_id"], dendro=FALSE)

## ----results='hide'-----------------------------------------------------------------------------------------
## run enrichment analysis with developmental effect score
res = aba_enrich(genes, dataset='dev_effect', test='wilcoxon', cutoff_quantiles=c(0.2,0.5,0.8), n_randsets=100)

## -----------------------------------------------------------------------------------------------------------
## see the 5 brain regions with the lowest FWERs
res[[1]][1:5,]

## -----------------------------------------------------------------------------------------------------------
## plot developmental effect score of the 5 brain strcutures with the lowest FWERs
plot_expression(res[[1]][1:5,"structure_id"])

## -----------------------------------------------------------------------------------------------------------
sessionInfo()

