## ----setup, include = FALSE-------------------------------------------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL
    ## Related to
    ## https://stat.ethz.ch/pipermail/bioc-devel/2020-April/016656.html
)

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE--------------------------------------------------------
## Track time spent on making the vignette
startTime <- Sys.time()

## Bib setup
library("knitcitations")

## Load knitcitations with a clean bibliography
cleanbib()
cite_options(hyperlink = "to.doc", citation_format = "text", style = "html")

## Write bibliography information
bib <- c(
    R = citation(),
    BiocStyle = citation("BiocStyle")[1],
    knitcitations = citation("knitcitations")[1],
    knitr = citation("knitr")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    ISAnalytics = citation("ISAnalytics")[1]
)

write.bibtex(bib, file = "collision_removal.bib")

## ----installBioc, eval=FALSE------------------------------------------------------------------------------------------
#  ## For release version
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  BiocManager::install("ISAnalytics")
#  
#  ## For devel version
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  # The following initializes usage of Bioc devel
#  BiocManager::install(version = "devel")
#  BiocManager::install("ISAnalytics")

## ----installGitHub, eval=FALSE----------------------------------------------------------------------------------------
#  # For release version
#  if (!require(devtools)) {
#      install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "RELEASE_3_12",
#      dependencies = TRUE,
#      build_vignettes = TRUE
#  )
#  
#  ## Safer option for vignette building issue
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "RELEASE_3_12"
#  )
#  
#  # For devel version
#  if (!require(devtools)) {
#      install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "master",
#      dependencies = TRUE,
#      build_vignettes = TRUE
#  )
#  
#  ## Safer option for vignette building issue
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "master"
#  )

## ---------------------------------------------------------------------------------------------------------------------
library(ISAnalytics)

## ----OptVerbose, eval=FALSE-------------------------------------------------------------------------------------------
#  # DISABLE
#  options("ISAnalytics.verbose" = FALSE)
#  
#  # ENABLE
#  options("ISAnalytics.verbose" = TRUE)
#  

## ----OptWidg, eval=FALSE----------------------------------------------------------------------------------------------
#  # DISABLE HTML REPORTS
#  options("ISAnalytics.widgets" = FALSE)
#  
#  # ENABLE HTML REPORTS
#  options("ISAnalytics.widgets" = TRUE)

## ----import_af--------------------------------------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    path_AF <- system.file("extdata", "ex_association_file.tsv",
        package = "ISAnalytics"
    )

    root_correct <- system.file("extdata", "fs.zip",
        package = "ISAnalytics"
    )
    root_correct <- unzip_file_system(root_correct, "fs")
    association_file <- import_association_file(path_AF, root_correct, dates_format = "dmy")
})

## ----importMatr-------------------------------------------------------------------------------------------------------
# This imports both sequence count and fragment estimate matrices
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    matrices <- import_parallel_Vispa2Matrices_auto(
        association_file = association_file, root = NULL,
        quantification_type = c("fragmentEstimate", "seqCount"),
        matrix_type = "annotated", workers = 2, patterns = NULL,
        matching_opt = "ANY", multi_quant_matrix = FALSE
    )
})

## ----removecoll1------------------------------------------------------------------------------------------------------
# Pass the whole named list
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    matrices_processed <- remove_collisions(
        x = matrices,
        association_file = association_file,
        date_col = "SequencingDate",
        reads_ratio = 10
    )
})

## ----removecoll2------------------------------------------------------------------------------------------------------
# Pass the sequence count matrix only
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    matrices_processed_single <- remove_collisions(
        x = matrices$seqCount,
        association_file =
            association_file,
        date_col = "SequencingDate",
        reads_ratio = 10
    )
})

## ----removecoll3------------------------------------------------------------------------------------------------------
# Obtain multi-quantification matrix
multi <- comparison_matrix(matrices)
multi
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    matrices_processed_multi <- remove_collisions(
        x = multi,
        association_file =
            association_file,
        date_col = "SequencingDate",
        reads_ratio = 10,
        seq_count_col = "seqCount"
    )
})

## ----realign, R.options=options(ISAnalytics.widgets = FALSE)----------------------------------------------------------
seq_count_proc <- matrices_processed_single
other_matrices <- matrices[!names(matrices) %in% "seqCount"]
# Select only matrices that are not relative to sequence count
other_realigned <- realign_after_collisions(seq_count_proc, other_matrices)

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

## ----results = "asis", echo = FALSE, warning = FALSE, message = FALSE-------------------------------------------------
## Print bibliography
bibliography()

