## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL
    ## Related to
    ## https://stat.ethz.ch/pipermail/bioc-devel/2020-April/016656.html
)

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE----------------
## Track time spent on making the vignette
startTime <- Sys.time()

## Bib setup
library("knitcitations")

## Load knitcitations with a clean bibliography
cleanbib()
cite_options(hyperlink = "to.doc", citation_format = "text", style = "html")

## Write bibliography information
bib <- c(
    R = citation(),
    BiocStyle = citation("BiocStyle")[1],
    knitcitations = citation("knitcitations")[1],
    knitr = citation("knitr")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    ISAnalytics = citation("ISAnalytics")[1]
)

write.bibtex(bib, file = "aggregate_function_usage.bib")

## ----installBioc, eval=FALSE--------------------------------------------------
#  ## For release version
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  BiocManager::install("ISAnalytics")
#  
#  ## For devel version
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  # The following initializes usage of Bioc devel
#  BiocManager::install(version = "devel")
#  BiocManager::install("ISAnalytics")

## ----installGitHub, eval=FALSE------------------------------------------------
#  # For release version
#  if (!require(devtools)) {
#      install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "RELEASE_3_12",
#      dependencies = TRUE,
#      build_vignettes = TRUE
#  )
#  
#  ## Safer option for vignette building issue
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "RELEASE_3_12"
#  )
#  
#  # For devel version
#  if (!require(devtools)) {
#      install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "master",
#      dependencies = TRUE,
#      build_vignettes = TRUE
#  )
#  
#  ## Safer option for vignette building issue
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "master"
#  )

## -----------------------------------------------------------------------------
library(ISAnalytics)

## ----OptVerbose, eval=FALSE---------------------------------------------------
#  # DISABLE
#  options("ISAnalytics.verbose" = FALSE)
#  
#  # ENABLE
#  options("ISAnalytics.verbose" = TRUE)
#  

## ----OptWidg, eval=FALSE------------------------------------------------------
#  # DISABLE HTML REPORTS
#  options("ISAnalytics.widgets" = FALSE)
#  
#  # ENABLE HTML REPORTS
#  options("ISAnalytics.widgets" = TRUE)

## -----------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    path_AF <- system.file("extdata", "ex_association_file.tsv",
        package = "ISAnalytics"
    )

    root_correct <- system.file("extdata", "fs.zip",
        package = "ISAnalytics"
    )
    root_correct <- unzip_file_system(root_correct, "fs")
    association_file <- import_association_file(path_AF, root_correct)
})

## -----------------------------------------------------------------------------
aggregated_meta <- aggregate_metadata(association_file,
    grouping_keys = c(
        "SubjectID",
        "CellMarker",
        "Tissue", "TimePoint"
    ),
    import_stats = FALSE
)

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(aggregated_meta)

## -----------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    aggregated_meta <- aggregate_metadata(association_file,
        grouping_keys = c(
            "SubjectID", "CellMarker",
            "Tissue", "TimePoint"
        ),
        import_stats = TRUE
    )
})

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(aggregated_meta)

## -----------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    matrices <- import_parallel_Vispa2Matrices_auto(
        association_file = association_file, root = NULL,
        quantification_type = c("fragmentEstimate", "seqCount"),
        matrix_type = "annotated", workers = 2, patterns = NULL,
        matching_opt = "ANY", multi_quant_matrix = FALSE
    )
})

## -----------------------------------------------------------------------------
# Takes the whole list and produces a list in output
aggregated_matrices <- aggregate_values_by_key(matrices, association_file)

# Takes a single matrix and produces a single matrix as output
aggregated_matrices_single <- aggregate_values_by_key(
    matrices$seqCount,
    association_file
)

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(head(aggregated_matrices_single))

## -----------------------------------------------------------------------------
agg1 <- aggregate_values_by_key(
    x = matrices$seqCount,
    association_file = association_file,
    key = c("SubjectID", "ProjectID")
)

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(head(agg1))

## -----------------------------------------------------------------------------
agg2 <- aggregate_values_by_key(
    x = matrices$seqCount,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(mean = ~ mean(.x, na.rm = TRUE))
)

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(head(agg2))

## -----------------------------------------------------------------------------
agg3 <- aggregate_values_by_key(
    x = matrices$seqCount,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(describe = psych::describe)
)

agg3

## -----------------------------------------------------------------------------
## Obtaining multi-quantification matrix
comp <- comparison_matrix(matrices)

agg4 <- aggregate_values_by_key(
    x = comp,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(sum = sum, mean = mean),
    value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(head(agg4))

## -----------------------------------------------------------------------------
agg5 <- aggregate_values_by_key(
    x = matrices$seqCount,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(sum = sum, mean = mean),
    group = c(mandatory_IS_vars())
)

## ----echo=FALSE---------------------------------------------------------------
knitr::kable(head(agg5))

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

## ----results = "asis", echo = FALSE, warning = FALSE, message = FALSE-------------------------------------------------
## Print bibliography
bibliography()

