## ---- echo=FALSE, results="hide"-------------------------------------------
knitr::opts_chunk$set(error=FALSE, warning=FALSE, message=FALSE)
library(BiocStyle)

## ----quickstart-load-data, message=FALSE, warning=FALSE--------------------
library(scater)
example_sce <- mockSCE()
example_sce

## ----quickstart-add-exprs, results='hide'----------------------------------
str(counts(example_sce))

## --------------------------------------------------------------------------
example_sce$whee <- sample(LETTERS, ncol(example_sce), replace=TRUE)
colData(example_sce)
rowData(example_sce)$stuff <- runif(nrow(example_sce))
rowData(example_sce)

## --------------------------------------------------------------------------
per.cell <- perCellQCMetrics(example_sce, subsets=list(Mito=1:10))
summary(per.cell$sum)
summary(per.cell$detected)
summary(per.cell$subsets_Mito_percent)

## --------------------------------------------------------------------------
colData(example_sce) <- cbind(colData(example_sce), per.cell)

## ----plot-pdata-pct-exprs-controls-----------------------------------------
plotColData(example_sce, x = "sum", y="subsets_Mito_percent",
    colour_by = "Mutation_Status") + theme(legend.position = "top") +
    stat_smooth(method = "lm", se = FALSE, size = 2, fullrange = TRUE)

## ----plot-sceset-blocking--------------------------------------------------
plotScater(example_sce, block1 = "Mutation_Status", block2 = "Treatment",
     colour_by = "Cell_Cycle", nfeatures = 300, exprs_values = "counts")

## --------------------------------------------------------------------------
example_sce2 <- example_sce
example_sce2$plate_position <- paste0(
     rep(LETTERS[1:5], each = 8), 
     rep(formatC(1:8, width = 2, flag = "0"), 5)
)
plotPlatePosition(example_sce2, colour_by = "Gene_0001",
    by_exprs_values = "counts") 

## --------------------------------------------------------------------------
keep.total <- example_sce$sum > 1e5
keep.n <- example_sce$detected > 500
filtered <- example_sce[,keep.total & keep.n]
dim(filtered)

## --------------------------------------------------------------------------
keep.total <- isOutlier(per.cell$sum, type="lower", log=TRUE)
filtered <- example_sce[,keep.total]

## --------------------------------------------------------------------------
qc.stats <- quickPerCellQC(per.cell, percent_subsets="subsets_Mito_percent")
colSums(as.matrix(qc.stats))

## --------------------------------------------------------------------------
per.feat <- perFeatureQCMetrics(example_sce, subsets=list(Empty=1:10))
summary(per.feat$mean)
summary(per.feat$detected)
summary(per.feat$subsets_Empty_ratio)

## --------------------------------------------------------------------------
ave <- calculateAverage(example_sce)
summary(ave)

## --------------------------------------------------------------------------
summary(nexprs(example_sce, byrow=TRUE))

## ----plot-highest, fig.asp=1, fig.wide=TRUE--------------------------------
plotHighestExprs(example_sce, exprs_values = "counts")

## ----filter-no-exprs-------------------------------------------------------
keep_feature <- rowSums(counts(example_sce) > 0) > 0
example_sce <- example_sce[keep_feature,]
dim(example_sce)

## --------------------------------------------------------------------------
example_sce <- logNormCounts(example_sce)
vars <- getVarianceExplained(example_sce)
head(vars)

## --------------------------------------------------------------------------
plotExplanatoryVariables(vars)

## --------------------------------------------------------------------------
cpm(example_sce) <- calculateCPM(example_sce)

## --------------------------------------------------------------------------
example_sce <- logNormCounts(example_sce)
assayNames(example_sce)

## --------------------------------------------------------------------------
summary(librarySizeFactors(example_sce))

## --------------------------------------------------------------------------
assay(example_sce, "is_expr") <- counts(example_sce)>0

## ----plot-expression-------------------------------------------------------
plotExpression(example_sce, rownames(example_sce)[1:6],
    x = "Mutation_Status", exprs_values = "logcounts")

## ----plot-expression-scatter-----------------------------------------------
plotExpression(example_sce, rownames(example_sce)[1:6],
    x = "Gene_0001")

## ----plot-expression-col---------------------------------------------------
plotExpression(example_sce, rownames(example_sce)[1:6],
    colour_by = "Cell_Cycle", shape_by = "Mutation_Status",
    size_by = "Gene_0002")

## ----plot-expression-theme-bw----------------------------------------------
plotExpression(example_sce, rownames(example_sce)[7:12],
    x = "Mutation_Status", exprs_values = "counts",
    colour = "Cell_Cycle", show_median = TRUE,
    xlab = "Mutation Status", log = TRUE)

## ----plot-expression-many--------------------------------------------------
plotExpression(example_sce, rownames(example_sce)[1:6])

## --------------------------------------------------------------------------
example_sce <- runPCA(example_sce)
reducedDimNames(example_sce)

## ----plot-reduceddim-4comp-colby-shapeby-----------------------------------
plotReducedDim(example_sce, dimred = "PCA",
    colour_by = "Treatment", shape_by = "Mutation_Status")

## ----plot-reduceddim-4comp-colby-sizeby-exprs------------------------------
plotReducedDim(example_sce, dimred = "PCA",
    colour_by = "Gene_1000", size_by = "Gene_0500")

## ----plot-pca-default------------------------------------------------------
plotPCA(example_sce)

## ----plot-pca-feature-controls---------------------------------------------
chosen.genes <- 1:100
example_sce2 <- runPCA(example_sce, subset_row=chosen.genes)
plotPCA(example_sce2)

## ----plot-pca-4comp-colby-shapeby------------------------------------------
example_sce <- runPCA(example_sce, ncomponents=20)
plotPCA(example_sce, ncomponents = 4, colour_by = "Treatment",
        shape_by = "Mutation_Status")

## ----plot-pca-4comp-colby-sizeby-exprs-------------------------------------
plotPCA(example_sce, colour_by = "Gene_0001", size_by = "Gene_1000")

## ----plot-tsne-1comp-colby-sizeby-exprs------------------------------------
# Perplexity of 10 just chosen here arbitrarily.
set.seed(1000)
example_sce <- runTSNE(example_sce, perplexity=10)
plotTSNE(example_sce, colour_by = "Gene_0001", size_by = "Gene_1000")

## ----plot-tsne-from-pca----------------------------------------------------
set.seed(1000)
example_sce <- runTSNE(example_sce, perplexity=10, use.dimred="PCA", n_dimred=10)
plotTSNE(example_sce, colour_by="Treatment")

## ----plot-difmap-1comp-colby-sizeby-exprs----------------------------------
example_sce <- runDiffusionMap(example_sce)
plotDiffusionMap(example_sce, colour_by = "Gene_0001", size_by = "Gene_1000")

## --------------------------------------------------------------------------
example_sce <- runUMAP(example_sce)
plotUMAP(example_sce, colour_by = "Gene_0001", size_by = "Gene_1000")

## --------------------------------------------------------------------------
sessionInfo()

