## ----style, echo=FALSE, results='hide', message=FALSE-------------------------
library(BiocStyle)
library(knitr)
opts_chunk$set(error=FALSE, message=FALSE, warning=FALSE)
knitr::opts_chunk$set(echo = TRUE)

## ---- eval = FALSE------------------------------------------------------------
#  if(!requireNamespace("BiocManager", quietly = TRUE))
#      install.packages("BiocManager")
#  BiocManager::install("DepecheR")
#  
#  # Example data description
#  
#  The data used in this example is a semi-synthedic dataset, consisting of 1000
#  cytotoxic lymphocytes from each of 20 individuals. These have been categorized
#  into two groups, and after this, alterations have been added to the sizes of
#  some cell populations in both groups. This means that the groups can be
#  separated based onthe sizes of certain cell types in the data. And this
#  excersize will show how to identify these, and tell us what markers that define
#  the separating cell types in question.
#  
#  Importantly, DepecheR does not provide any pre-processing tools, such as for
#  compensation/spectral unmixing of flow cytometry files. The clustering function
#  does have an internal algorithm to detect data with extreme tails, but this does
#  not circumvent the need to transform flow- or mass cytometry data. This can be
#  done using either commercially available software or with R packages, such as
#  Biocpkg("flowCore") or Biocpkg("flowVS").
#  

## -----------------------------------------------------------------------------
library(DepecheR)
data('testData')
str(testData)

## ----eval=FALSE---------------------------------------------------------------
#  testDataDepeche <- depeche(testData[, 2:15])

## -----------------------------------------------------------------------------
## [1] "Files will be saved to ~/Desktop"
## [1] "As the dataset has less than 100 columns, peak centering is applied."
## [1] "Set 1 with 7 iterations completed in 14 seconds."
## [1] "Set 2 with 7 iterations completed in 6 seconds."
## [1] "Set 3 with 7 iterations completed in 6 seconds."
## [1] "The optimization was iterated 21 times."

## ----echo=FALSE---------------------------------------------------------------
data("testDataDepeche")

## -----------------------------------------------------------------------------
str(testDataDepeche)

## ----eval=FALSE---------------------------------------------------------------
#  library(Rtsne)
#  testDataSNE <- Rtsne(testData[,2:15], pca=FALSE)

## ----echo=FALSE---------------------------------------------------------------
    data("testDataSNE")

## -----------------------------------------------------------------------------
dColorPlot(colorData = testDataDepeche$clusterVector, xYData = testDataSNE$Y, 
           colorScale = "dark_rainbow", plotName = "Cluster")

## -----------------------------------------------------------------------------
dColorPlot(colorData = testData[2], xYData = testDataSNE$Y)


## -----------------------------------------------------------------------------
densContour <- dContours(testDataSNE$Y)

dDensityPlot(xYData = testDataSNE$Y, plotName = 'All_events', 
             colorScale="purple3", densContour = densContour)

#Here the data for the first group is plotted
dDensityPlot(xYData = testDataSNE$Y[testData$label==0,], plotName = 'Group_0', 
             colorScale="blue", densContour = densContour)

#And here comes the second group
dDensityPlot(xYData = testDataSNE$Y[testData$label==1,], plotName = 'Group_1', 
             colorScale="red", densContour = densContour)


## -----------------------------------------------------------------------------
dResidualPlot(
    xYData = testDataSNE$Y, groupVector = testData$label,
    clusterVector = testDataDepeche$clusterVector)

## -----------------------------------------------------------------------------
dWilcoxResult <- dWilcox(
    xYData = testDataSNE$Y, idsVector = testData$ids,
    groupVector = testData$label, clusterVector = testDataDepeche$clusterVector)

## ----eval=FALSE---------------------------------------------------------------
#  sPLSDAObject <- dSplsda(xYData = testDataSNE$Y, idsVector = testData$ids,
#                          groupVector = testData$label,
#                          clusterVector = testDataDepeche$clusterVector)
#  ## Saving 3 x 3 in image
#  
#  ## [1] "The separation of the datasets was perfect, with no overlap between the groups"
#  
#  ## [1] "Files were saved at /Users/jakthe/Labbet/GitHub/DepecheR/vignettes"
#  

## ----eval---------------------------------------------------------------------
dViolins(testDataDepeche$clusterVector, inDataFrame = testData, 
         plotClusters = 3, plotElements = testDataDepeche$essenceElementList)

## -----------------------------------------------------------------------------
sessionInfo()

