/*
 * main.cpp
 *
 *  Created on: 14.04.2013
 *      Author: mu
 */

#include <iostream>
#include <iomanip>
#include <sstream>
#include <cstring>
#include <cstdlib>
#include <string>
#include "stdint.h"

/* --Utilities. */
#include <sys/time.h>
#include "popt.h"

/* --Project includes. */
#include "canbus.h"
#include "canframe.h"
#include "obd2.h"


/* --OBD2-Datenbasis (Ausschnitt). */

const ParameterInfo Parameter[]={
		{0x05,ParameterInfo::CELSIUS,"Kühlmitteltemperatur",-40.,215.,1},
		{0x0c,ParameterInfo::RPM,"Motordrehzahl",0,16383.75,2},
		{0x0d,ParameterInfo::KM_PER_HOUR,"Fahrzeuggeschwindigkeit",0.,255.,1},
		{0x11,ParameterInfo::PERCENTAGE,"Gaspedalstellung",0.,100.,1},
		{0x1f,ParameterInfo::SECONDS,"Zeit seit Motorstart",0.,65535.,2},
		{0}
};

const char* TextualUnit[]={0,"","%","°C","km/h","rpm","sec"};

/* --Unterschiedliche Modi stehen zur Verfügung. */

/* --Monitor-Modus. */
void BusMonitor(const char* cdevice) {
	/* --CAN-Objekt anlegen und initialisieren. */
	ControllerAreaNetwork can;

	if (can.configure(cdevice)==false) {
		std::cerr << "CAN konnte nicht konfiguriert werden." << std::endl;
		return;
	}

	if (can.initialize()==false) {
		std::cerr << "CAN konnte nicht gestartet werden." << std::endl;
		return;
	}
	std::cout << "Monitor ist initialisiert..." << std::endl;

	/* --CAN Datentelegramm lesen und ausgeben. */
	can_frame frame;
	timeval previous={0},current,diff;
	while (can.receive(frame,current)) {
		/* --Zeitberechnungen. */
		if (previous.tv_sec==0)
			previous=current;
		timersub(&current,&previous,&diff);
		previous=current;
		/* --CAN Datentelegramm auf Konsole ausgeben. */
		printCANFrame(frame,diff,std::cout);
	}
}

/* --Sender-Modus. */
void BusSender(const char* cdevice) {

	/* --CAN-Objekt anlegen und initialisieren. */
	ControllerAreaNetwork can;

	if (can.configure(cdevice)==false) {
		std::cerr << "CAN konnte nicht konfiguriert werden." << std::endl;
		return;
	}

	if (can.initialize()==false) {
		std::cerr << "CAN konnte nicht gestartet werden." << std::endl;
		return;
	}
	std::cout << "Sender ist initialisiert..." << std::endl;

	std::string msg;
	can_frame frame;
	uint32_t tmp;
	/* --Zeile von Konsole einlesen. */
	while (std::getline(std::cin,msg)) {
		std::istringstream istream(msg);
		/* --CAN Id extrahieren (Standard Format). */
		istream >> std::hex >> tmp;
		frame.can_id=tmp&0x7ff;
		/* --CAN Daten extrahieren. */
		frame.can_dlc=0;
		while (!istream.eof() && frame.can_dlc<8) {
			istream >> std::hex >> tmp;
			frame.data[frame.can_dlc++]=(uint8_t)tmp;
		}
		/* --Versende das CAN-Datentelegramm. */
		if (can.post(frame)==false)
			return;
	}
}

/* --Diagnose-Modus. */
void OBD2(const char* cdevice) {
	/* --OBD2-Objekt anlegen und initialisieren. */
	OnBoardDiagnostics tester;

	if (tester.configure(cdevice)==false) {
		std::cerr << "OBD konnte nicht konfiguriert werden." << std::endl;
		return;
	}

	if (tester.initialize()==false) {
		std::cerr << "OBD konnte nicht gestartet werden." << std::endl;
		return;
	}
	std::cout << "Diagnosetester ist initialisiert..." << std::endl;
	/* --Diagnostiziere verschiedene Parameter. */
	uint8_t response[5];
	uint32_t i=0;
	double value=0;
	while (Parameter[i].physical) {
		std::cout << "Überprüfe " << Parameter[i].textual << "... " << std::flush;
		/* --Daten abfragen. */
		if (tester.diagnose(Parameter[i].pid,Parameter[i].nbytes,response)) {
			/* --Rohwert in physikalischen Wert umwandeln. */
			value=Parameter[i].toPhysical(response);
			/* --Textuelle Bildschirmausgabe. */
			std::cout << std::setprecision(1) << std::fixed << value
					  << ' ' << TextualUnit[Parameter[i].physical]
			          << "." << std::endl;
		} else
			std::cout << "fehlgeschlagen." << std::endl;
		i++;
	}
}

/* --Hauptprogramm. */

int main(int argc, const char* argv[]) {

	/* --Willkommen. */
	std::cout << "*** CANDY -- SWEET CAN BUS TEST APP ***" << std::endl;

	char *mode=0;

	/* --Definition of command line options (popt). */
	poptOption optionsTable[] = {
			{ "mode", 'm', POPT_ARG_STRING, &mode, 0, "specifies the mode of intercation with can bus",NULL},
	        POPT_AUTOHELP
	        { NULL, 0, 0, NULL, 0 }
	};

	/* --Process the command line options. */
	poptContext optCon = poptGetContext(NULL, argc, argv, optionsTable, 0);
    poptSetOtherOptionHelp(optCon, "[OPTIONS]* <can port>");

    if (argc<2) {
    	poptPrintUsage(optCon, stderr, 0);
        exit(1);
    }

    int32_t c;
    while ((c=poptGetNextOpt(optCon))>=0) {
    	switch (c) {
    	case 'm' : std::cout << '>' << mode << std::endl; break;
           }
        }

    /* --HW-Gerät. */
    const char* cdevice=poptGetArg(optCon);
    std::cout << "Verwende CAN-Schnittstelle " << cdevice << "..." << std::endl;

  	/* --Select the mode. */
	if (!strcmp(mode,"monitor")) {
		/* --Monitor-Modus. */
		BusMonitor(cdevice);
	} else if (!strcmp(mode,"sender")) {
		/* --Sender-Modus. */
		BusSender(cdevice);
	} else if (!strcmp(mode,"obd")) {
		/* --OBD2-Modus. */
		OBD2(cdevice);
	} else {
		std::cerr << "Falscher Modus: {sender|monitor|obd}." << std::endl;
		exit(1);
	}

    poptFreeContext(optCon);

	return 0;
}


